// -----------------------------------------------------------------------
// ubidiagqr.c, MATLAB version 6+
//
// The calling syntax is:
//
// [Q, u, vu = lbidiagqr (gamma, delta, mu)
//
// This procedure computes the QR-decomposition of the 2n-by-n matrix:
// [ U    ]
// [ mu*I ]
//
// where U is an n-by-n upper bidiagonal matrix:
//
//      [ gamma(1)  delta(1)                  ]
//      [           ...       ...             ]
//  U = [                     ...             ]
//      [                          delta(n-1) ]
//      [                          gamma(n)   ]
//
// original code by Urs von Matt     October 26, 1994
// this     code by Oleg Grodzevich     July 05, 2004
// -----------------------------------------------------------------------

#include <math.h>
#include "mex.h"
#include "blas.c"

static void ubidiagQR (int n, double* gamma, double* delta, double mu,
                              double* Q, double* u, double* v)
{
    int i, ldQ ;
    double t, mutmp ;

    ldQ   = n*2 - 1  ;
    u[0]  = gamma[0] ;
    mutmp = mu ;

    rotg (&u[0], &mutmp, &Q[0], &Q[ldQ]) ;

    for (i = 1 ; i < n ; ++i)
    {
        v[i-1] = delta[i-1] ;
        t = 0.0 ;

        rot (&v[i-1], &t, Q[i*2-2], Q[i*2-2+ldQ]) ;
        mutmp = mu ;

        rotg (&mutmp, &t,     &Q[i*2-1], &Q[i*2-1+ldQ]) ; u[i] = gamma[i] ;
        rotg (&u[i],  &mutmp, &Q[i*2],   &Q[i*2  +ldQ]) ;
    }
}

// input arguments
#define gamma prhs[0]
#define delta prhs[1]
#define mu    prhs[2]

// output arguments
#define Q     plhs[0]
#define u     plhs[1]
#define v     plhs[2]

void mexFunction (int nlhs, mxArray* plhs[], int nrhs, const mxArray* prhs[])
{
    int n ;

    // check for proper number of arguments
    if (nrhs != 3) mexErrMsgTxt ("ubidiagQR requires three input arguments.")  ; else
    if (nlhs != 3) mexErrMsgTxt ("ubidiagQR requires three output arguments.") ;

    // check the dimensions of gamma
    n = max (mxGetM (gamma), mxGetN (gamma)) ;
    if (min (mxGetM (gamma), mxGetN (gamma)) != 1)
        mexErrMsgTxt ("gamma must be an n-by-1 or a 1-by-n matrix.") ;

    // check the dimensions of delta
    if (n > 1)
    {
        if ((min (mxGetM (delta), mxGetN (delta)) != 1) ||
            (max (mxGetM (delta), mxGetN (delta)) != n-1))
            mexErrMsgTxt ("delta must be an (n-1)-by-1 or a 1-by-(n-1) matrix.") ;
    }

    // check the dimensions of mu
    if ((mxGetM (mu) != 1) || (mxGetN (mu) != 1))
        mexErrMsgTxt ("mu must be a scalar.") ;

    // create matrices for the return arguments
    Q = mxCreateDoubleMatrix (n*2-1, 2, mxREAL) ;
    u = mxCreateDoubleMatrix (n,     1, mxREAL) ;
    v = mxCreateDoubleMatrix (n-1,   1, mxREAL) ;

    // do the actual computations in a subroutine
    ubidiagQR (n, mxGetPr (gamma), mxGetPr (delta), *mxGetPr (mu),
                  mxGetPr (Q), mxGetPr (u), mxGetPr (v)) ;
}
