// -----------------------------------------------------------------------
// lbidiagsolve.c, MATLAB version 6+
//
// The calling syntax is:
//
// x = lbidiagsolve (gamma, delta, b)
//
// This procedure solves the linear system L*x=b,
// where L is an n-by-n lower bidiagonal matrix:
//
//      [ gamma(1)                                 ]
//      [ delta(1)  ...                            ]
//  L = [           ...  ...                       ]
//      [                ...  ...                  ]
//      [                     delta(n-1)  gamma(n) ]
//
// original code by Urs von Matt     October 26, 1994
// this     code by Oleg Grodzevich     July 05, 2004
// -----------------------------------------------------------------------

#include <math.h>
#include "mex.h"

static void lbidiagSolve (int n, double* gamma, double* delta, double* b, double* x)
{
    int i ;
    x[0] = b[0] / gamma[0] ;
    for (i = 1 ; i < n ; ++i) x[i] = (b[i] - delta[i-1] * x[i-1]) / gamma[i] ;
}

// input arguments
#define gamma prhs[0]
#define delta prhs[1]
#define b     prhs[2]

// output arguments
#define x     plhs[0]

void mexFunction (int nlhs, mxArray* plhs[], int nrhs, const mxArray* prhs[])
{
    int n ;

    // check for proper number of arguments
    if (nrhs != 3) mexErrMsgTxt ("lbidiagSolve requires three input arguments.") ; else
    if (nlhs != 1) mexErrMsgTxt ("lbidiagSolve requires one output arguments.")  ;

    // check the dimensions of gamma
    n = max (mxGetM (gamma), mxGetN (gamma)) ;
    if (min (mxGetM (gamma), mxGetN (gamma)) != 1)
        mexErrMsgTxt ("gamma must be an n-by-1 or a 1-by-n matrix.") ;

    // check the dimensions of delta
    if (n > 1)
    {
        if ((min (mxGetM (delta), mxGetN (delta)) != 1) ||
            (max (mxGetM (delta), mxGetN (delta)) != n-1))
            mexErrMsgTxt ("delta must be an (n-1)-by-1 or a 1-by-(n-1) matrix.") ;
    }

    // check the dimensions of b
    if ((mxGetM (b) != n) || (mxGetN (b) != 1))
        mexErrMsgTxt ("b must be an n-by-1 matrix.") ;

    // create matrices for the return arguments
    x = mxCreateDoubleMatrix (n, 1, mxREAL) ;

    // do the actual computations in a subroutine
    lbidiagSolve (n, mxGetPr (gamma), mxGetPr (delta), mxGetPr (b), mxGetPr (x)) ;
}
