// -----------------------------------------------------------------------
// lbidiagqr.c, MATLAB version 6+
//
// The calling syntax is:
//
// [Q, u, v] = lbidiagqr (gamma, delta, mu)
//
// This procedure computes the QR-decomposition of the (2n+1)-by-n matrix:
// [ L    ]
// [ mu*I ]
//
// where L is an (n+1)-by-n lower bidiagonal matrix:
//
//      [ gamma(1)                     ]
//      [ delta(1)  ...                ]
//  L = [           ...  ...           ]
//      [                ...  gamma(n) ]
//      [                     delta(n) ]
//
// original code by Urs von Matt     October 26, 1994
// this     code by Oleg Grodzevich     July 05, 2004
// -----------------------------------------------------------------------

#include <math.h>
#include "mex.h"
#include "blas.c"

static void lbidiagQR (int n, double* gamma, double* delta, double mu,
                              double* Q, double* u, double* v)
{
    int i, ldQ ;
    double tmp ;

    ldQ  = n*2 ;
    u[0] = gamma[0] ;
    for (i = 0 ; i < n ; ++i)
    {
        tmp = mu ;
        rotg (&u[i], &tmp, &Q[i*2],   &Q[i*2  +ldQ]) ;

        tmp = delta[i] ;
        rotg (&u[i], &tmp, &Q[i*2+1], &Q[i*2+1+ldQ]) ;

        if (i < n-1)
        {
            v[i]   = 0.0 ;
            u[i+1] = gamma [i+1] ;

            rot (&v[i], &u[i+1], Q[i*2+1], Q[i*2+1+ldQ]) ;
        }
    }
}

// input arguments
#define gamma prhs[0]
#define delta prhs[1]
#define mu    prhs[2]

// output arguments
#define Q     plhs[0]
#define u     plhs[1]
#define v     plhs[2]

void mexFunction (int nlhs, mxArray* plhs[], int nrhs, const mxArray* prhs[])
{
    int n ;

    // check for proper number of arguments
    if (nrhs != 3) mexErrMsgTxt ("lbidiagQR requires three input arguments.")  ; else
    if (nlhs != 3) mexErrMsgTxt ("lbidiagQR requires three output arguments.") ;

    // check the dimensions of gamma
    n = max (mxGetM (gamma), mxGetN (gamma)) ;
    if (min (mxGetM (gamma), mxGetN (gamma)) != 1)
        mexErrMsgTxt ("gamma must be an n-by-1 or a 1-by-n matrix.") ;

    // check the dimensions of delta
    if ((min (mxGetM (delta), mxGetN (delta)) != 1) ||
        (max (mxGetM (delta), mxGetN (delta)) != n))
        mexErrMsgTxt ("delta must be an n-by-1 or a 1-by-n matrix.") ;

    // check the dimensions of mu
    if ((mxGetM (mu) != 1) || (mxGetN (mu) != 1))
        mexErrMsgTxt ("mu must be a scalar.") ;

    // create matrices for the return arguments
    Q = mxCreateDoubleMatrix (n*2, 2, mxREAL) ;
    u = mxCreateDoubleMatrix (n,   1, mxREAL) ;
    v = mxCreateDoubleMatrix (n-1, 1, mxREAL) ;

    // do the actual computations in a subroutine
    lbidiagQR (n, mxGetPr (gamma), mxGetPr (delta), *mxGetPr (mu),
                  mxGetPr (Q), mxGetPr (u), mxGetPr (v)) ;
}
