function [optval,L,P,iter]=primcomp(A,H,toler)
%  function call is:   [optval,L,P,iter]=primcomp(A,H,toler)
% 
% **Primal step** algorithm for the approximate matrix completion problem with
%  weight matrix H. A primal-dual interior-point method is used.
%  This program uses the primal step to evaluate the dual step.
%  It is more efficient if H has many elements which are
%   'sufficiently large'.  If H is very sparse,
%   then use the dual step algorithm.
%  We ASSUME DIAGONAL IS ***NOT*** FIXED!!!!!
%
% User INPUT:  A -  the symmetric matrix A 
%                 (the sparse matlab package is assumed)
%              H -  weight matrix
%                  (diag(H) not 0 is ensured!!)
%                 i.e. 0 corresponds to free elements in A;
%                  >= 100 corresponds to definitely fixed elements in A.
%                  The program fails if a definitely 
%                       fixed principal minor is not >0.
%          toler -  tolerance for duality gap 
%
%  OUTPUT:  optval - optimal value
%           L      - optimal value of dual variable Lambda 
%                      (and it equals the gradient of obj fn.)
%           P      - optimal primal value
%           iter   - iteration count
%
%%%%%%%%%%%%%%%%%%%%
% Initialization:
disp(['   ']);
disp([' ----------------------------- ']);
disp(['Start of new approximate completion problem using primal-first-step']);
n=size(A,1);
n2=n^2;
H=abs(full(H));
H=sparse(H);
if sum(diag(H) <=0 | diag(H)>=100) > 0,
    disp(['   ']);
    disp(['diagonal of H had 0 or large elements - WARNING']);
    disp(['USER - ensure that no diagonal elements are fixed']);
    disp(['solutions of linear systems may become ill-conditioned']);
    disp(['diagonal is perturbed to be nonzero']);
    H=H+diag( (100*eps)*rand(n,1) );
end
%Initialization:
epsilon=toler;  % duality gap stopping tolerance
dualtol=toler;  % dual feasibility tolerance
A=sparse(A);
H=sparse(H);
indzs=find(H==0);
indnzs=find(H);
nnzsH= nnz(H); % number of nonzeros
% The following is done only for testing purposes, i.e. so as
%  not to have a complete pos semidef A
A(indzs)=zeros(size(indzs));  % zero out free elements of A
A=sparse(A);
indinfty=find(H>=100);
indninfty=find(H<100);
[ity,jty]=find(H<100);  % row and column indices for unspecified els
disp(['The dimension n is: ',num2str(n)]);
disp(['The number of free elements is: ',num2str(length(indninfty))]);
HA=sparse(H.*A);
ha=sparse(HA(indninfty));
th2=2*sparse(H(indninfty).*H(indninfty));
th2a=sparse( th2.*A(indninfty)  );
% Additional information for suff. large els
dth2= sparse(diag(th2));  
%%%%% heuristic for starting point
[ icol, ad, irow, ar] = setsp( A);  % set up data for Lanczos call
ee=ones(n,1); % !!!Need better initial estimate for SMALLEST eig
[ae, xnew] = mineig1( A, icol, ad, irow, ar, ee); % Lanczos version
ae=-min(0,ae);
P=A+(ae+1)*diag(ee);    % check theory here for initial feas point?????
P=sparse(P);
L=zeros(n);
th2p=th2.*P(indninfty);
L(indninfty)=th2p-th2a;
L=sparse(L);
[R,lmin]=chol(L);   % check pos def - dual feas.
i=0;step=1.1;
while lmin > 0,
  i=i+1;
  th2pn=(step^i)*th2p;
  L(indninfty)=th2pn-th2a; 
  L=sparse(L);
  [R,lmin]=chol(L);   % check pos def - dual feas.
end
P=sparse((step^i)*P);
gap=trace (P*L);
mu=(1/(2*n))*gap;
disp(['  ']);
disp(['START of ALGORITHM: ']);
disp(['initial duality gap value:  ',num2str(full(gap))]);
optval=(norm( (H.*P)-HA,'fro' ))^2;
disp(['initial value of objective function is:  ',num2str(optval)]);
disp(['  ']);
iter=0;
LO=zeros(length(indninfty));
tqp=zeros(size(indninfty));
%
%%%%MAIN WHILE LOOP%%%%%%%%%%%%%
while min((gap/(optval+1)),optval) > epsilon,  % iterate while duality gap is large
  iter=iter+1;    %  iteration count
  disp(['   -  ']);
  disp(['iteration number:  ',num2str(iter)]);
%  solve for primal direction h
tLO=cputime;
etLO=clock;
  th2p=th2.*P(indninfty);
  errdual=sparse( (th2p-th2a)-L(indninfty)  );
   for s=1:length(indninfty),
  LO(s,1:s)=(1/mu)*(full(L(ity(s),jty(1:s))).*full(L(jty(s),ity(1:s))) );
  LO(1:s,s)=LO(s,1:s)';
     LO(s,s)=LO(s,s)+th2(s);
     tqp(s)= L(jty(s),:)*P*L(:,ity(s));
     tqp(s)=L(indninfty(s))-(1/mu)*tqp(s)-errdual(s);
   end
   LO=sparse(LO);
   tqp=sparse(tqp);
disp(['cpu  time for operator LO:  ',num2str(cputime-tLO)]);
disp([' elapsed time for operator LO:  ',num2str(etime(clock,etLO))]);
  dnorm=norm(full(errdual));
  disp(['dual feasibility before updates:  ',num2str(dnorm)]);
tsolve=cputime;
etsolve=clock;
  h=LO\tqp;
disp(['cpu time for system solve:  ',num2str(cputime-tsolve)]);
disp([' elapsed time for system solve:  ',num2str(etime(clock,etsolve))]);
  h2=zeros(n2,1);
  h2(indninfty)=h;
  h=reshape(h2,n,n);
  h=sparse(h);
  h=sparse( (h+h')/2 );   % ensure symmetry
  q=sparse ( -(1/mu)*L*(P+h)*L+L);  % step for dual L
%  safeguard positive definiteness - primal and dual
%  we use the Cholesky factor. to test pos. def.
    [R,lmind]=chol((L+q));   % check dual feasibility
    [R,lminp]=chol((P+h));   % check primal feasibility
    step=.5;
    i=0;
    qo=q;ho=h;
    while max(lmind,lminp) > 0,
        i=i+1;
        q=sparse( step^i*qo );
        h=sparse ( step^i*ho );
        [R,lminp]=chol(P+h);   % check primal feasibility
        [R,lmind]=chol(L+q);   % check dual feasibility
    end
    if i>0,   % Use the Newton step if possible
       q=sparse( (.97*step^i)*qo );
       h=sparse( (.97*step^i)*ho );
    else
     disp(['Using a Newton step length 1 ']);
       q=qo;
       h=ho;
    end
%%%%%%%%
 %  update mu and P and L
    P=sparse(P+h);
    L=sparse(L+q);
    gap=trace (P*L);
    mu=(1/(2*n))*gap;
    disp(['duality gap value after updates:  ',num2str(full(gap))]);
    fP=(H.*P)-HA;
    optval=(norm( fP,'fro' ))^2;
    disp(['value of objective function after updates: ',num2str(optval)]);
    disp(['  ']);
end   
%%%%%%%%%%%%%%%%
% end of main while  loop 
% duality gap and dual feas must hold
disp(['  ']);
disp(['optimal P and optimal dual L found']);
optval=(norm( (H.*(P-A)),'fro' ))^2;  
disp(['value of objective function is:  ',num2str(optval)]);
disp(['  ']);
