function [optval,L,P,iter]=complalg(A,H,toler)
%  function call is:   [optval,L,P,iter]=complalg(A,H,toler)
% Solves the approximate matrix completion problem with
%  weight matrix H. A primal-dual interior-point method is used.
%  The program decides on a primal or dual solve
%  step algorithm, depending on the sparsity type of H.
%
% User INPUT:  A -  the symmetric matrix A 
%                 (the sparse matlab package is assumed)
%              H -  weight matrix
%                  (diag(H) not 0 is ensured!!)
%                 i.e. 0 corresponds to free elements in A;
%                  >= 1000 corresponds to definitely fixed elements in A.
%                  The program fails if a definitely 
%                       fixed principal minor is not >0.
%          toler -  tolerance for duality gap 
%                        and dual feasibilty
%
%  OUTPUT:  optval - optimal value
%           L      - optimal value of dual variable Lambda 
%                      and it equals the gradient of obj fn.
%           P      - optimal primal value
%           iter   - iteration count
%
n=size(A,1);
H=abs(full(H));
H=sparse(H);
while sum(diag(H) <=0) > 0,
    disp(['   ']);
    disp(['diagonal of H had 0 elements - WARNING']);
    disp(['solutions of linear systems may become ill-conditioned']);
    disp(['diagonal is perturbed to be nonzero']);
    H=H+diag( (1000*eps)*rand(n,1) );
end
diagfix=find(diag(H) >=1000);
;   % indicator for diagonal elements fixed
if length(diagfix) > 0,
    diagfix=1;
    disp(['   ']);
    disp(['diagonal of H had infinity elements - WARNING']);
    disp(['corresponding diagonal elements of A are fixed']);
    disp(['problems could arise if no completion exists']);
    disp(['i.e. if the existing minors are not nonnegative']);
end
%Initialization:
epsilon=toler;  % duality gap stopping tolerance
dualtol=toler;  % dual feasibility tolerance
A=sparse(A);
n2=n^2;
H=sparse(H);
hz=spones(H);
indinfty=find(H>=1000);
notinfty=reshape(   spones(H<1000)  ,n2,1) ;
notinfty=sparse(diag(notinfty));
nninfty=size(indinfty,1);
nnzsH= size(find(H),1);
indzs=find(H==0);
nzsH=size(indzs,1);
H2=H.*H;    % Hadamard product
tH2=sparse(2*H2);   
vtH2= sparse(reshape(tH2,n2,1));  % column vector
dvtH2= sparse(diag(vtH2));  % diagonal matrix with 2*HoH 
%%%%% heuristic for starting point
ae=min(eig(full(A)));ae=-min(0,ae); 
ee=ones(n,1);
ee(diagfix)=zeros(diagfix); 
P=A+(ae+1)*diag(ee);  
if length(diagfix) > 0,
  [R,lmin]=chol(P);   % check pos def primal
  if lmin > 0,
  disp(['   ']);
  disp(['No initial pos def matrix P can be found. ']);
  disp(['Probable cause is that a fixed principal minor is not p.d.']);
  disp(['Try to not fix the diagonal elements. ']);
  end
end
P=sparse(P);
tH2A=tH2.*A;
tH2P=tH2.*P;
L=tH2P-tH2A; 
L=sparse(L);
[R,lmin]=chol(L);   % check pos def - dual feas.
i=0;step=1.1;
while lmin > 0,
  i=i+1;
  tH2Pn=(step^i)*tH2P;
  L=tH2Pn-tH2A; 
  L=sparse(L);
  [R,lmin]=chol(L);   % check pos def - dual feas.
end
P=sparse((step^i)*P);
mu=(1/(2*n))*trace (P*L);
disp(['  ']);
disp(['START of ALGORITHM: ']);
disp(['initial duality gap value:  ',num2str(full(mu))]);
iter=0;
%
%%%%MAIN WHILE LOOP%%%%%%%%%%%%%
while mu > epsilon,  % iterate while duality gap is large
  iter=iter+1;    %  iteration count
  disp(['   -  ']);
  disp(['iteration number:  ',num2str(iter)]);
  if nzsH >= nninfty,    % H is more sparse than infinite
%  solve for dual direction q
  Li=inv(L);
  Li=sparse(Li);
  mLi=sparse(mu*Li );
  LO=sparse(mu*sparse(dvtH2)*sparse(kron(sparse(Li),sparse(Li))));
  LO=sparse(LO+speye(n2) );
  tqp= sparse(tH2.*(mLi-P) );
  q=LO\sparse( (reshape( tqp ,n2,1 )  ) );
  q=reshape(q,n,n);
%%% ensure sparsity pattern for q - loss from roundoff error
  q(indzs)=zeros(size(indzs));
  q=sparse( .5*(q+q') );   % ensure symmetry
  h=sparse ( mLi-mLi*q*Li-P );  % step for primal P
  else
%  solve for primal direction h
  Pi=inv(P);
  Pi=sparse(Pi);
  mPi=sparse(mu*Pi );
  LO=sparse(sparse(dvtH2)-mu*(notinfty*sparse(kron(sparse(Li),sparse(Li)))));
  tqp= sparse(-tH2.*(P-A) +mu*Pi);
  h=LO\(notinfty*sparse( (reshape( tqp ,n2,1 )  )) );
  h=reshape(h,n,n);
%%% ensure sparsity pattern for h - loss from roundoff error
  h(indinfty)=zeros(size(indinfty));
  h=sparse( .5*(h+h') );   % ensure symmetry
  q=sparse ( tH2.*h );  % step for dual L
  end
%  safeguard positive definiteness - primal and dual
%  we use the Cholesky factor. to test pos. def.
    [R,lmind]=chol((L+q));   % check dual feasibility
    [R,lminp]=chol((P+h));   % check primal feasibility
    step=.5;
    i=0;
    qo=q;ho=h;
    while max(lmind,lminp) > 0,
        i=i+1;
        q=sparse( step^i*qo );
        h=sparse ( step^i*ho );
        [R,lminp]=chol(P+h);   % check primal feasibility
        [R,lmind]=chol(L+q);   % check dual feasibility
    end
    q=sparse( (.97*step^i)*qo );
    h=sparse( (.97*step^i)*ho );
%%%%%%%%
 %  update mu and P and L
    P=sparse(P+h);
    L=sparse(L+q);    
    mu=(1/(2*n))*trace (P*L);
    disp(['   ']);
    disp(['duality gap value:  ',num2str(full(mu))]);
% check dual feasibility and reset if necessary
    dnorm=norm(full(L-(tH2.*(P-A)) )); 
    if dnorm > dualtol,
      L=sparse( (tH2.*P)-tH2A);    
      L=sparse(L);
      [R,lmin]=chol(L);   % check pos def - dual feas.
      while lmin > 0,
        P=P+1.1*eye(n);
        L=sparse( (tH2.*P)-tH2A);    
        L=sparse(L);
        [R,lmin]=chol(L);   % check pos def - dual feas.
      end
    end
%  end dual check
end   
%%%%%%%%%%%%%%%%
% end of main while  loop 
% duality gap and dual feas must hold
disp(['  ']);
disp(['optimal P and optimal dual L found']);
optval=(norm( (H.*(P-A)),'fro' ))^2;  
disp(['value of objective function is:  ',num2str(optval)]);
disp(['  ']);
