function [Xorig,A,Dpartial] = SNLSDPclique_gen(n,R)
% SNLSDPclique_gen  Generate a random instance of the Sensor Network 
%                   Localization problem in dimension r = 2
%
% [Xorig,A,Dpartial] = SNLSDPclique_gen(n,R) generates the original sensor
% positions Xorig, the anchor positions A, and the partial square distance
% matrix Dpartial where entry (i,j) is ||pi-pj||^2 if the distance between
% sensor i and senor j, ||pi-pj||, is less than the radio range R.  
% Note that the matrix [Xorig; A] has n rows and r = 2 columns.
%
% The sensors are uniformly distributed in a 10x10 square region with a 3x3
% grid of m = 9 anchors.
% 
% =====================
% Citation information:
% =====================
% [1]	N.Krislock and H.Wolkowicz. Explicit sensor network 
%       localization using semidefinite representations and clique 
%       reductions. Technical report, University of Waterloo, 2009.
% <http://www.optimization-online.org/DB_HTML/2009/05/2297.html>

% SNLSDPclique_gen, version 0.1
% Copyright (C) 2009 by
% Nathan Krislock, Henry Wolkowicz
% Last Modified May 27, 2009

%% Input data
r = 2;

xbsize = 10;  ybsize = 10;  % size of region
xm = 3;       ym = 3;       % number of anchors along each dimension

m = xm*ym;  % total number of anchors


%% Process inputs and do error-checking

if (nargout > 3)
    error('SNLSDPclique_gen:TooManyOutputs', 'Too many output arguments.')
elseif (nargout < 3)
    error('SNLSDPclique_gen:TooManyOutputs', 'Too few output arguments.')
end

if (nargin < 2)
    error('SNLSDPclique_gen:TooFewInputs','Too few input arguments.');
elseif (nargin > 2)
    error('SNLSDPclique_gen:TooManyInputs','Too many input arguments.');
end


%% add anchors for r = 2
xs = linspace(.05*xbsize,.95*xbsize,xm);
ys = linspace(.05*ybsize,.95*ybsize,ym);
[Ax,Ay] = meshgrid(xs,ys);
A = [Ax(:),Ay(:)];


%% generate random points in dim r
Porig = [rand(n-m,r); zeros(m,r)];
Porig(:,1) = xbsize*Porig(:,1)/(max(abs(Porig(:,1))));
Porig(:,2) = ybsize*Porig(:,2)/(max(abs(Porig(:,2))));
Xorig = Porig(1:n-m,:);
Porig(n-m+1:n,:) = A;


%% Compute Dorig and Dpartial
Dorig = K(Porig*Porig');          % complete EDM

Dcfull = sparse(Dorig < R^2);     % distances within radio range are known
Dcfull(n-m+1:end,n-m+1:end) = 1;  % add anchors as a clique
Dcfull(1:n+1:end) = 0;            % zero out the diagonal
Dpartial = Dcfull.*Dorig;         % sparse partial EDM



%% ========= Nested functions ==========

%% K
    function D = K(B)
        % D = K(B)
        %
        % Operator K of B  - forms EDM if B is psd
        
        v = diag(B);
        vt = v';
        D = bsxfun(@plus,v,vt);
        D = D - 2*B;
    end

end