function [X,exitflag,OUTPUT] = SNLSDPclique(Dpartial,A,R,OPTS)
% SNLSDPclique      Solve the (noiseless) Sensor Network Localization 
%                   problem using the SDP Clique method
%
% X = SNLSDPclique(Dpartial,A,R) solves the sensor network problem with
% partial square distance matrix Dpartial, anchor positions A, and radio
% range R.  Note that the distances between the anchors must be in the
% bottom right corner of the matrix Dpartial.
%
% X = SNLSDPclique(Dpartial,A,R,OPTS) solves the sensor network problem
% with the default parameters replaced by those in the structure OPTS with:
%
%   OPTS.level      =   {1,[2]} where:
%                           1 : perform only Rigid Clique Union
%                           2 : perform 1 and Rigid Node Absorption
%
%   OPTS.verbose    =   {[0],1,2,3} where 0 is the default no output,
%                       1 prints minimal output of the progress of the
%                       algorithm, 2 prints full output, and 3 provides
%                       graphical output
%
% [X,exitflag] = SNLSDPclique(...) returns an exitflag with:
%
%   exitflag = 1 if some sensors were localized
%   exitflag = 0 if no sensors were localized
%
% [X,exitflag,OUTPUT] = SNLSDPclique(...) returns a structure OUTPUT with:
%
%   OUTPUT.cliques              =   entry i of this sparse vector contains
%                                   the size of clique i
%
%   OUTPUT.cliquematrix         =   column i of this sparse matrix is the
%                                   indicator vector of clique i
%
%   OUTPUT.cliqueintersections  =   (i,j) entry of this sparse matrix
%                                   gives the size of the intersection of
%                                   clique i and clique j
%
%   OUTPUT.cliquefaces          =   the {i} entry of this cell array 
%                                   contains the matrix describing the face 
%                                   of the remaining clique i
%
%   OUTPUT.disjointcliques      =   entry i of this sparse vector contains
%                                   the size of clique i if clique i has
%                                   been identified as a disjoint clique
% 
% =====================
% Citation information:
% =====================
% [1]	N.Krislock and H.Wolkowicz. Explicit sensor network 
%       localization using semidefinite representations and clique 
%       reductions. Technical report, University of Waterloo, 2009.
% <http://www.optimization-online.org/DB_HTML/2009/05/2297.html>

% SNLSDPclique, version 0.1
% Copyright (C) 2009 by
% Nathan Krislock, Henry Wolkowicz
% Last Modified May 27, 2009

%% Initialize parameters

eigOPTS.disp = 0;  % for eigs
condtolerscaling = 1e2;  % number of digits that are possibly lost
                         % from scaling in the subspace intersection
maxscaling = 1e8;  % maximum increase of the condition scaling


%% Process inputs and do error-checking

if (nargout > 3)
    error('SNLSDPclique:TooManyOutputs', 'Too many output arguments.')
end

if (nargin < 3)
    error('SNLSDPclique:TooFewInputs','Too few input arguments.');
elseif (nargin == 3)
    OPTS.level = 2;
    OPTS.verbose = 0;
elseif (nargin > 4)
    error('SNLSDPclique:TooManyInputs','Too many input arguments.');
end

n = length(Dpartial);
[m,r] = size(A);

% Check that all anchor distances exist
if ( nnz(Dpartial(end-m+1:end,end-m+1:end)) + m ) ~= m^2,
    error('SNLSDPclique:MissingAnchorDistances',...
        'Some distances for anchors are missing.');
end

if OPTS.verbose >= 1
    disp(' ');
    disp('=================== Start of SNLSDPclique ===================');
end


%% Initialization

% Compute initial half-range (R/2) cliques
Dcq = (0<Dpartial) & (Dpartial<=(R/2)^2);
Dcq(1:n+1:end) = 1;

% add anchor clique
Dcq(n-m+1:n,n+1) = 1;

% Initialize set of cliques and set of disjointcliques
Cp = sparse(true(n+1,1));
disjointcliques = sparse(zeros(n+1,1));

% Compute sizes of clique intersections and clique sizes
intcliques = Dcq'*Dcq;
csizes = sparse(diag(intcliques));

eigvs = cell(n+1,1);  % Cell array used to store the eigenvectors


%% Remove cliques that are subsets of other cliques

T = bsxfun(@eq,intcliques,csizes);
ii = any(triu(T,1),2);
clear T

% Cliques in ii are subsets of other cliques
Dcq(:,ii) = 0;  Cp(ii) = 0;

if OPTS.verbose >= 1
    disp('Removed cliques that are subsets of other cliques.');
    disp(['Cliques remaining: ',num2str(sum(Cp))]);
end


%% First stage grow cliques

tic
GrowCliques
tt = toc;

if OPTS.verbose >= 1
    disp(['Time to growcliques is  ', num2str(tt)]);
end


%% Compute intcliques, csizes, Dcqinit, csizesinit

Cp  = logical(spones(Cp));  % save memory
Dcq = logical(spones(Dcq));

if OPTS.verbose >= 3
    figure(2)
    spy(Dcq);
    pause(.000001);
end

intcliques = Dcq'*Dcq;
csizes = sparse(diag(intcliques));
intcliques = triu(intcliques,1);

Dcqinit = Dcq;        % needed at the end and in absorption step
csizesinit = full(csizes);


%% Find disjoint cliques

% Find the cliques that have no intersection with other cliques
inds = find( (sum(intcliques)==0) & Cp' );

for ic = inds %#ok<FXUP>
    p = Dcq(:,ic);
    if nnz(Dpartial(~p,p)) == 0
        disjointcliques(ic) = sum(p);
        Cp(ic) = 0;
    end
end


%% Initialize indicators
Co = find(Cp)';

grow = 1;
paramchange = 0;
mainwiters = 0;          % # iterations through while loop

if OPTS.verbose >= 1
    disp('Finished initialization -- starting WHILE loop');
end


%% MAIN WHILE LOOP:
%  while number of cliques is reduced or a clique grows or a parameter
%  changes paramchange = 1 if flagred = 1 occurs. This signals that an
%  increase in condtolerscaling should be tried.

while length(Co)>1 && (grow || paramchange),
    
    %% Reset indicators
    grow = 0;
    paramchange = 0;
    mainwiters = mainwiters + 1;
     
    if OPTS.verbose >= 1
        disp('______________________________________');
        disp(['# cliques remaining is ',num2str(sum(Cp))]);
    end
    
    
    %% Rigid Clique Union
    
    if OPTS.verbose >= 1
        disp('   Rigid Clique Union');
    end
    
    for ic = Co  %#ok<FXUP> % for each clique ic
        
        if Cp(ic) % if ic is still a clique

            if sum(Dcq(:,ic)) == 2
                % clique ic has exactly two nodes
                
                % find a common neighbour of nodes k(1) and k(2)
                k = find(Dcq(:,ic),2);
                kk = find( Dpartial(k(1),:) & Dpartial(k(2),:), 1 );
                
                if ~isempty(kk),
                    % common neighbour found
                    
                    Dcq(kk,ic) = 1;
                    grow = 1;
                    
                elseif sum(Dpartial(k(1),:) | Dpartial(k(2),:)) == 2
                    % found a disjoint clique of size 2
                    
                    disjointcliques(ic) = sum(Dcq(:,ic));
                    Cp(ic) = 0;
                    
                end
                
            elseif sum(Dcq(:,ic)) >= r+1
                
                % Find the cliques intersecting clique ic
                icintcliques = Dcq'*Dcq(:,ic);
                icintcliques(ic) = 0;
                Copj = find(icintcliques)';
                
                for jc = Copj    %#ok<FXUP>
                    
                    flagred = RigidCliqueUnion(ic,jc);
                    
                    if flagred == 1,
                        paramchange = 1; %indicates to try lower precision
                    end
                    
                end
                
            end
            
        end
        
    end
    
    
    %% Rigid Node Absorption
    
    if ~grow && OPTS.level >= 2
        
        if OPTS.verbose >= 1
            disp('     Rigid Node Absorption');
        end
        
        for ic = Co %#ok<FXUP>
            
            p = Dcq(:,ic);
            
            if sum(p) >= r+1  % clique ic >= r+1,
                
                % Find nodes connected to at least r+1 nodes in clique ic
                NeighbourDegrees = sum(Dpartial(:,p)>0,2) .* ~p;
                [sND,inds] = sort(NeighbourDegrees,'descend');
                icConnectedNodes = inds(sND >= r+1)';
                
                numnodes = min(length(icConnectedNodes),3);
                
                for jc = icConnectedNodes(1:numnodes) %#ok<FXUP>
                    
                    flagred = RigidNodeAbsorption(ic,jc);
                    
                    if flagred == 1,
                        paramchange = 1; %indicates to try lower precision
                    end
                    
                end
                
            end
            
        end
        
    end
    
    
    %% Reset everything
    
    Cp  = logical(spones(Cp));  % save memory
    Dcq = logical(spones(Dcq));
    
    % Compute sizes of clique intersections and clique sizes
    intcliques = Dcq'*Dcq;
    csizes = sparse(diag(intcliques));
    intcliques = triu(intcliques,1);
    
    Co = find(Cp)';
    
    if OPTS.verbose >= 3
        figure(2)
        spy(Dcq);
        pause(.000001);
    end

    
    %% Scaling
    if ~grow && paramchange
        if  condtolerscaling < maxscaling
            condtolerscaling = 10*condtolerscaling;
            if OPTS.verbose >= 1
                disp(['condtolerscaling increased to ',...
                    num2str(condtolerscaling)]);
            end
        else
            paramchange = 0;
        end
    end
    
    
end


%% Find disjoint cliques

inds = find( (sum(intcliques)==0) & Cp' );

for ii = inds %#ok<FXUP>
    p = Dcq(:,ii);
    if nnz(Dpartial(~p,p)) == 0
        disjointcliques(ii) = sum(p);
        Cp(ii) = 0;
    end
end

% Add disjointcliques back into Cp
Cp( disjointcliques > 0 ) = 1;

Co = find(Cp)';


%% Print Output

if OPTS.verbose >= 1
    disp('______________________________________');
    disp('End of WHILE loop');
    disp(['number of remaining cliques is: ', ...
        num2str(sum(Cp))])
    disp(['sizes of remaining cliques is: ', ...
        num2str(csizes(csizes>0)')]);
    disp(['number of disjointcliques is: ', ...
        num2str(sum(disjointcliques>0))]);
end
clear NeighbourDegrees e22 e33 e11 p a1 a2


%% Compute sensor positions

% let ic be the clique of maximum size
[mcsize,ic] = max(csizes);

if csizes(ic) >= r+1
    [P,flagred] = CompleteClique(ic);
else
    flagred = 1;
end

if flagred
    if OPTS.verbose >= 2
        disp('completeclique FLAGRED during final completion')
    end
    X = [];
else
    
    inds = find(Dcq(:,ic));
    
    if all(ismember(n-m+1:n,inds))
        
        if OPTS.verbose >= 2
            disp(['Final completion:  ',...
                'anchors contained in largest clique']);
        end
        
        % Translate P so that P2 is centred at zero
        P2 = P(n-m+1:n,:);
        mp2 = sum(P2)/m;  % centre of P2
        P = P - repmat(mp2,n,1);  % translate P
        
        % Translate A so that A is centred at zero
        ma = sum(A)/m;  % centre of A
        A = A - repmat(ma,m,1);  % centre A at zero
        
        % solve min ||A-P2*Q|| s.t. Q'*Q = I
        % this is the orthogonal Procrustes problem
        %   -- Golub, Van Loan, Algorithm 12.4.1
        C = P2'*A;
        [Uc,Sc,Vc] = svd(C);
        Q = Uc*Vc';
        
        % Compute final X
        P = P*Q;  % rotate P to align P2 with anchors A
        
        P = P + repmat(ma,n,1);  % translate P back
        A = A + repmat(ma,m,1);  % translate A back
        
        P2 = P(n-m+1:n,:);
        
        if OPTS.verbose >= 1
            disp(['Anchor agreement error = ',...
                num2str(norm(A-P2,'fro'))]);
        end
        
        inds = setdiff(inds,n-m+1:n);
        
        X = P(inds,:);  % return X = P1
        
    else
        
        X = [];
        
    end
    
end

%end


%% Set exitflag and OUTPUT

if isempty(X)
    exitflag = 0;
else
    exitflag = 1;
end

OUTPUT.cliques              =   csizes;
OUTPUT.cliquematrix         =   Dcq;
OUTPUT.cliqueintersections  =   intcliques;
OUTPUT.cliquefaces          =   eigvs;
OUTPUT.disjointcliques      =   disjointcliques;

if OPTS.verbose >= 1
    disp('=================== End of SNLSDPclique ===================');
    disp(' ');
end



%% ========= Nested functions ==========

%% GrowCliques
    function GrowCliques
        
        if OPTS.verbose >= 1
            disp('Try to grow cliques');
        end
        
        Co = find(Cp)';
        
        for ic = Co %#ok<FXUP>
            
            icgrow = 0;
            
            % Find nodes that are connected to every node in clique ic
            p = Dcq(:,ic);
            q = all(Dpartial(:,p),2);
            neighboursofwholeclique = find(q)';
            
            for jc = neighboursofwholeclique %#ok<FXUP>
                % node jc is not a member of clique ic
                
                e22 = Dpartial(:,jc) & p;
                % e22 = nhbrs of jc in clique ic
                
                if sum(e22) == csizes(ic)
                    % every node in clique ic is a nhbr of jc
                    
                    p(jc) = 1;  % add node jc to clique ic
                    csizes(ic) = csizes(ic) + 1;
                    icgrow = icgrow + 1;
                end
                
            end
            
            if icgrow
                Dcq(:,ic) = p;
            end
            
            if OPTS.verbose >= 2 && icgrow > 0
                disp([num2str(icgrow),' nodes added to clique ',...
                    num2str(ic)]);
            end
            
        end
        
        intcliques = Dcq'*Dcq;
        csizes = sparse(diag(intcliques));
        
        % Again remove cliques that are subsets of other cliques
        T = bsxfun(@eq,intcliques,csizes + ~Cp);
        ii = any(triu(T,1),2);
        
        % Cliques in ii are subsets of other cliques
        Dcq(:,ii) = 0;  Cp(ii) = 0;
        
        if OPTS.verbose >= 1
            disp('Removed cliques that are subsets of other cliques.');
            disp(['Cliques remaining: ',num2str(sum(Cp))]);
        end
        
    end


%% RigidCliqueUnion
    function flagred = RigidCliqueUnion(ic,jc)
        
        flagred = 0;
        
        e22 = Dcq(:,ic) & Dcq(:,jc);  % intersect cliques ic and jc
        
        e11 = (Dcq(:,ic) - e22)>0;   e33 = (Dcq(:,jc) - e22)>0;
        
        inds = [find(e11); find(e22); find(e33)];
        
        if ~any(e11)   % nothing in e11
            % clique ic is a subset of clique jc,
            % so copy clique jc onto clique ic and delete clique jc
            
            Cp(jc) = 0;
            Dcq(:,ic) = Dcq(:,jc);
            Dcq(:,jc) = 0;
            eigvs{ic} = eigvs{jc};
            eigvs{jc} = [];
            grow = 1;
            
        elseif ~any(e33)  % nothing in e33
            % clique jc is a subset of clique ic, so delete clique jc
            
            Cp(jc) = 0;
            Dcq(:,jc) = 0;
            eigvs{jc} = [];
            grow = 1;
            
        elseif sum(e22) >= r+1
            
            nvec = full([sum(e11) sum(e22) sum(e33)]);
            
            a1 = 1:sum(nvec(1:2));   a2 = nvec(1)+1:sum(nvec);
            a1inds = inds(a1);       a2inds = inds(a2);
            
            % Find Ub1
            if isempty(eigvs{ic})
                Dbar = full(Dpartial(a1inds,a1inds));
                B = Kdag(Dbar);
                [Ub,Db] = eigs(B,r,'LA',eigOPTS); %#ok<NASGU>
                k = length(a1);  e = ones(k,1);
                Ub1 = [ Ub, e/sqrt(k) ];
            else
                Ub1 = full(eigvs{ic}(a1inds,:));
            end
            
            % Find Ub2
            if isempty(eigvs{jc})
                Dbar = full(Dpartial(a2inds,a2inds));
                B = Kdag(Dbar);
                [Ub,Db] = eigs(B,r,'LA',eigOPTS); %#ok<NASGU>
                k = length(a2);  e = ones(k,1);
                Ub2 = [ Ub, e/sqrt(k) ];
            else
                Ub2 = full(eigvs{jc}(a2inds,:));
            end
            
            % Find U
            [U,flagred] = SubspaceIntersection(nvec,Ub1,Ub2);
            
            if ~flagred
                
                % Store U
                ii = repmat(inds,1,r+1);
                jj = repmat(1:r+1,length(inds),1);
                eigvs{ic} = sparse(ii(:),jj(:),U(:),n,r+1);
                
                % Update Dcq
                Cp(jc) = 0;
                Dcq(:,ic) = Dcq(:,ic) | Dcq(:,jc);
                Dcq(:,jc) = 0;
                eigvs{jc} = [];
                grow = 1;
                
                if OPTS.verbose >= 2
                    disp(['RigidCliqueUnion:  [ic jc |e22|] = [',...
                        num2str([ic jc nvec(2)]),']']);
                end
                
                if OPTS.verbose >= 3
                    Dcq = logical(spones(Dcq));
                    figure(2);
                    spy(Dcq);
                    pause(.000001);
                end
                
            end
            
        end
        
    end


%% RigidNodeAbsorption
    function flagred = RigidNodeAbsorption(ic,jc)
        
        flagred = 0;
        
        e22 = Dpartial(:,jc) & Dcq(:,ic);  % e22 = nhbrs of jc in clique ic
        ne22 = sum(e22);
        
        if ne22 == sum(Dcq(:,ic)) && isempty(eigvs{ic})
            % node jc is connected to every node in clique ic
            % and there are no eigvs for clique ic,
            % so clique ic is an original clique -- 
            % simply add jc to clique ic
            
            % Update Dcq
            Dcq(jc,ic) = 1;  % absorb jc into clique ic
            grow = 1;
            
            % Output
            if OPTS.verbose >= 2
                disp(['RigidNodeAbsorption:  [ic jc |e22|] = [',...
                    num2str([ic jc sum(e22)]),'] -- simple add']);
            end
        else
            
            % Complete clique ic, if necessary
            if nnz(Dpartial(e22,e22)) ~= ne22*(ne22-1)
                
                % Complete clique ic
                [P,flagred] = CompleteClique(ic);
                
                if ~flagred
                    
                    % Store distances
                    
                    Dpartial(e22,e22) = ...
                        (Dpartial(e22,e22)==0) .* K(P(e22,:)*P(e22,:)') ...
                        + Dpartial(e22,e22);
                    
                end
            end
            
            % If complete clique was successful, perform node absorption
            if flagred
                if OPTS.verbose >= 2
                    disp('completeclique FLAGRED in RigidNodeAbsorption');
                end
            else
                
                if ne22 == sum(Dcq(:,ic))
                    % node jc connected to every node in clique ic
                    
                    p = Dcq(:,ic);
                    p(jc) = 1;  % add node jc to clique ic indices
                    
                    B = Kdag(full(Dpartial(p,p)));
                    [Ub,Db] = eigs(B,r,'LA',eigOPTS); %#ok<NASGU>
                    k = sum(p);  e = ones(k,1);
                    U = [ Ub, e/sqrt(k) ];
                    
                    % Store U
                    inds = find(p);
                    ii = repmat(inds,1,r+1); 
                    jj = repmat(1:r+1,length(inds),1);
                    eigvs{ic} = sparse(ii(:),jj(:),U(:),n,r+1);
                    
                    % Update Dcq
                    Dcq(jc,ic) = 1;  % absorb jc into clique ic
                    grow = 1;
                    
                    % Output
                    if OPTS.verbose >= 2
                        disp(['RigidNodeAbsorption:  [ic jc |e22|] = [',...
                            num2str([ic jc ne22]),...
                            '] - simple node absorb']);
                    end
                    
                elseif ne22 >= r+1
                    % at least r+1 neighbours in clique jc
                    
                    e11 = (Dcq(:,ic) - e22)>0; 
                    e33 = sparse(jc,1,true,n,1);
                    
                    inds = [find(e11); find(e22); find(e33)];
                    nvec = full([sum(e11) sum(e22) sum(e33)]);
                    
                    a1 = 1:sum(nvec(1:2));   a2 = nvec(1)+1:sum(nvec);
                    a1inds = inds(a1);       a2inds = inds(a2);
                    
                    % Find Ub1
                    if isempty(eigvs{ic})
                        Dbar = full(Dpartial(a1inds,a1inds));
                        B = Kdag(Dbar);
                        [Ub,Db] = eigs(B,r,'LA',eigOPTS); %#ok<NASGU>
                        k = length(a1);  e = ones(k,1);
                        Ub1 = [ Ub, e/sqrt(k) ];
                    else
                        Ub1 = full(eigvs{ic}(a1inds,:));
                    end
                    
                    % Find Ub2
                    Dbar = full(Dpartial(a2inds,a2inds));
                    B = Kdag(Dbar);
                    [Ub,Db] = eigs(B,r,'LA',eigOPTS); %#ok<NASGU>
                    k = length(a2);  e = ones(k,1);
                    Ub2 = [ Ub, e/sqrt(k) ];
                    
                    % Find U
                    [U,flagred] = SubspaceIntersection(nvec,Ub1,Ub2);
                    
                    if ~flagred
                        
                        % Store U
                        ii = repmat(inds,1,r+1);
                        jj = repmat(1:r+1,length(inds),1);
                        eigvs{ic} = sparse(ii(:),jj(:),U(:),n,r+1);
                        
                        % Update Dcq
                        Dcq(jc,ic) = 1;  % absorb jc into clique ic
                        grow = 1;
                        
                        % Output
                        if OPTS.verbose >= 2
                            disp(['RigidNodeAbsorption:  ',...
                                '[ic jc |e22|] = [',...
                                num2str([ic jc ne22]),']']);
                        end
                        
                    end
                    
                end
                
            end
            
        end
        
        if OPTS.verbose >= 3
            Dcq = logical(spones(Dcq));
            figure(2);
            spy(Dcq);
            pause(.000001);
        end
        
    end


%% SubspaceIntersection
    function [U,flagred] = SubspaceIntersection(nvec,Ub1,Ub2)
        
        flagred = 0;
        
        kb1 = nvec(1);
        kb2 = nvec(2);
        %kb3 = nvec(3);
        
        cond1 = cond(Ub1(kb1+1:end,:));
        cond2 = cond(Ub2(1:kb2,:));
        
        if min(cond1,cond2) > condtolerscaling*kb2,
            flagred = 1;
        else
            if cond1 < cond2,
                utemp = Ub1(kb1+1:end,:) \ Ub2(1:kb2,:);
                U = [Ub1(1:kb1,:)*utemp; Ub2];
            else
                utemp = Ub2(1:kb2,:) \ Ub1(kb1+1:end,:) ;
                U = [Ub1; Ub2(kb2+1:end,:)*utemp];
            end
        end
        
        if flagred,
            if OPTS.verbose >= 2
                disp('FLAGRED in SubspaceIntersection: large cond number');
            end
            U = [];
        else
            kk = sum(nvec);
            ek = ones(kk,1)/sqrt(kk);
            [U,rtemp] = qr( [ ek U(:,1:end-r+1) ], 0 ); %#ok<NASGU>
            U = [ U(:,2:end) ek ];
        end
        
    end


%% CompleteClique
    function [P,flagred] = CompleteClique(ic)
        % Given a clique ic, compute all the sensor positions P;
        % the clique has the eigenvectors or all the distances specified
        
        
        % Complete the EDM of the clique ic
        inds = find(Dcq(:,ic));
        
        if length(inds) <= r
            
            disp('length(inds) <= r in completeclique');
            
            P = [];
            flagred = 1;
            return
            
        else
            
            if isempty(eigvs{ic})
                % clique ic must be a clique in the original graph
                
                % Check that we have a complete EDM
                p = Dcq(:,ic);
                np = sum(p);
                if nnz(Dpartial(p,p)) ~= np*(np-1)
                    disp('EDM not complete in completecliques');
                    keyboard
                end
                
                D = full(Dpartial(inds,inds));
                B = Kdag(D);
                [Ubs,Dbs] = eigs(B,r,'LA',eigOPTS);
                
                k = length(inds);  e = ones(k,1);
                Ub = [ Ubs, e/sqrt(k) ];
                
                ii = repmat(inds,1,r+1); jj = repmat(1:r+1,length(inds),1);
                eigvs{ic} = sparse(ii(:),jj(:),Ub(:),n,r+1);
                
                % Compute P
                P = Ubs*sqrt(Dbs);
                ii = repmat(inds,1,r);  jj = repmat(1:r,length(inds),1);
                P = sparse(ii(:),jj(:),P(:),n,r);
                
                flagred = 0;
                return
                
            else
                
                % Find a good original half-range clique inside clique ic
                
                indsPlusAnchorClique = [inds; n+1];
                
                [temps,tempi] = sort(...
                    csizesinit(indsPlusAnchorClique),'descend');
                inds3 = sum( temps >= r+1 );  % # of orig cliques >= r+1
                
                % cliques with sizes >= r+1
                cliqueinds = indsPlusAnchorClique(tempi(1:inds3))';
                
                for im = cliqueinds
                    % for orig cliques im -- sorted, size >= r+1
                    
                    % determine if orig clique im is inside clique ic
                    e22 = Dcqinit(:,im) & Dcq(:,ic);
                    
                    if sum(e22) == csizesinit(im)
                        % clique im is inside clique ic
                        
                        b = find(Dcqinit(:,im));
                        mm = length(b);
                        
                        % Find U and V
                        U = eigvs{ic};
                        
                        UTe = sum(U)';
                        [V,Rtemp] = qr(UTe);
                        if norm(V*Rtemp-UTe) > .000001
                            disp('||V*Rtemp-U^T*e|| > .000001');
                            keyboard
                        end
                        V = V(:,2:end);  UV = U*V;
                        
                        % Compute AA and B
                        Jb = eye(mm) - ones(mm)/mm;
                        Ub = U(b,:);
                        AA = Jb*Ub*V;
                        
                        B  = Kdag(full(Dpartial(b,b)));
                        [Vb,Db] = eigs(B,r,'LA',eigOPTS);
                        
                        condB = cond(Db);
                        
                        if condB > 50
                            
                            if OPTS.verbose >= 2
                                disp(['******* completeclique condB = ',...
                                    num2str(condB)]);
                            end
                            
                            P = [];
                            flagred = 1;
                            
                            return
                        else
                            
                            % Compute P
                            
                            Zb = linsolve(AA,Vb*sqrt(Db));
                            P = UV(inds,:)*Zb;
                            
                            ii = repmat(inds,1,r);
                            jj = repmat(1:r,length(inds),1);
                            P = sparse(ii(:),jj(:),P(:),n,r);
                            
                            flagred = 0;
                            
                            return
                        end
                        
                    end
                    
                end
                
                P = [];
                flagred = 1;
                return
                
            end
            
        end
        
    end


%% K
    function D = K(B)
        % D = K(B)
        %
        % Operator K of B  - forms EDM if B is psd
        
        v = diag(B);
        vt = v';
        D = bsxfun(@plus,v,vt);
        D = D - 2*B;
    end


%% Kdag
    function  B = Kdag(D)
        %  B = Kdag(D)
        
        nn = length(D);
        
        D = 0.5*D;
        vn = sum(D,2)/nn;
        vnt = vn';
        evnn = sum(vn)/nn;
        
        B = bsxfun(@plus,vn,vnt);
        B = B - D;
        B = B - evnn;
    end


end