function [resultSemi,resultCvx] = runMCQAP(problemList,optionsProbGen,options)
%function [resultSemi,resultCvx] = runMCQAP(problem,options,optionsProbGen)

% This function generates instances of elliptope/QAP
% and pass the instance to the solver.
% Two solvers are used; semi-smooth Newton and SDPT3.
% The instance is converted to SOCP for feeding into SDPT3.
% This function has an ability to generate problems that have rank-one
% optimal point (specified by the optionsProbGen.isXrankOne).

% input:
%    problemList : array of strings ; each can be MC, QAPnoFR, QAPFR
%    optionsProbGen : problem generation options
%                     .isXrankOne: boolean;
%                           true for generating rank-one optimal X
%                           false for random optimal X
%                     .probOrder: natural number; problem size
%                     .genNum: number of instance generated
%    options : solver options ; they are passed to the solver
% output:
%    resultSemi :numerical results from semi-smooth Newton method
%         contains average of primal/dual feasbility, complementarity
%         algorithm time, iterations, condition numbers of Jacobian
%    resultCvx : numerical results from sdtp3
%         contains average of primal/dual feasbility, complementarity
%         algorithm time, iterations


if isfield(optionsProbGen,'probOrder')
    probOrder = optionsProbGen.probOrder;
else
    probOrder = 6;
end
if isfield(optionsProbGen,'isXrankOne')
    isXrankOne = optionsProbGen.isXrankOne;
else
    isXrankOne = false;
end
if isfield(optionsProbGen,'genNum')
    genNum = optionsProbGen.genNum;
else
    genNum = 1;
end


for instanceNum = 1:genNum
    fprintf('Generating problem, isXrankOne=%d, instanceNum=%d\n',...
        isXrankOne, instanceNum)
    for problem = problemList
        %% Problem generation  %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    	%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
        if strcmp(problem,'MC')

            % Generate an elliptope data A,b and options.Acell
            [A,b,options] = MakeProblemMC(probOrder,options);
            if isXrankOne  % generate problem with rank-1 opt.sol
                xtemp = sign(randn(probOrder,1));
        		xtemp(xtemp == 0) = 1;  % ensure NO zeros
                X = xtemp*xtemp';  % rank-one optimal solution of MC
                %[Vbar,~] = qr(X); % changed by henry
                %Vbar(:,1) = [];    % Vbar spans the orthogonal compl. of X
        		Vbar = null(xtemp'); % better? spans orthog. compl. of X?
                Ztemp = rand(probOrder-1);
                ZtempV = Vbar*Ztemp;
                Z = ZtempV*ZtempV'; % dual optimal solution
                y = randn(length(b),1);  % for dual opt
        		% construct W so that optimality conditions hold for X
                W = X - Z - ssMat(A'*y,probOrder,options.sMindn);
            else
                W = randn(probOrder);  %?not symmetric??????
        		W = (W + W')/2;  % added by Henry
            end

        elseif strcmp(problem,'QAPnoFR')
            % Generate QAP instance without FR

            isFR = false;
            [A,b,options] = MakeProblemQAP(probOrder,isFR,options);
            % col size of A = t( n^2+1 )

            if isXrankOne % generate problem with rank-1 opt.sol
                y = randn(length(b),1);
                In = eye(probOrder);
                Xperm = In(randperm(probOrder),:);
                X = [1;Xperm(:)]*[1,Xperm(:)']; % rank-one optimal solution

                [Vbar,~] = qr(X);
                Vbar(:,1) = [];
                Ztemp = rand((probOrder)^2);
                ZtempV = Vbar*Ztemp;
                Z = ZtempV*ZtempV';   %dual optimal solution
                W = X - Z - ssMat(A'*y,probOrder^2+1,options.sMindn);
            else
                W = randn(probOrder^2+1);
            end



        elseif strcmp(problem,'QAPFR')
            % Generate QAP instnace after FR

            isFR = true;
            [A,b,options] = MakeProblemQAP(probOrder,isFR,options);
            % col size of A = t( (n-1)^2+1 )
            if isXrankOne  % generate problem with rank-1 opt.sol
                y = randn(length(b),1);

                en = ones(probOrder,1);
                In = eye(probOrder);
                Hn = [kron(en',In);kron(In,en')];
                Kend = [-ones(2*probOrder,1) Hn];
                Kendo = Kend(1:end-1,:);   % last row is redundant; better conditioned!
                Vhat = null(Kendo);  % facial range vector

                Xperm = In(randperm(probOrder),:);
                X = [1;Xperm(:)]*[1,Xperm(:)'];  % optimal solution to the original QAP
                R = Vhat'*X*Vhat;  % optimal solution to facially reduced QAP
                R = (R+R')/2;

                [Vbar,~] = qr(R); % Vbar spans the orthogonal compl. of X
                Vbar(:,1) = [];
                Ztemp = rand((probOrder-1)^2);
                ZtempV = Vbar*Ztemp;
                Z = ZtempV*ZtempV';   %dual optimal solution
                W = R - Z - ssMat(A'*y,(probOrder-1)^2+1,options.sMindn);
            else
                W = randn((probOrder-1)^2+1);
            end


        else
            fprintf('Incorrect problem name\n')
            return;
        end

        w = ssvec(W,options.svindn);  % objective data



        %% Call the algorithm
        startTimeSemi = tic;
        output = semismoothsolverK_preconditioner(A,b,w,options);
        endTimeSemi = toc(startTimeSemi );

        % save conditiong number for reporting
        if output.condJac <1e12
            countGood = 1; countBad = 0;
        else
            countGood = 0; countBad = 1;
        end



        %% Call SDPT3
        startTimeCvx = tic;

        % form our problem as  -- second order cone problem
        [blk,At_sdpt3,C_sdpt3,b_sdpt3] = formSDPT3Const(W,options.Acell,b);

        % solve using sdpt3
        opts_sdpt3 = [];
        opts_sdpt3.printlevel = 0; % level 2 --> 0 by henry jun4/24
        %[obj,Xsdpt3,ysdpt3,Zsdpt3,infosdpt3,runhist] = ...
        [~,Xsdpt3,~,~,infosdpt3,~] = ...
            sqlp(blk,At_sdpt3,C_sdpt3,b_sdpt3, opts_sdpt3);

        pfSdpt3 = infosdpt3.pinfeas;
        dfSdpt3 = infosdpt3.dinfeas;
        csSdpt3 = infosdpt3.gap;
        iterSdpt3 = infosdpt3.iter;
        timeSdpt3 = toc(startTimeCvx );

        %         ntemp = size(W,1);
        %         tntemp = ntemp*(ntemp+1)/2;
        %         cvx_begin sdp
        %         cvx_precision best;
        %         variable xCvx(tntemp,1) ;
        %         dual variable yCvx ;
        %         dual variable ZCvx ;
        %         minimize( 0.5*norm( w - xCvx ) );
        %         subject to
        %         yCvx: A*xCvx == b
        %         ZCvx: ssMat(xCvx,ntemp,options.sMindn) >=0
        %         cvx_end
        %         zCvx = ssvec(ZCvx, options.svindn);




        %% Save/collect result

        xSemi = output.xfinal;
        ySemi = output.yfinal;
        zSemi = output.zfinal;
        [pfSemi, dfSemi, csSemi] = ...
            computeOptCond(xSemi,ySemi,zSemi,A,b,w,size(W,1),options);



        condInfo = [countGood,countBad,output.condJac];
        semiInfo = [pfSemi,dfSemi,csSemi,output.iter,endTimeSemi];
        cvxInfo = [pfSdpt3,dfSdpt3,csSdpt3,iterSdpt3, timeSdpt3];
        if strcmp(problem,'MC')
            saveResult.Semi.MC(instanceNum,1:8) = [semiInfo,condInfo];
            saveResult.Cvx.MC(instanceNum,1:5) = cvxInfo;
        elseif strcmp(problem,'QAPnoFR')
            saveResult.Semi.QAPnoFR(instanceNum,1:8) = [semiInfo, condInfo];
            saveResult.Cvx.QAPnoFR(instanceNum,1:5) = cvxInfo;
        elseif strcmp(problem,'QAPFR')
            saveResult.Semi.QAPFR(instanceNum,1:8) =  [semiInfo, condInfo];
            saveResult.Cvx.QAPFR(instanceNum,1:5) = cvxInfo;
        end

    end % end of  'for problem = problemList'
end % end of  'for instanceNum = 1:genNum '

%saveResult.Semi.MC(:,8)
%saveResult.Semi.QAPFR(:,8)
%saveResult.Semi.QAPnoFR(:,8)

resultSemi = [
    mean(saveResult.Semi.MC(:,1:5),1), mean(saveResult.Semi.MC(:,8)) ;
    mean(saveResult.Semi.QAPFR(:,1:5),1), mean(saveResult.Semi.QAPFR(:,8)) ;
    mean(saveResult.Semi.QAPnoFR(:,1:5),1), mean(saveResult.Semi.QAPnoFR(:,8))
    ];
temp = resultSemi(:,end);
temp(temp<0) = inf;
resultSemi(:,end) = temp;

resultCvx = [ mean(saveResult.Cvx.MC(:,1:5),1) ;
    mean(saveResult.Cvx.QAPFR(:,1:5),1);
    mean(saveResult.Cvx.QAPnoFR(:,1:5),1)];


end % end of  'function runMCQAP(problemList,options,optionsProbGen)'


function [pf, df, cs] = computeOptCond(x,y,z,A,b,w,n,options)
% compute first order optimality conditions

X = ssMat(x,n,options.sMindn);
Z = ssMat(z,n,options.sMindn);
% primal feasibility
wAty = w + A'*y;
% primal feasibility
pf = norm(A*x-b)/norm(b) + abs(min(eig(X))/n);
% dual feasibility
df = norm(z-(x-wAty))/norm(w) + abs(min(eig(Z))/n) ;
% complementarity
cs = abs(trace(Z*X))/(1+(norm(Z,'fro')+norm(X,'fro'))/2);


end % end of  'function computeOptCond(xsemi,ysemi,zsemi)'
