%%%%% MakeProblemQAP
%
%Create data for an QAP = {X psd : diag(X) = e}
% input:  n : order of permutation matrices
%         isFaciallyReduced : boolean, true : create facially reduced QAP
%                                      false : create original QAP
%         options : solver options passed to this function to append svindn, sMindn
% output: (A,b) data for A*svec(X) = b
%         options: solver options  


function [A,b,options] = MakeProblemQAP(n,isFaciallyReduced,options)
%function [A,b,X0] = MakeProblemQAP(n,isFaciallyReduced,options)

fprintf('In MakeProblemQAP.m: Generating the ')
if (isFaciallyReduced)
    fprintf('facially reduced ')
else
    fprintf('original ')
end
fprintf('QAP instance of order=%d \n',n)


en = ones(n,1);
n2 = n^2;
En = ones(n);
In = eye(n);
Zn = zeros(n);
%% Forming the gangster index set Jbar  - this does NOT include Y(1,1)
%% off-diagonals of diagonal blocks and diagonals off off-diagonal blocks
Es21 = triu(En,1);
YJ = (kron(In,Es21) + kron(Es21,In));
%%%remove redundant constraints
YJ(1:n*(n-1),n2-n+1:n2) = 0;  % last col. of off diag. blocks
YJ(n2-3*n+1:n2-2*n,n2-2*n+1:n2-n) = 0;  % last col. of off diag. blocks
YJ = blkdiag(0,YJ);
Jbar = sparse(YJ~=0);     % fix 0 elements in barycenter to stay 0



trn1 = (n-1)*n/2;  % t(n-1)
%% Helper data: Form the basis for the diag elements 
EiiCell = cell(n,1);  %
% for ii = 1:trn1
for ii = 1:n
    Eii = Zn;
    Eii(ii,ii) = 1;  % = e_i*e_i^T (size n-by-n)
    EiiCell{ii} = Eii;
end
%% Helper data: Form the basis for the off-diag elements (in Sn)
offDiagCell = cell(trn1,1);  %
offDiagInds = find(triu(En,1));
for ii = 1:trn1
    temp = Zn;
    temp(offDiagInds(ii)) = 1;
    temp = temp+temp';
    offDiagCell{ii} = temp;
end




% Save the constraint data in Acell, b
Acell = cell( 2*trn1 + 2*n + n^2 ,1 );  % n*trn1 + trn1 +1 is the total # of equality constraints before FR
b = [];
totalCnt = 0;  % consrtaint counting


%% <1> Realize Gangster constraint  
% Gangster type 1: off-diagonals of diagonal blocks = 0 ( #.const = n*t(n-1))
for blkCnt = 1:n
    Eii = Zn;
    Eii(blkCnt,blkCnt) = 1;  % = e_i*e_i^T (size n-by-n)
    for jj = 1:trn1
        totalCnt = totalCnt+1;
        Acell{totalCnt} = blkdiag( 0, kron(Eii, offDiagCell{jj}) );      
        b(totalCnt) = 0; 
    end
end

 % Gangster type 2: diagonals of off-diagonal blocks = 0 ( #.const = n*t(n-1))
for ii = 1:trn1
    for jj = 1:n
        Eii = EiiCell{jj};
        totalCnt = totalCnt+1;
        Acell{totalCnt} = blkdiag( 0, kron(offDiagCell{ii},Eii) );
        b(totalCnt) = 0; 
    end
end


%% <2> Realize bdiag(Y)=I ; sum of n*n block diagonal matrices = I 
% sum of (i,i)-th element of the diagonal blocks = 1
for blkCnt = 1:n
    Eii = EiiCell{blkCnt};
    totalCnt = totalCnt+1;
    Acell{totalCnt} = blkdiag( 0, kron( In, Eii ) );
    b(totalCnt) = 1; 
end

% sum of (i,j)-th element of the diagonal blocks = 0
%  -> Implied by Gangster type 1; hence commented out
for jj = 1:trn1
    totalCnt = totalCnt+1;
    Acell{totalCnt} = blkdiag( 0, kron( In, offDiagCell{jj} ) ); 
    b(totalCnt) = 0; 
end


%% <3> Realize odiag(Y)=I ; trace of n*n diagonal matrices = 1, 
%                       trace of n*n off-diagonal matrices = 0
%trace of n*n diagonal matrices = 1
for blkCnt = 1:n
    Eii = EiiCell{blkCnt};  % = e_i*e_i^T (size n-by-n)
    totalCnt = totalCnt+1;
    Acell{totalCnt} = blkdiag( 0, kron( Eii, In ) ); 
    b(totalCnt) = 1; 
end

%trace of n*n off-diagonal matrices = 0
%  -> Implied by Gangster type 2; hence commented out
for jj = 1:trn1
    totalCnt = totalCnt+1;
    Acell{totalCnt} = blkdiag( 0, kron( offDiagCell{jj} , In ) );
    b(totalCnt) = 0; 
end


%% <4> Realize arrow(Y) = e0 ; first column = diagonal
Zn21 = zeros(n^2+1);
temp = Zn21;
temp(1,1) = 1;
%top-left corner Y_{00}=1
totalCnt = totalCnt+1;
Acell{totalCnt} = temp;
b(totalCnt) = 1; 

%first through the last column/row are equal to the diagonal
for jj = 2:n^2+1
    temp = Zn21;
    temp(jj,jj) = 1;
    temp(jj,1) = -0.5;
    temp(1,jj) = -0.5;
    totalCnt = totalCnt+1;
    Acell{totalCnt} = temp;
    b(totalCnt) = 0; 
end

b = reshape(b,length(b),1);



% total number of constraints = 2*trn1 + 2n + n^2+ 2
m = totalCnt; % #.const (= n*trn1 + n*trn1 +1)


% for ssMat/ssvec
newn = n2+1;
[svindnewn, sMindnewn] = svecIsoInd(newn);
options.svindn = svindnewn;
options.sMindn = sMindnewn;



%% Form the matrix representation 
% Matrix representation of G_J(Y), i.e., AcellMat*svec(Y)
AMat = zeros( m, (newn)*(newn+1)/2 );
for ii = 1:m
    temp = ssvec(Acell{ii},svindnewn);
    AMat(ii,:) = temp';
end

% Discard a redundant constraint
% There is one redundant constraint - trace(Y11 block) = 1; \
% We remove one constrant here rather than at the construction for clarity.
[PAMat,indepInd] = licols(AMat');
A = PAMat';
b = b(indepInd);
Acell = Acell(indepInd);

options.Acell = Acell;
m = length(indepInd);

if isFaciallyReduced
    
    Hn = [kron(en',In);kron(In,en')];
    Kend = [-ones(2*n,1) Hn];
    Kendo = Kend(1:end-1,:);   % last row is redundant; better conditioned!
    Vhat = null(Kendo);  % facial range vector

    %% Facial Reduction
    % Construct 'AcellFR' for data matricex after FR
    % Transform from G_J(Y) = <Ai,Y>
    % to <Ai,Y> = <Ai, VRV'> = <V'*Ai*V,R>
    AcellFR = cell( m ,1 );
    for ii = 1:m
        temp = (Vhat'*Acell{ii})*Vhat;
        temp = (temp+temp')/2;
        AcellFR{ii} = temp;  % V'*Ai*V
    end
    
    r = (n-1)^2+1;   % new matrix size after FR
 
    % Matrix representation of G_J(VRV'), i.e., AcellMatFR*svec(R)
    [svindr, sMindr] = svecIsoInd(r);
    options.svindn = svindr;
    options.sMindn = sMindr;
    
    
    AMatFR = zeros( m,  r*(r+1)/2 );
    for ii = 1:m
        temp = ssvec(AcellFR{ii},svindr);
        AMatFR(ii,:) = temp';
    end
    

    [PAMatFR,indpenInd] = licols(AMatFR');
    PAMatFR = PAMatFR';
    AcellFR = AcellFR(indpenInd);  % discard redundant constraints
    mbar = length(indpenInd); % number of lin.indep. constraints

    
    
    if mbar ~= n^3-2*n^2+1 % theoretical gaurantee; (see Zhao p.92)
        fprintf(2,'Error: # of constraints are incorrect after FR\n')
        keyboard
    end
    
    A = AMatFR(indpenInd,:);
    b = b(indpenInd);
    options.Acell = AcellFR;

    
end % end of    if isFaciallyReduced

% Report
if isFaciallyReduced
    %fprintf('After FR\n')
    %fprintf('size of AMatFR = %d-by-%d\n',size(AMatFR))
    %fprintf('rank(AMatFR) = %d\n', rank(AMatFR))
    fprintf('                     implicit redundancy = %d\n',...
             size(AMatFR,1)-length(indpenInd)  )
    fprintf('                     size of A = %d-by-%d\n',size(PAMatFR))
else
   fprintf('                      size of AMat = %d-by-%d\n',size(A))
end

end     %end function










