function output = semismoothsolverK_preconditioner(A,b,v,options)
%function [xfinal,yk,output] = semismoothsolverK_preconditioner(A,b,v,options)
%
%  When STRICT FEASIBILITY holds:
%  Solve the nearest/proj. problem min ||v-x|| s.t. Ax=b, x>=0
%  Here x>=0 denotes SDP as does the _+ projection
%  Use the opt condiitons   F(y) = A(v+A'*y)_+ - b = 0
%
% If strict feasibility DOES NOT holds:
% identifies the constraints causing strict feasibility to fail,
% returns them and recommends to perform facial reduction
%
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% INPUT and problem description:
% v - svec initialization point in R^n
% A - svec matrix in R^(m*n) for equal. constraints; assumed onto/full row rank
% b vector in R^m
% options - Struct: maxiter - Maximum number of iterations for while loop
%                   tol - tolerance of stopping criteria
%                   tol - tolerance for the lsqr method
%                   saveegis - true for saving the eigenvalues of Jac at
%                   every iteration
%           See in Initialization below for other options
% OUTPUT:  output Struct:
%  xfinal  - final solution
%  yk      - final dual solution in opt conditions
%  noSlater   - (false,true) = (Slater hold, Slater does not hold need FR)
%  see output below for other variables in output

% Initialization
warning off   % since Jacobian can be ill-conditioned
% Options
if isfield(options,'maxiter')
    maxiter = options.maxiter;
else
    maxiter = 100;
end
if isfield(options,'modprint')     % #iters between outputs
    modprint = options.modprint;
else
    modprint = 10;
end
if isfield(options,'verbose')
    verbose = options.verbose;
else
    verbose = true;
end
if isfield(options,'verbosePlot')
    verbosePlot = options.verbosePlot;
else
    verbosePlot = true;
end
if isfield(options,'tol')
    tol = options.tol;
else
    tol = 1e-7;
end


sqrt2 = sqrt(2);
[m,tn] = size(A);
n = (-1+round(sqrt(1+8*tn)))/2;
n2 = n^2;

noSlater= false; % we assume Slater holds, if not we will change it later

if isfield(options,'svindn')
    svindn = options.svindn;
else
    svindn = find(triu(ones(n),1)); % matrix indices for the strictly upper triangular
end

if isfield(options,'sMindn')
    sMindn = options.sMindn;
else
    sMindn = zeros(n2,1); %matrix indices with respect to linear indices of the diagonal and the strictly upper triangle
    %tn = n*(n+1)/2;
    sMindn(svindn) = 1:tn-n; %strictly upper triangle
    sMindn(1:n+1:n2) = tn-n+1:tn;%the diagonal
    %calculate the linear indices of the strictly lower triangle
    indMn = zeros(n); %matrix indices
    indMn(:) = 1:n2;
    indMn = indMn';
    lsvindn = indMn(svindn); %strictly lower triangular
    sMindn(lsvindn) = 1:tn-n; %strictly lower triangular
end
if isfield(options,'saveeigs')  %for saving the eigenvalues after each iteration
    saveeigs = options.saveeigs;
    eigenvaluesk = zeros(maxiter,m);
else
    saveeigs = false;
end
Acell = cell(m,1);
for ii = 1:m
	Acell{ii} =  ssMat(A(ii,:),n,sMindn);
end
OptionsJac.Acell = Acell;

%%%% OptionsJac: options for Jacobian Computation
OptionsJac.A = A;
OptionsJac.svindn = svindn;
OptionsJac.n = n;
OptionsJac.sMindn = sMindn;


%%%%%%%%%% if yk exists then we are doing a warm start
if isfield(options,'yk')
    yk = options.yk;
else   %%%%%%%%NO warm start
    yk = zeros(m,1);
end
k = 1;   % iter for while loop
bAv = b-A*v;
Ndir = yk;
normNdir = norm(Ndir);
vAty = v+A'*yk;        %svec vector
vAtyM = ssMat(vAty,n,sMindn);
[lpfdM,xkM,zkM] = projK(vAtyM);      %  xk = max(vAty,0);
%xk = ssvec(xkM,svindn);
xk = [xkM(svindn)*sqrt2; xkM(1:n+1:end)'];
zk = [zkM(svindn)*sqrt2; zkM(1:n+1:end)'];

%zk = xk-vAty;   % from the optimality conditions  ??? is Mat(zk) psd????
normb = (1+norm(b)); % for relative error
Fk = A*xk-b;
relresid = norm(Fk)/normb;
lbnd = (-norm(zk+A'*yk)^2/2)+yk'*bAv-zk'*v; % from dual problem
lambdareg = 0;
%%% save norm(xk), norm(zk), norm(yk)
normsx = [];
normsz = [];
normsy = [];

condestJacs = []; % save condition number at every iteration

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%



% Print desirable strings
if verbose
    VarValue = {relresid,lbnd,normNdir};
    VarPrint = {'relres/optcond','lower bnd','||Ndir||',  ...
        '#eig(X)>=0','lambda'};
    %'|#pos x_j|','setxor cols','zerosx+z','lambda'};
    %'|#pos x_j|','setxor pos','(act+m)','deg: n-(act+m)'};
    fprintf('\nUsing Inexact nonsmooth Newton on ')
    fprintf(' A*svec(x)=b, sMat(X) psd with A %i by %i \n',m,n)
    fprintf('%-9s', 'iter')
    fprintf('%-15s', VarPrint{:})
    fprintf('\n')
    
    % Print desirable information
    fprintf('%-9i', k)
    fprintf('%-15e', VarValue{:})
    fprintf('%-15i', lpfdM)
    fprintf('%-15e', 0)
    fprintf('\n')
end



lnormv = log(norm(v)); % to help with regularization
m = length(Fk);
Jac = zeros(m);   % NOT NOT sparse so do not use speye but eye
%Jactemp = zeros(m);

%Im = eye(m);   % unit vectors no longer used/just rows of A
while relresid > tol && k < maxiter  % k # iters

    
    %%%%%%%%%%%%%%%%%%%%%%%%%
    %   Find the Jacobian   %
    %%%%%%%%%%%%%%%%%%%%%%%%%
    
    % Compute some auxiliary inputs (Omega, UJ) for Jacobian computation
    [UJ,lambdas] = eig(vAtyM,'vector');
    % Order the eigenvalues and eigenvectors
    [lambdas, Ord] = sort(lambdas,'descend');  % lambdas = lambdas(n:-1;1)
    UJ = UJ(:,Ord);
    % Build B(\lambda(X)):
    p  = sum(lambdas>0); % number of positive eigenvalues
    
    % Compute B (equals Omega in the Sun paper)
    Btemp = zeros(n);
    for i=1:n
        for j=i:n
            if i <= p && j > p
                Btemp(i,j) = lambdas(i)/(lambdas(i)-lambdas(j));
		%%%?????????Henry???? denominators can be zero???tiny??
		%%%% is B symmetric???? make this more efficient???
            elseif i > p && j <= p
                Btemp(i,j) = lambdas(j)/(lambdas(j)-lambdas(i));
            end
        end
    end
    Btemp = Btemp + triu(Btemp,1)';
    Omega = Btemp+blkdiag(ones(p),zeros(n-p)); 
    
    % Update options for Jacobian computation
    OptionsJac.vAtyM = vAtyM;
    OptionsJac.Omega = Omega;
    OptionsJac.UJ = UJ;
    OptionsJac.p = p;
    
    % Compute each column of Jacobian
    for ii = 1:m
        %Jactemp(:,ii) = JacCol(Im(:,ii),OptionsJac);
        %Jactemp(:,ii) = JacCol(ii,OptionsJac);
        Jac(:,ii) = JacCol(ii,OptionsJac);
    end
    %Jac = Jactemp;
    Jac = (Jac+Jac')/2;   % symmetrized here so no need to symmetrize below
    
    if k < 20  
        normNdirp = normNdir(end);
    else
        normNdirp = prod(normNdir(end-5:end));
    end
    
    %%% Next for saving the eigenvalues after every iteration
    if saveeigs  %for saving the eigenvalues after each iteration
        [~,DJac] = eig(Jac);
        eigenvaluesk(k,:) = diag(DJac);
    end
    %%%
    
    
    % Determine whether to stop the algorithm or regularize
    if m >1
        condestJac = condest(Jac);
        expcondestJac = floor(log10(condestJac));
    elseif m == 1
        condestJac = Jac;
        expcondestJac = -floor(log10(condestJac));
    end
    condestJacs = [condestJacs;condestJac];
    %% The if below is NOT correct; needs fixing
    if expcondestJac - floor(log10(relresid)) > 19 
        %if the order of cond number, then we exit the algorithm
        [UJac,DJac] = eig(Jac);
        [~, index0] = min(diag(DJac)); % index of eigenvector associated to the 0 eigenvalue
        eig0 = UJac(:,index0);
        mmFR= find(abs(eig0)>10^-4); % position constraint affecting to strict feasibility
        output.AFR = sign(eig0(mmFR)).*A(mmFR,:); % position constraint affecting to strict feasibility right symbol
        output.bFR = sign(eig0(mmFR)).*b(mmFR);
        noSlater = true;
        fprintf('\nExiting the algorithm: Jacobian is ill-conditioned;')
        fprintf('log10(cond(Jk))=%d, log10(relresid)=%d\n',expcondestJac,floor(log10(relresid)))
        break % break the while loop
    else
        if condestJac> 1e3*n
            lambdareg = ...
                1e-2*mean([...
                relresid*max(1,log(normNdirp)), ...
                1e-1*relresid*max(1,lnormv), ...  %log(norm(v))
                1e-1*relresid, ...
                ]);
            %Jac = (Jac+lambdareg*speye(m));
            Jac = (Jac+lambdareg*eye(m)); % Henry changed to eye!!NOT sparse
        else
            lambdareg = 0;
        end
    end
    
    
    
    %%%%%%%%%%%%%%%%%%%%%%%%%
    % Find Search Direction %
    %%%%%%%%%%%%%%%%%%%%%%%%%

    %Solve Jac*Ndir = -Fk using Cholesky 
    % Jac = A*derivPKy*A'; % Jacobian m-by-m positive definite
    %Jac  = (Jac + Jac')/2;   symmetrized above!!
    R = chol(Jac);
    Ndir = -R\(R'\Fk);  
    normNdir(k) = norm(Ndir);
    
    
 
        
    %%%%%%%%%%%%%%%%%%%%%%%%%
    %    Iterate Updates    %
    %%%%%%%%%%%%%%%%%%%%%%%%%
    yk = yk + Ndir;   %update
    vAty = v+A'*yk;
    vAtyM = ssMat(vAty,n,sMindn);
    [lpfdM,xkM,zkM] = projK(vAtyM);      
    %xk = ssvec(xkM,svindn);
    xk = [xkM(svindn)*sqrt2; xkM(1:n+1:end)'];
    zk = [zkM(svindn)*sqrt2; zkM(1:n+1:end)'];
    %zk = xk-vAty;  % only xk-vAty is needed; max used to ensure c.s.
    
    
    
    %%%%%%%%%%%%%%%%%%%%%%%%%
    %       Reporting       %
    %%%%%%%%%%%%%%%%%%%%%%%%%
    
    normsx = [normsx;norm(xk)];
    normsz = [normsz;norm(zk)];
    normsy = [normsy;norm(yk)];
    
         
    Fk = A*xk-b;
    relresid = norm(Fk)/normb;
    
    lbnd = (-norm(zk+A'*yk)^2/2)+yk'*bAv-zk'*v; % lower bound
    k = k+1;
    if verbose && mod(k,modprint) == 0 % modprint set at while above
        VarValue = {relresid,lbnd};
        fprintf('%-9i', k)
        fprintf('%-15e', VarValue{:},norm(yk))
        fprintf('%-15i', lpfdM)
        fprintf('%-15e', lambdareg)
        fprintf('\n')
    end
    
end %%% End of while loop

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%



%%% observe the magnitude of iterates upon termination
if verbosePlot
    h=figure(1);
    clf(1)
    plot(normsx,'r')
    hold on
    plot(normsz,'b')
    plot(normsy,'g')
    plotTexOpt = {'FontSize',18,'fontname','times','Interpreter','latex'};
    legend('$||X_k||$','$||Z_k||$','$||y_k||$',plotTexOpt{:},'Location','northwest')
    xlabel('iterations',plotTexOpt{:})
    ylabel('$|| \cdot ||$',plotTexOpt{:})
    title('Convergence Behaviour of $(X_k,y_k,Z_k)$',plotTexOpt{:})
    hold off
    
    set(h,'Units','Inches');
    pos = get(h,'Position');
    set(h,'PaperPositionMode','Auto','PaperUnits','Inches','PaperSize',[pos(3), pos(4)])
    filename = 'ConvXYZ2.pdf';
    print(h,filename,'-dpdf','-r0');
    
    % plot the condition numbers
    figure(2)
    clf(2)
    plot( condestJacs ,'r')
    hold on
    plotTexOpt = {'FontSize',18,'fontname','times','Interpreter','latex'};
    %legend('$||x_k||$','$||z_k||$','$||y_k||$',plotTexOpt{:},'Location','northwest')
    xlabel('iteration',plotTexOpt{:})
    ylabel('condest(Jac)',plotTexOpt{:})
    title('Condition number',plotTexOpt{:})
    hold off
end
if ~verbose || modprint > 0   % print last iteration
    VarValue = {relresid,lbnd};
    VarPrint = {'relres/optcond','lower bnd','||Ndir||',  ...
        '#eig(X)>=','lambda'};
    %'|#pos x_j|','setxor cols','zerosx+z','lambda'};
    %'|#pos x_j|','setxor pos','(act+m)','deg: n-(act+m)'};
    fprintf('\nLast iterstion of nonsmooth Newton proj. alg: \n')
    fprintf('%-9s', 'iter')
    fprintf('%-15s',VarPrint{:})
    fprintf('\n')
    fprintf('%-9i', k)
    fprintf('%-15e', VarValue{:},norm(Ndir))
    fprintf('%-15e', lpfdM);
    fprintf('%-15e', lambdareg)
    fprintf('\n')
    fprintf('\n')
end
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% remove roundoff error
if lpfdM > m  
    [~,indsp] = sort(xk,'descend');
    if xk(indsp(m+1))/norm(xk) < tol
        xk(indsp(m+1:lpfdM)) = 0;
        fprintf('setting %i tiny els of last xk to 0\n',lpfdM-m)
        zk = xk-vAty;
    end
end


% Report the optimality conditions
Xk = ssMat(xk,n,sMindn);
Zk = ssMat(zk,n,sMindn);
if verbose
    fprintf('Report: Optimality Conditions at(Xk,Zk,yk)\n')
    % primal feasibility
    fprintf('primal feasibility: ||A*xk-b||/||b||=%e, min.eig(Xk)=%e\n',...
             norm(A*xk-b)/norm(b), min(eig(Xk)) )
    % primal feasibility
    fprintf('dual feasibility  : ||Xk-Zk-vAty||=%e, min.eig(Zk)=%e\n',...
            norm(zk-(xk-vAty)), min(eig(Zk)) )
            %norm(xk-zk-vAty), min(eig(Zk)) )   % replaced
      % since zk = xk-vAty was used
      if size(A,1) == n && norm(zk-(xk-vAty)) > 1e-10
      	keyboard
      end
    % complementarity
    XkZk = ssMat(xk+zk,n,sMindn);
    strcompl = min(eig(XkZk));  % strict compl.
    compl = trace(Zk*Xk);  %  Xk'*Zk=0 naturally holds due to algo. construction
    fprintf('complementarity   : CS: <Xk,Zk>= %e, Strict CS: min.eig(Xk+Zk) = %e  \n', ...
             compl,strcompl)
    objfn = norm(xk-v)^2/2;
    fprintf('                    rel-gap = %e\n',abs(objfn-lbnd)/(1+objfn))
    fprintf('\n')
end


% Return last iterate
output.xfinal = xk;
output.zfinal =  zk;
output.yfinal = yk;
output.relresid = relresid;
output.Xfinal = Xk;
output.Zfinal =  Zk;
output.lbnd = lbnd;
output.iter = k;
output.mineig = min(eig(Xk));
output.noSlater = noSlater;
output.condestJac = condestJac;
if saveeigs
    output.saveeigs = eigenvaluesk;
end

eigJac = eig(Jac);
TOLsmall = max(size(Jac)) * eps(norm(Jac));

output.condJac = cond(Jac);   % max(eigJac)/min(eigJac);
fprintf('The condition number of the Jacobian at termination: %e \n', ...
         output.condJac)   % max(eigJac)/min(eigJac) )
fprintf('   number of eig.val of Jacobian < %.2e = %d\n', ...
         TOLsmall, sum(eigJac < 10*TOLsmall)  )

%% only report the degeneracy if Xk is close to feasible
relResid = norm(A*xk-b)/norm(b);
if relResid < tol
    isdegen = degenChecker(Xk,A,options);
    fprintf('Evaluating degeneracy of Xk with rel.residual=%e\n',relResid)
    if isdegen
        fprintf('   -> Xk is degenerate')
    else
        fprintf('   -> Xk is nondegenerate and strict feasibility holds\n')
    end
end


% Below is not true: need to consider unboundedness of dual multipliers
% if noSlater
%     fprintf('USER: Conclusion: Slater does not hold. FR needed. \n')
% else
%     fprintf('USER: Conclusion: Slater holds, problem solved. \n')
% end
warning on
end   % end of main inexactsemismooth function


%function y = JacCol(x,optionsafun)
function y = JacCol(ii,optionsafun)
%%%%% computes   A of x   or A adjoint of x
sqrt2 = sqrt(2);
Acell = optionsafun.Acell;
A = optionsafun.A;
n = optionsafun.n;
svindn = optionsafun.svindn;
%sMindn = optionsafun.sMindn;  % no longer needed

Omega = optionsafun.Omega;
U = optionsafun.UJ;
p = optionsafun.p;

%%???????changed Hv 
%Hv = A'*x;  % A' acting on unit vector replaced by ii-th row of A below
%H = ssMat(A(ii,:),n,sMindn);
%H = Acell{ii};

tildeH = U'*Acell{ii}*U;  % tilde J
% Next, the big matrix in the middle of the final product
OmegatildeH = Omega(1:p,p+1:n).*tildeH(1:p,p+1:n);
temp = [tildeH(1:p,1:p) , OmegatildeH;...
    OmegatildeH', zeros(n-p)];

PdH = U*temp*U'; % final directional derivative
%PdH = (PdH+PdH')/2;   % skip symmetrization due to ssvec next line
%PdHv = ssvec(PdH,svindn);
PdHv = [PdH(svindn)*sqrt2; PdH(1:n+1:end)'];
y = A*PdHv;
%y = y+1e-10*x;  % trick pcg with precond.

end




