function [output, Yp, Rp, Zp] = ...
		ADMMsolver(V, n, k, d, Gtmp, Dbar0, params, options)
% [output, Yp, Rp, Zp] = ...
%	ADMMsolver(V, n, k, d, Gtmp, Dbar0, params, options)
    %%%% Solver: DNN relaxation of coloured EDM  size: pts/sets/embed dim n/k/d
%   %INPUT:  V is facial reduction vector
%            problem size: pts/sets/embed-dim are n/k/d
%%           values n,k,d are NOT needed; can use size(V,1) !!!!
%%  if varyingn the n is a ***VECTOR*** OF SIZES
%            Gtmp   logical matrix, gangster indices polyhedral constraints
%            Dbar0   objective function matrix

    %%External functions needed/used
    %   curLB    for lower bound
    %   curUB    for upper bound
    %   simplex_proj  projection onto the simplex (for eig proj.)

    %%%%%%%%%%%%% Initializations:
        %initialization of options
    nko = size(V,1);   % this is   n*k+1; do NOT use n as it may be changing
    if isfield(options,'maxiter')
        maxiter = options.maxiter;
    else
        maxiter = k*(nko)+10^2;
    end
    if isfield(options,'epsilon')
        epsilon = options.epsilon;
    else
        epsilon = 1e-11;
    end
    if isfield(options,'boundCounterMax')
        boundCounterMax = options.boundCounterMax;
    else
        boundCounterMax = 1000;
    end
    if isfield(options,'eta')
        eta = options.eta;
    else
        eta = 1e-10;
    end
    if isfield(options,'verbose')
        verbose = options.verbose;
    else
        verbose = true;
    end
    
        %initialization of parameters
    if isfield(params,'beta')
        beta = params.beta;
    else
        beta = (nko-1)/2;   % nko = n*k+1
    end
    if isfield(params,'gamma')
        gamma = params.gamma;
    else
        gamma = .9;
    end

        %initialization of variables
    Dbar = Dbar0(2:end,2:end);  %passed to bound functions
    iter = 0;
    Inko = speye(nko);   % nko = n*k+1
    PdistY = inf;
    PdistY0 = 0;   % counter for Y not changing
    PdistR = inf;
    normYVRVp = inf;
    KKTres = inf;
    LB = -inf;
    LBCounter = 0;
    UB = inf;
    UBCounter = 0;
    stalling = 0;
    nk = nko-1;
    %nko = nk+1;
    Yc = zeros(nko);
    Zc = zeros(nko);
    Rc = zeros(nko-k); %%Rc = zeros(n*k-k+1);
    if verbose
        fprintf('\n\n====== ADMM algorithm for colourful barycenter; ')
        fprintf(' NEW problem k,nko,d = %i %i %i \n',k,nko,d);
        fprintf('%-8s','iter#');
        fprintf('%-15s','beta','normYVRVp','PdistY','PdistR','Zdistp','Zdistpp')
        fprintf('%-15s','UB','LB','KKTres','rel. gap')
	    fprintf('%-10s','Stall:LB,UB,PY')
        fprintf('\n');
    end

    UBS = [];  % save all UB
    LBS = [];  % save all LB
    relgaps = Inf;
        %Arrow indices for dual variable Z
    %arrowInds = spalloc(nko,nko,3*nk);
    arrowInds = [0 true(1,nk) 
	       true(nk,1) (speye(nk)==1)];
    arrowInds = arrowInds == true;   % position of arrow constr. indices
        %Stopping criterion
    stopcrit = [(iter < maxiter) ...
                (stalling < boundCounterMax) ...
                (KKTres > epsilon) ...
                (relgaps(end) > epsilon)];
    %%%%%%%%%%%%%%%%%%%
    % Adaptive beta parameters
    mu = 2;
    tau_inc = 2;
    tau_dec = 2;
    %%%%%%%%%%%%%%%%%%%
    Vtemp = sparse(V);    % this should be done before the call?????
                            % and only if sparse version is used
    while all(stopcrit)
            %Adapt beta based on gap between primal res:normVRVp and 
            %                                dual res:max(PdistY, PdistR)
	    %%% add upper/lower bounds on beta??? polish??
        if normYVRVp > mu * max(PdistY, PdistR)
            beta = min(tau_inc * beta,10*d*nk);  % adaptive beta
        elseif max(PdistY, PdistR) > mu * normYVRVp
            beta = max(beta / tau_dec,1/(d*nk));
        end
            %Update R:  R = UDU^T
        %M = V'*(Yc+Zc/beta)*V; 
        M = Vtemp'*(Yc+Zc/beta)*Vtemp; 
	M = (M+M')/2; 
        [U, D] = eig(M);
        [dR , ~] = simplex_proj(diag(D),k+1);
        nullR = sum(dR==0);
                %Update by proper eigenvalued eigenvectors
        UR = U(:,nullR+1:end); DR = diag(dR(nullR+1:end));
        Rp = UR*DR*UR'; 
	Rp = (Rp + Rp')/2;
        %VRV = V*Rp*V'; 
        VRV = Vtemp*Rp*Vtemp'; 
	VRV = (VRV+VRV')/2;
        
            %Update Z: 
        Zp = Zc + gamma * beta * (Yc - VRV); Zp = (Zp + Zp')/2; %Why symmetrize?
        Zp(arrowInds) = 0;  

            %Update Y
        Yp = VRV - (Dbar0+Zp)/beta;  Yp = (Yp + Yp')/2; %Why symmetrize?
                %Average arrow entries of Y   
        dYp = (diag(Yp) + Yp(:,1))/2;
        Yp(Inko == 1) = dYp; Yp(:,1) = dYp; Yp(1,:) = dYp;   
                %Gangster projection
        Yp(Gtmp) = 0;
        Yp(1,1) = 1;
                %[0,1]-box projection
        Yp(Yp>1) = 1;
        Yp(Yp<0) = 0;
        
            %Check stalling
        if PdistY0 > min(100,boundCounterMax/2)  % if stalling, then round later
            eta = eta/2;
        end
        
            %Update Z: 
        Zpp = Zp + gamma * beta * (Yp - VRV);  Zpp = (Zpp + Zpp')/2; %Why symmetrize?
        Zpp(arrowInds) = 0;  

            %Bounding: 
                %Lower bound
        if mod(iter,10) == 0  % only every 10th iteration
            LBtemp = curLB(Dbar0, Zpp, Vtemp, Gtmp, k);
            if LBtemp > LB
                LB = LBtemp;
                LBS = [LBS LB];
                LBCounter = 0;
            else
                LBCounter = LBCounter + 1;
            end
        end
                %Upper bound(Williamson-Goemanns) 
        if mod(iter,10) == 0  % only every 10th iteration
            [UBtemp, ~] = curUB(Vtemp, UR, Yp, n, k, Dbar);
            if UBtemp < UB
                UB = UBtemp;
                UBS = [UBS UB];
                UBCounter = 0;
            else
                UBCounter = UBCounter + 1;
            end
        end
    
            %Update iter, PDres, KKTres, relgaps, PDvars, stopCri, stalling
        iter = iter + 1;
                %Presidual
        normYVRVp = norm(Yp-VRV,'fro'); 
                %Dresidual
        PdistY = norm(Yp-Yc, 'fro');
        PdistR = norm(Rp-Rc, 'fro');
                %Dual variable updates distances
        Zdistp = norm(Zc-Zp, 'fro');
        Zdistpp = norm(Zc-Zpp, 'fro');   % total change in dual vrble
                %Stalling
        if PdistY == 0
            PdistY0 = PdistY0 + 1;
        else
            PdistY0 = 0;  % reset to 0
        end
                %relKKT residual
        KKTres = max([PdistR, PdistY, normYVRVp]);
                %Relative gaps
        relgaps = [relgaps 2*(UB-LB)/(abs(UB)+abs(LB)+1)];
        if verbose
            if mod(iter,10) == 0
                fprintf('%-8i',iter);
                fprintf('%-15.2e',beta,normYVRVp,PdistY,PdistR,Zdistp,Zdistpp);
                fprintf('%-15.2e',UB,LB,KKTres,relgaps(end));
                fprintf('%-7i',LBCounter,UBCounter,PdistY0);
                fprintf('\n')
            end
        end
                %Primal dual variables update
        Rc = Rp;
        Yc = Yp;
        Zc = Zpp;
                %Stalling
        stalling = min([UBCounter LBCounter PdistY0]);
                %Stopping criterion
        stopcrit = [(iter < maxiter)...  
                    (stalling < boundCounterMax)...
                    (KKTres > epsilon)...  
                    (relgaps(end) > epsilon)];
    end   % end of while

    %Compute lower and upper bound for the last iterate
    LBtemp = curLB(Dbar0, Zc, Vtemp, Gtmp, k);
    if LBtemp > LB
        LB = LBtemp;
        LBS = [LBS LB];
    end
    [UBtemp, xapprx] = curUB(Vtemp, UR, Yc, n, k, Dbar);
    if UBtemp < UB
        UB = UBtemp;
        UBS = [UBS UB];
    end
        %Printouts
    if verbose
        fprintf('%-8s','iter#');
        fprintf('%-15s','beta','normYVRVp','PdistY','PdistR','Zdistp','Zdistpp')
        fprintf('%-15s','UB','LB','KKTres','rel. gap')
	    fprintf('%-10s','Stall:LB,UB,PY')
        fprintf('\n')
        fprintf('%-8i',iter);
        fprintf('%-15.2e',beta,normYVRVp,PdistY,PdistR,Zdistp,Zdistpp);
        fprintf('%-15.2e',UB,LB,KKTres,relgaps(end));
        fprintf('%-7i',LBCounter,UBCounter,PdistY0);
        fprintf('\n')
    end
    fprintf('\nstopping critera at end of ADMMsolver: \n')
    fprintf('  1-iter=%i 2-stalling=%i 3-KKTres=%g  4-relgaps(end)=%g', ... 
                   iter,stalling,KKTres,relgaps(end))
    fprintf('\n')
    stopind = find(stopcrit == 0,1);
    fprintf(' stopped corresp. to criterion # %i of above\n',stopind)
    fprintf('\n\n')
        %Outputs
    output.iter = iter;
    output.LBS = LBS;
    output.UBS = UBS;
    output.xapprx = xapprx;
    output.relgaps = relgaps;
    output.Y = Yc;
    output.Z = Zc;
    output.R = Rc;
    output.KKTres = KKTres;
    output.stalling = stalling;
    output.stopind = stopind;
    if min(UBS)-max(LBS) > 1e-4
    	fprintf('WARNING: upper bound - lower bound gt 1e-4\n')
    end
end   
