/* PostOrderHeap.cs, Version 1.0, May 1st, 2004.
  
  This file implements the Post-Order Heap and a simple application
  to test it. The implementation favors legibility over efficiency.
  A paper describing the Post-Order Heap was presented at the
  Third International Conference on Fun with Algorithms (FUN 2004),
  Elba, Italy, May 2004. This software is copyrighted under the
  following minor variant of the zlib license.
  
  --------------------------------------------------------------------------  
  Copyright (C) 2003-2004 Nicholas Harvey and Kevin Zatloukal

  This software is provided 'as-is', without any express or implied
  warranty.  In no event will the authors be held liable for any damages
  arising from the use of this software.

  Permission is granted to anyone to use this software for any purpose,
  including commercial applications, and to alter it and redistribute it
  freely, subject to the following restrictions:

  1. The origin of this software must not be misrepresented; you must not
     claim that you wrote the original software. If you use this software
     in a product, an acknowledgment in the product documentation and a
     notification to the authors would be appreciated but is not required.
  2. Altered source versions must be plainly marked as such, and must not be
     misrepresented as being the original software.
  3. This notice may not be removed or altered from any source distribution.

  Nick Harvey <nickh@mit.edu>
  Kevin Zatloukal <kevinz@mit.edu>
  --------------------------------------------------------------------------
*/

using System;
using System.Collections;
using System.Diagnostics;

namespace HeapCode {

    /// <summary>
    /// Implements a Post-Order Heap.
    /// </summary>
    class PostOrderHeap {

        //**********************************************************************************************
        // Private Members
        //**********************************************************************************************

        int         n;          // Number of nodes in heap
        int         H;          // The value of H(n)
        int         D;          // The value of D_n
        ArrayList   nodeArray;  // Array of heap nodes



        //**********************************************************************************************
        // Private Methods
        //**********************************************************************************************

        /// <summary>
        /// Compute 2^x.
        /// </summary>
        int Pow2( int x ) {
            return 1 << x;
        }

        /// <summary>
        /// Get the y'th bit from value x. The LSB is the 0th bit.
        /// </summary>
        int GetBit( int x, int y ) {
            if( (x & Pow2(y)) == 0 ) {
                return 0;
            } else {
                return 1;
            }
        }

        /// <summary>
        /// Set the y'th bit of value x to z. The LSB is the 0th bit.
        /// z should be either 0 or 1.
        /// </summary>
        int SetBit( int x, int y, int z ) {
            if ( z == 0 ) {
                x &= ~( Pow2(y) );
            } else {
                x |= Pow2(y);
            }
            return x;
        }

        /// <summary>
        /// Get index of node i's left child
        /// </summary>
        /// <param name="i">Node of interest</param>
        /// <param name="h">Height of node i</param>
        int LeftChild( int i, int h ) {
            return i - Pow2(h);
        }

        /// <summary>
        /// Get index of node i's right child
        /// </summary>
        /// <param name="i">Node of interest</param>
        /// <param name="h">Height of node i</param>
        int RightChild( int i, int h ) {
            return i - 1;
        }

        /// <summary>
        /// Swap elements at indices a and b.
        /// </summary>
        void SwapElements( int a, int b ) {
            Object temp = nodeArray[a];
            nodeArray[a] = nodeArray[b];
            nodeArray[b] = temp;
        }

        /// <summary>
        /// Restore heap property at node i
        /// </summary>
        /// <param name="i">Node index</param>
        /// <param name="h">Height of node i</param>
        void Heapify( int i, int h ) {
            int l, r, smallest;

            // No need to do anything when we're at the bottom.
            if ( h == 0 ) return;
           
            // Compare node i to its children and find the smallest.
            l = LeftChild( i, h );
            r = RightChild( i, h );
            if ( ((IComparable) nodeArray[l]).CompareTo( nodeArray[i] ) < 0 ) {
                smallest = l;
            } else {
                smallest = i;
            }
            if ( ((IComparable) nodeArray[r]).CompareTo( nodeArray[smallest] ) < 0 )
                smallest = r;

            // If node i is not the smallest, swap it with the smaller child,
            // and recursively heapify from that child.
            if ( smallest != i ) {
                SwapElements( i, smallest );
                Heapify( smallest, h-1 );
            }
        }

        /// <summary>
        /// Compute the size of a tree of height x.
        /// This is the function called s(x) in the paper.
        /// </summary>
        int TreeSize( int x ) {
            return Pow2(x+1)-1;
        }


        //**********************************************************************************************
        // Public Methods
        //**********************************************************************************************

        /// <summary>
        /// Default constructor
        /// </summary>
        public PostOrderHeap() {
            n = H = D = 0;

            // Initialize the node array and add a dummy null element at index 0.
            nodeArray = new ArrayList();
            nodeArray.Add( null );
        }

        /// <summary>
        /// Insert the given object into the heap
        /// </summary>
        public void Insert( IComparable obj ) {
            // First, compute D(n+1) and H(n+1)
            if( n>0 ) {
                if ( GetBit(D, H) == 0 ) {
                    D = SetBit(D, H, 1);
                    H = 0;
                } else {
                    D = SetBit(D, H, 0);
                    H++;
                }
            }
   
            // Add new node to the array and restore heap property
            n++;
            nodeArray.Add( obj );
            Heapify( n, H );
        }
        
        /// <summary>
        /// Remove the minimum node from the heap and return it
        /// </summary>
        public Object DeleteMin() {
            int minNodeLoc=n, minNodeHeight=H;

            // Enumerate all roots and find minimum element
            int x=n-TreeSize(H), h=H-1;
            while( x>0 ) {
                do { h++; } while( GetBit(D,h)==0 );
                if ( ((IComparable) nodeArray[x]).CompareTo(nodeArray[minNodeLoc]) < 0 ) {
                    minNodeLoc = x;
                    minNodeHeight = h;
                }
                x -= TreeSize(h);
            }
               
            // Extract minimum node and restore heap property
            if ( minNodeLoc < n ) {
                SwapElements( minNodeLoc, n );
                Heapify( minNodeLoc, minNodeHeight );
            }

            // Update n, H, and D
            Object minNode = nodeArray[ n ];
            nodeArray.RemoveAt(n); n--;
            if( n>0 ) {
                if( H>0 ) {
                    H--;
                    D = SetBit(D, H, 1);
                } else {
                    while( GetBit(D, H)==0 ) H++;
                    D = SetBit(D, H, 0);
                }
            }
               
            return minNode;
        }
    }


    /// <summary>
    /// An simple application to demonstrate use of the Post-Order Heap.
    /// </summary>
    class TestApp {

        /// <summary>
        /// The main entry point for the application.
        /// </summary>
        [STAThread]
        static void Main(string[] args) {
            TestApp app = new TestApp();
            app.Run();
        }

        /// <summary>
        /// The main method for testing the Post-Order Heap.
        /// </summary>
        void Run() {
            const int NUM_TESTS = 100;
            const int MAX_SIZE = 100000;

            PostOrderHeap heap = new PostOrderHeap();
            Random rng = new Random();
            int i, j;

            for( i=0; i<NUM_TESTS; i++ ) {
                int size = rng.Next( MAX_SIZE );
                Console.WriteLine("Test heap {0}, size = {1}", i, size);

                // Create array of random numbers to insert
                int[] values = new int[ size ];
                for( j=0; j<size; j++ ) {
                    values[j] = j;
                }
                for( j=0; j<size; j++ ) {
                    int r = rng.Next(size-j);
                    int t = values[j];
                    values[j] = values[j+r];
                    values[j+r] = t;
                }

                // Insert numbers into the heap
                for( j=0; j<size; j++ ) {
                    heap.Insert( values[j] );
                }

                // Delete numbers from the heap
                for( j=0; j<size; j++ ) {
                    int x = (int) heap.DeleteMin();
                    if( x != j ) {
                        Console.Write("Error! Got bogus value");
                        Debugger.Break();
                    }
                }
            }

            Console.WriteLine("All tests passed! Press Enter.");
            Console.ReadLine();
        }
    }
}