/****************************************************************************/
/*                                                                          */
/*  This file is part of CONCORDE                                           */
/*                                                                          */
/*  (c) Copyright 1995--1999 by David Applegate, Robert Bixby,              */
/*  Vasek Chvatal, and William Cook                                         */
/*                                                                          */
/*  Permission is granted for academic research use.  For other uses,       */
/*  contact the authors for licensing options.                              */
/*                                                                          */
/*  Use at your own risk.  We make no guarantees about the                  */
/*  correctness or usefulness of this code.                                 */
/*                                                                          */
/****************************************************************************/

/****************************************************************************/
/*                                                                          */
/*                    Interface to the Cutters                              */
/*                                                                          */
/*                           TSP CODE                                       */
/*                                                                          */
/*                                                                          */
/*  Written by:  Applegate, Bixby, Chvatal, and Cook                        */
/*  Date: February 17, 1997                                                 */
/*                                                                          */
/*                                                                          */
/*    EXPORTED FUNCTIONS:                                                   */
/*                                                                          */
/*  int CCtsp_connect_cuts (CCtsp_lpcut_in **cuts, int *cutcount,           */
/*      int ncount, int ecount, int *elist, double *x)                      */
/*    FINDS violated subtour inequalities via connectivity.                 */
/*     -cuts will return any new cuts found (they will be added to the      */
/*      head of the linked list)                                            */
/*     -cutcount will return the number of new cuts added                   */
/*     -ncount is the number of nodes                                       */
/*     -ecount is the number of edges                                       */
/*     -elist contains the LP edges in node node format                     */
/*     -x is an LP solution                                                 */
/*                                                                          */
/*  int CCtsp_segment_cuts (CCtsp_lpcut_in **cuts, int *cutcount,           */
/*      int ncount, int ecount, int *elist, double *x)                      */
/*    FINDS violated subtour inequalities via linsub.                       */
/*                                                                          */
/*  int CCtsp_exact_subtours (CCtsp_lpcut_in **cuts, int *cutcount,         */
/*      int ncount, int ecount, int *elist, double *x)                      */
/*    FINDS violated subtour inequalities via a mincut algorithm.           */
/*                                                                          */
/*  int CCtsp_tighten_lp (CCtsp_lpcuts *cuts, CCtsp_tighten_info *stats,    */
/*      CCtsp_lpcut_in **cutsout, int *cutcount, int ncount,                */
/*      int ecount, int *elist, double *x, double testtol,                  */
/*      int maxcuts, double *viol, CCrandstate *rstate)                     */
/*    CALLS tighten for each cut in the cuts.                               */
/*     -stats contains some running statistics of tighten                   */
/*     -cutsout returns the tightened cuts that are violated (they are      */
/*      added to the tail of the linked list)                               */
/*     -cutcount is the number of cuts in cutsout                           */
/*     -testtol is a tolerance for calling tighten (call only when the      */
/*      cut has slack value within testtol)                                 */
/*     -maxcuts is a bound on the number of cuts to be returned             */
/*                                                                          */
/*  int CCtsp_double_decker_lp (CCtsp_lpcuts *cuts,                         */
/*      CCtsp_tighten_info *stats, CCtsp_lpcut_in **cutsout,                */
/*      int *cutcount, int ncount, int ecount, int *elist, double *x,       */
/*      double testtol, int maxcuts, double *viol, CCrandstate *rstate)     */
/*    MISSING                                                               */
/*                                                                          */
/*  int CCtsp_cliquetree_lp (CCtsp_lpcuts *cuts,                            */
/*      CCtsp_tighten_info *stats, CCtsp_lpcut_in **cutsout,                */
/*      int *cutcount, int ncount, int ecount, int *elist, double *x,       */
/*      double testtol, int maxcuts, double *viol, CCrandstate *rstate)     */
/*    MISSING                                                               */
/*                                                                          */
/*  int CCtsp_star_lp (CCtsp_lpcuts *cuts, CCtsp_tighten_info *stats,       */
/*      CCtsp_lpcut_in **cutsout, int *cutcount, int ncount,                */
/*      int ecount, int *elist, double *x, double testtol,                  */
/*      int maxcuts, double *viol, CCrandstate *rstate)                     */
/*    MISSING                                                               */
/*                                                                          */
/*  int CCtsp_handling_lp (CCtsp_lpcuts *cuts, CCtsp_tighten_info *stats,   */
/*      CCtsp_lpcut_in **cutsout, int *cutcount, int ncount,                */
/*      int ecount, int *elist, double *x, double testtol,                  */
/*      int maxcuts, double *viol, CCrandstate *rstate)                     */
/*    CALLS CCtsp_comb_handling for each comb in cuts.                      */
/*     -agruments as in CCtsp_tighten_lp.                                   */
/*                                                                          */
/*  int CCtsp_handling_lp (CCtsp_lpcuts *cuts, CCtsp_tighten_info *stats,   */
/*      CCtsp_lpcut_in **cutsout, int *cutcount, int ncount,                */
/*      int ecount, int *elist, double *x, double testtol,                  */
/*      int maxcuts, double *viol, CCrandstate *rstate)                     */
/*    MISSING                                                               */
/*                                                                          */
/*  int CCtsp_teething_lp (CCtsp_lpcuts *cuts, CCtsp_tighten_info *stats,   */
/*      CCtsp_lpcut_in **cutsout, int *cutcount, int ncount,                */
/*      int ecount, int *elist, double *x, double testtol,                  */
/*      int maxcuts, double *viol, CCrandstate *rstate)                     */
/*    MISSING                                                               */
/*                                                                          */
/*  int CCtsp_file_cuts (char *cutfile, CCtsp_lpcut_in **cuts,              */
/*      int *cutcount, int ncount, int *tour)                               */
/*    READS a set of cuts from a file; the format of the cuts can be        */
/*     found by examining the code                                          */
/*     -cutfile is an asci file with a list of cuts                         */
/*     -cuts will return any new cuts found (they will be added to the      */
/*      tail of the linked list)                                            */
/*     -cutcount with return the number of new cuts added                   */
/*     -ncount is the number of nodes                                       */
/*     -tour the permutation tour (used to map the incoming nodes)          */
/*                                                                          */
/*  int CCtsp_file_cuts_write (const char *cutfile, CCtsp_lpcuts *cuts,     */
/*      int *tour)                                                          */
/*    WRITES a set of cuts in a text file that can be read by               */
/*     tsp_file_cuts                                                        */
/*     -cutfile is the name of the file to be written                       */
/*     -cuts is the set of cuts to be written                               */
/*     -tour is a permutation tour (used to map the outgoing nodes)         */
/*                                                                          */
/*  int CCtsp_test_pure_comb (int ncount, CCtsp_lpcut_in *c, int *yes_no,   */
/*      int *handle)                                                        */
/*    TEST if the cut is a comb (without flipped teeth or intersections)    */
/*     -ncount is the number of nodes in the TSP                            */
/*     -yes_no will be set to either 0 or 1, with 1 meaning yes             */
/*     -handle with return the index of the handle if the cut is a comb     */
/*      (handle can be NULL)                                                */
/*                                                                          */
/*  int CCtsp_test_pseudocomb (int ncount, CCtsp_lpcut_in *c, int handle,   */
/*      int *yes_no)                                                        */
/*    TEST if the cut is a pseudocomb.                                      */
/*     -handle gives the index of the handle of the pseudocomb              */
/*                                                                          */
/*  int CCtsp_test_teeth_disjoint (int ncount, CCtsp_lpcut_in *c,           */
/*      int handle, int *yes_no)                                            */
/*    TEST if the cliques other than handle are pairwise disjoint.          */
/*     -yes_no is 1 if disjoint and 0 otherwise.                            */
/*                                                                          */
/*  int CCtsp_find_pure_handle (int ncount, CCtsp_lpcut_in *c,              */
/*      int *handle)                                                        */
/*    FINDS a clique that is c's handle if c is a comb; the search          */
/*     assumes that the teeth are disjoint, so if the comb has              */
/*     extra intersections then a tooth may be returned.                    */
/*     -handle returns the potential handle (it will return -1 if no        */
/*      clique is a potential handle)                                       */
/*                                                                          */
/*  int CCtsp_buildcut_begin (CCtsp_cutinfo *cuts, int init_cliquecount)    */
/*    MISSING                                                               */
/*                                                                          */
/*  int CCtsp_buildcut_addclique (CCtsp_cutinfo *cuts, *arr, int size)      */
/*    MISSING                                                               */
/*                                                                          */
/*  int CCtsp_buildcut_finish (CCtsp_cutinfo *cuts, int rhs)                */
/*    MISSING                                                               */
/*                                                                          */
/*  void CCtsp_buildcut_abort (CCtsp_cutinfo *cuts)                         */
/*    MISSING                                                               */
/*                                                                          */
/*  int CCtsp_truncate_cutlist (CCtsp_lpcut_in **cuts, int ncount,          */
/*      int ecount, int *elist, double *x, int maxcuts,                     */
/*      CCrandstate *rstate)                                                */
/*    RETURNS the maxcuts most violated cuts in the linked list, the        */
/*     remaining cuts are freed.                                            */
/*                                                                          */
/****************************************************************************/

#include "machdefs.h"
#include "macrorus.h"
#include "util.h"
#include "tsp.h"
#include "cut.h"
#include "combs.h"
#include "verify.h"

#define X_FLUFF (1e-10)
#undef  DUMP_BUILDCUT

typedef struct exactsub_param {
    int             nodecount;
    int             cutcount;
    CCtsp_lpcut_in *cuts;
} exactsub_param;


static int
    add_segment (double val, int a, int b, void *pass_param),
    add_exact (double val, int count, int *cutarray, void *pass_param),
    work_on_combs_in_lp (CCtsp_lpcuts *cuts, CCtsp_tighten_info *stats,
        CCtsp_lpcut_in **cutsout, int *cutcount, int ncount, int ecount,
        int *elist, double *x, double testtol, int maxcuts, int caller,
        double *viol, CCrandstate *rstate),
    grab_nonzero_x (int ecount, int *elist, double *x, int *new_ecount,
        int **new_elist, double **new_x, double tol);



int CCtsp_connect_cuts (CCtsp_lpcut_in **cuts, int *cutcount, int ncount,
        int ecount, int *elist, double *x)
{
    int rval = 0;
    int i, k, ncomp;
    CCtsp_lpcut_in *c     = (CCtsp_lpcut_in *) NULL;
    int *comps      = (int *) NULL;
    int *compscount = (int *) NULL;

    *cutcount = 0;
    rval = CCcut_connect_components (ncount, ecount, elist, x, &ncomp,
                                     &compscount, &comps);
    if (rval) {
        fprintf (stderr, "CCcut_connect_components failed\n"); goto CLEANUP;
    }

    for (i = 0, k = 0; i < ncomp - 1; k += compscount[i], i++) {
        rval = CCtsp_array_to_subtour (&c, comps + k, compscount[i], ncount);
        if (rval) {
            fprintf (stderr, "CCtsp_array_to_subtour failed\n");
            rval = 1; goto CLEANUP;
        }
        c->next = *cuts;
        *cuts = c;
        (*cutcount)++;
    }

CLEANUP:

    CC_IFFREE (comps, int);
    CC_IFFREE (compscount, int);

    return rval;
}

int CCtsp_segment_cuts (CCtsp_lpcut_in **cuts, int *cutcount, int ncount,
        int ecount, int *elist, double *x)
{
    int rval;
    exactsub_param p;
    int i;
    int *endmark = (int *) NULL;

    *cutcount = 0;

    p.nodecount = ncount;
    p.cutcount  = 0;
    p.cuts      = *cuts;

    endmark = CC_SAFE_MALLOC (ncount, int);
    if (endmark == (int *) NULL) {
        fprintf (stderr, "Out of memory in CCtsp_segment_cuts\n");
        rval = 1; goto CLEANUP;
    }

    for (i=0; i<ncount; i++) {
        endmark[i] = 0;
    }
    for (i=0; i<ecount; i++) {
        if (x[i] >= 0.999999) {
            endmark[elist[2*i]]++;
            endmark[elist[2*i+1]]++;
        }
    }
    for (i=0; i<ncount; i++) {
        if (endmark[i] == 2) {
            endmark[i] = CC_LINSUB_NO_END;
        } else {
            endmark[i] = CC_LINSUB_BOTH_END;
        }
    }

    rval = CCcut_linsub (ncount, ecount, endmark, elist, x, 2.0 - 0.0001,
                         (void *) &p, add_segment);
    if (rval) {
        fprintf (stderr, "CCcut_linsub failed\n");
        goto CLEANUP;
    }

    *cutcount = p.cutcount;
    *cuts = p.cuts;

    rval = 0;

  CLEANUP:
    CC_IFFREE (endmark, int);
    return rval;
}

int CCtsp_shrink_subtours (CCtsp_lpcut_in **cuts, int *cutcount, int ncount,
            int ecount, int *elist, double *x)
{
    int rval;
    exactsub_param p;

    *cutcount = 0;
/*
    rval = CCtsp_connect_cuts (cuts, cutcount, ncount, ecount, elist, x);
    if (rval) {
        fprintf (stderr, "CCtsp_connect_cuts failed\n"); goto CLEANUP;
    }

    if (*cutcount > 0) {
        rval = 0; goto CLEANUP;
    }
*/

    p.nodecount = ncount;
    p.cutcount  = 0;
    p.cuts      = *cuts;

    rval = CCcut_shrink_cuts (ncount, ecount, elist, x, 2.0 - 0.0001,
                       add_exact, (void *) &p);
    if (rval) {
        fprintf (stderr, "CCcut_violated_cuts failed\n"); goto CLEANUP;
    }

    *cutcount = p.cutcount;
    *cuts = p.cuts;

CLEANUP:

    return rval;
}


int CCtsp_exact_subtours (CCtsp_lpcut_in **cuts, int *cutcount, int ncount,
            int ecount, int *elist, double *x)
{
    int rval;
    exactsub_param p;

    *cutcount = 0;
    rval = CCtsp_connect_cuts (cuts, cutcount, ncount, ecount, elist, x);
    if (rval) {
        fprintf (stderr, "CCtsp_connect_cuts failed\n"); goto CLEANUP;
    }

    if (*cutcount > 0) {
        rval = 0; goto CLEANUP;
    }

    p.nodecount = ncount;
    p.cutcount  = 0;
    p.cuts      = *cuts;

    rval = CCcut_violated_cuts (ncount, ecount, elist, x, 2.0 - 0.0001,
                       add_exact, (void *) &p);
    if (rval) {
        fprintf (stderr, "CCcut_violated_cuts failed\n"); goto CLEANUP;
    }

    *cutcount = p.cutcount;
    *cuts = p.cuts;

#if 0
  - this is just to check the values of the exact cuts
    if (*cutcount) {
        CCtsp_lpgraph lg;
        CCtsp_lpcut_in *c;
        double t;

        CCtsp_init_lpgraph_struct (&lg);

        rval = CCtsp_build_lpgraph (&lg, ncount, ecount, elist, (int *) NULL);
        if (rval) {
            fprintf (stderr, "CCtsp_build_lpgraph failed\n"); goto CLEANUP;
        }
        rval = CCtsp_build_lpadj (&lg, 0, ecount);
        if (rval) {
            CCtsp_free_lpgraph (&lg);
            fprintf (stderr, "CCtsp_build_lpadj failed\n"); goto CLEANUP;
        }
        for (c = p.cuts; c; c = c->next) {
            t = CCtsp_cutprice (&lg, c, x);
            printf ("[%f] ", 2.0 + t); fflush (stdout);
        }
        printf ("\n"); fflush (stdout);
        CCtsp_free_lpgraph (&lg);
    }
#endif
    rval = 0;

CLEANUP:

    return rval;
}

static int add_segment (double val, int a, int b, void *pass_param)
{
    int rval = 0;
    exactsub_param *p = (exactsub_param *) pass_param;
    CCtsp_lpcut_in *c = (CCtsp_lpcut_in *) NULL;

    if (val > 2.0) {
        printf ("Warning: Cut of value %f in add_segment\n", val);
        fflush (stdout);
        goto CLEANUP;
    }

    rval = CCtsp_segment_to_subtour (&c, a, b, p->nodecount);
    if (rval) {
        fprintf (stderr, "CCtsp_segment_to_subtour failed\n");
        rval = 1; goto CLEANUP;
    }
    c->next = p->cuts;
    p->cuts = c;
    p->cutcount++;

CLEANUP:

    return rval;
}

static int add_exact (double val, int count, int *cutarray, void *pass_param)
{
    int rval = 0;
    exactsub_param *p = (exactsub_param *) pass_param;
    CCtsp_lpcut_in *c = (CCtsp_lpcut_in *) NULL;

    if (count >= p->nodecount) goto CLEANUP;

    if (val > 2.0) {
        printf ("Warning: Cut of value %f in add_exact\n", val);
        fflush (stdout);
        goto CLEANUP;
    }

    rval = CCtsp_array_to_subtour (&c, cutarray, count, p->nodecount);
    if (rval) {
        fprintf (stderr, "CCtsp_array_to_subtour failed\n");
        rval = 1; goto CLEANUP;
    }
    c->next = p->cuts;
    p->cuts = c;
    p->cutcount++;

CLEANUP:

    return rval;
}

int CCtsp_tighten_lp (CCtsp_lpcuts *cuts, CCtsp_tighten_info *stats,
        CCtsp_lpcut_in **cutsout, int *cutcount, int ncount, int ecount,
        int *elist, double *x, double testtol, int maxcuts,
        double *viol, CCrandstate *rstate)
{
    CCtsp_lpcut_in new, old;
    CCtsp_lpcut_in *c;
    int i;
    int rval = 0;
    double improve;
    CCtsp_lpgraph lg;
    double *newx = (double *) NULL;
    int *newelist = (int *) NULL;
    int newecount;
    CCtsp_lpcut_in **clist = (CCtsp_lpcut_in **) NULL;
    double *vlist = (double *) NULL;
    double maxviol = 0.0;
    int clistsize = 0, vlistsize = 0;
    int count = 0;
    int *perm = (int *) NULL;
    double *cutval = (double *) NULL;

    *cutcount = 0;
    if (!cuts || !cuts->cutcount) return 0;


    rval = grab_nonzero_x (ecount, elist, x, &newecount, &newelist, &newx,
                           X_FLUFF);
    if (rval) {
        fprintf (stderr, "grab_nonzero_x failed\n"); goto CLEANUP;
    }

    cutval = CC_SAFE_MALLOC (cuts->cutcount, double);
    if (!cutval) {
        fprintf (stderr, "out of memory in CCtsp_tighten_lp\n");
        rval = 1; goto CLEANUP;
    }
    rval = CCtsp_price_cuts (cuts, ncount, newecount, newelist, newx, cutval);
    if (rval) {
        fprintf (stderr, "CCtsp_price_cuts failed\n"); goto CLEANUP;
    }

    CCtsp_init_lpgraph_struct (&lg);

    rval = CCtsp_build_lpgraph (&lg, ncount, newecount, newelist,
                                (int *) NULL);
    if (rval) {
        fprintf (stderr, "CCtsp_build_lpgraph failed\n"); goto CLEANUP;
    }
    CC_FREE (newelist, int);
    rval = CCtsp_build_lpadj (&lg, 0, newecount);
    if (rval) {
        fprintf (stderr, "CCtsp_build_lpadj failed\n"); goto CLEANUP;
    }

    for (i = 0; i < cuts->cutcount; i++) {
        if (cutval[i] < testtol && !cuts->cuts[i].branch
            && (cuts->cuts[i].cliquecount > 1 || cutval[i] < 0.1*testtol)
            /* && cuts->cuts[i].age < 3 */) {
            rval = CCtsp_lpcut_to_lpcut_in (cuts, &(cuts->cuts[i]), &old);
            if (rval) {
                fprintf (stderr, "CCtsp_lpcut_to_lpcut_in failed\n");
                goto CLEANUP;
            }
            rval = CCtsp_tighten_lpcut_in (&lg, &old, newx, &new, stats,
                                           &improve);
            if (rval) {
                fprintf (stderr, "CCtsp_tighten_lpcut failed\n");
                goto CLEANUP;
            }
            CCtsp_free_lpcut_in (&old);

            if (improve - cutval[i] > CCtsp_MIN_VIOL) {
                c = CC_SAFE_MALLOC (1, CCtsp_lpcut_in);
                if (!c) {
                    fprintf (stderr, "out of memory in CCtsp_tighten_lp\n");
                    rval = 1; goto CLEANUP;
                }
                *c = new;
                if (count >= clistsize) {
                    rval = CCutil_reallocrus_scale ((void **) &clist,
                                &clistsize, count + 1, 1.3,
                                sizeof (CCtsp_lpcut_in *));
                    if (rval) {
                        fprintf (stderr, "CCutil_reallocrus_scale failed\n");
                        rval = 1; goto CLEANUP;
                    }
                }
                if (count >= vlistsize) {
                    rval = CCutil_reallocrus_scale ((void **) &vlist,
                                &vlistsize, count + 1, 1.3, sizeof (double));
                    if (rval) {
                        fprintf (stderr, "CCutil_reallocrus_scale failed\n");
                        rval = 1; goto CLEANUP;
                    }
                }
                clist[count] = c;
                vlist[count] = cutval[i] - improve;
                count++;
            } else {
                CCtsp_free_lpcut_in (&new);
            }
        }
    }

    if (count) {
        perm = CC_SAFE_MALLOC (count, int);
        if (!perm) {
            fprintf (stderr, "out of memory in CCtsp_tighten_lp\n");
            rval = 1; goto CLEANUP;
        }
        for (i = 0; i < count; i++) {
            perm[i] = i;
        }
        if (count > maxcuts) {
            CCutil_rselect (perm, 0, count - 1, maxcuts, vlist, rstate);
            for (i = maxcuts; i < count; i++) {
                CCtsp_free_lpcut_in (clist[perm[i]]);
            }
            count = maxcuts;
        }
        for (i = 0; i < count; i++) {
            if (vlist[perm[i]] < maxviol)
                maxviol = vlist[perm[i]];
            clist[perm[i]]->next = *cutsout;
            *cutsout = clist[perm[i]];
        }
    }

    *cutcount = count;
    if (viol) *viol = -maxviol;

CLEANUP:

    CC_IFFREE (newelist, int);
    CC_IFFREE (newx, double);
    CC_IFFREE (clist, CCtsp_lpcut_in *);
    CC_IFFREE (vlist, double);
    CC_IFFREE (perm, int);
    CC_IFFREE (cutval, double);
    CCtsp_free_lpgraph (&lg);
    return rval;
}

#define CALL_TEETHING   1
#define CALL_DDECKER    2
#define CALL_CLIQUETREE 3
#define CALL_STAR       4
#define CALL_HANDLING   5

int CCtsp_double_decker_lp (CCtsp_lpcuts *cuts, CCtsp_tighten_info *stats,
        CCtsp_lpcut_in **cutsout, int *cutcount, int ncount, int ecount,
        int *elist, double *x, double testtol, int maxcuts,
        double *viol, CCrandstate *rstate)
{
    int rval = 0;

    rval = work_on_combs_in_lp (cuts, stats, cutsout, cutcount, ncount, ecount,
                elist, x, testtol, maxcuts, CALL_DDECKER, viol, rstate);
    if (rval) {
        fprintf (stderr, "work_on_combs_in_lp failed\n");
        goto CLEANUP;
    }

CLEANUP:

    return rval;
}

int CCtsp_teething_lp (CCtsp_lpcuts *cuts, CCtsp_tighten_info *stats,
        CCtsp_lpcut_in **cutsout, int *cutcount, int ncount, int ecount,
        int *elist, double *x, double testtol, int maxcuts,
        double *viol, CCrandstate *rstate)
{
    int rval = 0;

    rval = work_on_combs_in_lp (cuts, stats, cutsout, cutcount, ncount, ecount,
                elist, x, testtol, maxcuts, CALL_TEETHING, viol, rstate);
    if (rval) {
        fprintf (stderr, "work_on_combs_in_lp failed\n");
        goto CLEANUP;
    }

CLEANUP:

    return rval;
}

int CCtsp_cliquetree_lp (CCtsp_lpcuts *cuts, CCtsp_tighten_info *stats,
        CCtsp_lpcut_in **cutsout, int *cutcount, int ncount, int ecount,
        int *elist, double *x, double testtol, int maxcuts,
        double *viol, CCrandstate *rstate)
{
    int rval = 0;

    rval = work_on_combs_in_lp (cuts, stats, cutsout, cutcount, ncount, ecount,
                elist, x, testtol, maxcuts, CALL_CLIQUETREE, viol, rstate);
    if (rval) {
        fprintf (stderr, "work_on_combs_in_lp failed\n");
        goto CLEANUP;
    }

CLEANUP:

    return rval;
}

int CCtsp_star_lp (CCtsp_lpcuts *cuts, CCtsp_tighten_info *stats,
        CCtsp_lpcut_in **cutsout, int *cutcount, int ncount, int ecount,
        int *elist, double *x, double testtol, int maxcuts,
        double *viol, CCrandstate *rstate)
{
    int rval = 0;

    rval = work_on_combs_in_lp (cuts, stats, cutsout, cutcount, ncount, ecount,
                elist, x, testtol, maxcuts, CALL_STAR, viol, rstate);
    if (rval) {
        fprintf (stderr, "work_on_combs_in_lp failed\n");
        goto CLEANUP;
    }

CLEANUP:

    return rval;
}

int CCtsp_handling_lp (CCtsp_lpcuts *cuts, CCtsp_tighten_info *stats,
        CCtsp_lpcut_in **cutsout, int *cutcount, int ncount, int ecount,
        int *elist, double *x, double testtol, int maxcuts,
        double *viol, CCrandstate *rstate)
{
    int rval = 0;

    rval = work_on_combs_in_lp (cuts, stats, cutsout, cutcount, ncount, ecount,
                elist, x, testtol, maxcuts, CALL_HANDLING, viol, rstate);
    if (rval) {
        fprintf (stderr, "work_on_combs_in_lp failed\n");
        goto CLEANUP;
    }

CLEANUP:

    return rval;
}

static int work_on_combs_in_lp (CCtsp_lpcuts *cuts, CCtsp_tighten_info *stats,
        CCtsp_lpcut_in **cutsout, int *cutcount, int ncount, int ecount,
        int *elist, double *x, double testtol, int maxcuts, int caller,
        double *viol, CCrandstate *rstate)
{
    CCtsp_lpcut_in new, old;
    CCtsp_lpcut_in *c  = (CCtsp_lpcut_in *) NULL;
    CCtsp_lpcut_in *dd = (CCtsp_lpcut_in *) NULL;
    CCtsp_lpcut_in *ddnext;
    int i, test;
    int rval = 0;
    double improve, newslack, dslack;
    CCtsp_lpgraph lg;
    CC_GCgraph gg;
    double *newx = (double *) NULL;
    int *newelist = (int *) NULL;
    int newecount;
    CCtsp_lpcut_in **clist = (CCtsp_lpcut_in **) NULL;
    double *vlist = (double *) NULL;
    double maxviol = 0.0;
    int clistsize = 0, vlistsize = 0;
    int count = 0;
    int *perm = (int *) NULL;
    double *cutval = (double *) NULL;

    *cutcount = 0;
    if (!cuts || !cuts->cutcount) return 0;

    CCtsp_init_lpgraph_struct (&lg);
    CCcombs_GC_init_graph (&gg);

    rval = grab_nonzero_x (ecount, elist, x, &newecount, &newelist, &newx,
                           X_FLUFF);
    if (rval) {
        fprintf (stderr, "grab_nonzero_x failed\n"); goto CLEANUP;
    }

    cutval = CC_SAFE_MALLOC (cuts->cutcount, double);
    if (!cutval) {
        fprintf (stderr, "out of memory in CCtsp_tighten_lp\n");
        rval = 1; goto CLEANUP;
    }
    rval = CCtsp_price_cuts (cuts, ncount, newecount, newelist, newx, cutval);
    if (rval) {
        fprintf (stderr, "CCtsp_price_cuts failed\n"); goto CLEANUP;
    }

    rval = CCtsp_build_lpgraph (&lg, ncount, newecount, newelist,
                                (int *) NULL);
    if (rval) {
        fprintf (stderr, "CCtsp_build_lpgraph failed\n"); goto CLEANUP;
    }
    if (caller == CALL_DDECKER || caller == CALL_CLIQUETREE ||
        caller == CALL_STAR    || caller == CALL_HANDLING) {
        rval = CCcombs_GC_build_graph (&gg, ncount, newecount, newelist, newx);
        if (rval) {
            fprintf (stderr, "CCcombs_GC_build_graph failed\n"); goto CLEANUP;
        }
    }

    CC_FREE (newelist, int);
    rval = CCtsp_build_lpadj (&lg, 0, newecount);
    if (rval) {
        fprintf (stderr, "CCtsp_build_lpadj failed\n"); goto CLEANUP;
    }

    for (i = 0; i < cuts->cutcount; i++) {
        if (cuts->cuts[i].branch || cuts->cuts[i].cliquecount % 2 ||
            cuts->cuts[i].cliquecount < 4 || cutval[i] >= testtol) {
            continue;
        }
        rval = CCtsp_lpcut_to_lpcut_in (cuts, &(cuts->cuts[i]), &old);
        if (rval) {
            fprintf (stderr, "CCtsp_lpcut_to_lpcut_in failed\n"); goto CLEANUP;
        }
        rval = CCtsp_test_pure_comb (ncount, &old, &test, (int *) NULL);
        if (rval) {
            fprintf (stderr, "CCtsp_test_pure_comb failed\n");
            CCtsp_free_lpcut_in (&old);
            goto CLEANUP;
        }
        if (test == 1) {
            switch (caller) {
            case CALL_TEETHING:
                rval = CCtsp_teething (&lg, newx, &old, &dd);
                if (rval) {
                    fprintf (stderr, "CCtsp_teething failed\n"); goto CLEANUP;
                }
                break;
            case CALL_DDECKER:
                rval = CCtsp_comb_to_double_decker (&lg, &gg, newx, &old, &dd);
                if (rval) {
                    fprintf (stderr, "CCtsp_comb_to_double_decker failed\n");
                    goto CLEANUP;
                }
                break;
            case CALL_CLIQUETREE:
                rval = CCtsp_comb_to_cliquetree (&lg, &gg, newx, &old, &dd);
                if (rval) {
                    fprintf (stderr, "CCtsp_comb_to_cliquetree failed\n");
                    goto CLEANUP;
                }
                break;
            case CALL_STAR:
                rval = CCtsp_comb_to_star (&lg, &gg, newx, &old, &dd);
                if (rval) {
                    fprintf (stderr, "CCtsp_comb_to_star failed\n");
                    goto CLEANUP;
                }
                break;
            case CALL_HANDLING:
                rval = CCtsp_comb_handling (&lg, &gg, newx, &old, &dd);
                if (rval) {
                    fprintf (stderr, "CCtsp_comb_handling failed\n");
                    goto CLEANUP;
                }
                break;
            default:
                fprintf (stderr, "unknown caller in work_on_combs_in_lp\n");
                rval = 1; goto CLEANUP;
            }

            CCtsp_free_lpcut_in (&old);
            while (dd) {
                ddnext = dd->next;
                dslack = CCtsp_cutprice (&lg, dd, newx);
                if (dslack >= 1.0) {
                    CCtsp_free_lpcut_in (dd);
                    CC_FREE (dd, CCtsp_lpcut_in);
                } else {
                    rval = CCtsp_tighten_lpcut_in (&lg, dd, newx, &new,
                                                   stats, &improve);
                    if (rval) {
                        fprintf (stderr, "CCtsp_tighten_lpcut failed\n");
                        goto CLEANUP;
                    }
                    CCtsp_free_lpcut_in (dd);
                    CC_FREE (dd, CCtsp_lpcut_in);

                    newslack = dslack - 2.0*improve;
                    if (-newslack > CCtsp_MIN_VIOL) {
                        c = CC_SAFE_MALLOC (1, CCtsp_lpcut_in);
                        if (!c) {
                            fprintf (stderr,
                               "out of memory in work_on_combs_in_lp\n");
                            CCtsp_free_lpcut_in (&new);
                            rval = 1; goto CLEANUP;
                        }
                        *c = new;
                        if (count >= clistsize) {
                            rval = CCutil_reallocrus_scale ((void **) &clist,
                                    &clistsize, count + 1, 1.3,
                                    sizeof (CCtsp_lpcut_in *));
                            if (rval) {
                                fprintf (stderr,
                                    "CCutil_reallocrus_scale failed\n");
                                rval = 1; goto CLEANUP;
                            }
                        }
                        if (count >= vlistsize) {
                            rval = CCutil_reallocrus_scale ((void **) &vlist,
                                     &vlistsize, count + 1, 1.3,
                                     sizeof (double));
                            if (rval) {
                                fprintf (stderr,
                                    "CCutil_reallocrus_scale failed\n");
                                rval = 1; goto CLEANUP;
                            }
                        }
                        clist[count] = c;
                        vlist[count] = newslack;
                        count++;
                    } else {
                        CCtsp_free_lpcut_in (&new);
                    }
                }
                dd = ddnext;
            }
        } else {
            CCtsp_free_lpcut_in (&old);
        }
    }

    if (count) {
        perm = CC_SAFE_MALLOC (count, int);
        if (!perm) {
            fprintf (stderr, "out of memory in work_on_combs_in_lp\n");
            rval = 1; goto CLEANUP;
        }
        for (i = 0; i < count; i++) {
            perm[i] = i;
        }
        if (count > maxcuts) {
            CCutil_rselect (perm, 0, count - 1, maxcuts, vlist, rstate);
            for (i = maxcuts; i < count; i++) {
                CCtsp_free_lpcut_in (clist[perm[i]]);
            }
            count = maxcuts;
        }
        for (i = 0; i < count; i++) {
            if (vlist[perm[i]] < maxviol)
                maxviol = vlist[perm[i]];
            clist[perm[i]]->next = *cutsout;
            *cutsout = clist[perm[i]];
        }
    }

    *cutcount = count;
    if (viol) *viol = -maxviol;

CLEANUP:

    CC_IFFREE (newelist, int);
    CC_IFFREE (newx, double);
    CC_IFFREE (clist, CCtsp_lpcut_in *);
    CC_IFFREE (vlist, double);
    CC_IFFREE (perm, int);
    CC_IFFREE (cutval, double);
    CCtsp_free_lpgraph (&lg);
    CCcombs_GC_free_graph (&gg);
    if (dd) {
        CCtsp_free_lpcut_in (dd);
    }
    return rval;
}

int CCtsp_file_cuts (char *cutfile, CCtsp_lpcut_in **cuts, int *cutcount,
        int ncount, int *tour)
{
    FILE *in = (FILE *) NULL;
    int *inv = (int *) NULL;
    CCtsp_lpcut_in *c;
    CCtsp_lpcut_in **clast;
    int i, j, k;
    int ncliques, size;
    int *icliq = (int *) NULL;
    int rval = 0;

    *cutcount = 0;

    in = fopen (cutfile, "r");
    if  (in == (FILE *) NULL) {
        fprintf (stderr, "unable to open %s for reading\n", cutfile);
        return 0;
    }

    inv = CC_SAFE_MALLOC (ncount, int);
    if (!inv) {
        fprintf (stderr, "out of memory in CCtsp_file_cuts\n");
        rval = 1; goto CLEANUP;
    }
    for (i = 0; i < ncount; i++) {
        inv[tour[i]] = i;
    }

    clast = cuts;
    while ((*clast) != (CCtsp_lpcut_in *) NULL) {
        clast = &((*clast)->next);
    }
    
    while (fscanf (in, "%d", &ncliques) != EOF) {
        c = CC_SAFE_MALLOC (1, CCtsp_lpcut_in);
        if (!c) {
            fprintf (stderr, "out of memory in CCtsp_file_cuts\n");
            rval = 1; goto CLEANUP;
        }
        CCtsp_init_lpcut_in (c);
        c->cliquecount = ncliques;
        c->cliques = CC_SAFE_MALLOC (ncliques, CCtsp_lpclique);
        if (!c->cliques) {
            fprintf (stderr, "out of memory in CCtsp_file_cuts\n");
            rval = 1; goto CLEANUP;
        }
        for (i = 0; i < ncliques; i++) {
            fscanf (in, "%d", &size);
            icliq = CC_SAFE_MALLOC (size, int);
            if (!icliq) {
                fprintf (stderr, "out of memory in CCtsp_file_cuts\n");
                rval = 1; goto CLEANUP;
            }
            for (j = 0; j < size; j++) {
                fscanf (in, "%d", &k);
                icliq[j] = inv[k];
            }
            rval = CCtsp_array_to_lpclique (icliq, size, &(c->cliques[i]));
            if (rval) {
                fprintf (stderr, "CCtsp_array_to_lpclique failed\n");
                goto CLEANUP;
            }
            CC_FREE (icliq, int);
        }
        fscanf (in, "%d", &(c->rhs));
        c->sense = 'G';
        c->branch = 0;
        rval = CCtsp_construct_skeleton (c, ncount);
        if (rval) {
            fprintf (stderr, "CCtsp_construct_skeleton failed\n");
            goto CLEANUP;
        }

        (*clast) = c;
        c->next = (CCtsp_lpcut_in *) NULL;
        clast = &(c->next);
        (*cutcount)++;
#if 1
/*      printf ("File cut:\n");
        CCtsp_print_lpcut_in (c);*/
        rval = CCverify_cut (c, CC_TYPE_ALL, &i);
        if (rval) {
            fprintf (stderr, "Invalid file cut\n");
            goto CLEANUP;
        } else {
            printf ("File cut type %d\n", i);
        }
#endif
    }

CLEANUP:

    CC_IFFREE (inv, int);
    fclose (in);
    return  rval;
}

int CCtsp_file_cuts_write (const char *cutfile, CCtsp_lpcuts *cuts, int *tour)
{
    FILE *out = (FILE *) NULL;
    int i, j, k, p;
    int cutcount = cuts->cutcount;
    CCtsp_lpcut *c;
    CCtsp_lpclique *cl;
    int isize;

    out = fopen (cutfile, "w");
    if  (out == (FILE *) NULL) {
        fprintf (stderr, "unable to open %s for writing\n", cutfile);
        return 1;
    }

    for (i = 0; i < cutcount; i++) {
        c = &cuts->cuts[i];
        if (!c->branch) {
            fprintf (out, "%d\n", c->cliquecount);
            for (j = 0; j < c->cliquecount; j++) {
                cl = &cuts->cliques[c->cliques[j]];
                for (k = 0, isize = 0; k < cl->segcount; k++) {
                    isize += (cl->nodes[k].hi - cl->nodes[k].lo + 1);
                }
                fprintf (out, "%d  ", isize);
                CC_FOREACH_NODE_IN_CLIQUE (p, *cl, k) {
                    fprintf (out, "%d ", tour[p]);
                }
                fprintf (out, "\n");
            }
            fprintf (out, "%d\n", c->rhs);
        }
    }

    fclose (out);
    return 0;
}

int CCtsp_buildcut_begin (CCtsp_cutinfo *cuts, int init_cliquecount)
{
    cuts->current = CC_SAFE_MALLOC (1, CCtsp_lpcut_in);
    if (!cuts->current) return -1;
    CCtsp_init_lpcut_in (cuts->current);
    cuts->current->cliques = CC_SAFE_MALLOC (init_cliquecount, CCtsp_lpclique);
    if (!cuts->current->cliques) {
        CC_FREE (cuts->current, CCtsp_lpcut_in);
        return -1;
    }
    return 0;
}

int CCtsp_buildcut_addclique (CCtsp_cutinfo *cuts, int *arr, int size)
{
    int i;
    int *newarr = (int *) NULL;
    int newsize;
    int rval;
    CCtsp_lpcut_in *c = cuts->current;

    if (!c) {
        fprintf (stderr, "Trying to add to nonexistent clique\n");
        return -1;
    }

    rval = CCcut_SRK_expand (&cuts->expand, arr, size, &newarr, &newsize);
    if (rval) {
        fprintf (stderr, "CCcut_SRK_expand failed\n");
        CCtsp_buildcut_abort (cuts);
        return rval;
    }

    rval = CCutil_reallocrus_count ((void **) &(c->cliques), c->cliquecount+1,
                             sizeof (c->cliques[0]));
    if (rval) {
        fprintf (stderr, "couldn't realloc cliques\n");
        CC_IFFREE (newarr, int);
        CCtsp_buildcut_abort (cuts);
        return rval;
    }
    
    i = c->cliquecount;

    rval = CCtsp_array_to_lpclique (newarr, newsize, &(c->cliques[i]));
    if (rval) {
        fprintf (stderr, "CCtsp_array_to_lpclique failed\n");
        CC_IFFREE (newarr, int);
        CCtsp_buildcut_abort (cuts);
        return rval;
    }
    c->cliquecount++;
    CC_IFFREE (newarr, int);
    return 0;
}

void CCtsp_buildcut_abort (CCtsp_cutinfo *cuts)
{
    CCtsp_free_lpcut_in (cuts->current);
    CC_IFFREE (cuts->current, CCtsp_lpcut_in);
}

int CCtsp_buildcut_finish (CCtsp_cutinfo *cuts, int rhs)
{
    CCtsp_lpcut_in *c = cuts->current;
    int rval;

#ifdef DUMP_BUILDCUT
    {
        int i, j, tmp;
        printf ("new buildcut (%d):", c->cliquecount);
        for (i=0; i<c->cliquecount; i++) {
            printf (" (");
            CC_FOREACH_NODE_IN_CLIQUE (j, c->cliques[i], tmp) {
                printf ("%d ",j);
            }
            printf (")");
        }
        printf (" >= %d\n", rhs);
        fflush (stdout);
    }
#endif

    c->rhs = rhs;
    c->sense = 'G';
    c->branch = 0;

    rval = CCtsp_construct_skeleton (c,
            CCcut_SRK_original_ncount (&cuts->expand));
    if (rval) {
        fprintf (stderr, "CCtsp_construct_skeleton failed\n");
        goto CLEANUP;
    }

    c->next = *cuts->clist;
    (*cuts->clist) = c;
    cuts->current = (CCtsp_lpcut_in *) NULL;
    (*cuts->cutcount)++;

    rval = 0;
 CLEANUP:
    if (rval) {
        CCtsp_free_lpcut_in (c);
    }
    return rval;
}

static int grab_nonzero_x (int ecount, int *elist, double *x, int *new_ecount,
        int **new_elist, double **new_x, double tol)
{
    int i;
    int count;

    *new_ecount = 0;
    *new_elist = (int *) NULL;
    *new_x = (double *) NULL;

    for (i = 0, count = 0; i < ecount; i++) {
        if (x[i] > tol) {
            count++;
        }
    }

    *new_elist = CC_SAFE_MALLOC (2*count, int);
    *new_x = CC_SAFE_MALLOC (count, double);
    if (!(*new_elist) || !(*new_x)) {
        fprintf (stderr, "out of memory in grab_nonzero_x\n");
        CC_IFFREE (*new_elist, int);
        CC_IFFREE (*new_x, double);
        return 1;
    }

    for (i = 0, count = 0; i < ecount; i++) {
        if (x[i] > tol) {
            (*new_elist)[2*count] = elist[2*i];
            (*new_elist)[2*count+1] = elist[2*i+1];
            (*new_x)[count] = x[i];
            count++;
        }
    }
    *new_ecount = count;

    return 0;
}

int CCtsp_test_pure_comb (int ncount, CCtsp_lpcut_in *c, int *yes_no,
        int *handle)
{
    int rval = 0;
    int i, marked, ihandle;
    int *marks = (int *) NULL;

    *yes_no = 0;
    if (handle) *handle = -1;

    if (c->cliquecount < 4 || c->cliquecount % 2 ||
        c->sense != 'G') {
        goto CLEANUP;
    }

    rval = CCtsp_find_pure_handle (ncount, c, &ihandle);
    if (rval) {
        fprintf (stderr, "CCtsp_find_pure_handle failed\n");
        goto CLEANUP;
    }
    if (ihandle == -1) goto CLEANUP;

    marks = CC_SAFE_MALLOC (ncount, int);
    if (!marks) {
        fprintf (stderr, "out of memory in CCtsp_test_pure_comb\n");
        rval = 1; goto CLEANUP;
    }
    CCtsp_mark_cut (c, marks, 0);

    CCtsp_mark_clique (&c->cliques[ihandle], marks, 1);
    for (i = 0; i < c->cliquecount; i++) {
        if (i != ihandle) {
            CCtsp_is_clique_marked (&c->cliques[i], marks, 1, &marked);
            if (!marked) goto CLEANUP;
            CCtsp_is_clique_marked (&c->cliques[i], marks, 0, &marked);
            if (!marked) goto CLEANUP;
        }
    }
    CCtsp_mark_clique (&c->cliques[ihandle], marks, 0);

    for (i = 0; i < c->cliquecount; i++) {
        if (i != ihandle) {
            CCtsp_is_clique_marked (&c->cliques[i], marks, 1, &marked);
            if (marked) goto CLEANUP;
            CCtsp_mark_clique (&c->cliques[i], marks, 1);
        }
    }

    *yes_no = 1;
    if (handle) *handle = ihandle;

CLEANUP:

    CC_IFFREE (marks, int);
    return rval;
}

int CCtsp_test_pseudocomb (int ncount, CCtsp_lpcut_in *c, int handle,
        int *yes_no)
{
    int rval = 0;
    int i, k, marked;
    int *ends = (int *) NULL;
    int *marks = (int *) NULL;

    *yes_no = 0;
    if (c->cliquecount <= 1 || c->cliquecount % 2 || c->sense != 'G') {
        printf ("bad cliquecount or sense in pseudocomb\n"); fflush (stdout);
        goto CLEANUP;
    }

    marks = CC_SAFE_MALLOC (ncount, int);
    if (!marks) {
        fprintf (stderr, "out of memory in CCtsp_test_pseudocomb\n");
        rval = 1; goto CLEANUP;
    }
    CCtsp_mark_cut (c, marks, 0);

    /* Teeth intersect H and are not contained in H */

    CCtsp_mark_clique (&c->cliques[handle], marks, 1);
    for (i = 0; i < c->cliquecount; i++) {
        if (i != handle) {
            CCtsp_is_clique_marked (&c->cliques[i], marks, 1, &marked);
            if (!marked) goto CLEANUP;
            CCtsp_is_clique_marked (&c->cliques[i], marks, 0, &marked);
            if (!marked) goto CLEANUP;
        }
    }
    CCtsp_mark_clique (&c->cliques[0], marks, 0);

    /* Big teeth are pairwise disjoint */

    for (i = 0; i < c->cliquecount; i++) {
        if (i != handle) {
            CCtsp_clique_count (&c->cliques[i], &k);
            if (k >= 3) {
                CCtsp_is_clique_marked (&c->cliques[i], marks, 1, &marked);
                if (marked) goto CLEANUP;
                CCtsp_mark_clique (&c->cliques[i], marks, 1);
            }
        }
    }
    for (i = 1; i < c->cliquecount; i++) {
        CCtsp_mark_clique (&c->cliques[i], marks, 0);
    }

    /* No small tooth is contained in a big tooth */

    for (i = 0; i < c->cliquecount; i++) {
        if (i != handle) {
            CCtsp_clique_count (&c->cliques[i], &k);
            if (k >= 3) {
                CCtsp_mark_clique (&c->cliques[i], marks, i + 1);
            }
        }
    }
    for (i = 0; i < c->cliquecount; i++) {
        if (i != handle) {
            CCtsp_clique_count (&c->cliques[i], &k);
            if (k < 3) {
                rval = CCtsp_clique_to_array (&c->cliques[i], &ends, &k);
                if (rval) {
                    fprintf (stderr, "CCtsp_clique_to_array failed\n");
                    goto CLEANUP;
                }
                if (ends[0] != 0 && ends[0] == ends[1]) goto CLEANUP;
                CC_IFFREE (ends, int);
            }
        }
    }


    *yes_no = 1;


CLEANUP:

    CC_IFFREE (marks, int);
    CC_IFFREE (ends, int);
    return rval;
}

int CCtsp_test_teeth_disjoint (int ncount, CCtsp_lpcut_in *c, int handle,
        int *yes_no)
{
    int rval = 0;
    int i, marked;
    int *marks = (int *) NULL;

    *yes_no = 0;

    marks = CC_SAFE_MALLOC (ncount, int);
    if (!marks) {
        fprintf (stderr, "out of memory in CCtsp_teeth_disjoint\n");
        rval = 1; goto CLEANUP;
    }
    CCtsp_mark_cut (c, marks, 0);

    for (i = 0; i < c->cliquecount; i++) {
        if (i != handle) {
            CCtsp_is_clique_marked (&c->cliques[i], marks, 1, &marked);
            if (marked) goto CLEANUP;
            CCtsp_mark_clique (&c->cliques[i], marks, 1);
        }
    }

    *yes_no = 1;

CLEANUP:

    CC_IFFREE (marks, int);
    return rval;
}

int CCtsp_find_pure_handle (int ncount, CCtsp_lpcut_in *c, int *handle)
{
    int rval = 0;
    int *marks = (int *) NULL;
    int i, test;

    *handle = -1;
    if (c->cliquecount % 2 || c->cliquecount < 4) goto CLEANUP;

    marks = CC_SAFE_MALLOC (ncount, int);
    if (!marks) {
        fprintf (stderr, "out of memory in CCtsp_pure_find_handle\n");
        rval = 1; goto CLEANUP;
    }
    CCtsp_mark_cut (c, marks, 0);

    CCtsp_mark_clique (&c->cliques[0], marks, 1);
    CCtsp_is_clique_marked (&c->cliques[1], marks, 1, &test);
    if (test) {
        CCtsp_is_clique_marked (&c->cliques[2], marks, 1, &test);
        if (test) {
            *handle = 0; goto CLEANUP;
        } else {
            *handle = 1; goto CLEANUP;
        }
    } else {
        for (i = 2; i < c->cliquecount; i++) {
            CCtsp_is_clique_marked (&c->cliques[i], marks, 1, &test);
            if (test) {
                *handle = i;
                goto CLEANUP;
            }
        }
    }

CLEANUP:

    CC_IFFREE (marks, int);
    return rval;
}

int CCtsp_truncate_cutlist (CCtsp_lpcut_in **cuts, int ncount, int ecount,
        int *elist, double *x, int maxcuts, CCrandstate *rstate)
{
    int i;
    int count = 0;
    int rval = 0;
    CCtsp_lpcut_in *c, *cnext;
    CCtsp_lpcut_in **clist = (CCtsp_lpcut_in **) NULL;
    double *vlist = (double *) NULL;
    int *perm = (int *) NULL;
    CCtsp_lpgraph lg;

    CCtsp_init_lpgraph_struct (&lg);

    if (maxcuts <= 0) {
        for (c = *cuts; c; c = cnext) {
            cnext = c->next;
            CCtsp_free_lpcut_in (c);
        }
        *cuts = (CCtsp_lpcut_in *) NULL;
        goto CLEANUP;
    }

    for (c = *cuts; c; c = c->next) {
        count++;
    }

    if (count > maxcuts) {
        rval = CCtsp_build_lpgraph (&lg, ncount, ecount, elist, (int *) NULL);
        if (rval) {
            fprintf (stderr, "CCtsp_build_lpgraph failed\n"); goto CLEANUP;
        }
        rval = CCtsp_build_lpadj (&lg, 0, ecount);
        if (rval) {
            fprintf (stderr, "CCtsp_build_lpadj failed\n"); goto CLEANUP;
        }

        vlist = CC_SAFE_MALLOC (count, double);
        clist = CC_SAFE_MALLOC (count, CCtsp_lpcut_in *);
        perm  = CC_SAFE_MALLOC (count, int);
        if (!vlist || !clist || !perm) {
            fprintf (stderr, "out of memory in CCtsp_tighten_lp\n");
            rval = 1; goto CLEANUP;
        }
        for (i = 0, c = *cuts; c; c = c->next, i++) {
            clist[i] = c;
            vlist[i] = CCtsp_cutprice (&lg, c, x);
            perm[i] = i;
        }

        CCutil_rselect (perm, 0, count - 1, maxcuts, vlist, rstate);
        for (i = maxcuts; i < count; i++) {
            CCtsp_free_lpcut_in (clist[perm[i]]);
        }

        *cuts = (CCtsp_lpcut_in *) NULL;
        for (i = 0; i < maxcuts; i++) {
            clist[perm[i]]->next = *cuts;
            *cuts = clist[perm[i]];
        }
    }

CLEANUP:

    CC_IFFREE (clist, CCtsp_lpcut_in *);
    CC_IFFREE (vlist, double);
    CC_IFFREE (perm, int);
    CCtsp_free_lpgraph (&lg);

    return rval;
}
