/****************************************************************************/
/*                                                                          */
/*  This file is part of CONCORDE                                           */
/*                                                                          */
/*  (c) Copyright 1995--1999 by David Applegate, Robert Bixby,              */
/*  Vasek Chvatal, and William Cook                                         */
/*                                                                          */
/*  Permission is granted for academic research use.  For other uses,       */
/*  contact the authors for licensing options.                              */
/*                                                                          */
/*  Use at your own risk.  We make no guarantees about the                  */
/*  correctness or usefulness of this code.                                 */
/*                                                                          */
/****************************************************************************/

/****************************************************************************/
/*                                                                          */
/*                  THE MAIN PROGRAM FOR CONCORDE                           */
/*                                                                          */
/*                           TSP CODE                                       */
/*                                                                          */
/*                                                                          */
/*  Written by:  Applegate, Bixby, Chvatal, and Cook                        */
/*  Date: September 25, 1995                                                */
/*                                                                          */
/*  SEE short decsribtion in usage ().                                      */
/*                                                                          */
/*  NOTE:  When using CC_SPARSE edge sets, it is important to specify a     */
/*   a tour or an upperbound if you have one, since our heuristics are      */
/*   not designed for finding Hamilton circuits in sparse graphs.           */
/*                                                                          */
/****************************************************************************/

#include "machdefs.h"
#include "util.h"
#include "edgegen.h"
#include "tsp.h"
#include "profrus.h"
#include "linkern.h"
#include "heldkarp.h"
#include "bigguy.h"
#include "macrorus.h"
#if 0
#include "verify.h"
#endif

#define CC_LK_TRY  (1)    /* number of trials to generate initial tour */
#define EXTRA_OPTIONS     /* define to permit additional command line options */

static int norm = CC_EUCLIDEAN;
static char *datfname     = (char *) NULL;
static char *edgegenfname = (char *) NULL;
static char *problname    = (char *) NULL;
static char *probfname    = (char *) NULL;
static char *edgefname    = (char *) NULL;
static char *fullfname    = (char *) NULL;
static char *tourfname    = (char *) NULL;
static char *masterfname  = (char *) NULL;
static char *poolfname    = (char *) NULL;
static char *restartfname = (char *) NULL;
static char *xfname       = (char *) NULL;
static char *outfname     = (char *) NULL;
static int binary_in = 0;
static int tsplib_in = 1;
static int nnodes_want = 0;
static int gridsize = 0;
static int seed = 0;
static int just_subtour = 0;
static int just_blossom = 0;
static int just_subtour_and_blossom = 0;
static int valid_edges = 0;
static double initial_ub = CCtsp_LP_MAXDOUBLE;
static int usetighten = 0;
static int dfs_branching = 0;
static int bfs_branching = 1;
static int usebranchcliques = 1;
static int standalone_branch = 0;
static int dontcutroot = 0;
static int save_proof = 0;
static int tentative_branch_num = 0;
static int longedge_branching = 1;
static int do_profile = 0;
static int maxchunksize = 16;
static char *filecutname = (char *) NULL;
static unsigned short hostport = CCtsp_HOST_PORT;
static int be_nethost = 0;
static char *grunthostname = (char *) NULL;
static char *cutbossname = (char *) NULL;
static int eliminate_edges = -1;
static int eliminate_sparse = 0;
static int complete_price = 0;
static int run_silently = 1;
static int simple_branching = 0;
static int want_rcnearest = 0;
static int run_fast_cuts = 0;
static int multiple_chunker = 0;
static int output_tour_as_edges = 0;
static int unlink_files = 0;
static char *logfilename = (char *) NULL;


static void
    adjust_upbound (double *bound, int ncount, CCdatagroup *dat),
    usage (char *f);

static int
    daemonize (char *logfile),
    build_nodes (char *probname, int *p_ncount, CCdatagroup *p_dat,
        int **p_tour, int silent, CCrandstate *rstate),
    build_master (int *p_ncount, CCdatagroup *p_dat, int **p_tour,
        int silent, CCrandstate *rstate),
    build_edges (int *p_ecount, int **p_elist, int **p_elen,
        int ncount, int *ptour, CCdatagroup *dat, char *in_edgefname,
        char *in_edgegenfname, int in_just_subtour, int in_just_blossom,
        int in_just_subtour_and_blossom, int silent, CCrandstate *rstate),
    build_fulledges (int *p_excount, int **p_exlist, int **p_exlen,
        int ncount, int *ptour, char *in_fullfname),
    run_hk (int ncount, CCdatagroup *dat, int *hk_tour),
    parseargs (int ac, char **av),
    find_good_tour (int ncount, CCdatagroup *dat, int *perm, double *ub,
            int trials, int silent, CCrandstate *rstate),
    getedges (CCdatagroup *dat, CCedgegengroup *plan, int ncount, int *ecount,
            int **elist, int **elen, int silent, CCrandstate *rstate);



int main (int ac, char **av)
{
    double szeit;
    char *probname = (char *) NULL;
    int i, ncount;
    CCdatagroup dat;
    CCtsp_lp *lp = (CCtsp_lp *) NULL;
    CCtsp_cutselect sel, tentativesel;
    CCtsp_lpcuts *pool = (CCtsp_lpcuts *) NULL;
    int *ptour = (int *) NULL;
    int ecount = 0;
    int *elist = (int *) NULL;
    int *elen = (int *) NULL;
    int excount = 0;
    int *exlist = (int *) NULL;
    int *exlen = (int *) NULL;
    int *besttour = (int *) NULL;
    int is_infeasible = 0;
    double upbound = 0.0;
    int bbcount = 0;
    int rval = 0;
    double branchzeit = 0.0;
    CCrandstate rstate;
    char *cmdout = (char *) NULL;
    char rcnname[1024];
    int cmdlen;
    
    cmdlen = 0;
    for (i=0; i<ac; i++) {
        cmdlen += strlen(av[i]) + 1;
    }
    cmdout = CC_SAFE_MALLOC (cmdlen, char);
    if (cmdout == (char *) NULL) {
        fprintf (stderr, "Out of memory in main\n");
        rval = 1; goto CLEANUP;
    }
    
    cmdlen = 0;
    for (i=0; i<ac; i++) {
        strcpy (cmdout + cmdlen, av[i]);
        cmdlen += strlen(av[i]);
        cmdout[cmdlen] = ' ';
        cmdlen++;
    }
    cmdout[cmdlen-1] = '\0';
    
    szeit = CCutil_zeit ();
    seed = (int) CCutil_real_zeit ();

    if (parseargs (ac, av))
        return 0;

    if (logfilename != (char *) NULL) {
        rval = daemonize (logfilename);
        if (rval) {
            fprintf (stderr, "daemonize failed\n");
            goto CLEANUP;
        }
    }
        
    printf ("%s\n", cmdout);
    CCutil_printlabel ();
    CC_IFFREE (cmdout, char);

    CCutil_sprand (seed, &rstate);
    printf ("Using random seed %d\n", seed); fflush (stdout);

    if (do_profile) {
        CCprof_init ();
        CCprof_start ();
    }

    CCutil_signal_init ();
    CCutil_init_datagroup (&dat);
    
    if (grunthostname) {
#ifdef CC_NETREADY
        rval = CCtsp_grunt (grunthostname, hostport, poolfname, cutbossname,
                            problname, run_silently, &rstate);
        if (rval) {
            fprintf (stderr, "CCtsp_grunt failed\n");
        }
        goto CLEANUP;
#else  /* CC_NETREADY */
        fprintf (stderr, "Networking not enabled\n");
        rval = 1; goto CLEANUP;
#endif /* CC_NETREADY */
    }

    if (run_fast_cuts) {
        CCtsp_init_fast_cutselect (&sel);
    } else {
        CCtsp_init_cutselect (&sel);
    }
    CCtsp_init_tentative_cutselect (&tentativesel);
    sel.usetighten = usetighten;
    tentativesel.usetighten = usetighten;
    sel.maxchunksize = maxchunksize;
    if (filecutname != (char *) NULL) {
        sel.filecuts    = 1;
        sel.filecutname = filecutname;
    }

#ifdef CC_NETREADY
    if (cutbossname != (char *) NULL) {
        sel.remotepool  = 1;
        sel.remotehost  = cutbossname;
        sel.remoteport  = CCtsp_CUT_PORT;
        tentativesel.remotepool  = 1;
        tentativesel.remotehost  = cutbossname;
        tentativesel.remoteport  = CCtsp_CUT_PORT;
    }
#endif /* CC_NETREADY */

    if (!be_nethost) {
        hostport = 0;
    }

    if (problname)        probname = CCtsp_problabel (problname);
    else if (datfname)    probname = CCtsp_problabel (datfname);
    else if (masterfname) probname = CCtsp_problabel (masterfname);
    else                  probname = CCtsp_problabel ("unnamed");
    if (probname == (char *) NULL) {
        fprintf (stderr, "CCtsp_problabel failed\n");
        rval = 1; goto CLEANUP;
    }
    if (problname == (char *) NULL) {
        problname = probname;
    }
    sprintf (rcnname, "%s.rcn", probname);

    rval = build_nodes (probname, &ncount, &dat, &ptour, run_silently, &rstate);
    if (rval) {
        fprintf (stderr, "build_nodes failed\n");
        goto CLEANUP;
    }

    if (ncount < 10) {
        printf ("Less than 10 nodes: Using Held-Karp code.\n");
        fflush (stdout);

        besttour = CC_SAFE_MALLOC (ncount, int);
        CCcheck_NULL (besttour, "out of memory for besttour");

        rval = run_hk (ncount, &dat, besttour);
        CCcheck_rval (rval, "run_hk failed");

        rval = CCtsp_dumptour (ncount, &dat, ptour, probname, besttour,
                               outfname, output_tour_as_edges, run_silently);
        CCcheck_rval (rval, "CCtsp_dumptour failed");

        printf ("Total Running Time: %.2f (seconds)\n", CCutil_zeit () - szeit);
        fflush (stdout);
        goto CLEANUP;
    }

    adjust_upbound (&initial_ub, ncount, &dat);


    if (!probfname && !restartfname) {
        rval = build_edges (&ecount, &elist, &elen, ncount, ptour,
                            &dat, edgefname, edgegenfname, just_subtour,
                            just_blossom, just_subtour_and_blossom,
                            run_silently, &rstate);
        if (rval) {
            fprintf (stderr, "build_edges failed\n");
            goto CLEANUP;
        }
    }

    rval = build_fulledges (&excount, &exlist, &exlen, ncount, ptour,
                            fullfname);
    if (rval) {
        fprintf (stderr, "build_fulledges failed\n");
        goto CLEANUP;
    }
    
    /* Change CCtsp_init_lp to be after read.  build_edges & build_fulledges
       + first_lp become construct_lp */

    /***** Get the initial cutpool *****/

    rval = CCtsp_init_cutpool (&ncount, poolfname, &pool);
    if  (rval) {
        fprintf (stderr, "CCtsp_init_cutpool failed\n");
        goto CLEANUP;
    }

    /***** Initialize besttour to represent the permutation tour  ****/

    besttour = CC_SAFE_MALLOC (ncount, int);
    if (!besttour) {
        fprintf (stderr, "out of memory in allocating best tour\n");
        rval = 1; goto CLEANUP;
    }
    for (i = 0; i < ncount; i++) {
        besttour[i] = i;
    }

    if (restartfname) {
        upbound  = initial_ub;
        bbcount = 0;

        rval = CCtsp_bfs_restart (problname, restartfname, &sel,
                &tentativesel, &upbound, &bbcount, usebranchcliques, &dat,
                ptour, pool, ncount, besttour, hostport, &branchzeit,
                save_proof, tentative_branch_num, longedge_branching,
                (double *) NULL, (int *) NULL, run_silently, &rstate);
        if (rval) {
            fprintf (stderr, "CCtsp_bfs_restart failed\n");
            goto CLEANUP;
        }
        goto DONE;
    }

    rval = CCtsp_dumptour (ncount, &dat, ptour, probname, besttour,
                           (char *) NULL, 0, run_silently);
    if (rval) {
        fprintf (stderr, "CCtsp_dumptour failed\n"); goto CLEANUP;
    }

    /***** Get the initial LP *****/

    rval = CCtsp_init_lp (&lp, problname, -1, probfname, ncount, &dat,
                    ecount, elist, elen, excount, exlist, exlen, valid_edges,
                    ptour, initial_ub, pool, run_silently, &rstate);
    if (rval == 2) {
        printf ("CCtsp_init_lp reports an infeasible LP\n");
        rval = CCtsp_verify_infeasible_lp (lp, &is_infeasible, run_silently);
        if (rval) {
            fprintf (stderr, "CCtsp_verify_infeasible_lp failed\n");
            goto CLEANUP;
        }
        if (!is_infeasible) {
            printf ("Couldn't verify infeasible LP\n");
            fflush (stdout);
            rval = 1; goto CLEANUP;
        }
        upbound = CCtsp_LP_MAXDOUBLE;
        bbcount = 1;
        goto DONE;
    } else if (rval) {
        fprintf (stderr, "CCtsp_init_lp failed\n"); goto CLEANUP;
    }

    CCutil_start_timer (&lp->stats.total);
    
    ecount = 0;
    CC_IFFREE (elist, int);
    CC_IFFREE (elen, int);
    excount = 0;
    CC_IFFREE (exlist, int);
    CC_IFFREE (exlen, int);

    if (0 && lp->full_edges_valid) {
        if (CCtsp_inspect_full_edges (lp)) {
            fprintf (stderr, "full edge set does not contain all LP edges\n");
            rval = 1; goto CLEANUP;
        }
    }

    if (standalone_branch) {
        rval = CCtsp_do_interactive_branch (lp, run_silently, &rstate);
        if (rval) {
            fprintf (stderr, "CCtsp_do_interactive_branch failed\n");
            goto CLEANUP;
        } else {
            printf ("completed interative branch\n");
            printf ("Total Running Time: %.2f (seconds)\n",
                    CCutil_zeit () - szeit);
            goto CLEANUP;
        }
    }

    if (run_fast_cuts) {
        rval = CCtsp_cutselect_set_tols (&sel, lp, -1, 0);
    } else {
        rval = CCtsp_cutselect_set_tols (&sel, lp, 1, run_silently);
    }
    if (rval) {
        fprintf (stderr, "CCtsp_cutselect_set_tols failed\n");
        goto CLEANUP;
    }

    if (run_fast_cuts) {
        rval = CCtsp_cutting_loop (lp, &sel, 1, run_silently, &rstate);
        if (rval) {
            fprintf (stderr, "CCtsp_cutting_loop failed\n");
            goto CLEANUP;
        }
        printf ("Bound: %f\n", lp->lowerbound); fflush (stdout);
        CCutil_stop_timer (&lp->stats.total, 1);

        printf ("Final LP has %d rows, %d columns, %d nonzeros\n",
                CClp_nrows (lp->lp), CClp_ncols (lp->lp),
                CClp_nnonzeros (lp->lp));

        if (want_rcnearest) {
            rval = CCtsp_dump_rc_nearest (lp, want_rcnearest, rcnname, 0);
            if (rval) {
                fprintf (stderr, "CCtsp_dump_rc_nearest failed\n");
                fflush (stdout);
            }
        }
        if (xfname) {
            rval = CCtsp_dump_x (lp, xfname);
            if (rval) {
                fprintf (stderr, "CCtsp_dump_x failed\n");
                goto CLEANUP;
            }
        }
        goto DONE;
    } else if (just_subtour) {
        rval = CCtsp_subtour_loop (lp, run_silently, &rstate);
        if (rval) {
            fprintf (stderr, "CCtsp_subtour_loop failed\n");
            goto CLEANUP;
        }
        printf ("Bound: %f\n", lp->lowerbound); fflush (stdout);
        CCutil_stop_timer (&lp->stats.total, 1);
/*
        CCtsp_output_statistics (&lp->stats);
*/
        printf ("Final LP has %d rows, %d columns, %d nonzeros\n",
                CClp_nrows (lp->lp), CClp_ncols (lp->lp),
                CClp_nnonzeros (lp->lp));

        if (want_rcnearest) {
            rval = CCtsp_dump_rc_nearest (lp, want_rcnearest, rcnname, 1);
            if (rval) {
                fprintf (stderr, "CCtsp_dump_rc_nearest failed\n");
                fflush (stdout);
            }
        }
        if (xfname) {
            rval = CCtsp_dump_x (lp, xfname);
            if (rval) {
                fprintf (stderr, "CCtsp_dump_x failed\n");
                goto CLEANUP;
            }
        }
        goto DONE;
    } else if (just_blossom) {
        rval = CCtsp_blossom_loop (lp, run_silently, &rstate);
        if (rval) {
            fprintf (stderr, "CCtsp_blossom_loop failed\n");
            goto CLEANUP;
        }
        printf ("Bound: %f\n", lp->lowerbound); fflush (stdout);
        CCutil_stop_timer (&lp->stats.total, 1);
        printf ("Final LP has %d rows, %d columns, %d nonzeros\n",
                CClp_nrows (lp->lp), CClp_ncols (lp->lp),
                CClp_nnonzeros (lp->lp));

        if (want_rcnearest) {
            rval = CCtsp_dump_rc_nearest (lp, want_rcnearest, rcnname, 0);
            if (rval) {
                fprintf (stderr, "CCtsp_dump_rc_nearest failed\n");
                fflush (stdout);
            }
        }
        if (xfname) {
            rval = CCtsp_dump_x (lp, xfname);
            if (rval) {
                fprintf (stderr, "CCtsp_dump_x failed\n");
                goto CLEANUP;
            }
        }
        goto DONE;
    } else if (just_subtour_and_blossom) {
        rval = CCtsp_subtour_and_blossom_loop (lp, run_silently, &rstate);
        if (rval) {
            fprintf (stderr, "CCtsp_subtour_and_blossom_loop failed\n");
            goto CLEANUP;
        }
        printf ("Bound: %f\n", lp->lowerbound); fflush (stdout);
        CCutil_stop_timer (&lp->stats.total, 1);
        printf ("Final Root LP has %d rows, %d columns, %d nonzeros\n",
                CClp_nrows (lp->lp), CClp_ncols (lp->lp),
                CClp_nnonzeros (lp->lp));

        if (want_rcnearest) {
            rval = CCtsp_dump_rc_nearest (lp, want_rcnearest, rcnname, 0);
            if (rval) {
                fprintf (stderr, "CCtsp_dump_rc_nearest failed\n");
                fflush (stdout);
            }
        }

        if (xfname) {
            rval = CCtsp_dump_x (lp, xfname);
            if (rval) {
                fprintf (stderr, "CCtsp_dump_x failed\n");
                goto CLEANUP;
            }
        }

        goto DONE;
    } else if (dontcutroot == 0) {
        if (multiple_chunker) {
            rval = CCtsp_cutting_multiple_loop (lp, &sel, 1, maxchunksize,
                                    run_silently, &rstate);
        } else {
            rval = CCtsp_cutting_loop (lp, &sel, 1, run_silently, &rstate);
        }
        if (rval == 2) {
            printf ("CCtsp_cutting_loop reports an infeasible LP\n");
            rval = CCtsp_verify_infeasible_lp (lp, &is_infeasible,
                                               run_silently);
            if (rval) {
                fprintf (stderr, "CCtsp_verify_infeasible_lp failed\n");
                goto CLEANUP;
            }
            if (!is_infeasible) {
                printf ("Couldn't verify infeasibile LP\n");
                fflush (stdout);
                rval = 1; goto CLEANUP;
            }
            upbound = CCtsp_LP_MAXDOUBLE;
            bbcount = 1;
            CCutil_stop_timer (&lp->stats.total, 1);
/*
            CCtsp_output_statistics (&lp->stats);
*/
            printf ("Final LP has %d rows, %d columns, %d nonzeros\n",
                    CClp_nrows (lp->lp), CClp_ncols (lp->lp),
                    CClp_nnonzeros (lp->lp));

            goto DONE;
        } else if (rval) {
            fprintf (stderr, "cutting_loop failed\n");
            goto CLEANUP;
        }
    }

    {
        double tourval;
        CCutil_start_timer (&lp->stats.linkern);
        rval = CCtsp_call_x_heuristic (lp, &tourval, besttour, run_silently,
                                       &rstate);
        if (rval) {
            fprintf (stderr, "CCtsp_call_x_heuristic failed\n");
            goto CLEANUP;
        }
        if (!run_silently) {
            CCutil_stop_timer (&lp->stats.linkern, 1);
        } else {
            CCutil_stop_timer (&lp->stats.linkern, 0);
        }
        if (tourval < lp->upperbound) {
            printf ("New upperbound from x-heuristic: %.2f\n", tourval);
            lp->upperbound = tourval;
            rval = CCtsp_dumptour (ncount, &dat, ptour, probname, besttour,
                         (char *) NULL, 0, run_silently);
            if (rval) {
                fprintf (stderr, "CCtsp_dumptour failed\n"); goto CLEANUP;
            }
        }
    }

    printf ("Final lower bound %f, upper bound %f\n", lp->lowerbound,
                                                      lp->upperbound);
    fflush (stdout);

    if (xfname) {
        rval = CCtsp_dump_x (lp, xfname);
        if (rval) {
            fprintf (stderr, "CCtsp_dump_x failed\n");
            goto CLEANUP;
        }
    }

    if (want_rcnearest) {
        rval = CCtsp_dump_rc_nearest (lp, want_rcnearest, rcnname, 0);
        if (rval) {
            fprintf (stderr, "CCtsp_dump_rc_nearest failed\n");
            fflush (stdout);
        }
    }

    if (lp->graph.ncount < 100000 || complete_price) {
        CCbigguy bound;
        CCbigguy bupper;
        rval = CCtsp_exact_price (lp, &bound, complete_price, 0, run_silently);
        if (rval) {
            fprintf (stderr, "CCtsp_exact_price failed\n");
            goto CLEANUP;
        }
        lp->exact_lowerbound = bound;
        printf ("Exact lower bound: %.6f\n", CCbigguy_bigguytod (bound));
        if (!run_silently) {
            printf ("DIFF: %f\n", lp->lowerbound - CCbigguy_bigguytod (bound));
            fflush (stdout);
        }

        bupper = CCbigguy_dtobigguy (lp->upperbound);
        CCbigguy_sub (&bupper, CCbigguy_ONE);

        if (CCbigguy_cmp (lp->exact_lowerbound, bupper) > 0) {
            upbound = lp->upperbound;
            bbcount = 1;
            if (!dfs_branching && !bfs_branching) {
                printf ("Optimal Solution: %.2f\n", upbound);
                printf ("Number of bbnodes: %d\n", bbcount);
                fflush (stdout);
            }
            if (!run_silently) {
                CCutil_stop_timer (&lp->stats.total, 1);
            } else {
                CCutil_stop_timer (&lp->stats.total, 0);
            }
/*
            CCtsp_output_statistics (&lp->stats);
*/
            printf ("Final LP has %d rows, %d columns, %d nonzeros\n",
                    CClp_nrows (lp->lp), CClp_ncols (lp->lp),
                    CClp_nnonzeros (lp->lp));
            goto DONE;
        }

        if (eliminate_edges) {
            rval = CCtsp_eliminate_variables (lp, eliminate_sparse,
                                              run_silently);
            if (rval) {
                fprintf (stderr, "CCtsp_eliminate_variables failed\n");
                goto CLEANUP;
            }
        }
    } else {
        printf ("During testing, do not exact price large problems\n");
        fflush (stdout);
        CCutil_stop_timer (&lp->stats.total, 1);
/*
        CCtsp_output_statistics (&lp->stats);
*/
        printf ("Final LP has %d rows, %d columns, %d nonzeros\n",
                CClp_nrows (lp->lp), CClp_ncols (lp->lp),
                CClp_nnonzeros (lp->lp));

        goto DONE;
    }

    CCutil_stop_timer (&lp->stats.total, 1);

    printf ("Final Root LP has %d rows, %d columns, %d nonzeros\n",
            CClp_nrows (lp->lp), CClp_ncols (lp->lp),
            CClp_nnonzeros (lp->lp));
    
    if (dfs_branching) {
        upbound = lp->upperbound;
        bbcount = 0;

        if (simple_branching) CCtsp_init_simple_cutselect (&sel);
        rval = CCtsp_easy_dfs_brancher (lp, &sel, 0, &upbound, &bbcount,
                     usebranchcliques, besttour, longedge_branching,
                     simple_branching, run_silently, &rstate);
        if (rval) {
            fprintf (stderr, "CCtsp_easy_dfs_brancher failed\n");
            goto CLEANUP;
        }
    } else if (bfs_branching) {
        double lowbound = lp->lowerbound;
        int id          = lp->id;

        upbound  = lp->upperbound;
        bbcount = 0;

        rval = CCtsp_write_probroot_id (problname, lp);
        if (rval) {
            fprintf (stderr, "CCtsp_write_probroot_id failed\n");
            goto CLEANUP;
        }
        CCtsp_free_tsp_lp_struct (&lp);

        rval = CCtsp_bfs_brancher (problname, id, lowbound, &sel, 
                &tentativesel, &upbound, &bbcount, usebranchcliques, &dat,
                ptour, pool, ncount, besttour, hostport, &branchzeit,
                save_proof, tentative_branch_num, longedge_branching,
                (double *) NULL, (int *) NULL, run_silently, &rstate);
        if (rval) {
            fprintf (stderr, "CCtsp_bfs_brancher failed\n"); goto CLEANUP;
        }
    }

DONE:

    if (dfs_branching || bfs_branching || restartfname) {
        printf ("Optimal Solution: %.2f\n", upbound);
        printf ("Number of bbnodes: %d\n", bbcount);
        fflush (stdout);
        rval = CCtsp_dumptour (ncount, &dat, ptour, probname, besttour,
                               outfname, output_tour_as_edges, run_silently);
        CCcheck_rval (rval, "CCtsp_dumptour failed");
    }

    printf ("Total Running Time: %.2f (seconds)", CCutil_zeit () - szeit);
    if (branchzeit != 0.0) {
        printf ("  Branching Time: %.2f (seconds)", branchzeit);
    }
    printf ("\n");
    fflush (stdout);

    if (!dfs_branching && !bfs_branching && !restartfname) {
/*
        CCtsp_output_statistics (&lp->stats);
*/

        rval = CCtsp_write_probfile_sav (lp);
        if (rval) {
            fprintf (stderr, "CCtsp_write_probfile_sav failed\n");
            goto CLEANUP;
        }
    }

    if (pool && pool->cutcount) {
        char buf[1024];
        if (!run_silently) {
            printf ("Final Pool: %d cuts\n", pool->cutcount);
            fflush (stdout);
        }

        sprintf (buf, "%s.pul", probname);
        rval = CCtsp_write_cutpool (ncount, buf, pool);
        if (rval) {
            fprintf (stderr, "CCtsp_write_cutpool failed\n");
            goto CLEANUP;
        }
    }

    if (sel.remotepool && pool && pool->cutcount > pool->savecount) {
        rval = CCtsp_send_newcuts (ncount, pool, sel.remotehost,
                sel.remoteport);
        if (rval) {
            fprintf (stderr, "CCtsp_send_newcuts failed\n");
            rval = 0;
        }
    }
        
    rval = 0;

CLEANUP:

    if (unlink_files) {
        char buf[1024];
        if (!run_silently) {
            printf ("Delete the temporary files: pul sav mas\n");
            fflush (stdout);
        }

        sprintf (buf, "%s.pul", probname);
        rval = unlink (buf);
        if (rval && !run_silently) {
            printf ("CCutil_sdelete_file failed for %s\n", buf);
        }

        sprintf (buf, "O%s.pul", probname);
        rval = unlink (buf);
        if (rval && !run_silently) {
            printf ("CCutil_sdelete_file failed for %s\n", buf);
        }

        sprintf (buf, "%s.sav", probname);
        rval = unlink (buf);
        if (rval && !run_silently) {
            printf ("CCutil_sdelete_file failed for %s\n", buf);
        }

        sprintf (buf, "O%s.sav", probname);
        rval = unlink (buf);
        if (rval && !run_silently) {
            printf ("CCutil_sdelete_file failed for %s\n", buf);
        }

        sprintf (buf, "%s.mas", probname);
        rval = unlink (buf);
        if (rval && !run_silently) {
            printf ("CCutil_sdelete_file failed for %s\n", buf);
        }

        sprintf (buf, "O%s.mas", probname);
        rval = unlink (buf);
        if (rval && !run_silently) {
            printf ("CCutil_sdelete_file failed for %s\n", buf);
        }
    }

    CCtsp_free_tsp_lp_struct (&lp);

    if (pool) {
        CCtsp_free_cutpool (&pool);
    }

    CC_IFFREE (elist, int);
    CC_IFFREE (elen, int);
    CC_IFFREE (exlist, int);
    CC_IFFREE (exlen, int);
    CC_IFFREE (ptour, int);
    CC_IFFREE (besttour, int);
    CC_IFFREE (probname, char);
    CCutil_freedatagroup (&dat);
    if (do_profile) {
        CCprof_stop ();
        CCprof_end ();
    }
    CC_IFFREE (cmdout, char);

    return rval;
}

static int daemonize (char *logfile)
{
    int d;
    int pid;
    int rval = 0;

    pid = fork ();
    if (pid > 0) {
        /* parent */
        printf ("Created child process %d\n", pid);
        fflush (stdout);
        fflush (stderr);
        _exit (0);
    } else if (pid == 0) {
        /* child */
        d = open (logfile, O_WRONLY|O_APPEND|O_CREAT, 0644);
        if (d < 0) {
            perror (logfile);
            fprintf (stderr, "Unable to open log file %s for output\n",
                     logfile);
            rval = -1; goto CLEANUP;
        }
        if (dup2 (d, 1) < 0) {
            perror ("dup2 (d,1)");
            rval = -1; goto CLEANUP;
        }
        if (dup2 (d, 2) < 0) {
            perror ("dup2 (d,2)");
            rval = -1; goto CLEANUP;
        }
        if (close (d)) {
            perror ("close (d)");
            rval = -1; goto CLEANUP;
        }
        if (close (0)) {
            perror ("close (0)");
            rval = -1; goto CLEANUP;
        }
        rval = 0; goto CLEANUP;
    } else {
        /* error */
        perror ("fork");
        fprintf (stderr, "Unable to fork()\n");
        rval = -1; goto CLEANUP;
    }
 CLEANUP:
    return rval;
}

static int build_nodes (char *probname, int *p_ncount, CCdatagroup *p_dat,
        int **p_tour, int silent, CCrandstate *rstate)
{
    char buf[1024];
    int rval;

    if (masterfname == (char *) NULL) {
        rval = build_master (p_ncount, p_dat, p_tour, silent, rstate);
        if (rval) {
            fprintf (stderr, "build_master failed\n");
            return rval;
        }
        sprintf (buf, "%s.mas", probname);
        rval = CCutil_putmaster (buf, *p_ncount, p_dat, *p_tour);
        if (rval) {
            fprintf (stderr, "CCutil_putmaster failed\n");
            return rval;
        }
    } else {
        if (CCutil_getmaster (masterfname, p_ncount, p_dat, p_tour)) {
            fprintf (stderr, "CCutil_getmaster failed\n");
            return 1;
        }
    }

    return 0;
}

static int build_master (int *p_ncount, CCdatagroup *p_dat, int **p_tour,
        int silent, CCrandstate *rstate)
{
    int rval;
    int allow_dups;
    int use_gridsize;

    CCutil_init_datagroup (p_dat);
    if (tsplib_in && datfname != (char *) NULL) {
        rval = CCutil_gettsplib (datfname, p_ncount, p_dat);
        if (rval) {
            fprintf (stderr, "CCutil_gettsplib failed\n"); goto CLEANUP;
        }
    } else {
        *(p_ncount) = nnodes_want;
        if (gridsize < 0) {
            use_gridsize = -gridsize;
            allow_dups = 0;
        } else if (gridsize > 0) {
            use_gridsize = gridsize;
            allow_dups = 1;
        } else {
            use_gridsize = nnodes_want;
            allow_dups = 0;
        }
        rval = CCutil_getdata (datfname, binary_in, norm, p_ncount, p_dat,
                               use_gridsize, allow_dups, rstate);
        if (rval) {
            fprintf (stderr, "CCutil_getdata failed\n"); goto CLEANUP;
        }
    }

    if (*p_ncount < 4) {
        fprintf (stderr, "Have 3 cities or less, nothing to solve\n");
        rval = 1;  goto CLEANUP;
    }


    /***** Get the permutation tour and permute the data  *****/

    *p_tour = CC_SAFE_MALLOC (*p_ncount, int);
    if ((*p_tour) == (int *) NULL) {
        fprintf (stderr, "out of memory in build_master\n");
        rval = 1; goto CLEANUP;
    }

    if (tourfname) {
        rval = CCutil_getcycle (*p_ncount, tourfname, *p_tour, 0);
        if (rval) {
            fprintf (stderr, "CCutil_getcycle failed\n");
            goto CLEANUP;
        }
    } else {
        double bnd;
        if (just_subtour || just_blossom || just_subtour_and_blossom ||
            run_fast_cuts) {
            rval = find_good_tour (*p_ncount, p_dat, *p_tour, &bnd, -1,
                                   silent, rstate);
        } else if (initial_ub == CCtsp_LP_MAXDOUBLE) {
            rval = find_good_tour (*p_ncount, p_dat, *p_tour, &bnd, CC_LK_TRY,
                                   silent, rstate);
        } else {
            if (!silent) {
                printf ("Initial bnd %f given - use short tour run\n",
                        initial_ub);
                fflush (stdout);
            }
            rval = find_good_tour (*p_ncount, p_dat, *p_tour, &bnd, 0,
                                   silent, rstate);
        }
        if (rval) {
            fprintf (stderr, "find_good_tour failed\n");
            goto CLEANUP;
        }
    }

    rval = CCutil_datagroup_perm (*p_ncount, p_dat, *p_tour);
    if (rval) {
        fprintf (stderr, "CCutil_datagroup_perm failed\n"); goto CLEANUP;
    }

    rval = 0;

CLEANUP:

    return rval;
}

static void adjust_upbound (double *bound, int ncount, CCdatagroup *dat)
{
    double bnd;
    int i;

    bnd = CCutil_dat_edgelen (ncount - 1, 0, dat);
    for (i = 1; i < ncount; i++) {
        bnd += CCutil_dat_edgelen (i-1, i, dat);
    }
    if (bnd < *bound) {
        printf ("Set initial upperbound to %.0f (from tour)\n", bnd);
        fflush (stdout);
        *bound = bnd;
    }
}

static int build_edges (int *p_ecount, int **p_elist, int **p_elen,
        int ncount, int *ptour, CCdatagroup *dat, char *in_edgefname,
        char *in_edgegenfname, int in_just_subtour, int in_just_blossom,
        int in_just_subtour_and_blossom, int silent, CCrandstate *rstate)
{
    int rval;
    int *elist = (int *) NULL;
    int ecount;
    int i;
    
    if (in_edgefname) {
        int *invperm = (int *) NULL;

        printf ("Read initial edge set\n"); fflush (stdout);
        
        rval = CCutil_getedgelist (ncount, in_edgefname, p_ecount, p_elist,
                                   p_elen, 0);
        if (rval) {
            fprintf (stderr, "CCutil_getedgelist failed\n");
            goto CLEANUP;
        }
        ecount = *p_ecount;
        elist = *p_elist;
        printf ("Initial edgeset: %d edges (%d nodes)\n", ecount, ncount);
        printf ("Rearrange the edges to match the tour order\n");
        fflush (stdout);

        invperm = CC_SAFE_MALLOC (ncount, int);
        if (!invperm) {
            rval = 1;
            goto CLEANUP;
        }
        for (i = 0; i < ncount; i++)
            invperm[ptour[i]] = i;
        for (i = 0; i < 2*ecount; i++)
            elist[i] = invperm[elist[i]];
        CC_FREE (invperm, int);
    } else if (dat) {
        CCedgegengroup plan;
        
        if (in_edgegenfname) {
            rval = CCedgegen_read (in_edgegenfname, &plan);
            if (rval) {
                fprintf (stderr, "CCedgegen_read failed\n");
                return 1;
            }
        } else {
            CCedgegen_init_edgegengroup (&plan);
            if (in_just_subtour || in_just_blossom ||
                in_just_subtour_and_blossom) {
                plan.tour.greedy = 1;
                plan.f2match_nearest.number = 4;
            } else {
                plan.linkern.count = 10;
                plan.linkern.quadnearest = 2;
                plan.linkern.greedy_start = 0;
                plan.linkern.nkicks = (ncount / 100) + 1;
            }
        }

        rval = getedges (dat, &plan, ncount, p_ecount, p_elist, p_elen,
                         silent, rstate);
        if (rval) {
            fprintf (stderr, "getgraph failed\n");
            goto CLEANUP;
        }
    }
    rval = 0;
 CLEANUP:
    return rval;
}

static int build_fulledges (int *p_excount, int **p_exlist, int **p_exlen,
        int ncount, int *ptour, char *in_fullfname)
{
    int i;
    int rval;
    int *exlist;
    int excount;
    
    if (in_fullfname) {
        int *invperm = (int *) NULL;

        rval = CCutil_getedgelist (ncount, in_fullfname, p_excount, p_exlist,
                                   p_exlen, 0);
        if (rval) {
            fprintf (stderr, "CCutil_getedgelist failed\n");
            goto CLEANUP;
        }

        invperm = CC_SAFE_MALLOC (ncount, int);
        if (!invperm) {
            fprintf (stderr, "out of memory in main\n");
            rval = 1;
            goto CLEANUP;
        }
        for (i = 0; i < ncount; i++)
            invperm[ptour[i]] = i;
        excount = *p_excount;
        exlist = *p_exlist;
        for (i = 0; i < 2*excount; i++)
            exlist[i] = invperm[exlist[i]];
        CC_FREE (invperm, int);
    } else {
        *p_excount = 0;
    }
    rval = 0;
 CLEANUP:
    return rval;
}

static int find_good_tour (int ncount, CCdatagroup *dat, int *perm, double *ub,
        int trials, int silent, CCrandstate *rstate)
{
    int rval = 0;
    CCedgegengroup plan;
    int ecount;
    int *elist = (int *) NULL;
    int tcount;
    int *tlist = (int *) NULL;
    int *bestcyc = (int *) NULL;
    int *cyc     = (int *) NULL;
    int *tmp;
    double val, bestval, szeit;
    int kicks, i, istour;

    szeit = CCutil_zeit ();
    bestval = CCtsp_LP_MAXDOUBLE;

    if (trials == -1) {
        kicks = (ncount > 400 ? 100 : ncount/4);
    } else {
        kicks = (ncount > 1000 ? 500 : ncount/2);
    }

    if (!silent) {
        printf ("Finding a good tour for compression: %d\n", trials);
        fflush (stdout);
    }

    cyc     = CC_SAFE_MALLOC (ncount, int);
    bestcyc = CC_SAFE_MALLOC (ncount, int);
    if (!cyc || !bestcyc) {
        fprintf (stderr, "out of memory in find_good_tour\n");
        rval = 1; goto CLEANUP;
    }

    CCedgegen_init_edgegengroup (&plan);
    if (ncount < 9) {
        plan.nearest = ncount - 1;
    } else {
        plan.quadnearest = 2;
    }
    rval = CCedgegen_edges (&plan, ncount, dat, (double *) NULL, &ecount,
                            &elist, silent, rstate);
    if (rval) {
        fprintf (stderr, "CCedgegen_edges failed\n"); goto CLEANUP;
    }
    plan.quadnearest = 0;
    plan.nearest = 0;

    if (ncount < 9) {
        plan.tour.greedy  = 1;
    } else {
        plan.tour.nearest_count = 1;
    }
    rval = CCedgegen_edges (&plan, ncount, dat, (double *) NULL, &tcount,
                            &tlist, silent, rstate);
    if (rval) {
        fprintf (stderr, "CCedgegen_edges failed\n"); goto CLEANUP;
    }

    if (tcount != ncount) {
        fprintf (stderr, "wrong edgeset from CCedgegen_edges\n");
        rval = 1; goto CLEANUP;
    }

    rval = CCutil_edge_to_cycle (ncount, tlist, &istour, cyc);
    if (rval) {
        fprintf (stderr, "CCutil_edge_to_cycle failed\n");
        rval = 1; goto CLEANUP;
    }
    if (istour == 0) {
        fprintf (stderr, "Starting tour has an error\n");
        rval = 1; goto CLEANUP;
    }
    CC_FREE (tlist, int);

    rval = CClinkern_tour (ncount, dat, ecount, elist, ncount, kicks,
                    cyc, bestcyc, &bestval, silent, 0.0, 0.0,
                    (char *) NULL,
                    CC_LK_GEOMETRIC_KICK, rstate);
    if (rval) {
        fprintf (stderr, "CClinkern_tour failed\n"); goto CLEANUP;
    }

    for (i = 0; i < trials; i++) {
        rval = CClinkern_tour (ncount, dat, ecount, elist, ncount, kicks,
                        (int *) NULL, cyc, &val, silent, 0.0, 0.0,
                        (char *) NULL, CC_LK_GEOMETRIC_KICK, rstate);
        if (rval) {
            fprintf (stderr, "CClinkern_tour failed\n"); goto CLEANUP;
        }
        if (val < bestval) {
            CC_SWAP (cyc, bestcyc, tmp);
            bestval = val;
        }
    }

    if (trials > 0) {
        rval = CClinkern_tour (ncount, dat, ecount, elist, ncount, 2 * kicks,
                        bestcyc, perm, ub, silent, 0.0, 0.0,
                        (char *) NULL, CC_LK_GEOMETRIC_KICK, rstate);
        if (rval) {
            fprintf (stderr, "CClinkern_tour failed\n"); goto CLEANUP;
        }
    } else {
        for (i = 0; i < ncount; i++) {
            perm[i] = bestcyc[i];
        }
    }

    if (!silent) {
        printf ("Time to find compression tour: %.2f (seconds)\n",
                CCutil_zeit() - szeit);
        fflush (stdout);
    }

CLEANUP:

    CC_IFFREE (cyc, int);
    CC_IFFREE (bestcyc, int);
    CC_IFFREE (elist, int);
    CC_IFFREE (tlist, int);
    return rval;
}

static int getedges (CCdatagroup *dat, CCedgegengroup *plan, int ncount,
        int *ecount, int **elist, int **elen, int silent,
        CCrandstate *rstate)
{
    int i;

    *elist = (int *) NULL;
    *elen = (int *) NULL;

    if (dat == (CCdatagroup *) NULL || plan == (CCedgegengroup *) NULL) {
        fprintf (stderr, "getedges needs CCdatagroup and CCedgegengroup\n");
        return 1;
    }

    if (CCedgegen_edges (plan, ncount, dat, (double *) NULL, ecount, elist,
                         silent, rstate)) {
        fprintf (stderr, "CCedgegen_edges failed\n");
        return 1;
    }
    *elen = CC_SAFE_MALLOC(*ecount, int);
    if (!(*elen)) {
        CC_FREE (*elist, int);
        return 1;
    }
    for (i = 0; i < *ecount; i++) {
        (*elen)[i] = CCutil_dat_edgelen ((*elist)[2*i], (*elist)[(2*i) + 1],
                                         dat);
    }

    return 0;
}

static int run_hk (int ncount, CCdatagroup *dat, int *hk_tour)
{
    double hk_val;
    int hk_found, hk_yesno;
    int *hk_tlist = (int *) NULL;
    int rval = 0;

    hk_tlist = CC_SAFE_MALLOC (2*ncount, int);
    CCcheck_NULL (hk_tlist, "out of memory for hk_tlist");

    rval = CCheldkarp_small (ncount, dat, (double *) NULL, &hk_val,
                             &hk_found, 0, hk_tlist, 1000000, 2);
    CCcheck_rval (rval, "CCheldkarp_small failed");
    printf ("Optimal Solution: %.2f\n", hk_val); fflush (stdout);

    rval = CCutil_edge_to_cycle (ncount, hk_tlist, &hk_yesno, hk_tour);
    CCcheck_rval (rval, "CCutil_edge_to_cycle failed");

    if (hk_yesno == 0) {
        fprintf (stderr, "Held-Karp returned list that is not a tour\n");
        rval = 1;  goto CLEANUP;
    }

CLEANUP:

     CC_IFFREE (hk_tlist, int);
     return rval;
}

static int parseargs (int ac, char **av)
{
    int c, inorm;
    int boptind = 1;
    char *boptarg = (char *) NULL;

    /* remaining: cfjoOTW */
#ifdef EXTRA_OPTIONS
    while ((c = CCutil_bix_getopt (ac, av, "aAbBC:dD:e:E:F:fg:GhH:iIJ:k:K:l:LmM:n:N:o:pP:qQr:R:s:S:t:u:UvVwX:xyYZz:", &boptind, &boptarg)) != EOF)
#else /* EXTRA_OPTIONS */
    while ((c = CCutil_bix_getopt (ac, av, "BC:d:e:E:F:fg:hiIJ:k:K:l:LM:N:o:P:r:R:s:S:t:u:UvwX:x", &boptind, &boptarg)) != EOF)
#endif
        switch (c) {
#ifdef EXTRA_OPTIONS
        case 'a':
            eliminate_edges = 1;
            break;
        case 'A':
            eliminate_edges = 0;
            break;
        case 'b':
            binary_in = 1;
            break;
#endif
        case 'B':
            bfs_branching = 0;
            break;
        case 'C':
            maxchunksize = atoi(boptarg);
            break;
        case 'd':
            dfs_branching = 1;
	    bfs_branching = 0;
            break;
#ifdef EXTRA_OPTIONS
        case 'D':
            edgegenfname = boptarg;
            break;
#endif
        case 'e':
            edgefname = boptarg;
            break;
        case 'E':
            fullfname = boptarg;
	    valid_edges = 1;
            break;
        case 'F':
            filecutname = boptarg;
            break;
        case 'f':
            output_tour_as_edges = 1;
            break;
        case 'g':
            grunthostname = boptarg;
            break;
#ifdef EXTRA_OPTIONS
        case 'G':
            eliminate_sparse = 1;
            break;
#endif
        case 'h':
            be_nethost = 1;
            break;
#ifdef EXTRA_OPTIONS
        case 'H':
            hostport = (unsigned short) atoi(boptarg);
            break;
#endif
        case 'i':
            just_blossom = 1;
            break;
        case 'I':
            just_subtour = 1;
            break;
        case 'J':
            tentative_branch_num = atoi (boptarg);
            break;
        case 'k':
            nnodes_want = atoi (boptarg);
            break;
        case 'K':
            cutbossname = boptarg;
            break;
        case 'l':
            logfilename = boptarg;
            break;
        case 'L':
            longedge_branching = 0;
            break;
        case 'm':
            multiple_chunker = 1;
            break;
        case 'M':
            masterfname = boptarg;
            break;
#ifdef EXTRA_OPTIONS
        case 'n':
            problname = boptarg;
            break;
        case 'p':
            do_profile = 1;
            break;
#endif
        case 'P':
            poolfname = boptarg;
            break;
        case 'o':
            outfname = boptarg;
            break;
#ifdef EXTRA_OPTIONS
        case 'q':
            dontcutroot = 1;
            break;
        case 'Q':
            valid_edges = 0;
            break;
#endif
        case 'r':
            gridsize = atoi(boptarg);
            break;
        case 'R':
            restartfname = boptarg;
            break;
        case 's':
            seed = atoi (boptarg);
            break;
        case 'S':
            probfname = boptarg;
            break;
        case 't':
            tourfname =  boptarg;
            break;
        case 'u':
            initial_ub = atof (boptarg);
            break;
        case 'U':
            usebranchcliques = 0;
            break;
        case 'v':
            run_silently = 0;
            break;
#ifdef EXTRA_OPTIONS
        case 'V':
            run_fast_cuts = 1;
            maxchunksize = 0;
            break;
#endif
        case 'w':
            just_subtour_and_blossom = 1;
            break;
        case 'X':
            xfname = boptarg;
            break;
        case 'x':
            unlink_files = 1;
            break;
#ifdef EXTRA_OPTIONS
        case 'y':
            simple_branching = 1;
            break;
        case 'Y':
            standalone_branch = 1;
            break;
        case 'z':
            want_rcnearest = atoi (boptarg);
            break;
        case 'Z':
            save_proof = 1;
            break;
#endif
        case 'N':
            inorm = atoi (boptarg);
            switch (inorm) {
            case 0: norm = CC_MAXNORM; break;
            case 1: norm = CC_EUCLIDEAN_CEIL; break;
            case 2: norm = CC_EUCLIDEAN; break;
            case 3: norm = CC_EUCLIDEAN_3D; break;
            case 4: norm = CC_USER; break;
            case 5: norm = CC_ATT; break;
            case 6: norm = CC_GEOGRAPHIC; break;
            case 7: norm = CC_MATRIXNORM; break;
            case 8: norm = CC_DSJRANDNORM; break;
            case 9: norm = CC_CRYSTAL; break;
            case 10: norm = CC_SPARSE; break;
            case 11: norm = CC_RHMAP1; break;
            case 12: norm = CC_RHMAP2; break;
            case 13: norm = CC_RHMAP3; break;
            case 14: norm = CC_RHMAP4; break;
            case 15: norm = CC_RHMAP5; break;
            case 16: norm = CC_EUCTOROIDAL; break;
            case 17: norm = CC_GEOM; break;
            default:
                usage (av[0]);
                return 1;
            }
            tsplib_in = 0;
            break;
        case CC_BIX_GETOPT_UNKNOWN:
        case '?':
        default:
            usage (av[0]);
            return 1;
        }
    if (boptind < ac) {
        datfname = av[boptind++];
    }

    if (boptind != ac) {
        usage (av[0]);
        return 1;
    }

    if (datfname == (char *) NULL && nnodes_want == 0 && 
        probfname == (char *) NULL && edgefname == (char *) NULL &&
        masterfname == (char *) NULL && grunthostname == (char *) NULL) {
        usage (av[0]);
        return 1;
    }

    if (datfname == (char *) NULL && masterfname == (char *) NULL &&
        edgefname != (char *) NULL) {
        fprintf (stderr, "cannot give an edge file without a dat file or a master file\n");
        return 1;
    }

    if (eliminate_edges < 0) {
        if (bfs_branching || dfs_branching) {
            eliminate_edges = 1;
        } else {
            eliminate_edges = 0;
        }
    }
    
    return 0;
}

static void usage (char *f)
{
    fprintf (stderr, "Usage: %s [-see below-] [dat_file]\n", f);
#ifdef EXTRA_OPTIONS
    fprintf (stderr, "   -a    eliminate edges (default if branching)\n");
    fprintf (stderr, "   -A    don't eliminate edges (default if not branching)\n");
    fprintf (stderr, "   -b    datfile in integer binary format\n");
#endif
    fprintf (stderr, "   -B    do not branch\n");
    fprintf (stderr, "   -C #  maximum chunk size in localcuts (default 16)\n");
    fprintf (stderr, "   -d    use dfs branching instead of bfs\n");
#ifdef EXTRA_OPTIONS
    fprintf (stderr, "   -D f  edgegen file for initial edge set\n");
#endif
    fprintf (stderr, "   -e f  initial edge file\n");
    fprintf (stderr, "   -E f  full edge file (must contain initial edge set)\n");
    fprintf (stderr, "   -f    write optimal tour as edge file\n");
    fprintf (stderr, "   -F f  read extra cuts from file\n");
    fprintf (stderr, "   -g h  be a grunt for boss h\n");
#ifdef EXTRA_OPTIONS
    fprintf (stderr, "   -G    eliminate edges from full edge list (if valid)\n");
#endif
    fprintf (stderr, "   -h    be a boss for the branching\n");
#ifdef EXTRA_OPTIONS
    fprintf (stderr, "   -H p  use p for the boss port\n");
#endif
    fprintf (stderr, "   -i    just solve the blossom polytope\n");
    fprintf (stderr, "   -I    just solve the subtour polytope\n");
    fprintf (stderr, "   -J #  number of tentative branches\n");
    fprintf (stderr, "   -k #  number of nodes for random problem\n");
    fprintf (stderr, "   -K h  use cut server h\n");
    fprintf (stderr, "   -l f  run in background with output in f\n");
    fprintf (stderr, "   -L    do not use long-edge branch selection\n");
    fprintf (stderr, "   -M f  master file\n");
#ifdef EXTRA_OPTIONS
    fprintf (stderr, "   -m    use multiple passes of cutting loop\n");
    fprintf (stderr, "   -n s  problem location (just a name or host:name, not a file name)\n");
    fprintf (stderr, "   -o f  output file name (for optimal tour)\n");
    fprintf (stderr, "   -p    profile run\n");
#endif
    fprintf (stderr, "   -P f  cutpool file\n");
#ifdef EXTRA_OPTIONS
    fprintf (stderr, "   -q    do not cut the root lp\n");
    fprintf (stderr, "   -Q    full edge list not valid\n");
#endif
    fprintf (stderr, "   -r #  use #x# grid for random points, no dups if #<0\n");
    fprintf (stderr, "   -R f  restart file\n");
    fprintf (stderr, "   -s #  random seed\n");
    fprintf (stderr, "   -S f  problem file\n");
    fprintf (stderr, "   -t f  tour file (in node node node format)\n");
    fprintf (stderr, "   -u v  initial upperbound\n");
    fprintf (stderr, "   -U    do not permit branching on subtour inequalities\n");
    fprintf (stderr, "   -v    verbose (turn on lots of messages)\n");
    fprintf (stderr, "   -w    just subtours and trivial blossoms\n");
    fprintf (stderr, "   -x    delete files on completion (sav pul mas)\n");
    fprintf (stderr, "   -X f  write the last root fractional solution to f\n");
#ifdef EXTRA_OPTIONS
    fprintf (stderr, "   -y    use simple cutting and branching in DFS\n");
    fprintf (stderr, "   -Y    standalone branching\n");
    fprintf (stderr, "   -Z    save the proof\n");
#endif
    fprintf (stderr, "   -N #  norm (must specify if dat file is not a TSPLIB file)\n");
    fprintf (stderr, "         0=MAX, 1=JOHNSON, 2=L2, 3=3D, 4=USER, 5=ATT, 6=GEO, 7=MATRIX,\n");
    fprintf (stderr, "         8=DSJRAND, 9=CRYSTAL, 10=SPARSE, 11-15=RH-norm 1-5, 16=TOROIDAL\n");
    fprintf (stderr, "         17=GEOM\n");
}
