#include <stdio.h>
#include <stdlib.h>

#ifdef LINUX
#include <getopt.h>
#endif

#include "eg_util.h"
#include "eg_macros.h"
#include "eg_dpseparator.h"
#include "eg_kpseparator.h"
#include "eg_dptighten.h"
#include "eg_greedykp.h"

void usage (char *program)
{
	fprintf (stdout, "Usage: %s [options]\n", program);
	fprintf (stdout, "Options:\n");
	fprintf (stdout, "     -x d   x file name.\n");
	fprintf (stdout, "     -b     use binary file format.\n");
	fprintf (stdout, "     -p lf  percentage of ball around " 
	                 "s to use for domino generation.\n");
  fprintf (stdout, "     -h s   boss host.\n");
  fprintf (stdout, "     -k n   use greedy kp heuristic (n handles).\n");
}

int parseargs (int argc,
							 char **argv,
							 int *binary_file,
							 char **boss_host,
							 char **file_name,
							 double *percentage,
							 int *use_greedy,
               int *max_handles)
{

	int c;
	char noyes[2][4] = {[0] = "no",[1] = "yes" };

	*file_name = 0;
	*binary_file = 0;
	*boss_host = 0;
	*percentage = 1.0;
	*use_greedy = 0;
  *max_handles = 2;

	while ((c = getopt (argc, argv, "x:bh:p:k:")) != EOF)
	{
		switch (c)
		{

		case 'p':
			*percentage = atof (optarg);
			break;
		case 'h':
			*boss_host = optarg;
			break;
		case 'k':
			*use_greedy=1;
      *max_handles = atoi(optarg);
			break;
		case 'b':
			*binary_file = 1;
			break;
		case 'x':
			*file_name = optarg;
			break;
		default:
			usage (argv[0]);
			return 1;
		}
	}

	if (!*file_name)
	{
		usage (argv[0]);
		return 1;
	}

	/* reporting the options */
	fprintf (stdout, "\n");
	fprintf (stdout, "Parsed Options:\n");
	fprintf (stdout, "input         : %s\n", *file_name);
	fprintf (stdout, "binary files  : %s\n", noyes[*binary_file]);
	fprintf (stdout, "percentage    : %lf\n", *percentage);
	if (*boss_host)
	   fprintf (stdout, "boss host     : %s\n", *boss_host);
	fprintf (stdout, "greedy kp     : %s\n", noyes[*use_greedy]);
	fprintf (stdout, "\n");
	fflush (stdout);

	return 0;

}

int main (int argc,
					char **argv)
{
	int rval,
	  i,
	  d,
	  binary_file,
		use_greedy;
	char *file_name = 0,
	 *boss_host;

	int nnodes,
	  norig_edges,
	  *orig_edges,
    max_handles;

	double *orig_weight;
	double percentage = 1.0;


	/**** PARSING INPUT ********************************************************/

	rval =
		parseargs (argc, argv, &binary_file, &boss_host, &file_name, &percentage,
		           &use_greedy, &max_handles);
	CHECKRVAL (rval);

	/**** LOADING X FILE *******************************************************/

	if (binary_file)
	{
		rval =
			loadBGraph (file_name, &nnodes, &norig_edges, &orig_edges, &orig_weight);
		CHECKRVAL (rval);
	}
	else
	{
		rval =
			loadGraph (file_name, &nnodes, &norig_edges, &orig_edges, &orig_weight);
		CHECKRVAL (rval);
	}

	/**** CALLING DPSEPARATOR ***************************************************/

  if (!use_greedy)
	{

     int nIneq=0,
	      *ndominoes,
	      **naset,
	      **nbset,
	      *nhandle,
	      ***aset,
	      ***bset,
	      **handle;

	   rval = DPseparator (nnodes, norig_edges, orig_edges, orig_weight, &nIneq,
		   								   &ndominoes, &naset, &nbset, &nhandle, &aset, &bset,
		   								   &handle, boss_host, percentage, 10.0);
	   CHECKRVAL (rval);

	   /* here we call the tightening.... just to check */
   	 if (nIneq)
	   {
		   int ln, nd;
		   int *new_n_aset,
		    *new_n_bset,
		     new_n_handle,
		   **new_aset,
		   **new_bset,
		    *new_handle;
		   double violation;
		   for (ln = 0; ln < nIneq; ln++)
		   {
			   rval = DPtighten (nnodes, norig_edges, orig_edges, orig_weight,
												   ndominoes[ln], naset[ln], nbset[ln], nhandle[ln],
												   aset[ln], bset[ln], handle[ln], &new_n_aset,
												   &new_n_bset, &new_n_handle, &new_aset, &new_bset,
												   &new_handle, &violation);
			   if (new_aset)
			   {
				   for (nd = ndominoes[ln]; nd--;)
				   {
					   free (new_aset[nd]);
					   free (new_bset[nd]);
				   }
				   free (new_aset);
				   free (new_bset);
				   free (new_n_aset);
				   free (new_n_bset);
				   free (new_handle);
			   }
		   }
	   }

	   if (nIneq)
	   {
		   for (i = 0; i < nIneq; i++)
		   {
			   for (d = 0; d < ndominoes[i]; d++)
			   {
				    EGfree (aset[i][d]);
				    EGfree (bset[i][d]);
			   }
			   EGfree (naset[i]);
			   EGfree (nbset[i]);
			   EGfree (handle[i]);
			   EGfree (aset[i]);
			   EGfree (bset[i]);
		   }
		   EGfree (ndominoes);
		   EGfree (naset);
		   EGfree (nbset);
		   EGfree (aset);
		   EGfree (bset);
		   EGfree (handle);
		   EGfree (nhandle);
	   }
	}

	/**** CALLING KPSEPARATOR ***************************************************/

  if (use_greedy)
	{

     int nineq = 0,
         *nhandles = 0,
         **handle_size = 0,
         ***handles = 0,
         *nteeth = 0,
         **teeth_size = 0,
         ***teeth = 0,
         **teeth_k = 0,
         ***teeth_handle = 0,
         ***teeth_nhalf = 0,
         ****teeth_halves = 0;

	   rval = KPseparator (max_handles,
                         nnodes, 
                         norig_edges, 
                         orig_edges, 
                         orig_weight, 
                         &nineq,
                         &nhandles,
                         &handle_size,
                         &handles,
                         &nteeth, 
                         &teeth_size,
                         &teeth,
                         &teeth_k,
                         &teeth_handle,
                         &teeth_nhalf,
                         &teeth_halves, 
		                     boss_host, 
                         percentage);
	   CHECKRVAL (rval);

     for(i=0; i < nineq; i++)
        EGfreePKPCB(nhandles[i],
                    handle_size[i],
                    handles[i],
                    nteeth[i],
                    teeth_size[i],
                    teeth[i],
                    teeth_k[i],
                    teeth_handle[i],
                    teeth_nhalf[i],
                    teeth_halves[i]);

     free(nhandles);
     free(handle_size);
     free(handles);
     free(nteeth);
     free(teeth_size);
     free(teeth);
     free(teeth_k);
     free(teeth_handle);
     free(teeth_nhalf);
     free(teeth_halves);

	}

	/**** FREE ALLOCATIONS *****************************************************/


	EGfree (orig_edges);
	EGfree (orig_weight);

	return 0;

}
