#include <stdio.h>
#include "eg_mempool.h"

//#include "tsp.h"

#ifndef CCtsp_DOMINO_PORT
#define CCtsp_DOMINO_PORT ((unsigned short) 24869)
#endif

#ifndef CCtsp_DOMINO_WORK
#define CCtsp_DOMINO_WORK        'A'
#endif
#ifndef CCtsp_DOMINO_GRAPH
#define CCtsp_DOMINO_GRAPH       'G'
#endif
#ifndef CCtsp_DOMINO_NO
#define CCtsp_DOMINO_NO          'N'
#endif
#ifndef CCtsp_DOMINO_RECEIVE
#define CCtsp_DOMINO_RECEIVE     'R'
#endif
#ifndef CCtsp_DOMINO_SEND
#define CCtsp_DOMINO_SEND        'S'
#endif
#ifndef CCtsp_DOMINO_WAIT
#define CCtsp_DOMINO_WAIT        'W'
#endif
#ifndef CCtsp_DOMINO_YES
#define CCtsp_DOMINO_YES         'Y'
#endif
#ifndef CCtsp_DOMINO_EXIT
#define CCtsp_DOMINO_EXIT        'X'
#endif

#include "bc_util.h"
#include "eg_util.h"

#include "eg_mempool.h"
#include "eg_list.h"
#include "eg_dgraph.h"

#include "dp_config.h"

#include "eg_ddomino.h"
#include "eg_nettype.h"
#include "eg_util.h"

#define DOM_STAT_OPEN 0
#define DOM_STAT_DONE 1
#define DOM_STAT_WORK 2

int main (int ac,
					char **av)
{
	int id,
	  gid,
	  domino_k;
	unsigned int i,
	  nremain,
	  curloc,
	  try;
	int rval = 0,
	  ndominoes;
	CC_SPORT *lport = (CC_SPORT *) NULL;
	CC_SFILE *s;
	double rtime,
	  cumtime = 0.0,
	  graph_time = 0.0;
	unsigned int *status = (unsigned int *) NULL;
	char request;

	int new_id = -1,
	  current_g_id = -1;
	unsigned int nnodes = 0,
	  nedges = 0,
	 *edges = 0;
	double *weight = 0;
	double ddp_heuristic_maxtime;

	EGlist_t *D;
	EGlistNode_t *l_it;
	EGmemPool_t *mem;
	EGnetDdom_t *net_ddom;

	mem = EGnewMemPool (8192, EGmemPoolNewSize, EGmemPoolNewSize (1));
	D = EGnewList (mem);

	if (ac != 1)
	{
		fprintf (stderr, "Usage: %s (no arguments)\n", av[0]);
		rval = 1;
		goto CLEANUP;
	}

	printf ("BEGINNING DOMINO NET PROCESSING\n\n");
	fflush (stdout);

	lport = CCutil_snet_listen (CCtsp_DOMINO_PORT);
	if (lport == (CC_SPORT *) NULL)
	{
		fprintf (stderr, "CCutil_snet_listen failed\n");
		rval = 1;
		goto CLEANUP;
	}

	while (1)
	{

		/* First, wait until we receive a request of type CCtsp_DOMINO_GRAPH.
		 * or, until we are told to shut down.                                */

		request = 0;
		do
		{
			s = CCutil_snet_receive (lport);
			if (!s)
			{
				fprintf (stderr, "CCutil_snet_receive failed, ignoring\n");
				continue;
			}

			if (CCutil_sread_char (s, &request))
			{
				fprintf (stderr, "CCutil_sread_char failed, abort con\n");
				CCutil_sclose (s);
				continue;
			}

			switch (request)
			{
			case CCtsp_DOMINO_RECEIVE:
				rval = CCutil_swrite_char (s, CCtsp_DOMINO_NO);
				CCcheck_rval (rval, "CCutil_swrite_char failed (NO)");
				CCutil_sclose (s);
				break;
			case CCtsp_DOMINO_EXIT:
				printf ("Shutting down the domino boss\n");
				fflush (stdout);
				CCutil_sclose (s);
				rval = 1;
				goto CLEANUP;
			case CCtsp_DOMINO_GRAPH:
				break;
			case CCtsp_DOMINO_SEND:
				rval = CCutil_swrite_char (s, CCtsp_DOMINO_NO);
				CCcheck_rval (rval, "CCutil_swrite_char failed (NO)");
				CCutil_sclose (s);
				break;
				//fprintf (stderr, "No graph, cannot send dominos\n");
				//CCutil_sclose (s);
				//rval = 1;  goto CLEANUP;
			default:
				fprintf (stderr, "Invalid request %c\n", request);
			}
		} while (request != CCtsp_DOMINO_GRAPH);

		/* A request to send us a graph has come in. Receive the graph.       */

		/* Get the graph number:                                              */

		rval = CCutil_sread_int (s, &new_id);
		CCcheck_rval (rval, "receive_current_g_id failed");

		if (new_id <= current_g_id)
			new_id = current_g_id + 1;

		current_g_id = new_id;

		fprintf (stdout, "Domboss: New graph id = %d.\n", current_g_id);
		fflush (stdout);

		rval = CCutil_sread_int (s, &domino_k);
		CCcheck_rval (rval, "receive_domino_k failed");

		rval = CCutil_sread_double (s, &ddp_heuristic_maxtime);
		CCcheck_rval (rval, "receive_heuristic_maxtime failed");

		rval = EGreceiveNetGraph (s, &nnodes, &nedges, &edges, &weight, mem);
		CCcheck_rval (rval, "receive_graph failed");

		fprintf (stdout, "Domboss: nnodes = %d   nedges = %d.\n", nnodes, nedges);
		fflush (stdout);

		/* For each node we maintain a status (to keep track of if 
		 * the cycle corresponding to it has been obtained                    */

		status = CC_SAFE_MALLOC (nnodes, unsigned int);
		CCcheck_NULL (status, "out of memory for status array");

		for (i = 0; i < nnodes; i++)
			status[i] = DOM_STAT_OPEN;

		nremain = nnodes;
		curloc = 0;

		/* We wait until all of the node problems have been solved.          */

		while (nremain)
		{
			request = 0;
			do
			{

				/* Open up a port                                            */

				s = CCutil_snet_receive (lport);
				if (!s)
				{
					fprintf (stderr, "CCutil_snet_receive failed, ignoring\n");
					continue;
				}

				/* Get the next request.                                     */

				if (CCutil_sread_char (s, &request))
				{
					fprintf (stderr, "CCutil_sread_char failed, abort con\n");
					CCutil_sclose (s);
					continue;
				}

				/* We wait for the grunts to ask for some work or for them to
				 * send us dominoes. That is, we wait for the signal 
				 * DOMINO_RECEIVE.                                          */

				switch (request)
				{
				case CCtsp_DOMINO_RECEIVE:
					rval = CCutil_swrite_char (s, CCtsp_DOMINO_YES);
					CCcheck_rval (rval, "CCutil_swrite_char failed (YES)");
					break;
				case CCtsp_DOMINO_EXIT:
					printf ("Shutting down the domino boss\n");
					fflush (stdout);
					CCutil_sclose (s);
					rval = 1;
					goto CLEANUP;
				case CCtsp_DOMINO_GRAPH:
					fprintf (stderr, "Cannot receive new graph\n");
					CCutil_sclose (s);
					rval = 1;
					goto CLEANUP;
				case CCtsp_DOMINO_SEND:
					rval = CCutil_swrite_char (s, CCtsp_DOMINO_NO);
					CCcheck_rval (rval, "CCutil_swrite_char failed (NO)");
					CCutil_sclose (s);
					break;
				default:
					fprintf (stderr, "Invalid request %c\n", request);
				}
			} while (request != CCtsp_DOMINO_RECEIVE);

			/* We ask what graph id it has in memory.                         */

			rval = CCutil_sread_int (s, &gid);
			if (rval)
			{
				fprintf (stderr, "CCutil_sread_int failed, abort con\n");
				rval = 0;
				goto CLOSE_CONN;
			}

			/* We ask what node the grunt is solving the problem for.         */

			rval = CCutil_sread_int (s, &id);
			if (rval)
			{
				fprintf (stderr, "CCutil_sread_int failed, abort con\n");
				rval = 0;
				goto CLOSE_CONN;
			}

			/* If the grunt replies '-1' it means that he is not solving any */

			if (id != -1)
			{

				rval = CCutil_sread_double (s, &rtime);

				if (rval)
				{
					fprintf (stderr, "CCutil_sread_double failed, abort con\n");
					rval = 0;
					goto CLOSE_CONN;
				}

				/* If the grunt has a graph different than what we have.     */
				if (gid != current_g_id)
				{
					printf ("Finished node with graph %d, ignoring\n", gid);
					fflush (stdout);

					rval = CCutil_swrite_char (s, CCtsp_DOMINO_NO);
					CCcheck_rval (rval, "CCutil_swrite_char failed (NO)");
				}

				/* If the grunt solved a node which has already been solved. */
				else if (status[id] == DOM_STAT_DONE)
				{
					rval = CCutil_swrite_char (s, CCtsp_DOMINO_NO);
					CCcheck_rval (rval, "CCutil_swrite_char failed (NO)");

					printf ("Finished completed node %d, ignoring\n", id);
					fflush (stdout);
				}

				/* We accept the dominoes that the grunt has.                */
				else
				{
					rval = CCutil_swrite_char (s, CCtsp_DOMINO_YES);
					CCcheck_rval (rval, "CCutil_swrite_char failed (YES)");

					rval = CCutil_sread_int (s, &ndominoes);
					CCcheck_rval (rval, "receive_ndominoes failed");

					for (i = 0; i < (unsigned) ndominoes; i++)
					{
						rval = EGreceiveNetDomino (s, &net_ddom, mem);
						CCcheck_rval (rval, "receive_dominoes failed");
						EGlistPushBack (D, net_ddom);
					}

					status[id] = DOM_STAT_DONE;
					cumtime += rtime;
					graph_time += rtime;
					nremain--;

					printf
						("\rDONE %5d:  %6.2f sec, %9.2f graph, %9.2f total, %5d remaining.",
						 id, rtime, graph_time, cumtime, nremain);
					fflush (stdout);
				}
			}

			/* We must determine which job is to be assigned to the grunt. We 
			 * will assign the one with lowest cardinality which is still
			 * available. We loop around so as to re-assign ids.              */

			try = 0;
			while (status[curloc % nnodes] == DOM_STAT_DONE && try < nnodes)
			{
				curloc++;
				try++;
			}

			/* If there is some job which is not done, assign it.             */

			if (try < nnodes)
			{
				if (gid != current_g_id)
				{
					rval = CCutil_swrite_char (s, CCtsp_DOMINO_GRAPH);
					CCcheck_rval (rval, "CCutil_swrite_char failed (GRAPH)");
										/*** SEND GRAPH ***/
					rval = CCutil_swrite_int (s, current_g_id);
					CCcheck_rval (rval, "send_graph_id failed");
					rval = EGsendNetGraph (s, nnodes, nedges, edges, weight);
					CCcheck_rval (rval, "send_graph failed");
					rval = CCutil_swrite_int (s, domino_k);
					CCcheck_rval (rval, "send_domino_k failed");
					rval = CCutil_swrite_double (s, ddp_heuristic_maxtime);
					CCcheck_rval (rval, "send_ddp_heuristic_maxtime failed");
				}
				else
				{
					rval = CCutil_swrite_char (s, CCtsp_DOMINO_WORK);
					CCcheck_rval (rval, "CCutil_swrite_char failed");
				}

				rval = CCutil_swrite_int (s, (int) (curloc % nnodes));
				CCcheck_rval (rval, "CCutil_swrite_int failed");

				status[curloc % nnodes] = DOM_STAT_WORK;

				// printf ("  New = %u\n", curloc % nnodes); fflush (stdout);

				curloc++;
			}

			/* If all jobs are done, tell the grunt to wait.                  */

			else
			{
				printf ("\n");
				fflush (stdout);
				rval = CCutil_swrite_char (s, CCtsp_DOMINO_WAIT);
				CCcheck_rval (rval, "CCutil_swrite_char failed (WAIT)");
			}

		CLOSE_CONN:

			//printf ("  Closing connection (1).\n"); fflush (stdout);
			CCutil_sclose (s);
		}

		printf ("FINISHED Graph %d: %.2f seconds.\n\n", current_g_id, graph_time);
		fflush (stdout);

		/* We now communicate with concorde.                                  */

		request = 0;
		do
		{

			/* open the port                                                  */

			//printf ("  Opening Port (2).\n"); fflush (stdout);

			s = CCutil_snet_receive (lport);
			if (!s)
			{
				fprintf (stderr, "CCutil_snet_receive failed, ignoring\n");
				continue;
			}

			/* wait for request                                               */

			if (CCutil_sread_char (s, &request))
			{
				fprintf (stderr, "CCutil_sread_char failed, abort con\n");
				CCutil_sclose (s);
				continue;
			}

			/* wait until we get the 'DOMINO_SEND' request.                   */

			switch (request)
			{
			case CCtsp_DOMINO_RECEIVE:
				rval = CCutil_swrite_char (s, CCtsp_DOMINO_NO);
				CCcheck_rval (rval, "CCutil_swrite_char failed (NO)");
				CCutil_sclose (s);
				break;
			case CCtsp_DOMINO_EXIT:
				printf ("Shutting down the domino boss\n");
				fflush (stdout);
				CCutil_sclose (s);
				rval = 1;
				goto CLEANUP;
			case CCtsp_DOMINO_GRAPH:
				fprintf (stderr, "Cannot receive new graph\n");
				CCutil_sclose (s);
				rval = 1;
				goto CLEANUP;
			case CCtsp_DOMINO_SEND:
				break;
			default:
				fprintf (stderr, "Invalid request %c\n", request);
			}
		} while (request != CCtsp_DOMINO_SEND);

		/* send the dominoes back to concorde                                 */

		printf ("Sending %u dominos ... ", D->size);
		fflush (stdout);

		rval = CCutil_swrite_char (s, CCtsp_DOMINO_SEND);
		CCcheck_rval (rval, "send DOMINO_SEND failed");

		rval = CCutil_swrite_int (s, (int) D->size);
		CCcheck_rval (rval, "send ndominoes failed");

		for (l_it = D->begin; l_it; l_it = l_it->next)
		{
			net_ddom = (EGnetDdom_t *) (l_it->this);
			rval = EGsendNetDomino (s, net_ddom);
			CCcheck_rval (rval, "send_dominos failed");
		}

		printf ("done.\n");
		fflush (stdout);

		CCutil_sclose (s);

		EGlistClearMP (D, EGfreeNetDdomMP, mem);

		EGmemPoolFree (edges, sizeof (unsigned int) * 2 * nedges, mem);
		EGmemPoolFree (weight, sizeof (double) * nedges, mem);

		nedges = nnodes = 0;
		edges = 0;
		weight = 0;

		graph_time = 0;

	}

CLEANUP:

	EGfreeList (D);
	EGfreeMemPool (mem);
	CCutil_snet_unlisten (lport);
	EGfree (status);

	return rval;
}
