/* EGlib "Efficient General Library" provides some basic structures and
 * algorithms commons in many optimization algorithms.
 *
 * Copyright (C) 2005 Daniel Espinoza and Marcos Goycoolea.
 * 
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but 
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY 
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public 
 * License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA 
 * */
/* ========================================================================= */
/* Example of use of EGuGraph 
 *
 * This program ilustrate the use of EGuGraph_t structures, how to creatte (in
 * this case a random) graph, display it, and erase it afterwards.
 * */
/* ========================================================================= */

#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include "eg_config.h"
#if OS == LINUX
#include <getopt.h>
#endif
#include "eg_mempool.h"
#include "eg_ugraph.h"

void usage (char *program)
{
	fprintf (stderr, "Usage: %s [options]\n", program);
	fprintf (stderr, "Options:\n");
	fprintf (stderr, "     -n n   number of nodes.\n");
	fprintf (stderr, "     -m n   number of edges.\n");
	fprintf (stderr, "     -s n   initial random seed.\n");

}

static int parseargs (int argc,
											char **argv,
											unsigned int *n,
											unsigned int *m,
											unsigned int *seed)
{

	int c;

	while ((c = getopt (argc, argv, "n:m:s:")) != EOF)
	{
		switch (c)
		{
		case 's':
			*seed = atoi (optarg);
			break;
		case 'n':
			*n = atoi (optarg);
			break;
		case 'm':
			*m = atoi (optarg);
			break;
		default:
			usage (argv[0]);
			return 1;
		}
	}

	/* reporting the options */
	fprintf (stderr, "Parsed Options:\n");
	fprintf (stderr, "   n    : %u\n", *n);
	fprintf (stderr, "   m    : %u\n", *m);
	fprintf (stderr, "   seed : %u\n", *seed);
	return 0;

}

int main (int argc,
					char **argv)
{
	/* local variables */
	unsigned i;
	int status,
	  tail,
	  head;
	unsigned int n = 0,
	  m = 0;
	double *weight = 0;
	EGmemPool_t *mem;
	EGuGraph_t *G;
	unsigned int seed = time (0);
	EGuGraphNode_t **nodes = 0;

	/* now we parse the arguments, n is the number of nodes to create and m is
	 * the number of edges to create */
	status = parseargs (argc, argv, &n, &m, &seed);
	srand (seed);
	CHECKRVAL (status);

	/* basic set-up */
	mem = EGnewMemPool (100, EGmemPoolNewSize, EGmemPoolNewSize (1));
	G = EGnewUGraph (mem);
	G->id = 0;

	/* if there is a node we create them */
	if (n)
	{

		/* to access the nodes we create this array of pointers to them, this is
		 * nod needed, is yust for convinence of use here */
		nodes =
			(EGuGraphNode_t **) EGmemPoolMalloc (mem, sizeof (EGuGraphNode_t *) * n);

		/* we create all the nodes */
		for (i = 0; i < n; i++)
		{
			nodes[i] = EGuGraphAddNode (G, 0);
		}

		/* and now we create edges with endpoints random */
		if (m)
		{
			weight = (double *) EGmemPoolMalloc (mem, sizeof (double) * m);
		}
		for (i = 0; i < m; i++)
		{
			head = random () % n;
			tail = random () % n;
			weight[i] = random ();
			weight[i] /= EGRAND_MAX;
			EGuGraphAddEdge (G, weight + i, nodes[head], nodes[tail]);
		}

		/* we don't need any longer the array of pointers to nodes */
	}

	/* now we display the graph */
	EGuGraphDisplay (G, 0, 0, 0, stderr);

	/* now we will unattach node by node and print the resoult */
	for (i = 0; i < n; i++)
	{
		EGuGraphUnattachNode (G, nodes[i]);
		EGuGraphDisplay (G, 0, 0, 0, stderr);
	}

	/* now we will attach node by node (in exactly ther inverse order than we
	 * unattach then so to obtain a consistent graph at each step, you could add
	 * then in any order and print the final resoult, but the intermidium graphs
	 * wouldn't be consistents. */
	for (i = n; i--;)
	{
		EGuGraphAttachNode (G, nodes[i]);
		EGuGraphDisplay (G, 0, 0, 0, stderr);
	}

	/* clear it */
	EGuGraphClear (G, nullFree, nullFree, nullFree);

	/* now we display the graph */
	EGuGraphDisplay (G, 0, 0, 0, stderr);

	/* freing all memory */
	EGmemPoolFree (nodes, sizeof (EGuGraphNode_t *) * n, mem);
	EGfreeUGraph (G);
	EGfreeMemPool (mem);
	return 0;
}
