/* EGlib "Efficient General Library" provides some basic structures and
 * algorithms commons in many optimization algorithms.
 *
 * Copyright (C) 2005 Daniel Espinoza and Marcos Goycoolea.
 * 
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but 
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY 
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public 
 * License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA 
 * */
#include "eg_ugraph.h"

EGuGraphNode_t *EGuGraphOtherEnd (EGuGraphEdge_t * e,
																	EGuGraphNode_t * n)
{

	if (e->head == n)
		return (e->tail);

	return (e->head);

}

EGlistNode_t *EGuGraphOtherEit (EGuGraphEdge_t * e,
																EGuGraphNode_t * n)
{

	if (e->head == n)
		return (e->tail_cn);

	return (e->head_cn);

}

int EGuGraphSetOtherEit (EGuGraphEdge_t * e,
												 EGuGraphNode_t * n,
												 EGlistNode_t * other_eit)
{

	if (e->head == n)
	{
		e->tail_cn = other_eit;
		return 0;
	}

	e->head_cn = other_eit;

	return 0;

}

/* ========================================================================= */
/* attach/unattach does not change id's, but it does mantain the number of nodes
 * and edges. */
int EGuGraphAttachNode (EGuGraph_t * G,
												EGuGraphNode_t * n)
{

	/* local variable */
	EGlistNode_t *eit,
	 *other_eit;
	EGuGraphEdge_t *e;
	EGuGraphNode_t *other_n;

	/* update total number of edges */
	G->nedges += n->edges->size;
	EGlistAttach (G->nodes, n->cn);

	for (eit = n->edges->begin; eit; eit = eit->next)
	{

		e = (EGuGraphEdge_t *) (eit->this);
		other_eit = EGuGraphOtherEit (e, n);

		if (!other_eit)
		{
			other_n = EGuGraphOtherEnd (e, n);
			EGlistPushBack (other_n->edges, e);
			EGuGraphSetOtherEit (e, n, other_n->edges->end);
		}

	}															/* end for eit */

	/* ending */
	return 0;
}

int EGuGraphAttachEdge (EGuGraph_t * G,
												EGuGraphEdge_t * e)
{

	/* update adjacency list of tail and head and update number of edges */
	EGlistAttach (e->head->edges, e->head_cn);
	EGlistAttach (e->tail->edges, e->tail_cn);
	G->nedges++;

	/* ending */
	return 0;
}

int EGuGraphUnattachNode (EGuGraph_t * G,
													EGuGraphNode_t * n)
{

	/* local variable */
	EGlistNode_t *eit,
	 *other_eit;
	EGuGraphEdge_t *e;
	EGuGraphNode_t *other_n;

	/* update total number of edges */
	G->nedges -= n->edges->size;
	EGlistUnattach (G->nodes, n->cn);

	for (eit = n->edges->begin; eit; eit = eit->next)
	{
		e = (EGuGraphEdge_t *) (eit->this);
		other_n = EGuGraphOtherEnd (e, n);
		other_eit = EGuGraphOtherEit (e, n);
		if (other_eit)
		{
			EGlistErase (other_n->edges, other_eit, nullFree);
			EGuGraphSetOtherEit (e, n, 0);
		}
	}															/* end for eit */

	/* ending */
	return 0;
}

int EGuGraphUnattachEdge (EGuGraph_t * G,
													EGuGraphEdge_t * e)
{

	/* update adjacency list of tail and head and update number of edges */
	EGlistUnattach (e->head->edges, e->head_cn);
	EGlistUnattach (e->tail->edges, e->tail_cn);
	G->nedges--;

	/* ending */
	return 0;
}


/* ========================================================================= */
EGuGraph_t *EGnewUGraph (EGmemPool_t * mem)
{

	/* local variables */
	EGuGraph_t *G;

	/* basic set-up */
	G = (EGuGraph_t *) EGmemPoolMalloc (mem, sizeof (EGuGraph_t));
	memset (G, 0, sizeof (EGuGraph_t));
	G->nodes = EGnewList (mem);
	G->mem = mem;
	G->id = UINT_MAX;

	/* ending */
	return G;
}

EGuGraphNode_t *EGnewUGraphNode (EGmemPool_t * mem)
{

	/* local variables */
	EGuGraphNode_t *v;

	/* basic set-up */
	v = (EGuGraphNode_t *) EGmemPoolMalloc (mem, sizeof (EGuGraphNode_t));
	memset (v, 0, sizeof (EGuGraphNode_t));
	v->edges = EGnewList (mem);
	v->id = UINT_MAX;

	/* ending */
	return v;
}

EGuGraphEdge_t *EGnewUGraphEdge (EGmemPool_t * mem)
{

	/* local variables */
	EGuGraphEdge_t *e;

	/* basic set-up */
	e = (EGuGraphEdge_t *) EGmemPoolMalloc (mem, sizeof (EGuGraphEdge_t));
	memset (e, 0, sizeof (EGuGraphEdge_t));
	e->id = UINT_MAX;

	/* ending */
	return e;
}

/* ========================================================================= */
void EGfreeUGraph (void *G)
{

	/* clear the structures */
	EGuGraphClear (G, nullFree, nullFree, nullFree);

	/* and now we free the structure itself */
	EGfreeList (((EGuGraph_t *) G)->nodes);
	EGmemPoolFree (G, sizeof (EGuGraph_t), ((EGuGraph_t *) G)->mem);

	/* ending */
	return;
}

void EGfreeUGraphNode (void *v)
{

	/* local variables */
	MESSAGE (9, "Free Node %u:", ((EGuGraphNode_t *) v)->id);
	EGmemPool_t *mem = ((EGuGraphNode_t *) v)->edges->mempool;

	/* we free the edges */
	EGlistClearMP (((EGuGraphNode_t *) v)->edges, EGfreeUGraphEdge, mem);
	EGfreeList (((EGuGraphNode_t *) v)->edges);

	/* free the structure itself */
	EGmemPoolFree (v, sizeof (EGuGraphNode_t), mem);

	/* ending */
	return;
}

void EGfreeUGraphEdge (void *v,
											 EGmemPool_t * mem)
{

	MESSAGE (UG_LEVEL, "Free Edge %u:(%u,%u)",
					 ((EGuGraphEdge_t *) v)->id,
					 ((EGuGraphEdge_t *) v)->head->id, ((EGuGraphEdge_t *) v)->tail->id);
	EGmemPoolFree (v, sizeof (EGuGraphEdge_t), mem);
	return;
}

/* ========================================================================= */
/* clear functions leave the structure 'as new' */
void EGuGraphClear (EGuGraph_t * G,
										EGfree_f userFreeEdgeData,
										EGfree_f userFreeNodeData,
										EGfree_f userFreeUGraphData)
{
	MESSAGE (UG_LEVEL, "Entering for graph (%p:%u)", (void *) G, G->id);
	/* free the graph internal data */
	if (userFreeUGraphData)
	{
		userFreeUGraphData (G->data);
		G->data = 0;
	}

	/* now we loop through the nodes */
	while (G->nodes->begin)
	{
		EGuGraphEraseNode (G, (EGuGraphNode_t *) G->nodes->begin->this,
											 userFreeEdgeData, userFreeNodeData);
	}

	/* ending */
	G->node_id_max = 0;
	G->edge_id_max = 0;
	MESSAGE (UG_LEVEL, "Ending for graph");
	return;
}

void EGuGraphClearMP (EGuGraph_t * G,
											EGfreeMP_f userFreeEdgeDataMP,
											EGfreeMP_f userFreeNodeDataMP,
											EGfreeMP_f userFreeUGraphDataMP,
											EGmemPool_t * edge_data_mem,
											EGmemPool_t * node_data_mem,
											EGmemPool_t * uGraph_data_mem)
{

	/* free the graph internal data */
	if (userFreeUGraphDataMP)
	{
		userFreeUGraphDataMP (G->data, uGraph_data_mem);
		G->data = 0;
	}

	/* now we loop through the nodes */
	while (G->nodes->begin)
	{
		EGuGraphEraseNodeMP (G, (EGuGraphNode_t *) G->nodes->begin->this,
												 userFreeEdgeDataMP, userFreeNodeDataMP, node_data_mem,
												 edge_data_mem);
	}

	/* ending */
	G->node_id_max = 0;
	G->edge_id_max = 0;
	return;
}

/* ========================================================================= */
/* erase functions clear all memory for the structure and it's sub-structure, it
 * has the ability to call 'frees' for the internal 'data' pointers if wanted */
void EGuGraphEraseEdge (EGuGraph_t * G,
												EGuGraphEdge_t * e,
												EGfree_f userFreeEdgeData)
{

	MESSAGE (UG_LEVEL, "Entering for graph (%p:%u) edge (%p:%u)", (void *) G,
					 G->id, (void *) e, e->id);
	/* decrease the number of edges and free the data if needed */
	G->nedges--;
	if (userFreeEdgeData)
	{
		MESSAGE (UG_LEVEL, "Freing Edge Data %p", e->data);
		userFreeEdgeData (e->data);
	}

	/* now we have to free the head_cn and tail_cn and the edge itself */
	if (e->tail != e->head)
	{
		MESSAGE (UG_LEVEL, "Removing head_cn %p from list %p", (void *) e->head_cn,
						 (void *) e->head->edges);
		EGlistErase (e->head->edges, e->head_cn, nullFree);
		EXITL (1, e->tail->edges->begin && !e->tail->edges->end, "bad!!");
		MESSAGE (UG_LEVEL, "Removing tail_cn %p from list %p",
						 (void *) e->tail_cn, (void *) e->tail->edges);
		EGlistErase (e->tail->edges, e->tail_cn, nullFree);
		MESSAGE (UG_LEVEL, "Removing edge %p", (void *) e);
		EGfreeUGraphEdge (e, e->tail->edges->mempool);
	}
	else
	{
		MESSAGE (UG_LEVEL, "Removing tail_cn %p", (void *) e->tail_cn);
		EXITL (1, e->tail->edges->begin && !e->tail->edges->end, "bad!!");
		EGlistEraseMP (e->tail->edges, e->tail_cn, EGfreeUGraphEdge,
									 e->tail->edges->mempool);
	}

	/* ending */
	MESSAGE (UG_LEVEL, "Ending for graph (%p:%u) edge.", (void *) G, G->id);
	return;
}

void EGuGraphEraseEdgeMP (EGuGraph_t * G,
													EGuGraphEdge_t * e,
													EGfreeMP_f userFreeEdgeDataMP,
													EGmemPool_t * edge_data_mem)
{

	/* decrease the number of edges and free the data if needed */
	G->nedges--;
	if (userFreeEdgeDataMP)
	{
		userFreeEdgeDataMP (e->data, edge_data_mem);
		e->data = 0;
	}

	/* now we have to free the head_cn and tail_cn and the edge itself */
	if (e->tail != e->head)
	{
		EGlistErase (e->head->edges, e->head_cn, nullFree);
		EXITL (1, e->tail->edges->begin && !e->tail->edges->end, "bad!!");
		EGlistEraseMP (e->tail->edges, e->tail_cn, EGfreeUGraphEdge,
									 e->tail->edges->mempool);
	}
	else
	{
		EXITL (1, e->tail->edges->begin && !e->tail->edges->end, "bad!!");
		EGlistEraseMP (e->tail->edges, e->tail_cn, EGfreeUGraphEdge,
									 e->tail->edges->mempool);
	}

	/* ending */
	return;
}

void EGuGraphEraseNode (EGuGraph_t * G,
												EGuGraphNode_t * v,
												EGfree_f userFreeEdgeData,
												EGfree_f userFreeNodeData)
{

	MESSAGE (UG_LEVEL, "Entering for graph (%p:%u) node (%p:%u)", (void *) G,
					 G->id, (void *) v, v->id);
	/* free user data if needed */
	if (userFreeNodeData)
	{
		userFreeNodeData (v->data);
	}

	/* now we call the erase for all edges */
	while (v->edges->begin)
	{
		EGuGraphEraseEdge (G, (EGuGraphEdge_t *) v->edges->begin->this,
											 userFreeEdgeData);
	}

	/* take out the node from the graph and free it */
	EGlistErase (G->nodes, v->cn, EGfreeUGraphNode);

	/*ending */
	return;
	MESSAGE (UG_LEVEL, "Ending for graph (%p:%u) node", (void *) G, G->id);
}

void EGuGraphEraseNodeMP (EGuGraph_t * G,
													EGuGraphNode_t * v,
													EGfreeMP_f userFreeEdgeDataMP,
													EGfreeMP_f userFreeNodeDataMP,
													EGmemPool_t * node_data_mem,
													EGmemPool_t * edge_data_mem)
{

	/* free user data if needed */
	if (userFreeNodeDataMP)
	{
		userFreeNodeDataMP (v->data, node_data_mem);
	}

	/* now we call the erase for all edges */
	while (v->edges->begin)
	{
		EGuGraphEraseEdgeMP (G, (EGuGraphEdge_t *) v->edges->begin->this,
												 userFreeEdgeDataMP, edge_data_mem);
	}

	/* take out the node from the graph and free it */
	EGlistErase (G->nodes, v->cn, EGfreeUGraphNode);

	/*ending */
	return;
}

/* ========================================================================= */
/* this create a new structure and update internal data values */
EGuGraphNode_t *EGuGraphAddNode (EGuGraph_t * G,
																 void *ndata)
{

	/* local variable */
	EGuGraphNode_t *v;

	/* BASIC SET-UP */
	v = EGnewUGraphNode (G->mem);
	EGlistPushBack (G->nodes, v);
	v->cn = G->nodes->end;
	v->data = ndata;
	v->id = G->node_id_max++;

	/* ending */
	return v;
}

EGuGraphEdge_t *EGuGraphAddEdge (EGuGraph_t * G,
																 void *edata,
																 EGuGraphNode_t * head,
																 EGuGraphNode_t * tail)
{

	/* local variables */
	EGuGraphEdge_t *e;

	/* basic set-up */
	e = EGnewUGraphEdge (G->mem);
	e->head = head;
	e->tail = tail;
	EGlistPushBack (head->edges, e);
	e->head_cn = head->edges->end;
	if (head != tail)
		EGlistPushBack (tail->edges, e);
	e->tail_cn = tail->edges->end;
	e->data = edata;
	e->id = G->edge_id_max++;
	G->nedges++;

	/* ending */
	return e;
}

/* ========================================================================= */
/* this functions compress the Id's in use */
int EGuGraphResetNodeId (EGuGraph_t * G)
{
	TEST (G && 1, "Not implemented yet");
	return (G == 0);
}

int EGuGraphResetEdgeId (EGuGraph_t * G)
{
	TEST (G && 1, "Not implemented yet");
	return (G == 0);
}


/* this function resets the node ids so they are numbered from 0 .. nnodes -1 */
int EGuGraphRenumberNodeId (EGuGraph_t * G)
{

	int cnt = 0;
	EGlistNode_t *it;
	EGuGraphNode_t *n;

	for (it = G->nodes->begin; it; it = it->next)
	{
		n = (EGuGraphNode_t *) (it->this);
		n->id = cnt;
		cnt += 1;
	}

	G->node_id_max = cnt;

	return 0;

}

/* ========================================================================= */
/* exports graph from Cook format. If no gdata, edata, or ndata, use 0. */
/* assumes that the number of nodes is G->node_id_max!! */
/* assumes that edges initialized appropriately */
/* assumes if the data != 0, then initialized as above */
int EGuGraphExport (EGuGraph_t * G,
										int *edges,
										void **gdata,
										void **edata,
										void **ndata)
{

	int cnt = 0;
	EGuGraphNode_t *n,
	 *m;
	EGuGraphEdge_t *e;
	EGlistNode_t *nit,
	 *eit;

	if (gdata)
		*gdata = G->data;

	//fprintf(stderr, "Gn = %u   Ge = %u\n", G->nodes->size, G->nedges);
	//EGuGraphDisplay (G, 0, 0, 0, stderr);

	for (nit = G->nodes->begin; nit; nit = nit->next)
	{
		n = (EGuGraphNode_t *) (nit->this);
		if (ndata)
			ndata[n->id] = n->data;

		for (eit = n->edges->begin; eit; eit = eit->next)
		{

			e = (EGuGraphEdge_t *) (eit->this);
			m = EGuGraphOtherEnd (e, n);

			if ((m->id > n->id))
			{
				edges[2 * cnt] = e->tail->id;
				edges[2 * cnt + 1] = e->head->id;
				if (edata)
					edata[cnt] = e->data;
				cnt += 1;
			}

		}
	}

	return 0;

}


/* ========================================================================= */
/* import graph from Cook format. If no gdata, edata, or ndata, use 0.  */
EGuGraph_t *EGuGraphImport (int nnodes,
														int nedges,
														int *edges,
														void *gdata,
														void **edata,
														void **ndata,
														EGmemPool_t * mem)
{

	int i;
	EGuGraph_t *G;
	EGuGraphNode_t **nodes;

	G = EGnewUGraph (mem);
	G->data = gdata;
	G->id = 0;

	nodes = (EGuGraphNode_t **) EGmalloc (sizeof (EGuGraphNode_t *) * nnodes);

	if (ndata)
		for (i = 0; i < nnodes; i++)
			nodes[i] = EGuGraphAddNode (G, ndata[i]);
	else
		for (i = 0; i < nnodes; i++)
			nodes[i] = EGuGraphAddNode (G, 0);

	if (edata)
		for (i = 0; i < nedges; i++)
			EGuGraphAddEdge (G, edata[i], nodes[edges[2 * i]],
											 nodes[edges[2 * i + 1]]);
	else
		for (i = 0; i < nedges; i++)
			EGuGraphAddEdge (G, 0, nodes[edges[2 * i]], nodes[edges[2 * i + 1]]);

	EGfree (nodes);

	return G;

}


/* ========================================================================= */
/* print in some format the graph and its internal data */
int EGuGraphDisplay (EGuGraph_t * G,
										 EGdisplay_f userDisplayUGraphData,
										 EGdisplay_f userDisplayNodeData,
										 EGdisplay_f userDisplayEdgeData,
										 FILE * file)
{

	/* local variables */
	EGlistNode_t *It1,
	 *It2;
	EGuGraphNode_t *v;
	EGuGraphEdge_t *e;

	/* this function display an output for the graph */
	/* first main graph-related displays */
	fprintf (file, "\nGraph %u: ", G->id);
#if __EG_UGRAPH_CHECK__
	fprintf (file, " (%p){nodes=%p data=%p mem=%p id=%u nedges=%u node_id_max="
					 "%u edge_id_max=%u} ", (void *) G, (void *) G->nodes,
					 (void *) G->data, (void *) G->mem, G->id, G->nedges, G->node_id_max,
					 G->edge_id_max);
#endif
	if (userDisplayUGraphData)
	{
		fprintf (file, "[");
		userDisplayUGraphData (G->data, file);
		fprintf (file, "]\n");
	}
	else
		fprintf (file, "\n");
	/* now node-related displays */
	for (It1 = G->nodes->begin; It1; It1 = It1->next)
	{
		v = (EGuGraphNode_t *) It1->this;
		fprintf (file, "Node %u: ", v->id);
#if __EG_UGRAPH_CHECK__
		fprintf (file, "(%p){edges=%p cn=%p data=%p id=%u} ", (void *) v,
						 (void *) v->edges, (void *) v->cn, (void *) v->data, v->id);
#endif
		if (userDisplayNodeData)
		{
			fprintf (file, "[");
			userDisplayNodeData (v->data, file);
			fprintf (file, "]\n");
		}
		else
			fprintf (file, "\n");
		for (It2 = v->edges->begin; It2; It2 = It2->next)
		{
			e = (EGuGraphEdge_t *) It2->this;
			fprintf (file, "%u: (%u,%u) ", e->id, e->tail->id, e->head->id);
#if __EG_UGRAPH_CHECK__
			fprintf (file, "(%p){head=%p tail=%p head_cn=%p tail_cn=%p data=%p "
							 "id=%u}\n", (void *) e, (void *) e->head, (void *) e->tail,
							 (void *) e->head_cn, (void *) e->tail_cn, (void *) e->data,
							 e->id);
#endif
			if (userDisplayEdgeData)
			{
				fprintf (file, "[");
				userDisplayEdgeData (e->data, file);
				fprintf (file, "] ");
			}
		}														/* end for It2 */
		fprintf (file, "\n");
	}															/* end for It1 */

	/* ending */
	return 0;
}

/* will empty graph G, the components will have gdata set to null, their id
   set to which component it is, and the id_max values as in G */
int EGuGraphBreakUpComponents (EGuGraph_t * G,
															 EGlist_t * G_list,
															 EGmemPool_t * mem)
{

	int i,
	  n;
	int *comps;
	int num_components = 0;

	EGuGraph_t *nG;
	EGuGraphNode_t **nodes;

	comps = (int *) EGmemPoolSMalloc (mem, int,
																		G->node_id_max);
	nodes =
		(EGuGraphNode_t **) EGmemPoolSMalloc (mem, EGuGraphNode_t *,
																					G->node_id_max);

	EGuGraphComponents (G, &num_components, nodes, comps, mem);

	for (i = 0; i < num_components; i++)
	{

		nG = EGnewUGraph (G->mem);
		nG->id = i;
		nG->node_id_max = G->node_id_max;
		nG->edge_id_max = G->edge_id_max;

		for (n = 0; n < (int) G->node_id_max; n++)
			if (comps[n] == i)
				EGuGraphUnattachNode (G, nodes[n]);

		for (n = 0; n < (int) G->node_id_max; n++)
			if (comps[n] == i)
				EGuGraphAttachNode (nG, nodes[n]);

		nG->nedges = nG->nedges / 2;

		EGlistPushBack (G_list, nG);

	}

	//fprintf(stderr, "nedges = %u   nnodes = %u\n", G->nedges, G->nodes->size);
	//EGuGraphDisplay(G, 0,0,0, stderr);

	EGmemPoolSFree (comps, int,
									G->node_id_max,
									mem);
	EGmemPoolSFree (nodes, EGuGraphNode_t *, G->node_id_max, mem);

	return 0;

}

int EGuGraphIsConnected (EGuGraph_t * G,
												 EGmemPool_t * mem)
{

	int *comps;
	int num_components = 0;
	EGuGraphNode_t **nodes;

	comps = (int *) EGmemPoolSMalloc (mem, int,
																		G->node_id_max);
	nodes =
		(EGuGraphNode_t **) EGmemPoolSMalloc (mem, EGuGraphNode_t *,
																					G->node_id_max);

	EGuGraphComponents (G, &num_components, nodes, comps, mem);

	EGmemPoolSFree (comps, int,
									G->node_id_max,
									mem);
	EGmemPoolSFree (nodes, EGuGraphNode_t *, G->node_id_max, mem);

	if (num_components > 1)
		return 0;

	return 1;

}

/* ========================================================================= */
/* get graph components */
/* assumes that nodes is an array of at least G->node_id_max EGuGraphNode_t* */
/* assumes that comps is an array of at least G->node_id_max ints */
int EGuGraphComponents (EGuGraph_t * G,
												int *num_components,
												EGuGraphNode_t ** nodes,
												int *comps,
												EGmemPool_t * mem)
{

	int i;

	EGlist_t *dfs_list;
	EGlistNode_t *it;
	EGuGraphEdge_t *e;
	EGuGraphNode_t *n,
	 *current_node;

	dfs_list = EGnewList (mem);

	memset (nodes, 0, sizeof (EGuGraphNode_t *) * G->node_id_max);
	memset (comps, -1, sizeof (int) * G->node_id_max);

	for (it = G->nodes->begin; it; it = it->next)
	{
		current_node = (EGuGraphNode_t *) (it->this);
		nodes[current_node->id] = current_node;
	}

	/* DFS */

	*num_components = 0;
	current_node = (EGuGraphNode_t *) (G->nodes->begin->this);

	while (current_node)
	{

		*num_components = *num_components + 1;
		EGlistPushBack (dfs_list, current_node);
		comps[current_node->id] = (*num_components) - 1;

		while (dfs_list->size)
		{

			current_node = (EGuGraphNode_t *) (dfs_list->end->this);
			EGlistErase (dfs_list, dfs_list->end, nullFree);

			for (it = current_node->edges->begin; it; it = it->next)
			{
				e = (EGuGraphEdge_t *) (it->this);
				n = EGuGraphOtherEnd (e, current_node);
				if (comps[n->id] == -1)
				{
					comps[n->id] = (*num_components) - 1;
					EGlistPushBack (dfs_list, n);
				}
			}

		}

		current_node = 0;
		for (i = 0; i < (int) G->node_id_max; i++)
			if (comps[i] == -1 && nodes[i])
			{
				current_node = nodes[i];
				break;
			}

	}

	EGfreeList (dfs_list);

	return 0;

}
