/* EGlib "Efficient General Library" provides some basic structures and
 * algorithms commons in many optimization algorithms.
 *
 * Copyright (C) 2005 Daniel Espinoza and Marcos Goycoolea.
 * 
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but 
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY 
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public 
 * License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA 
 * */
/* ========================================================================= */
/* Implementation of spanning tree for EGuGraph
 *
 * - 2003-06-23 
 * 						- First implementation (this uses the Tarjan algorithm for
 * 							minimum/maximum weight spanning trees ).
 * */
/* ========================================================================= */
#ifndef __EG_SPANNING_H__
#define __EG_SPANNING_H__
#include <stdlib.h>
#include <stdio.h>
#include "eg_config.h"
#include "eg_macros.h"
#include "eg_listint.h"
#include "eg_list.h"
#include "eg_mempool.h"
#include "eg_ugraph.h"
#include "eg_equiset.h"

/* ========================================================================= */
/* this is the definition of the type to be used as cost for the edsges, you can
 * change this definition at compile time. */
/* ========================================================================= */
typedef double EGspanningTreeCost_t;

/* ========================================================================= */
/** @brief minimum / maximum spanning tree algorithm.
 *
 * This function implement the Tarjan and Cheriton ( D. Cheriton and R.E.
 * Tarjan, Finding minimum spanning trees, SIAM J. Comput., 5 (1976), pp.
 * 724-742 ) algorithm for minimum/maximum weight spanning trees. 
 *
 * The type of the cost is EGspanningTreeCost_t as defined above. 
 *
 * This version assum that the user is providing (almost) all the memory for its
 * operation, and will not check if that memory exists.
 *
 * @param G is the uGraph where to perform the minimum / maximum spanning tree.
 * @param voidOS offset of the external data.
 * @param intArray internal array.
 * @param treeEdges is an array of pointers to EGuGraphEdge_t* where we will 
 * store wich edges has been used in the minimum / maximum spanning tree.
 * @param mem is a memory poool from where we will take all internal memory. 
 * @param costOS is the cost offset in the edge data structure.  
 * @param sense define if the problem is of minimization (-1) or maximization 
 * (1), note that other values will resoult in wrong answers.
 * @param value where we will store the final value of our spanning tree.
 * */
/* ========================================================================= */
int EGspanningTreeADV (EGuGraph_t * G,
											 size_t costOS,
											 size_t voidOS,
											 int sense,
											 EGuGraphEdge_t ** treeEdges,
											 void **intArray,
											 EGspanningTreeCost_t * value,
											 EGmemPool_t * mem);

/* ========================================================================= */
/** @brief this function compute a maximum / minimum spanning tree. 
 *
 * This function implement the Tarjan and Cheriton ( D. Cheriton and R.E.
 * Tarjan, Finding minimum spanning trees, SIAM J. Comput., 5 (1976), pp.
 * 724-742 ) algorithm for minimum/maximum weight spanning trees. 
 *
 * The type of the cost is EGspanningTreeCost_t as defined above. 
 *
 * This version only need the uGraph, the array to save edges and where to store
 * the value, it is a simplify version in the sense that it ask for its local
 * memory and then free it.
 *
 * @param G is the uGraph where to perform the minimum / maximum spanning tree.
 * @param voidOS offset of the external data.
 * @param treeEdges is an array of pointers to EGuGraphEdge_t* where we will 
 * store wich edges has been used in the minimum / maximum spanning tree.
 * @param mem is a memory poool from where we will take all internal memory. 
 * @param costOS is the cost offset in the edge data structure.  
 * @param sense define if the problem is of minimization (-1) or maximization 
 * (1), note that other values will resoult in wrong answers.
 * @param value where we will store the final value of our spanning tree.
 * */
/* ========================================================================= */
int EGspanningTree (EGuGraph_t * G,
										size_t costOS,
										size_t voidOS,
										int sense,
										EGuGraphEdge_t ** treeEdges,
										EGspanningTreeCost_t * value,
										EGmemPool_t * mem);

/* ========================================================================= */
/** @brief create a uGraph to be used in the EGspanningTree algorithm.
 *
 * Given a uGraph in standard format and an array of weights for its edges we
 * create a uGraph G whose node and uGraph data are NULL and whose edge data is
 * the pointer to the cost of the edge, this implies that the cost offset is
 * zero. the void1OS is zero and void2OS is sizeof(void*)
 *
 * @note Since the algorithm is implemented for undirected uGraphs, but the uGraph
 * structure is directed, we assume that the 'edge' parameter contains both
 * copies of each undirected edge.
 * 
 * @param nnodes number of nodes in the uGraph.
 * @param weight the weight of all edges.
 * @param mem memory pool used for internal allocations.
 * @param nedges number of edges in the uGraph.
 * @param edges array of lenggth 2*nedges that store in positions [2*i] and
 * [2*i+1] the tail and the head of the i-th edge.
 * */
/* ========================================================================= */
EGuGraph_t *EGnewSpanningTreeUGraph (int nnodes,
																		 int nedges,
																		 int *edges,
																		 EGspanningTreeCost_t * weight,
																		 EGmemPool_t * mem);

void EGfreeSTNodeData (void *data,
											 EGmemPool_t * mem);

/* end of eg_spanning.h */
#endif
