/* EGlib "Efficient General Library" provides some basic structures and
 * algorithms commons in many optimization algorithms.
 *
 * Copyright (C) 2005 Daniel Espinoza and Marcos Goycoolea.
 * 
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but 
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY 
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public 
 * License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA 
 * */
/* ========================================================================= */
/** @defgroup EGmemSlab EGmemSlab
 *
 * This is a basic interface for slab pool managment. The idea comes from Slabs
 * as defined in both Linux and Solaris (see "The Slab Allocator: An
 * Object-Caching Kernel Memory Allocator", by Jeff Bonwick, Sun Microsystems),
 * the basic idea is
 * to provide pool for a specific type of object, and to store them in an
 * initialized state, so that initialization and destruction only is done while
 * growing/freeing the memory slabs, thus this approach should provide greater
 * advantages for complitated to initialize structures. and in theory
 * (althought not yet implemented) this structure can be managed so as to
 * provide a shrinkable memory managment on the fly.
 *
 * In this implementation we only allow small caches (i.e. objects must be
 * smaller than EG_SLAB_LIMIT), with buffer controls and slab structures stored
 * within a unique memory page. We could allow in the future for more flexible
 * slabs. This implementation also uses colored slabs (see the paper for
 * further details).
 *
 * Here we can see a schematic drawing of the slab allocator structure and
 * functions:
 *
 * @version 0.0.1
 * @par History:
 * - 2005-07-30
 * 						- First Implpementation.
 * */
/** @file
 * @ingroup EGmemSlab */
/** @addtogroup EGmemSlab */
/** @{ */
/** @example eg_memslab.ex.c */
/* ========================================================================= */
#ifndef __EG_MEM_SLAB_H__
#define __EG_MEM_SLAB_H__
#define _XOPEN_SOURCE 600
#include <stdlib.h>
#include <unistd.h>
#include <limits.h>
#include "eg_mem.h"
#include "eg_elist.h"

/* ========================================================================= */
/* declare both the slab structure and the slab pool structure */
struct EGmemSlab_t;
struct EGmemSlabPool_t;

/* ========================================================================= */
/** @brief maximum size of the objects that can be allocated via slab. */
#define EG_SLAB_LIMIT 1023U

/* ========================================================================= */
/** @brief size of the memory slabs (in bytes ) */
#define EG_SLAB_SIZE 0x1000U

/* ========================================================================= */
/** @brief mask to detect the position of a piece of memory within a slab */
#define EG_SLAB_MASK (~0x0fffU)

/* ========================================================================= */
/** @brief address used to check consistency if enabled */
#define EG_SLAB_POISON (0xdeadbeef)

/* ========================================================================= */
/** @brief if set to one, enable profiling for the slab allocator */
#define EG_SLAB_PROFILE 100

/* ========================================================================= */
/** @brief size of the buffer manager, this size should be memory aligned, it
 * is used to compute the actual pointer to the object. */
#define EG_SLAB_BFCNTL_SIZE sizeof(char*)

/* ========================================================================= */
/** @brief local verbose level for the slab allocator, the lower the level, the
 * more information will be printed on screen. */
#define EG_SLAB_VERBOSE 100

/* ========================================================================= */
/** @brief local debug level for the slab allocator, the lower the level, the
 * more testing will be done. */
#define EG_SLAB_DEBUG 0

#ifndef EG_SLAB_REDUCE_TO_MALLOC
/* ========================================================================= */
/** @brief if set to one, reduce the slab pool allocator to a simple malloc
 * call */
#define EG_SLAB_REDUCE_TO_MALLOC 0
#endif


/* ========================================================================= */
/** @brief Given a pointer, return a pointer to the beginning of the containing
 * page. */
#define EG_SLAB_PAGE(ptr) (((size_t)ptr)&EG_SLAB_MASK)

/* ========================================================================= */
/** @brief structure that holds the information relevant to each slab */
typedef struct EGmemSlab_t
{
	unsigned int n_used:10;				/**< Number of used elements in this slab */
	unsigned int elem_sz:10;			/**< Size of the elements in the slab, 
																		 including extra space for pointer next.*/
	unsigned int n_elem:10;				/**< Total number of elements in the slab */
	unsigned int flags:2;					/**< Internal use flags bits */
	unsigned int color:10;				/**< Offset for memory bus balancing */
	unsigned int unused:22;				/**< Unused space in the structure */
	EGeList_t slab_cn;						/**< Connector into the list of slabs */
	char *next_elem;							/**< Next free element in this slab */
	struct EGmemSlabPool_t *pool;	/**< Pointer to the slab pool structure */
}
EGmemSlab_t;

/* ========================================================================= */
/** @brief given a piece of memory that should be within a slab, return the
 * pointer to the related slab structure, remember that the slab structure is
 * at the end of the slab piece of memory. */
#define EGmemSlabGetSlab(ptr) \
		((EGmemSlab_t*)(EG_SLAB_PAGE(ptr) + EG_SLAB_SIZE - EG_MEM_ALIGN(sizeof(EGmemSlab_t))))

/* ========================================================================= */
/** @brief initialize a slab structure. This include calling the constructor
 * for all elements in the slab. Note that this function asumes that all memory
 * has been previously set to NULL. It will also place this slab in the list 
 * of empty slabs in the given pool.
 * @param slab pointer within the memory range of the slab to be initialized.
 * @param Pool Slab Pool where this slab will bellong from. The slab pool
 * should be initialized (i.e. should have a constructor and destructor, and an
 * element size set.
 * @param Color offset of the memory address to be used, this is for improving
 * memory bus usage balancing.
 * */
#define EGmemSlabInit(slab,Pool,Color) ({\
	EGmemSlab_t*const _EGmSlb = EGmemSlabGetSlab(slab);\
	const EGconstructor_f _EGconstr = (Pool)->constr;\
	const size_t _EGesz = (Pool)->elem_sz;\
	unsigned _EGnel = (EG_SLAB_SIZE - EG_MEM_ALIGN(sizeof(EGmemSlab_t)))/_EGesz;\
	char* _EGcel = (char*)EG_SLAB_PAGE(_EGmSlb) + (Color);\
	char** _EGcptr = 0;\
	MESSAGE(EG_SLAB_VERBOSE,"Creating slab at %p with %u elements of effective"\
					" size %zd and color %u", (void*)EG_SLAB_PAGE(_EGmSlb), _EGnel, \
					_EGesz, (Color));\
	_EGmSlb->n_used = 0;\
	_EGmSlb->elem_sz = _EGesz;\
	_EGmSlb->n_elem = _EGnel;\
	_EGmSlb->flags = 0;\
	_EGmSlb->color = (Color);\
	_EGmSlb->next_elem = _EGcel;\
	_EGmSlb->pool = (Pool);\
	while(_EGnel--){\
		/* call constructor for the element if provided */\
		if(_EGconstr) _EGconstr(((void*)_EGcel));\
		/* set the next pointer to the next element in the slab */\
		_EGcptr = (char**)(_EGcel + _EGesz - EG_SLAB_BFCNTL_SIZE);\
		_EGcel += _EGesz;\
		*_EGcptr = _EGcel;}\
	/* position the slab in the pool */\
	EGeListAddAfter(&(_EGmSlb->slab_cn),&((Pool)->empty));\
	/* set the next pointer of the last element in the slab to NULL */\
	*_EGcptr = 0;})

/* ========================================================================= */
/** @brief given an initialized slab, clear all internally allocated memory,
 * and leave the slab ready to be freed by 'free', this include calling the
 * destructor for all elements in the slab.
 * @param slab pointer to an area of the slab memory to be clear.
 * @note If debugging is enabled, then all fields will be poisoned so that
 * subsequent use of this structure will fail (but for the free call). Also, if
 * debugging is enabled, we will check that the slab has no element in use.
 * */
#define EGmemSlabClear(slab) ({\
	EGmemSlab_t*const _EGmSlb = EGmemSlabGetSlab(slab);\
	const EGdestructor_f _EGdest = _EGmSlb->pool->dest;\
	const size_t _EGesz = _EGmSlb->elem_sz;\
	unsigned _EGnel = _EGmSlb->n_elem;\
	char* _EGcel = (char*)EG_SLAB_PAGE(_EGmSlb) + _EGmSlb->color;\
	char** _EGcptr = 0;\
	MESSAGE(EG_SLAB_VERBOSE,"Clearing slab at %p with %u elements of effective"\
					" size %zd and color %u", (void*)EG_SLAB_PAGE(_EGmSlb), _EGnel, \
					_EGesz, _EGmSlb->color);\
	/* set the clear flag to the slab */\
	_EGmSlb->flags = 1;\
	while(_EGnel--){\
		/* call destructor for the element if provided */\
		if(_EGdest) _EGdest(((void*)_EGcel));\
		/* if debugging, poison the next pointer to the next element */\
		if(EG_SLAB_DEBUG <= DEBUG){\
			_EGcptr = (char**)(_EGcel + _EGesz - EG_SLAB_BFCNTL_SIZE);\
			*_EGcptr = (char*)EG_SLAB_POISON;}\
		_EGcel += _EGesz;}\
	/* remove the slab from it's containing list */\
	EGeListDel(&(_EGmSlb->slab_cn));\
	/* if debugging, then poison the slab */\
	if(EG_SLAB_DEBUG <= DEBUG){\
		EXIT(_EGmSlb->n_used,"Clearing a slab with %u elements in use",\
				 _EGmSlb->n_used);\
		_EGmSlb->next_elem = (char*)EG_SLAB_POISON;\
		_EGmSlb->slab_cn = (EGeList_t){0,0};\
		_EGmSlb->pool = (EGmemSlabPool_t*)EG_SLAB_POISON;}\
	/* ending */\
	0;})

/* ========================================================================= */
/** @brief given a pointer to a zone within a slab, check if the slab is OK. If
 * not, then the program will exit execution with a message. Note that this
 * test is performed only when we are debugging.
 * @param slab pointer within the slab memory range.
 * */
#if EG_SLAB_DEBUG <= DEBUG
#define EGmemSlabCheck(_EGmSlb) ({\
	EXIT(_EGmSlb->flags,"Slab at %p has been cleared, and can't be used", \
			(void*)EG_SLAB_PAGE(_EGmSlb));\
	EXIT( _EGmSlb->next_elem == (char*)EG_SLAB_POISON , "Slab at %p is "\
			"poisoned", (void*)EG_SLAB_PAGE(_EGmSlb));\
	EXIT(!(_EGmSlb->pool) || _EGmSlb->pool == (EGmemSlabPool_t*)EG_SLAB_POISON ,\
			"Slab at %p has no pool asociated with it", \
			(void*)EG_SLAB_PAGE(_EGmSlb));\
	0;})
#else
#define EGmemSlabCheck(slab)
#endif

/* ========================================================================= */
/** @brief Given a non-full slab, extract a pointer to the next unused element
 * in the slab, and update all internal data. and if it becomes full, then move
 * it to the full list within the pool. Also, if debugging, poison the pointer
 * to the enext element in the returned element. If the slab is not full, and
 * the number of active elements is one, then move the slab to the half-full
 * slab list in the pool.
 * @param slab pointer within a slab memory.
 * @return pointer to a void* of initialized memory by the given contructor in
 * the slab pool.
 * */
#define EGmemSlabPopElement(slab) ({\
	EGmemSlab_t*const _EGmSlb = EGmemSlabGetSlab(slab);\
	EGmemSlabPool_t*const _EGPlRf = _EGmSlb->pool;\
	void*const _EGelem = (void*)(_EGmSlb->next_elem);\
	char** _EGnext_elem = (char**)(((char*)_EGelem) + _EGmSlb->elem_sz - \
																	EG_SLAB_BFCNTL_SIZE);\
	/* some consistency checks */\
	EGmemSlabCheck(_EGmSlb);\
	EXITL(EG_SLAB_DEBUG, *_EGnext_elem == (char*)EG_SLAB_POISON, "Poisoned "\
				"element");\
	EXITL(EG_SLAB_DEBUG, _EGmSlb->n_used == _EGmSlb->n_elem, "Trying to get"\
				" an element from slab at %p, but slab is full", \
				(void*)EG_SLAB_PAGE(_EGmSlb));\
	/* now update the slab */\
	_EGmSlb->next_elem = *_EGnext_elem;\
	_EGmSlb->n_used++;\
	if(_EGmSlb->n_used == _EGmSlb->n_elem)\
		EGeListMoveAfter(&(_EGmSlb->slab_cn),&(_EGPlRf->full));\
	else if(_EGmSlb->n_used == 1U)\
		EGeListMoveAfter(&(_EGmSlb->slab_cn),&(_EGPlRf->half));\
	/* if debugging, poison the pointer to next in the returned element */\
	if(EG_SLAB_DEBUG <= DEBUG) *_EGnext_elem = (char*)EG_SLAB_POISON;\
	/* return the element */\
	_EGelem;})

/* ========================================================================= */
/** @brief Given an used object within a slab, give it back to the slab for
 * future used.
 * @param ptr pointer to the element to be given back to its containing slab.
 * */
#define EGmemSlabPushElement(ptr) ({\
	EGmemSlab_t*const _EGmSlb = EGmemSlabGetSlab(ptr);\
	EGmemSlabPool_t*const _EGmPl = _EGmSlb->pool;\
	void*const _EGelem = (void*)(ptr);\
	char** _EGnext_elem = (char**)(((char*)_EGelem) + _EGmSlb->elem_sz - \
																	EG_SLAB_BFCNTL_SIZE);\
	/* if debugging, check for poison in the pointer to the next element in the \
	 * given element */\
	EXITL(EG_SLAB_DEBUG, !_EGelem, "Trying to free a NULL ptr");\
	EGmemSlabCheck(_EGmSlb);\
	EXITL(EG_SLAB_DEBUG, *_EGnext_elem != (char*)EG_SLAB_POISON, "This can't"\
				" happen, memory %p is corrupted!", (void*)_EGelem);\
	EXITL(EG_SLAB_DEBUG, !_EGmSlb->n_used, "Trying to free elements in an "\
				"unused slab!");\
	/* now, if the slab was full, put it as half-full */\
	if(_EGmSlb->n_elem == _EGmSlb->n_used)\
		EGeListMoveAfter(&(_EGmSlb->slab_cn),&(_EGmPl->half));\
	/* decrease the number of used elements, and if zero, then move to the list\
	 * of empty slabs. */\
	_EGmSlb->n_used -= 1;\
	if(!(_EGmSlb->n_used))\
		EGeListMoveAfter(&(_EGmSlb->slab_cn),&(_EGmPl->empty));\
	/* now actually put the element into the slab */\
	*_EGnext_elem = _EGmSlb->next_elem;\
	_EGmSlb->next_elem = _EGelem;})

/* ========================================================================= */
/** @brief structure used to store a slab memory pool */
typedef struct EGmemSlabPool_t
{
	EGeList_t half;					/**< Head of the list for half-full slabs */
	EGeList_t empty;				/**< Head of the list for non used slabs */
	EGeList_t full;					/**< Head of the list for fully used slabs*/
	EGconstructor_f constr;	/**< Constructor for the local elements */
	EGdestructor_f dest;		/**< Destructor for the local elements */
	unsigned int elem_sz:10;/**< Size of the elements in the slab, including 
															 extra space for pointer to next. */
	unsigned int n_elem:10;	/**< Total number of elements in the slab */
	unsigned int c_color:10;/**< Last used color while creating slabs. */
	unsigned int unused1:2;	/**< Padding for unused bits */
	unsigned int max_color:10;/**< Maximum valid value for colors in this pool */
	#if EG_SLAB_PROFILE <= DEBUG
	unsigned int real_sz:10;/**< Actual size of the elements asked by the user */
	unsigned int unused2:12;/**< Padding for unused space */
	char const *file;				/**< File where the structure was initialized */
	char const *func;				/**< Function where the structure was initialized */
	int line;								/**< Line where the structure was initialized */
	unsigned n_slabs;				/**< Number of slabs */
	unsigned n_tot;					/**< Total number of elements in use by the user */
	#else
	unsigned int unused2:22;/**< Padding for unused space */
	#endif
}
EGmemSlabPool_t;

/* ========================================================================= */
/** @brief Initialize the profiling data of a slab pool */
#if EG_SLAB_PROFILE <= DEBUG
#define _EGmemSlabPoolInitProfile(_EGmPl,sz) ({\
	_EGmPl->file = __FILE__;\
	_EGmPl->func = __func__;\
	_EGmPl->line = __LINE__;\
	_EGmPl->real_sz = (sz);\
	_EGmPl->n_slabs = _EGmPl->n_tot = 0;})
#else
#define _EGmemSlabPoolInitProfile(_EGmPl,sz)
#endif

/* ========================================================================= */
/** @brief initialize a slab pool as an empty pool for elements of the given
 * size, and with te given constructor and destructors. 
 * @param constr_fn constructor fnctioin for the elements to be stored in the
 * pool.
 * @param dest_fn destructor function for the elements to be stored in the
 * pool.
 * @param pool pointer to the slab pool to initialize.
 * @param sz (real) size (in bytes) of the elements to be hold. in the pool.
 * This means that sz is the result of sizeof(TYPE), where TYPE is the
 * structure to be pooled. */
#define EGmemSlabPoolInit(pool,sz,constr_fn,dest_fn) ({\
	EGmemSlabPool_t*const _EGmPl = (pool);\
	unsigned const _EGesz = EG_MEM_ALIGN(sz)+EG_MEM_ALIGN(EG_SLAB_BFCNTL_SIZE);\
	unsigned const _EGnel = (EG_SLAB_SIZE - EG_MEM_ALIGN(sizeof(EGmemSlab_t))) \
													/ _EGesz;\
	/* check that the real element size is within bounds */\
	EXIT(_EGesz > EG_SLAB_LIMIT, "element size (user %u, real %u) for slab is"\
			" above limits %u", (sz), _EGesz, EG_SLAB_LIMIT);\
	/* initialize the structure */\
	EGeListInit(&(_EGmPl->half));\
	EGeListInit(&(_EGmPl->empty));\
	EGeListInit(&(_EGmPl->full));\
	_EGmPl->constr = constr_fn;\
	_EGmPl->dest = dest_fn;\
	_EGmPl->elem_sz = _EGesz;\
	_EGmPl->n_elem = _EGnel;\
	_EGmPl->c_color = 0;\
	_EGmPl->max_color = EG_SLAB_SIZE - EG_MEM_ALIGN(sizeof(EGmemSlab_t)) - \
											(_EGnel*_EGesz);\
	_EGmemSlabPoolInitProfile(_EGmPl,sz);\
	/* verbose output */\
	MESSAGE(EG_SLAB_VERBOSE,"Creating Slab Pool at %p:\n\tmax colors %u\n"\
					"\telements per slab %u\n\tuser size %u\n\treal size %u\n", \
					(void*)_EGmPl, _EGmPl->max_color, _EGnel, (sz), _EGesz);\
	0;})

/* ========================================================================= */
/** @brief show profile information after clearing a pool. */
#if EG_SLAB_PROFILE <= DEBUG
#define _EGmemSlabPoolProfile(_EGmPl) ({\
	fprintf(stderr,"Profile for SLAB_POOL %p (%s:%s:%d)\n", (void*)_EGmPl, \
					_EGmPl->func, _EGmPl->file, _EGmPl->line);\
	fprintf(stderr,"\tSlabs: %u\n", _EGmPl->n_slabs);\
	fprintf(stderr,"\tTotal Elements Requested: %u\n\tReal Elements "\
					"Available: %u\n", _EGmPl->n_tot, \
					_EGmPl->n_elem * _EGmPl->n_slabs);\
	fprintf(stderr,"\tuser size: %u\n\telem size: %u\n\tmax color: %u\n\t"\
					"elems per slab: %u\n", _EGmPl->real_sz, _EGmPl->elem_sz, \
					_EGmPl->max_color,_EGmPl->n_elem);\
	fprintf(stderr,"\tWaste per Slab: %.2lf%%\n", ((double)100*(EG_SLAB_SIZE - \
					_EGmPl->n_elem*_EGmPl->real_sz))/EG_SLAB_SIZE);})
#else
#define _EGmemSlabPoolProfile(_EGmPl)
#endif

/* ========================================================================= */
/** @brief clear a slab pool and all internal sub-structures and data, no
 * further calls to this structure are posible after this (but for freeing the
 * memory containing this data, or to re-initialize it).
 * @param Pool slab pool to be cleared.
 * */
#define EGmemSlabPoolClear(Pool) ({\
	EGmemSlabPool_t*const _EGmPl = (Pool);\
	void* _EGptr;\
	while(!EGeListIsEmpty(&(_EGmPl->half))){\
		_EGptr = _EGmPl->half.next;\
		EGmemSlabClear(_EGptr);\
		free((void*)EG_SLAB_PAGE(_EGptr));}\
	while(!EGeListIsEmpty(&(_EGmPl->empty))){\
		_EGptr = _EGmPl->empty.next;\
		EGmemSlabClear(_EGptr);\
		free((void*)EG_SLAB_PAGE(_EGptr));}\
	while(!EGeListIsEmpty(&(_EGmPl->full))){\
		_EGptr = _EGmPl->full.next;\
		EGmemSlabClear(_EGptr);\
		free((void*)EG_SLAB_PAGE(_EGptr));}\
	/* profile the pool if we are prpfiling */\
	_EGmemSlabPoolProfile(_EGmPl);\
	/* if we are debugging, then poison the pool */\
	if(EG_SLAB_DEBUG <= DEBUG){\
		_EGmPl->constr = 0;\
		_EGmPl->dest = 0;\
		_EGmPl->elem_sz = _EGmPl->n_elem = _EGmPl->c_color = \
		_EGmPl->max_color = 0;}\
	;})

/* ========================================================================= */
/** @brief add one to the given pointer, if profiling is enabled, otherwise, do
 * nothing */
#if EG_SLAB_PROFILE <= DEBUG
#define EGSLB_UPDATE(a) ((a)++)
#else
#define EGSLB_UPDATE(a)
#endif

/* ========================================================================= */
/** @brief Given a slab pool, return an element from the pool. 
 * @param Pool slab pool from where we will get the memory. 
 * @return pointer to an initialize element. */
#if EG_SLAB_REDUCE_TO_MALLOC
#define EGmemSlabPoolAlloc(Pool) ({\
	EGmemSlabPool_t*const _EGmPl = (Pool);\
	void*_EGmb = EGmalloc(_EGmPl->elem_sz+sizeof(void*));\
	void**_EGpt = (void**)_EGmb;\
	(*_EGpt) = _EGmPl;\
	_EGmb=((void*)(_EGpt+1));\
	_EGmPl->constr(_EGmb);\
	_EGmb;})
#else
#define EGmemSlabPoolAlloc(Pool) ({\
	EGmemSlabPool_t*const _EGmPl = (Pool);\
	void* _EGSmbRf = 0;\
	if(!EGeListIsEmpty(&(_EGmPl->half))) _EGSmbRf = _EGmPl->half.next;\
	else{\
		if(!EGeListIsEmpty(&(_EGmPl->empty))) _EGSmbRf = _EGmPl->empty.next;\
		else{\
			if(posix_memalign(&_EGSmbRf,EG_SLAB_SIZE,EG_SLAB_SIZE))\
				EXIT(1,"posix_memalign falied");\
			else{\
				EGmemSlabInit(_EGSmbRf,_EGmPl,_EGmPl->c_color);\
				_EGmPl->c_color += EG_MEM_ALIGNMENT;\
				EGSLB_UPDATE(_EGmPl->n_slabs);\
				if(_EGmPl->c_color > _EGmPl->max_color) _EGmPl->c_color = 0;}}}\
	EGSLB_UPDATE(_EGmPl->n_tot);\
	EGmemSlabPopElement(_EGSmbRf);})
#endif

/* ========================================================================= */
/** @brief Given a pointer to an element allocated through a slab pool, give it
 * back to the pool.
 * @param ptr pointer to be returned to the pool.
 * */
#if EG_SLAB_REDUCE_TO_MALLOC
#define EGmemSlabPoolFree(ptr) ({\
	void**_EGptr = ((void**)(ptr))-1;\
	EGmemSlabPool_t*const _EGmPl = (EGmemSlabPool_t*)(*_EGptr);\
	_EGmPl->dest(((void*)(_EGptr+1)));\
	EGfree(_EGptr);})
#else
#define EGmemSlabPoolFree(ptr) EGmemSlabPushElement(ptr)
#endif

/* ========================================================================= */
/** @brief Given a slab pool, free all unused slabs 
 * @param Pool slab pool to be shrinked. */
#define EGmemSlabPoolShrink(Pool) ({\
	EGmemSlabPool_t*const _EGmPl = (Pool);\
	void* _EGptr;\
	while(!EGeListIsEmpty(&(_EGmPl->empty))){\
		_EGptr = (void*)(_EGmPl->empty.next);\
		EGmemSlabClear(_EGptr);\
		free((void*)EG_SLAB_PAGE(_EGptr));}\
	0;})

/* ========================================================================= */
/* end of eg_memslab.h */
/** @} */
#endif
