/* EGlib "Efficient General Library" provides some basic structures and
 * algorithms commons in many optimization algorithms.
 *
 * Copyright (C) 2005 Daniel Espinoza and Marcos Goycoolea.
 * 
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but 
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY 
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public 
 * License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA 
 * */
#include "eg_mempool.h"
/* ========================================================================= */
/* this type is to write addres to memory pointers */
/* ========================================================================= */
typedef union
{
	void *addres;
	void **memory;
}
EGmemory_t;

/* ========================================================================= */
/* this function resize a memory pool */
/* ========================================================================= */
static inline void EGmemPoolResize (EGmemPool_t * mypool)
{
	/* local variables */
	void **tmpMemoryList;
	size_t size;

	/* basic set-up */
	size = mypool->newsize (mypool->MemoryListSize + 1);
	if (size < mypool->manageLimit)
		size = mypool->manageLimit;

	/* looking for memory */
	tmpMemoryList =
		(void **) EGmalloc ((mypool->MemoryListSize + 1) * sizeof (void *));

	/* moving old information */
	memcpy (tmpMemoryList, mypool->MemoryList,
					sizeof (void *) * mypool->MemoryListSize);
	EGfree (mypool->MemoryList);
	mypool->MemoryList = tmpMemoryList;

	/* if we can save the memory in the tail we do-it, if the tail memory is
	 * bellow sizeof(void*) we discard it, the total amount of discarded memory
	 * should not be too large becouse we call the resize function just a litle.
	 * for some odd reason this code does not run correctly on SUN, it might be a
	 * good idea to post a question about this */
	if (mypool->freeArraySize >= sizeof (void *))
	{
		MESSAGE (__EG_MEMPOOL_DEBUGL__, "something odd about to happen source=%p "
						 "destine=%p size = %zd",
						 mypool->freedMemory[mypool->
																 freeArraySize >> EG_MEM_ALIGNMENT_SHIFT],
						 mypool->freeArrayHead, (unsigned) (mypool->freeArraySize));
		*((void **) mypool->freeArrayHead) =
			mypool->freedMemory[mypool->freeArraySize >> EG_MEM_ALIGNMENT_SHIFT];
		MESSAGE (__EG_MEMPOOL_DEBUGL__, "tail memory saved");
		mypool->freedMemory[mypool->freeArraySize >> EG_MEM_ALIGNMENT_SHIFT] =
			mypool->freeArrayHead;
		MESSAGE (__EG_MEMPOOL_DEBUGL__, "freedMemory updated");
	}

	/* asking for new chunk of memory */
	mypool->freeArrayHead = mypool->MemoryList[mypool->MemoryListSize] =
		(void *) EGmalloc (size);

	/* we update the mempool values */
	mypool->freeArraySize = size;
	mypool->MemoryListSize++;
#if __EG_MEMPOOL_PROFILE__
	mypool->totalMem += size;
#endif

	/* ending */
	return;
}

/* ========================================================================= */
/* this fucntion allocate a new memory pool, note that
  a memory pool with 'manageLimit == 0' is equivalent
 * to malloc/EGfree */
/* ========================================================================= */
EGmemPool_t *__EGnewMemPool (const size_t manageLimit,
														 size_t (*newsize) (size_t),
														 const size_t initSize)
{

	/* if we reduce to malloc, we just return */
#if __EG_MEMPOOL_REDUCE_TO_MALLOC__
	if (manageLimit && newsize && initSize)
		return 0;
	else
	{
		EXIT (1, "Wrong parameters");
		exit (1);
	}
	return 0;
}
#else

	/* local variables */
	EGmemPool_t *mypool = EGsMalloc (EGmemPool_t, 1);
	EGmemPoolInit (mypool, manageLimit, newsize, initSize);
	/* ending */
	return mypool;
}
#endif

/* ========================================================================= */
/* this function EGfree a memory pool, note that we only need to EGfree
 * the memory pointed by MemoryList */
/* ========================================================================= */
void EGfreeMemPool (EGmemPool_t * mypool)
{

	/* if we reduce to malloc, we just return */
#if __EG_MEMPOOL_REDUCE_TO_MALLOC__
	if (mypool)
	{
		EXIT (1, "non null pointer");
		exit (1);
	}
	return;
}
#else

	/* local variables */
	EGmemPoolClear (mypool);

	/* and EGfree the memory pool itself */
	EGfree (mypool);

	/* ending */
	return;
}
#endif

/* ========================================================================= */
/* this function ask to a memory pool a memory block of size 'size' */
/* ========================================================================= */
void *__EGmemPoolMalloc (EGmemPool_t * const mypool,
												 size_t size
#if DEBUG && (__EG_MEMPOOL_REDUCE_TO_MALLOC__==0)
												 ,
												 const char *file,
												 const int line
#endif
	)
{
	/* local variables */
	void *res;

/* if we reduce to malloc, we just call malloc */
#if __EG_MEMPOOL_REDUCE_TO_MALLOC__
	EXITL (__EG_MEMPOOL_DEBUGL__, mypool, "non null pointer parameter");

	res = EGmalloc (size);
	return res;
}
#else
	unsigned char *tmp;
	size_t pos;

	MESSAGE (__EG_MEMPOOL_DEBUGL__, "entering");
	/* if we alloc zero bytes throw a warning and return null */
#if __EG_MEMPOOL_PROFILE__ && (DEBUG>0)
	WARNING (!size, "Allocating 0 bytes from %s:%d", file, line);
	if (!size)
	{
		mypool->nNullAlloc++;
		mypool->nullAllocFile = file;
		mypool->nullAllocLine = line;
		return 0;
	}
#elif (DEBUG>0)
	if (!size)
	{
		fprintf (stderr, "WARNING: Allocating 0 bytes\n");
		return 0;
	}
#endif

	/* if we are checking EGfree and malloc we have to add the overhead to the size
	 * */
#if __EG_MEMPOOL_MALLOC_FREE_CHECK__ && \
			!__EG_MEMPOOL_REDUCE_TO_MALLOC__ && \
			DEBUG
	size += __EG_MEMPOOL_OVERHEAD__;
#endif

	/* if we need to do some alligment, we do-it here, note that this force the
	 * size to be at least EG_MEM_ALIGNMENT */
	size = ((size & (EG_MEM_ALIGNMENT - 1)) ?
					(size + EG_MEM_ALIGNMENT) & (~(EG_MEM_ALIGNMENT - 1)) : size);

#if __EG_MEMPOOL_PROFILE__ && (DEBUG>0)
	mypool->allocMem += size;
	if (mypool->maxBlock < size)
	{
		mypool->maxBlock = size;
		mypool->maxBlockFile = file;
		mypool->maxBlockLine = line;
	}
#endif

	/* if the memory is bigger than the manage limit we just alloc */
	if (size > mypool->manageLimit)
	{
		res = EGmalloc (size);
#if DEBUG && (__EG_MEMPOOL_REDUCE_TO_MALLOC__==0)
		EXIT ((!res),
					"when allocating %zd bytes, not enough memory\nCalled from %s:%d",
					size, file, line);
#endif
		/* profiling work */
#if __EG_MEMPOOL_PROFILE__
		mypool->noManageMem += size;
		mypool->nBigBlocks[__EG_MEMPOOL_BIGBUCK__ (size)]++;
		mypool->sBigBlocks[__EG_MEMPOOL_BIGBUCK__ (size)]++;
		if (mypool->sBigBlocks[__EG_MEMPOOL_BIGBUCK__ (size)] >
				mypool->cBigBlocks[__EG_MEMPOOL_BIGBUCK__ (size)])
			mypool->cBigBlocks[__EG_MEMPOOL_BIGBUCK__ (size)] =
				mypool->sBigBlocks[__EG_MEMPOOL_BIGBUCK__ (size)];
#endif

		/* setting the EGfree/malloc information */
#if __EG_MEMPOOL_MALLOC_FREE_CHECK__ && \
				!__EG_MEMPOOL_REDUCE_TO_MALLOC__ && \
				DEBUG
		__EG_MEMPOOL_SET_FILE (res, file);
		__EG_MEMPOOL_SET_LINE (res, line);
		__EG_MEMPOOL_SET_SIZE (res, size);
		__EG_MEMPOOL_SET_ALLOCED (res);
		res = ((char *) res) + __EG_MEMPOOL_OVERHEAD__;
#endif
		return res;
	}

	/* compute for once the position in the list of the current size */
	pos = size >> EG_MEM_ALIGNMENT_SHIFT;
	/* profiling work */
#if __EG_MEMPOOL_PROFILE__
	mypool->curUsedMem[pos] += size;
	if (mypool->maxUsedMem[pos] < mypool->curUsedMem[pos])
		mypool->maxUsedMem[pos] += size;
#endif

	/* first we check if we have memory previously freed to return */
	if (mypool->freedMemory[pos])
	{
		MESSAGE (__EG_MEMPOOL_DEBUGL__, "recycling");
		res = mypool->freedMemory[pos];
		mypool->freedMemory[pos] = *((void **) res);
		memset(res,0,(size_t)size);

		/* setting the EGfree/malloc information */
#if __EG_MEMPOOL_MALLOC_FREE_CHECK__ && \
				!__EG_MEMPOOL_REDUCE_TO_MALLOC__ && \
				DEBUG
		__EG_MEMPOOL_SET_FILE (res, file);
		__EG_MEMPOOL_SET_LINE (res, line);
		__EG_MEMPOOL_SET_SIZE (res, size);
		__EG_MEMPOOL_SET_ALLOCED (res);
		res = ((char *) res) + __EG_MEMPOOL_OVERHEAD__;
#endif
		return res;
	}

	/* if there is no previously freed memory, we check if we have enough space 
	 * in the memory array,in that case we return the memory from the array */
	if (mypool->freeArraySize > size)
	{
		MESSAGE (__EG_MEMPOOL_DEBUGL__, "using new");
		tmp = res = mypool->freeArrayHead;
		tmp += size;
		mypool->freeArrayHead = tmp;
		mypool->freeArraySize -= size;
		memset(res,0,(size_t)size);

		/* setting the EGfree/malloc information */
#if __EG_MEMPOOL_MALLOC_FREE_CHECK__ && \
				!__EG_MEMPOOL_REDUCE_TO_MALLOC__ && \
				DEBUG
		__EG_MEMPOOL_SET_FILE (res, file);
		__EG_MEMPOOL_SET_LINE (res, line);
		__EG_MEMPOOL_SET_SIZE (res, size);
		__EG_MEMPOOL_SET_ALLOCED (res);
		res = ((char *) res) + __EG_MEMPOOL_OVERHEAD__;
#endif
		return res;
	}

	/* if everythig else fail we resize the local memory and return 
	 * from the new allocated memory */
	MESSAGE (__EG_MEMPOOL_DEBUGL__, "resizing");
	EGmemPoolResize (mypool);
	MESSAGE (__EG_MEMPOOL_DEBUGL__, "done resizing");
	tmp = res = mypool->freeArrayHead;
	tmp += size;
	mypool->freeArrayHead = tmp;
	mypool->freeArraySize -= size;
	memset(res,0,(size_t)size);

	/* ending */
	/* setting the EGfree/malloc information */
#if __EG_MEMPOOL_MALLOC_FREE_CHECK__ && \
			!__EG_MEMPOOL_REDUCE_TO_MALLOC__ && \
			DEBUG
	__EG_MEMPOOL_SET_FILE (res, file);
	__EG_MEMPOOL_SET_LINE (res, line);
	__EG_MEMPOOL_SET_SIZE (res, size);
	__EG_MEMPOOL_SET_ALLOCED (res);
	res = ((char *) res) + __EG_MEMPOOL_OVERHEAD__;
#endif
	return res;
}
#endif

/* ========================================================================= */
/* this function EGfree to the memory pool a memory block of size 'size; 
 * It is __VERY__ important to EGfree a block memory this way and to the 
 * apropiate memory pool, any error in this __WILL__ cause serious memory 
 * errors */
/* ========================================================================= */
void __EGmemPoolFree (void *mem,
											size_t size,
											EGmemPool_t * mypool
#if DEBUG && (__EG_MEMPOOL_REDUCE_TO_MALLOC__==0)
											,
											const char *file,
											const int line
#endif
	)
{

	/* if we reduce to malloc, we just call malloc */
#if __EG_MEMPOOL_REDUCE_TO_MALLOC__
	EGfree (mem);
	EXITL (__EG_MEMPOOL_DEBUGL__, mypool, "non null parameter");
	mypool = (void *) size;
	return;
}
#else
	/* check if the mem is zero but not the size */
#if DEBUG && (__EG_MEMPOOL_REDUCE_TO_MALLOC__==0)
	EXIT ((!mem
				 && size),
				"in EGmemPoolFree, wrong parameter values\nCalled by %s:%d\n", file,
				line);
#endif

	/* if the memory is null we return */
	if (!mem)
		return;

	/* test that the size is non zero */
#if DEBUG && (__EG_MEMPOOL_REDUCE_TO_MALLOC__==0)
	EXIT (!size,
				"in EGmemPoolFree, trying to liberate 0 bytes of a non NULL "
				"memory\nCalled by %s:%d\n", file, line);
#else
	EXIT (!size,
				"in EGmemPoolFree, trying to liberate 0 bytes of a non NULL memory");
#endif

	/* if we are checking EGfree and malloc we have to add the overhead to the size
	 * */
#if __EG_MEMPOOL_MALLOC_FREE_CHECK__ && \
			!__EG_MEMPOOL_REDUCE_TO_MALLOC__ && \
			DEBUG
	size += __EG_MEMPOOL_OVERHEAD__;
#endif

	/* if we need to do some alligment, we do-it here, note that this force the
	 * size to be at least EG_MEM_ALIGNMENT */
	size = ((size & (EG_MEM_ALIGNMENT - 1)) ?
					(size + EG_MEM_ALIGNMENT) & (~(EG_MEM_ALIGNMENT - 1)) : size);

	/* first we check size, if it is too big we just EGfree it */
	if (size > mypool->manageLimit)
	{
		/* check the EGfree if we have debug information */
#if __EG_MEMPOOL_MALLOC_FREE_CHECK__ && \
				!__EG_MEMPOOL_REDUCE_TO_MALLOC__ && \
				DEBUG
		mem = (char *) mem - __EG_MEMPOOL_OVERHEAD__;
		EXIT (__EG_MEMPOOL_GET_STATUS (mem),
					"Freeing %p twice at %s:%d with size %zd"
					" allocated at %s:%d with size %zd",
					(void *) (((char *) mem) + __EG_MEMPOOL_OVERHEAD__), file, line,
					size - __EG_MEMPOOL_OVERHEAD__, __EG_MEMPOOL_GET_FILE (mem),
					__EG_MEMPOOL_GET_LINE (mem),
					__EG_MEMPOOL_GET_SIZE (mem) - __EG_MEMPOOL_OVERHEAD__);
		EXIT (size != __EG_MEMPOOL_GET_SIZE (mem),
					"Freeing %p at %s:%d with size "
					"%zd while it was allocated at %s:%d with size %zd",
					(void *) (((char *) mem) + __EG_MEMPOOL_OVERHEAD__), file, line,
					size - __EG_MEMPOOL_OVERHEAD__, __EG_MEMPOOL_GET_FILE (mem),
					__EG_MEMPOOL_GET_LINE (mem),
					__EG_MEMPOOL_GET_SIZE (mem) - __EG_MEMPOOL_OVERHEAD__);
		__EG_MEMPOOL_SET_FREEED (mem);
#endif

#if __EG_MEMPOOL_PROFILE__ && (DEBUG>0)
		mypool->sBigBlocks[__EG_MEMPOOL_BIGBUCK__ (size)]--;
#endif
		EGfree (mem);
		return;
	}

	/* check the EGfree if we have debug information */
#if __EG_MEMPOOL_MALLOC_FREE_CHECK__ && \
			!__EG_MEMPOOL_REDUCE_TO_MALLOC__ && \
			DEBUG
	mem = (char *) mem - __EG_MEMPOOL_OVERHEAD__;
	EXIT (__EG_MEMPOOL_GET_STATUS (mem),
				"Freeing %p twice at %s:%d with size %zd"
				" allocated at %s:%d with size %zd, previously freed at %s:%d",
				(void *) (((char *) mem) + __EG_MEMPOOL_OVERHEAD__), file, line,
				size - __EG_MEMPOOL_OVERHEAD__, __EG_MEMPOOL_GET_FILE (mem),
				__EG_MEMPOOL_GET_LINE (mem),
				__EG_MEMPOOL_GET_SIZE (mem) - __EG_MEMPOOL_OVERHEAD__,
				__EG_MEMPOOL_GET_FFILE (mem), __EG_MEMPOOL_GET_FLINE (mem));
	EXIT (size != __EG_MEMPOOL_GET_SIZE (mem),
				"Freeing %p at %s:%d with size "
				"%zd while it was allocated at %s:%d with size %zd",
				(void *) (((char *) mem) + __EG_MEMPOOL_OVERHEAD__), file, line,
				size - __EG_MEMPOOL_OVERHEAD__, __EG_MEMPOOL_GET_FILE (mem),
				__EG_MEMPOOL_GET_LINE (mem),
				__EG_MEMPOOL_GET_SIZE (mem) - __EG_MEMPOOL_OVERHEAD__);
	__EG_MEMPOOL_SET_FREEED (mem);
	__EG_MEMPOOL_SET_FFILE (mem, file);
	__EG_MEMPOOL_SET_FLINE (mem, line);
#endif

	/* now we store this memory for later re-use */
	*((void **) mem) = mypool->freedMemory[size >> EG_MEM_ALIGNMENT_SHIFT];
	mypool->freedMemory[size >> EG_MEM_ALIGNMENT_SHIFT] = mem;

#if __EG_MEMPOOL_PROFILE__
	mypool->curUsedMem[size >> EG_MEM_ALIGNMENT_SHIFT] -= size;
#endif


	/* ending */
	MESSAGE (__EG_MEMPOOL_DEBUGL__, "done");
	return;
}
#endif

/* ========================================================================= */
/* this function offer a default memory resizer strategy, this function 
 * return 1 << (i+10) */
/* ========================================================================= */
size_t EGmemPoolNewSize (size_t i)
{
	return (size_t) (1 << (i + 10));
}
