/* EGlib "Efficient General Library" provides some basic structures and
 * algorithms commons in many optimization algorithms.
 *
 * Copyright (C) 2005 Daniel Espinoza and Marcos Goycoolea.
 * 
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but 
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY 
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public 
 * License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA 
 * */
#ifndef _EGHEAP2
#define _EGHEAP2

#include <stdio.h>
#include <limits.h>
#include <float.h>
#include "eg_mempool.h"
#include "eg_fp.h"

/* ========================================================================= */
/* Here we define what kind of cost type to use, there are several choices, as
 * default we use double */
#ifndef EG_HEAP_COST_TYPE
#define EG_HEAP_COST_TYPE FP25_TYPE
#endif

/* ========================================================================= */
/* case the cost is any of the internal types */
#if EG_HEAP_COST_TYPE == DBL_TYPE
typedef double EGheapCost_t;
#endif
#if EG_HEAP_COST_TYPE == FLT_TYPE
typedef float EGheapCost_t;
#endif
#if EG_HEAP_COST_TYPE == INT_TYPE
typedef int EGheapCost_t;
#endif
#if ( (EG_HEAP_COST_TYPE == DBL_TYPE) || (EG_HEAP_COST_TYPE == FLT_TYPE)\
	|| (EG_HEAP_COST_TYPE == INT_TYPE) )
#define EGheapCostAdd(a,b) ((a)+(b))
#define EGheapCostSub(a,b) ((a)-(b))
#define EGheapCostIsLess(a,b) ((a)<(b))
#define EGheapCostDiv(a,b) ((a)/(b))
#define EGheapCostMul(a,b) ((a)*(b))
#define EGheapCostToLf(a) ((double)a)
#define EGheapCostMinus(a) (-1*a)
#define EGheapToCost(a) ((EGheapCost_t)(a))
#define EG_HEAP_COST_MAX INT_MAX

/* ========================================================================= */
/* case the cost is Fixed points with 10 bits for fractional values */
#elif EG_HEAP_COST_TYPE == FP10_TYPE
typedef EGfp10_t EGheapCost_t;
#define EGheapCostAdd(a,b) EGfpAdd10(a,b)
#define EGheapCostSub(a,b) EGfpSub10(a,b)
#define EGheapCostIsLess(a,b) ((a)<(b))
#define EGheapCostDiv(a,b) EGfpDiv10(a,b)
#define EGheapCostMul(a,b) EGfpMul10(a,b)
#define EGheapCostToLf(a) fptolf10(a)
#define EGheapCostMinus(a) EGfpMinus10(a)
#define EGheapToCost(a) lftofp10(a)
#define EG_HEAP_COST_MAX EGFP_MAX10

/* ========================================================================= */
/* case the cost is Fixed points with 20 bits for fractional values */
#elif EG_HEAP_COST_TYPE == FP20_TYPE
typedef EGfp20_t EGheapCost_t;
#define EGheapCostAdd(a,b) EGfpAdd20(a,b)
#define EGheapCostSub(a,b) EGfpSub20(a,b)
#define EGheapCostIsLess(a,b) ((a)<(b))
#define EGheapCostDiv(a,b) EGfpDiv20(a,b)
#define EGheapCostMul(a,b) EGfpMul20(a,b)
#define EGheapCostToLf(a) fptolf20(a)
#define EGheapCostMinus(a) EGfpMinus20(a)
#define EGheapToCost(a) lftofp20(a)
#define EG_HEAP_COST_MAX EGFP_MAX20

/* ========================================================================= */
/* case the cost is Fixed points with 28 bits for fractional values */
#elif EG_HEAP_COST_TYPE == FP28_TYPE
typedef EGfp28_t EGheapCost_t;
#define EGheapCostAdd(a,b) EGfpAdd28(a,b)
#define EGheapCostSub(a,b) EGfpSub28(a,b)
#define EGheapCostIsLess(a,b) ((a)<(b))
#define EGheapCostDiv(a,b) EGfpDiv28(a,b)
#define EGheapCostMul(a,b) EGfpMul28(a,b)
#define EGheapCostToLf(a) fptolf28(a)
#define EGheapCostMinus(a) EGfpMinus28(a)
#define EGheapToCost(a) lftofp28(a)
#define EG_HEAP_COST_MAX EGFP_MAX28

/* ========================================================================= */
/* case the cost is Fixed points with 25 bits for fractional values */
#elif EG_HEAP_COST_TYPE == FP25_TYPE
typedef EGfp25_t EGheapCost_t;
#define EGheapCostAdd(a,b) EGfpAdd25(a,b)
#define EGheapCostSub(a,b) EGfpSub25(a,b)
#define EGheapCostIsLess(a,b) ((a)<(b))
#define EGheapCostDiv(a,b) EGfpDiv25(a,b)
#define EGheapCostMul(a,b) EGfpMul25(a,b)
#define EGheapCostToLf(a) fptolf25(a)
#define EGheapCostMinus(a) EGfpMinus25(a)
#define EGheapToCost(a) lftofp25(a)
#define EG_HEAP_COST_MAX EGFP_MAX25
/* ========================================================================= */
/* if we reach this line this means that the type is unsupported */
#else
#error UNSUPORTED VALUE TYPE
#endif

/* ========================================================================= */
/* this version usses a fixed d=2 */
typedef struct
{

	void *this;
	unsigned int pos;
	EGheapCost_t val;

}
EGheapConnector_t;

typedef struct
{

	unsigned int size,
	  max_size,
	  d;
	EGheapConnector_t **heap;

}
EGheap_t;

extern inline EGheapCost_t EGheapGetMinVal (EGheap_t * h)
{

	EXITL (9, !h->size, "Empty heap");
	return (h->heap[0]->val);

}

extern inline void *EGheapGetMinThis (EGheap_t * h)
{

	EXITL (9, !h->size, "Empty heap");
	return (h->heap[0]->this);

}

extern inline EGheapConnector_t *EGheapGetMin (EGheap_t * h)
{

	EXITL (9, !h->size, "Empty heap");
	return (h->heap[0]);

}

extern inline EGheapConnector_t *EGnewHeapConnector (EGmemPool_t * mem,
																										 void *obj,
																										 EGheapCost_t val,
																										 unsigned int pos)
{

	EGheapConnector_t *hc;

	hc = (EGheapConnector_t *) EGmemPoolMalloc (mem, sizeof (EGheapConnector_t));

	hc->this = obj;
	hc->val = val;
	hc->pos = pos;

	return (hc);

}

extern inline int EGheapSiftup (EGheap_t * h,
																unsigned int current_pos)
{

	unsigned int father;
	EGheapConnector_t *swap;

	TESTL (9, current_pos >= h->size, "Position out of range");

	father = ((current_pos + 1) >> 1) - 1;

	while ((current_pos != 0)
				 && EGheapCostIsLess (h->heap[current_pos]->val, h->heap[father]->val))
	{

		swap = h->heap[current_pos];
		h->heap[current_pos] = h->heap[father];
		h->heap[father] = swap;

		h->heap[father]->pos = father;
		h->heap[current_pos]->pos = current_pos;

		current_pos = father;
		father = ((current_pos + 1) >> 1) - 1;

	}

	return 0;

}

extern inline EGheapConnector_t *EGheapInsert (EGheap_t * h,
																							 void *obj,
																							 EGheapCost_t val,
																							 EGmemPool_t * mem)
{

	EGheapConnector_t *h_con;

	EXITL (9, h->size >= h->max_size, "Heap full");

	h_con = EGnewHeapConnector (mem, obj, val, h->size);
	h->heap[h->size] = h_con;
	h->size++;

	EGheapSiftup (h, h->size - 1);

	return h_con;

}

extern inline int EGheapDecreaseVal (EGheap_t * h,
																		 EGheapConnector_t * hc,
																		 EGheapCost_t val)
{

	unsigned int rval;

	hc->val = val;
	rval = EGheapSiftup (h, hc->pos);
	CHECKRVAL (rval);

	return 0;

}

extern inline unsigned int EGheapGetMinChild (EGheap_t * h,
																							unsigned int pos)
{

	unsigned int first_pos,
	  last_pos;

	if (!h->size)
		return UINT_MAX;
	first_pos = (pos << 1) + 1;
	last_pos = h->size - 1;

	/* case both children exists */
	if (first_pos < last_pos)
	{
		last_pos = first_pos + 1;
		if (EGheapCostIsLess (h->heap[last_pos]->val, h->heap[first_pos]->val))
			return last_pos;
		return first_pos;
	}
	/* case of children out of range */
	if (first_pos > last_pos)
		return UINT_MAX;

	/* case there is only one child */
	return first_pos;
}

extern inline int EGheapSiftDown (EGheap_t * h,
																	unsigned int pos)
{
	unsigned int first_pos,
	  last_pos;
	unsigned int child_pos;
	EGheapConnector_t *swap;

	/* get min child */
	if (!h->size)
	{
		return 0;
	}
	first_pos = (pos << 1) + 1;
	last_pos = h->size - 1;

	/* case both chindren exists */
	if (first_pos < last_pos)
	{
		last_pos = first_pos + 1;
		if (EGheapCostIsLess (h->heap[last_pos]->val, h->heap[first_pos]->val))
			child_pos = last_pos;
		else
			child_pos = first_pos;
		goto pos1;
	}
	/* case of children out of range */
	if (first_pos > last_pos)
	{
		return 0;
	}
	/* case there is only one child */
	child_pos = first_pos;

	/* end getminchild */
pos1:


	while (child_pos < h->size
				 && EGheapCostIsLess (h->heap[child_pos]->val, h->heap[pos]->val))
	{
		swap = h->heap[pos];
		h->heap[pos] = h->heap[child_pos];
		h->heap[child_pos] = swap;

		h->heap[pos]->pos = pos;
		h->heap[child_pos]->pos = child_pos;

		pos = child_pos;
		/* get min child */
		first_pos = (pos << 1) + 1;
		last_pos = h->size - 1;

		/* case both children exists */
		if (first_pos < last_pos)
		{
			last_pos = first_pos + 1;
			if (EGheapCostIsLess (h->heap[last_pos]->val, h->heap[first_pos]->val))
				child_pos = last_pos;
			else
				child_pos = first_pos;
		}
		/* case of children out of range */
		else if (first_pos > last_pos)
		{
			return 0;
		}
		/* case there is only one child */
		else
		{
			child_pos = first_pos;
		}
		/* end getminchild */

	}

	return 0;

}

extern inline int EGheapDeleteMin (EGheap_t * h,
																	 EGmemPool_t * mem)
{

	EXITL (9, !h || !h->size, "Empty heap");

	EGmemPoolFree (h->heap[0], sizeof (EGheapConnector_t), mem);

	h->heap[0] = h->heap[h->size - 1];
	if (h->size - 1)
		h->heap[0]->pos = 0;

	h->heap[h->size - 1] = 0;

	h->size--;

	EGheapSiftDown (h, 0);

	return 0;

}

extern inline EGheap_t *EGnewHeap (EGmemPool_t * mem,
																	 unsigned int d,
																	 unsigned int n)
{

	EGheap_t *h;

	h = (EGheap_t *) EGmemPoolMalloc (mem, sizeof (EGheap_t));

	h->heap =
		(EGheapConnector_t **) EGmemPoolMalloc (mem,
																						sizeof (EGheapConnector_t *) * n);
	h->size = 0;
	h->max_size = n;
	h->d = d = 2;

	return (h);

}

extern inline void EGfreeHeapConnector (void *v,
																				EGmemPool_t * mem)
{

	EGmemPoolFree (v, sizeof (EGheapConnector_t), mem);
	return;

}

extern inline void EGheapClear (EGheap_t * h,
																EGmemPool_t * mem)
{

	unsigned int i;

	for (i = 0; i < h->size; i++)
		EGfreeHeapConnector (h->heap[i], mem);

	h->size = 0;

	return;

}



extern inline void EGfreeHeap (void *v,
															 EGmemPool_t * mem)
{

	EGheap_t *h;

	h = (EGheap_t *) (v);

	EGheapClear (h, mem);

	EGmemPoolFree (h->heap, sizeof (EGheapConnector_t *) * (h->max_size), mem);
	EGmemPoolFree (h, sizeof (EGheap_t), mem);

	return;

}

/* end eg_heap.h */
#endif
