/* EGlib "Efficient General Library" provides some basic structures and
 * algorithms commons in many optimization algorithms.
 *
 * Copyright (C) 2005 Daniel Espinoza and Marcos Goycoolea.
 * 
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but 
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY 
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public 
 * License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA 
 * */
/*****************************************************************************/
/*                                                                           */
/* This program reads a list of double values from a text file and uses a    */
/* heap to sort them. It also allows the user to change the value of one     */
/* of the read numbers after they have been placed in the heap. The purpose  */
/* of this program is to illustrate the use of the EGheap structure and its  */
/* associated functions.                                                     */
/*                                                                           */
/* The input file format is:                                                 */
/*                                                                           */
/* n                                                                         */
/* Value_0                                                                   */
/* Value_1                                                                   */
/* Value_2                                                                   */
/* Value_3                                                                   */
/*   ...                                                                     */
/* Value_n                                                                   */
/*                                                                           */
/*****************************************************************************/

#include <stdio.h>
#include <stdlib.h>

#ifdef LINUX
#include <getopt.h>
#endif

#include "eg_mempool.h"
#include "eg_heap.h"

void usage (char *program)
{
	fprintf (stdout, "Usage: %s [options]\n", program);
	fprintf (stdout, "Options:\n");
	fprintf (stdout, "     -d n   'd' value.\n");
	fprintf (stdout, "     -f n   file name.\n");
	fprintf (stdout, "     -c n   item whose value to change.\n");
	fprintf (stdout, "     -v n   new item value (0 by default).\n");

}

int parseargs (int argc,
							 char **argv,
							 unsigned int *d,
							 unsigned int *ch,
							 double *v,
							 char **file_name)
{

	int c;

	while ((c = getopt (argc, argv, "f:d:c:v:")) != EOF)
	{
		switch (c)
		{
		case 'f':
			*file_name = optarg;
			break;
		case 'd':
			*d = atoi (optarg);
			break;
		case 'c':
			*ch = atoi (optarg);
			break;
		case 'v':
			*v = atof (optarg);
			break;
		default:
			usage (argv[0]);
			return 1;
		}
	}

	/* reporting the options */
	fprintf (stdout, "Parsed Options:\n");
	fprintf (stdout, "input         : %s\n", *file_name);
	fprintf (stdout, "d             : %u\n", *d);
	if (*ch != UINT_MAX)
	{
		fprintf (stdout, "c             : %u\n", *ch);
		fprintf (stdout, "v             : %lf\n", *v);
	}

	return 0;

}

int main (int argc,
					char **argv)
{

	int rval;
	unsigned int i,
	  c = UINT_MAX,
	  d = 7;
	char *file_name = 0;
	FILE *file;

	unsigned int nval;
	double *val,
	  v = 0;

	EGmemPool_t *mem;
	EGheap_t *my_heap;
	EGheapConnector_t *hc = 0,
	 *chc = 0;

	/* Parse command line input to get 'file name' and 'd'. */
	rval = parseargs (argc, argv, &d, &c, &v, &file_name);
	CHECKRVAL (rval);

	/* Read the objects to sort from the file */
	file = fopen (file_name, "r");
	TEST (!file, "unable to open file %s", file_name);

	fscanf (file, "%u", &nval);

	val = (double *) malloc (sizeof (double) * nval);

	for (i = 0; i < nval; i++)
		fscanf (file, "%lf", &val[i]);

	fclose (file);

	/* Check if change value is in range */
	TEST (c != UINT_MAX
				&& c >= nval, "Change item (%u) is out of range (only %u objects)", c,
				nval);

	/* Create a new memory pool and heap */
	mem = EGnewMemPool (100, EGmemPoolNewSize, EGmemPoolNewSize (1));
	my_heap = EGnewHeap (mem, d, nval);

	/* Inserting the values into the heap */
	fprintf (stderr, "\nInserting:\n\n");
	for (i = 0; i < nval; i++)
	{
		hc = EGheapInsert (my_heap, (void *) i, EGheapToCost (val[i]), mem);
		fprintf (stderr, "%2i: val: %9.6lf \n", i, EGheapCostToLf (hc->val));
	}

	/* Popping the values from the heap */
	fprintf (stderr, "\nRemoving:\n\n");
	for (i = 0; i < nval; i++)
	{
		hc = EGheapGetMin (my_heap);
		fprintf (stderr, "%2i: item: %2u   val: %9.6lf\n", i,
						 (unsigned int) hc->this, EGheapCostToLf (hc->val));
		EGheapDeleteMin (my_heap, mem);
	}

	if (c == UINT_MAX)
		goto END;

	/* Re-inserting the values into the heap */
	fprintf (stderr, "\nRe-Inserting.\n\n");
	for (i = 0; i < nval; i++)
	{
		hc = EGheapInsert (my_heap, (void *) i, EGheapToCost (val[i]), mem);
		if (i == c)
			chc = hc;
	}

	/* Changing value of an item */

	TEST (chc->val <= v, "Can only decrease values");

	fprintf (stderr, "Changing value of item %u from %lf to %lf.\n", c,
					 EGheapCostToLf (chc->val), v);
	EGheapDecreaseVal (my_heap, chc, EGheapToCost (v));

	/* Popping the values from the heap */
	fprintf (stderr, "\nRemoving:\n\n");
	for (i = 0; i < nval; i++)
	{
		hc = EGheapGetMin (my_heap);
		fprintf (stderr, "%2i: item: %2u   val: %9.6lf\n", i,
						 (unsigned int) hc->this, EGheapCostToLf (hc->val));
		EGheapDeleteMin (my_heap, mem);
	}


	/* Liberating allocated memory */
END:

	EGfreeHeap (my_heap, mem);
	free (val);

	return 0;

}
