/* EGlib "Efficient General Library" provides some basic structures and
 * algorithms commons in many optimization algorithms.
 *
 * Copyright (C) 2005 Daniel Espinoza and Marcos Goycoolea.
 * 
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but 
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY 
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public 
 * License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA 
 * */
/* ========================================================================= */
/* Hash implementation over array of lists
 *
 * Version 0.0.1 2003-04-11
 * - 2003-06-17
 * 						- Add Display function.
 * - 2003-06-11 
 * 						- Mayor change, use internal list type.
 * - 2003-05-27 
 * 						- Add more debugging support, for example, if we set the
 * 							__EG_HASH_CHECK__ to non zero value, we will store how many
 * 							items are stored in the hash table.
 * 						- Add suport to check for repeated elements in the hash table at
 *	 						adding time when we __EG_HASH_CHECK__ is on 
 * - 2003-05-23 
 * 						- Fix a memory leak in EGfreeHash
 * - 2003-05-05 
 * 						- Change the definition of the internal prime numbers in an 
 * 							effort to improve the distributuin int the hash table
 *
 * */
/* ========================================================================= */

#ifndef __EG_HASH_H__
#define __EG_HASH_H__
#include "eg_macros.h"
#include "eg_listint.h"
#include <limits.h>
#include <inttypes.h>

/* ========================================================================= */
/* to store the key, if the key are too big we can  change that definition 
 * to long long int or wathever */
typedef uintmax_t EGkey_t;

/* ========================================================================= */
/* this is for the hash function */
/* prime number, this is one of the 1024 biggest prime for unsigned int */
extern EGkey_t EGhashPrime;
/* other prime number */
extern EGkey_t EGhashB;
/* and yet other prime number */
extern EGkey_t EGhashA;

/* ========================================================================= */
/* this is the hashing function */
#define EGhashFunction(key,maxsize,vptr)	{*(vptr) = ((unsigned int)( \
	(((EGkey_t)(key)) * EGhashB  + EGhashA )% EGhashPrime)) % (maxsize);}

/* ========================================================================= */
/* structure to store in the hash taable, for a key we store an integer 
 * this should be general enough */
typedef struct EGhashListNode__
{
	void *this;
	struct EGhashListNode__ *next;
	struct EGhashListNode__ *prev;
	EGkey_t key;
}
EGhashListNode_t;

/* ========================================================================= */
/* just to have some back compability */
typedef EGhashListNode_t EGhashData_t;

/* ========================================================================= */
/* structure that holds the internal lists */
typedef struct
{
	EGhashListNode_t *begin;
	EGhashListNode_t *end;
}
EGhashList_t;

/* ========================================================================= */
/* this is the structure of the hash table */
typedef struct
{
	size_t maxsize;
	EGmemPool_t *mempool;
	EGhashList_t **entry;
	size_t size;
#if __EG_HASH_PROFILE__
	const char *hashAllocationFile;
	int hashAllocationLine;
#endif
}
EGhash_t;

/* ========================================================================= */
/* just to make some testing */
void EGhashTest (void);

/* we only need to create hash tables, liberate them, insert, find and delete 
 * this return a pointer to an initialized hash table of maxsize 'size' */
#if __EG_HASH_PROFILE__
#define EGnewHash(A,B) __EGnewHash(A,B,__FILE__,__LINE__);
EGhash_t *__EGnewHash (size_t maxsize,
											 EGmemPool_t * mempool,
											 const char *file,
											 const int line);
#else
#define EGnewHash(A,B) __EGnewHash(A,B);
EGhash_t *__EGnewHash (size_t maxsize,
											 EGmemPool_t * mempool);
#endif

/* ========================================================================= */
/* this free a hash table, it need a free function for the internal data 
 * stored in 'data' note that you can give the nullFree function as parameter 
 * the hash will not free internal data stored with the hash, that is 
 * responsability if the external user */
void EGfreeHash (void *hash);

/* ========================================================================= */
/* this is to insert a new element to the hash */
int EGhashInsert (EGhash_t * hash,
									const EGkey_t key,
									void *data);

/* ========================================================================= */
/* this is to find an element in the hash table, if the key is not found, it 
 * return 0, if the adress of list node is not zero, it will store there the
 * addres of the pointer containing the first entry found, this is useful when
 * we have multiple entries with the same hash_key */
extern inline EGhashData_t *EGhashFind (const EGhash_t * hash,
																				const EGkey_t key)
{

	/* local variables */
	unsigned int pos;
	EGhashListNode_t *It;

	/* we compute the hash function */
	EGhashFunction (key, hash->maxsize, &pos);

	/* now we lookup for the key in the corresponding list */
	if (!hash->entry[pos])
		return 0;
	for (It = hash->entry[pos]->begin; It; It = It->next)
	{
		if (It->key == key)
		{
			return It;
		}
	}

	/* ending */
	return 0;
}

/* ========================================================================= */
/* this is to eliminate an element from the hash table the hash will not free 
 * internal data stored with the hash, that is responsability if the external 
 * user */
int EGhashErase (EGhash_t *,
								 EGkey_t);

/* ========================================================================= */
/* this function display the internal state of the hash table */
int EGhashDisplay (const EGhash_t * hash,
									 FILE * logFile);

/* ========================================================================= */
/* this function leaves the hash table 'as new', it recives a free function for
 * the internal data */
extern inline void EGhashClear (EGhash_t * hash,
																EGfree_f dataFree)
{
	/* local variables */
	register int i;

	/* we loop through the entries of the hash */
	for (i = ((EGhash_t *) hash)->maxsize; i;)
	{
		i--;
		if (((EGhash_t *) hash)->entry[i])
		{
			__EGlistIntClear (((EGhash_t *) hash)->entry[i], dataFree,;,
												hash->mempool, EGhashListNode_t);
			__EGlistIntFreeList (((EGhash_t *) hash)->entry[i], hash->mempool,
													 EGhashList_t, EGhashListNode_t);
			hash->entry[i] = 0;
		}														/* end if */
	}															/* end for i */


}

/* end eg_hash.h */
#endif
