/* EGlib "Efficient General Library" provides some basic structures and
 * algorithms commons in many optimization algorithms.
 *
 * Copyright (C) 2005 Daniel Espinoza and Marcos Goycoolea.
 * 
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but 
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY 
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public 
 * License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA 
 * */
/* ========================================================================= */
/* this is the header file for the CPLEX structure
 *
 * 2004-04-07 - Fiz free of null arrays (Renan Garcia)
 * 2003-05-19 - Fix a bug in EGcplexAdd{Col,Row}* related to malloc if size zero
 * 2003-05-07 - Fix some memory leacks in EGcplexSetAll*
 * 2003-05-06 - Add the EGcplexNullCutCallback.
 *						- Add suport for getting the number of nonzeros of a given column.
 * 						- Fix EGcplexGetPi function
 * 2003-04-11 - Change macros to extern inline functions support EGmemPool
 * 2003-04-14 - Fix include problems with inline declarations
 * 2003-04-15 - Fix CPXENVptr problems with compile warnings
 * 						- Add more compiling warning messages and options
 *
 * */
/* ========================================================================= */

#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include "eg_macros.h"
#include "eg_mempool.h"
#include "eg_cplexTypes.h"
#include "eg_list.h"
#include "eg_col.h"
#include "eg_row.h"
#include "eg_io.h"
#include "cplex.h"
#ifdef HAVE_CPLEX
#ifndef __EG_CPLEX_H__
#define __EG_CPLEX_H__

/* ========================================================================= */
/* first some definitions to be able to work with cplex prior 8.1 and after 81 */
/* ========================================================================= */
#if CPX_VERSION < 800
#define EGCPXPROB_MIP CPXPROB_MIP
typedef CPXENVptr EGCPXENVptr;
#else
#define EGCPXPROB_MIP CPXPROB_MILP
typedef CPXCENVptr EGCPXENVptr;
#endif

/* ========================================================================= */
/* if we are debuggin then we try to explain what have happend with cplex, if 
 * not, we just don't care, this allows us to eliminate some overhead and also 
 * to define SOME cplex operations as macros, that is WAY faster than a 
 * function */
/* ========================================================================= */
#if DEBUG >= 0
extern char EGcplexErrMsg[];

/* ========================================================================= */
/* the use of CPLEXTEST depends on two things, first, that the status of the 
 * CPLEX call was stored on Prob->status and that the macro recived as 
 * parameter the EGcplex_t* as A and that B is a printable char* */
/* ========================================================================= */
#define CPLEXTEST(Prob,Mesg) {if(Prob->status){ \
													CPXgeterrorstring(Prob->env,Prob->status,EGcplexErrMsg);\
													fprintf(stderr,"%s, in file %s, line %d\n",Mesg,__FILE__,__LINE__);\
													fprintf(stderr,"%s\n",EGcplexErrMsg); }}
#else
#define CPLEXTEST(Prob,B) ;
#endif

/* ========================================================================= */
/* structure to store a CPLEX problem */
/* ========================================================================= */
typedef struct
{
	CPXENVptr env;
	EGmemPool_t *mempool;
	CPXLPptr lp;
	int status;
	char internal;
}
EGcplex_t;

/* ========================================================================= */
/* return a pointer to an initialized EGcplex object if the invironment is 
 * null, it will create an ionternal environment that will be released when we 
 * free it through EGfreeCplex */
/* ========================================================================= */
EGcplex_t *EGnewCplex (CPXENVptr env,
											 EGmemPool_t * mempool);

/* ========================================================================= */
/* create a new CPLEX environment */
/* ========================================================================= */
CPXENVptr EGnewCplexEnv (void);

/* ========================================================================= */
/* liberate the memory returned by EGnewCplex, __NEVER__ liberate such a 
 * memory space with free */
/* ========================================================================= */
void EGfreeCplex (void *);

/* ========================================================================= */
/* liberate the memory returned by EGnewCplexEnv, __NEVER__ liberate such a 
 * memory space with free */
/* ========================================================================= */
void EGfreeCplexEnv (void *);

/* ========================================================================= */
/* set all variables to type binary */
/* ========================================================================= */
int EGcplexSetAllBin (EGcplex_t *);

/* ========================================================================= */
/* set all variables to type integer */
/* ========================================================================= */
int EGcplexSetAllInt (EGcplex_t *);

/* ========================================================================= */
/* set all variables to type continue */
/* ========================================================================= */
int EGcplexSetAllCon (EGcplex_t *);

/* ========================================================================= */
/* add a callback-cut to the problem */
/* ========================================================================= */
int EGcplexAddCallbackCut (EGcplex_t * Prob,
													 void *cbdata,
													 int wherefrom,
													 EGrow_t * row);

/* ========================================================================= */
/* add a list of rows to the problem */
/* ========================================================================= */
int EGcplexAddRowList (EGcplex_t *,
											 EGlist_t *);

/* ========================================================================= */
/* add an array of rows to the problem */
/* ========================================================================= */
int EGcplexAddRowArray (EGcplex_t *,
												EGrow_t **,
												unsigned int);

/* ========================================================================= */
/* add a list of columns to the problem */
/* ========================================================================= */
int EGcplexAddColList (EGcplex_t *,
											 EGlist_t *);

/* ========================================================================= */
/* add an array of rows to the problem */
/* ========================================================================= */
int EGcplexAddColArray (EGcplex_t *,
												EGcol_t **,
												int);

/* ========================================================================= */
/* Change problem type to LP. */
/* ========================================================================= */
extern inline int EGcplexSetToLp (EGcplex_t * Prob)
{
	CPXgetprobtype (Prob->env, Prob->lp);
	TEST ((Prob->status == -1), "no problem or enviroment");
	if (Prob->status)
		Prob->status = CPXchgprobtype (Prob->env, Prob->lp, CPXPROB_LP);
	CHECKRVAL (Prob->status);
	return Prob->status;
}

/* ========================================================================= */
/* Change problem type to MIP. */
/* ========================================================================= */
extern inline int EGcplexSetToMip (EGcplex_t * Prob)
{
	CPXgetprobtype (Prob->env, Prob->lp);
	TEST ((Prob->status == -1), "in EGcplexSetToMip, no problem or enviroment");
	if (Prob->status != 1)
		Prob->status = CPXchgprobtype (Prob->env, Prob->lp, EGCPXPROB_MIP);
	else
		Prob->status = 0;
	CPLEXTEST (Prob, "in EGcplexSetToMip");
	return Prob->status;
}

/* ========================================================================= */
/* Primal Simplex , store the optimization status in Prob->status status 0 
 * means a solution is found (not necesarily optimal) (see CPLEX manual for 
 * other values). */
/* ========================================================================= */
extern inline int EGcplexPrimopt (EGcplex_t * Prob)
{
	EGcplexSetToLp (Prob);
	TEST ((Prob->status), "in EGcplexPrimopt");
	Prob->status = CPXprimopt (Prob->env, Prob->lp);
	CPLEXTEST (Prob, "in EGcplexPrimopt");
	return Prob->status;
}

/* ========================================================================= */
/* hybrid Network Simplex with metod B; B = 'p' (primal) or 'd' (dual), return 
 * the optimization status in Prob->status, 0 means a solution is found (not 
 * necesarily optimal) see CPLEX manual for other values). */
/* ========================================================================= */
extern inline int EGcplexNetopt (EGcplex_t * Prob,
																 const char B)
{
	EGcplexSetToLp (Prob);
	TEST ((Prob->status), "in EGcplexNetopt");
	Prob->status = CPXhybnetopt (Prob->env, Prob->lp, B);
	CPLEXTEST (Prob, "in EGcplexNetopt");
	return Prob->status;
}

/* ========================================================================= */
/* Dual Simplex, return the optimization status in Prob->status, 0 means a 
 * solution is found (not necesarily optimzal) (see CPLEX manual for other 
 * values). */
/* ========================================================================= */
extern inline int EGcplexDualopt (EGcplex_t * Prob)
{
	EGcplexSetToLp (Prob);
	TEST ((Prob->status), "in EGcplexDualopt");
	Prob->status = CPXdualopt (Prob->env, Prob->lp);
	CPLEXTEST (Prob, "in EGcplexDualopt");
	return Prob->status;
}

/* ========================================================================= */
/* Mixed Integer Methods , return the optimization status 0 means a solution 
 * is found (not necesarily optimzal) (see CPLEX manual for other values). 
 * where <B>maxIntSol</B> is the maximun number of integer solution to be found 
 * before stoping. There are other issues as precision, for that you have to 
 * configure the environment, see the environment information */
/* ========================================================================= */
extern inline int EGcplexMipopt (EGcplex_t * Prob)
{
	EGcplexSetToMip (Prob);
	TEST ((Prob->status), "in EGcplexMipopt");
	Prob->status = CPXmipopt (Prob->env, Prob->lp);
	CPLEXTEST (Prob, "in EGcplexMipopt");
	return Prob->status;
}

/* ========================================================================= */
/* change the type of the i-th variable to 'type', type must	be 'B' for 
 * binary, 'I' for integer or 'C' for continue */
/* ========================================================================= */
extern inline int EGcplexSetCtype (EGcplex_t * Prob,
																	 int nz,
																	 int *ind,
																	 char *types)
{
	EGcplexSetToMip (Prob);
	TEST ((Prob->status), "in EGcplexSetCtype");
	Prob->status = CPXchgctype (Prob->env, Prob->lp, nz, ind, types);
	CPLEXTEST (Prob, "in EGcplexSetCtype");
	return Prob->status;
}

/* ========================================================================= */
/* set the problem for minimize (default) */
/* ========================================================================= */
extern inline int EGcplexSetToMin (EGcplex_t * Prob)
{
	CPXchgobjsen (Prob->env, Prob->lp, CPX_MIN);
	return 0;
}

/* ========================================================================= */
/* set the problem for maximize */
/* ========================================================================= */
extern inline int EGcplexSetToMax (EGcplex_t * Prob)
{
	CPXchgobjsen (Prob->env, Prob->lp, CPX_MAX);
	return 0;
}

/* ========================================================================= */
/* set the right hand side of the rows in ind to val, nz indicate the number 
 * of entries in the arrays name and val. Prob is the EGcplex_t structure */
/* ========================================================================= */
extern inline int EGcplexSetRhs (EGcplex_t * Prob,
																 int nz,
																 int *ind,
																 double *val)
{
	Prob->status = CPXchgrhs (Prob->env, Prob->lp, nz, ind, val);
	CPLEXTEST (Prob, "in EGcplexSetRhs");
	return Prob->status;
}

/* ========================================================================= */
/* set the sense of a set of constraints */
/* ========================================================================= */
extern inline int EGcplexSetSense (EGcplex_t * Prob,
																	 int nz,
																	 int *ind,
																	 char *sense)
{
	Prob->status = CPXchgsense (Prob->env, Prob->lp, nz, ind, sense);
	CPLEXTEST (Prob, "in EGcplexSetSense");
	return Prob->status;
}

/* ========================================================================= */
/* set the objetive coefficient of the pos-th variable to val Prob is the 
 * EGcplex_t structure, nz is the length of the arrays ind is the array of 
 * indices, type is the array of types of bounds to change and val is the new 
 * value of the bound */
/* ========================================================================= */
extern inline int EGcplexSetObj (EGcplex_t * Prob,
																 int nz,
																 int *ind,
																 double *val)
{
	Prob->status = CPXchgobj (Prob->env, Prob->lp, nz, ind, val);
	CPLEXTEST (Prob, "in EGcplexSetObj");
	return Prob->status;
}

/* ========================================================================= */
/* set for all the variables in 'bound' the limit 'type' to the double stored 
 * in the map; type must be one of the folowin values : 'U' for the upper 
 * limit, 'L' for the lower limit and 'B' for both the lower and the upper 
 * limit. Prob is the EGcplex_t structure, nz is the length of the arrays ind 
 * is the array of indices, type is the array of types of bounds to change and 
 * val is the new value of the bound */
/* ========================================================================= */
extern inline int EGcplexSetBound (EGcplex_t * Prob,
																	 int nz,
																	 int *ind,
																	 char *type,
																	 double *val)
{
	Prob->status = CPXchgbds (Prob->env, Prob->lp, nz, ind, type, val);
	CPLEXTEST (Prob, "in EGcplexSetBound");
	return Prob->status;
}

/* ========================================================================= */
/* indicates the number of rows (or restrictions) in the problem */
/* ========================================================================= */
extern inline int EGcplexGetNumRows (EGcplex_t * Prob,
																		 int *nrowptr)
{
	*nrowptr = CPXgetnumrows (Prob->env, Prob->lp);
	return 0;
}

/* ========================================================================= */
/* indicates the number of cols (or variables) in the problem */
/* ========================================================================= */
extern inline int EGcplexGetNumCols (EGcplex_t * Prob,
																		 int *ncolptr)
{
	*ncolptr = CPXgetnumcols (Prob->env, Prob->lp);
	return 0;
}

/* ========================================================================= */
/* directly add columns to cplex */
/* ========================================================================= */
int EGcplexAddCol (EGcplex_t * Prob,
									 int ccnt,
									 int cnz,
									 double *cobj,
									 int *cmbeg,
									 int *cmind,
									 double *cmval,
									 double *clb,
									 double *cub,
									 char **cname);

/* ========================================================================= */
/* directyly add rows to cplex */
/* ========================================================================= */
int EGcplexAddRow (EGcplex_t * Prob,
									 int rcnt,
									 int rnz,
									 double *rhs,
									 char *sense,
									 int *rmbeg,
									 int *rmind,
									 double *rmval,
									 char **rname);

/* ========================================================================= */
/* directyly add rows to cplex (in a callback function) */
/* ========================================================================= */
extern inline int EGcplexAddCut (EGcplex_t * Prob,
																 void *cbdata,
																 int wherefrom,
																 int rnz,
																 double rhs,
																 char sense,
																 int *rmind,
																 double *rmval)
{
	Prob->status =
		CPXcutcallbackadd (Prob->env, cbdata, wherefrom, rnz, rhs, sense, rmind,
											 rmval);
	CPLEXTEST (Prob, "in EGcplexAddCut");
	return Prob->status;
}

/* ========================================================================= */
/* deletes rows with index bigger or equal than 'beg' but smaller than 'end' */
/* ========================================================================= */
extern inline int EGcplexDelRows (EGcplex_t * Prob,
																	int beg,
																	int end)
{
	Prob->status = CPXdelrows (Prob->env, Prob->lp, beg, end - 1);
	CPLEXTEST (Prob, "in EGcplexDelRows");
	return Prob->status;
}

/* ========================================================================= */
/* deletes cols with index bigger or equal than 'beg' but smaller than 'end' */
/* ========================================================================= */
extern inline int EGcplexDelCols (EGcplex_t * Prob,
																	int beg,
																	int end)
{
	Prob->status = CPXdelcols (Prob->env, Prob->lp, beg, end - 1);
	CPLEXTEST (Prob, "in EGcplexDelCols");
	return Prob->status;
}

/* ========================================================================= */
/* return the CPLEX status of the problem, that status indicates optimality, 
 * infeasibility, etc., see (CPLEX manual for further information). */
/* ========================================================================= */
extern inline int EGcplexGetStat (EGcplex_t * Prob,
																	int *statusptr)
{
	*statusptr = CPXgetstat (Prob->env, Prob->lp);
	return 0;
}

/* ========================================================================= */
/* return the current objetive value for the problem. */
/* ========================================================================= */
extern inline int EGcplexGetBestObjval (EGcplex_t * Prob,
																				double *valptr)
{
	Prob->status = CPXgetprobtype (Prob->env, Prob->lp);
	if (Prob->status == 0)
		return 1;
	else if (Prob->status == 1)
		Prob->status = CPXgetbestobjval (Prob->env, Prob->lp, valptr);
	else
		TEST (1, "in EGcplexGetBestObjval, unknown problem type");
	CPLEXTEST (Prob, "in EGcplexGetBestObjval");
	return Prob->status;
}

/* ========================================================================= */
/* return the current objetive value for the problem. */
/* ========================================================================= */
extern inline int EGcplexGetObjval (EGcplex_t * Prob,
																		double *valptr)
{
	Prob->status = CPXgetprobtype (Prob->env, Prob->lp);
	if (Prob->status == 0)
		Prob->status = CPXgetobjval (Prob->env, Prob->lp, valptr);
	else if (Prob->status == 1)
		Prob->status = CPXgetmipobjval (Prob->env, Prob->lp, valptr);
	else
		EXIT (1, "in EGcplexGetObjval, unknown problem type");
	CPLEXTEST (Prob, "in EGcplexGetObjval");
	return Prob->status;
}

/* ========================================================================= */
/* given an array x of size end - begin and a problem, return the current 
 * values for the variables in the range [begin,end[, i.e. we obtain 
 * x[begin],....,x[end-1] */
/* ========================================================================= */
extern inline int EGcplexGetX (EGcplex_t * Prob,
															 double *x,
															 int begin,
															 int end)
{
	Prob->status = CPXgetprobtype (Prob->env, Prob->lp);
	if (Prob->status == 0)
		Prob->status = CPXgetx (Prob->env, Prob->lp, x, begin, end - 1);
	else if (Prob->status == 1)
		Prob->status = CPXgetmipx (Prob->env, Prob->lp, x, begin, end - 1);
	else
		EXIT (1, "in EGcplexGetX, unknown problem type");
	CPLEXTEST (Prob, "in EGcplexGetX");
	return Prob->status;
}

/* ========================================================================= */
/* given an array x of size end - begin and a problem, return the current 
 * dual values for the constraint in the range [begin,end[, i.e. we obtain 
 * x[begin],....,x[end-1] */
/* ========================================================================= */
extern inline int EGcplexGetPi (EGcplex_t * Prob,
																double *x,
																int begin,
																int end)
{
	Prob->status = CPXgetprobtype (Prob->env, Prob->lp);
	if (Prob->status == 0)
		Prob->status = CPXgetpi (Prob->env, Prob->lp, x, begin, end - 1);
	else
		EXIT (1, "in EGcplexGetPi, unknown problem type or problem is MIP");
	CPLEXTEST (Prob, "in EGcplexGetPi");
	return Prob->status;
}

/* ========================================================================= */
/* given an array x of size end - begin and a problem, return the current 
 * slack for the constraints in the range [begin,end[, i.e. we obtain 
 * x[begin],....,x[end-1] */
/* ========================================================================= */
extern inline int EGcplexGetSlack (EGcplex_t * Prob,
																	 double *x,
																	 int begin,
																	 int end)
{
	Prob->status = CPXgetprobtype (Prob->env, Prob->lp);
	if (Prob->status == 0)
		Prob->status = CPXgetslack (Prob->env, Prob->lp, x, begin, end - 1);
	else if (Prob->status == 1)
		Prob->status = CPXgetmipslack (Prob->env, Prob->lp, x, begin, end - 1);
	else
		EXIT (1, "in EGcplexGetSlack, unknown problem type");
	CPLEXTEST (Prob, "in EGcplexGetSlack");
	return Prob->status;
}

/* ========================================================================= */
/* given an array x of size end - begin and a problem, return the current 
 * reduced cost values for the variables in the range [begin,end[, i.e. we 
 * obtain x[begin],....,x[end-1] */
/* ========================================================================= */
extern inline int EGcplexGetDj (EGcplex_t * Prob,
																double *x,
																int begin,
																int end)
{
	Prob->status = CPXgetprobtype (Prob->env, Prob->lp);
	if (Prob->status == 0)
		Prob->status = CPXgetdj (Prob->env, Prob->lp, x, begin, end - 1);
	else
		EXIT (1, "in EGcplexGetDj, unknown problem type or problem is MIP");
	CPLEXTEST (Prob, "in EGcplexGetDj");
	return Prob->status;
}

/* ========================================================================= */
/* return in how many constraints a particular column bellongs */
/* ========================================================================= */
extern inline int EGcplexGetColNZ (EGcplex_t * Prob,
																	 int cind,
																	 int *cnz)
{

	/* local variables */
	int colnz,
	  matbeg;

	Prob->status =
		CPXgetcols (Prob->env, Prob->lp, &colnz, &matbeg, 0, 0, 0, cnz, cind,
								cind);
	if (Prob->status != 1207)
		CPLEXTEST (Prob, "in EGcplexGetColNZ");
	*cnz *= -1;

	/* ending */
	return 0;
}

/* ========================================================================= */
/* Return the column names Prob is a pointer to a EGcplex_t structure begin 
 * denote the first column name to take end denote the last plus one column 
 * name to take name is an an array of size end - begin of pointers to char* 
 * name store is an array of chars of size storesize where the names will be 
 * stored surplusptr is a pointer to an integer where the size of the non used 
 * space in store will be stored if *surplusptr is negative, the call had fail
 * */
/* ========================================================================= */
extern inline int EGcplexGetColname (EGcplex_t * Prob,
																		 char **names,
																		 char *store,
																		 int storesize,
																		 int *splusptr,
																		 int begin,
																		 int end)
{
	Prob->status =
		CPXgetcolname (Prob->env, Prob->lp, names, store, storesize, splusptr,
									 begin, end - 1);
	if (Prob->status != CPXERR_NEGATIVE_SURPLUS)
		CPLEXTEST (Prob, "in EGcplexGetColname");
	return Prob->status;
}

/* ========================================================================= */
/* Return a range of row coefficients and indices. Prob is a pointer to a 
 * EGcplex_t structure, begin denote the first row to take, end denote the last 
 * plus one row name to take, nz is the size of the arrays rbeg, rind and rval.
 * if *surplusptr is negative, the call had fail */
/* ========================================================================= */
extern inline int EGcplexGetRows (EGcplex_t * Prob,
																	int *nz,
																	int *rbeg,
																	int *rind,
																	double *rval,
																	int size,
																	int *surplus,
																	int begin,
																	int end)
{
	Prob->status =
		CPXgetrows (Prob->env, Prob->lp, nz, rbeg, rind, rval, size, surplus,
								begin, end - 1);
	if (Prob->status == CPXERR_NEGATIVE_SURPLUS)
		return 0;
	CPLEXTEST (Prob, "in EGcplexGetRows");
	return Prob->status;
}

extern inline int EGcplexGetCols (EGcplex_t * Prob,
																	int *nz,
																	int *cbeg,
																	int *cind,
																	double *cval,
																	int size,
																	int *surplus,
																	int begin,
																	int end)
{
	Prob->status =
		CPXgetcols (Prob->env, Prob->lp, nz, cbeg, cind, cval, size, surplus,
								begin, end - 1);
	if (Prob->status == CPXERR_NEGATIVE_SURPLUS)
		return 0;
	CPLEXTEST (Prob, "in EGcplexGetCols");
	return Prob->status;
}

/* ========================================================================= */
/* Return the row names Prob is a pointer to a EGcplex_t structure begin 
 * denote the first row name to take end denote the last plus one row name to 
 * take name is an an array of size end - begin of pointers to char* name 
 * store is an array of chars of size storesize where the names will be stored 
 * surplusptr is a pointer to an integer where the size of the non used space 
 * in store will be stored if *surplusptr is negative, the call had fail */
/* ========================================================================= */
extern inline int EGcplexGetRowname (EGcplex_t * Prob,
																		 char **names,
																		 char *store,
																		 int storesize,
																		 int *splusptr,
																		 int begin,
																		 int end)
{
	Prob->status =
		CPXgetrowname (Prob->env, Prob->lp, names, store, storesize, splusptr,
									 begin, end - 1);
	if (Prob->status != CPXERR_NEGATIVE_SURPLUS)
		CPLEXTEST (Prob, "in EGcplexGetRowname");
	return Prob->status;
}

/* ========================================================================= */
/* given an array x of size end - begin and a problem, return the current 
 * upper bounds values for the variables in the range [begin,end[, i.e. we 
 * obtain x[begin],....,x[end-1] */
/* ========================================================================= */
extern inline int EGcplexGetUb (EGcplex_t * Prob,
																double *x,
																int begin,
																int end)
{
	Prob->status = CPXgetub (Prob->env, Prob->lp, x, begin, end - 1);
	CPLEXTEST (Prob, "in EGcplexGetUb");
	return Prob->status;
}

/* ========================================================================= */
/* given an array x of size end - begin and a problem, return the current 
 * lower bounds values for the variables in the range [begin,end[, i.e. we 
 * obtain x[begin],....,x[end-1] */
/* ========================================================================= */
extern inline int EGcplexGetLb (EGcplex_t * Prob,
																double *x,
																int begin,
																int end)
{
	Prob->status = CPXgetlb (Prob->env, Prob->lp, x, begin, end - 1);
	CPLEXTEST (Prob, "in EGcplexGetLb");
	return Prob->status;
}

/* ========================================================================= */
/* given an array x of size end - begin and a problem, return the objective 
 * values for the variables in the range [begin,end[, i.e. we obtain 
 * x[begin],....,x[end-1] */
/* ========================================================================= */
extern inline int EGcplexGetObjCoeff (EGcplex_t * Prob,
																			double *x,
																			int begin,
																			int end)
{
	Prob->status = CPXgetobj (Prob->env, Prob->lp, x, begin, end - 1);
	CPLEXTEST (Prob, "in EGcplexGetObjCoeff");
	return Prob->status;
}

/* ========================================================================= */
/* given an array x of size end - begin and a problem, return the sense 
 * values for the constraints in the range [begin,end[, i.e. we obtain 
 * x[begin],....,x[end-1] */
/* ========================================================================= */
extern inline int EGcplexGetSense (EGcplex_t * Prob,
																	 char *x,
																	 int begin,
																	 int end)
{
	Prob->status =
		CPXgetsense ((EGCPXENVptr) (Prob->env), Prob->lp, x, begin, end - 1);
	CPLEXTEST (Prob, "in EGcplexGetSense");
	return Prob->status;
}

/* ========================================================================= */
/* given an array x of size end - begin and a problem, return the sense values 
 * for the constraints in the range [begin,end[, i.e. we obtain 
 * x[begin],....,x[end-1] */
/* ========================================================================= */
extern inline int EGcplexGetRhs (EGcplex_t * Prob,
																 double *x,
																 int begin,
																 int end)
{
	Prob->status = CPXgetrhs (Prob->env, Prob->lp, x, begin, end - 1);
	CPLEXTEST (Prob, "in EGcplexGetRhs");
	return Prob->status;
}

/* ========================================================================= */
/* save the problem structure in format type, where type must be "SAV" for 
 * binary matrix and basis file, "MPS" for MPS format, "LP" for CPLEX LP 
 * format, "REW" or "RMP" for MPS format with all names changed to generic 
 * names or "RLP" for LP format with all names changed to generic names. */
/* ========================================================================= */
extern inline int EGcplexWriteProb (EGcplex_t * Prob,
																		const char *file,
																		const char *type)
{
#if CPX_VERSION > 800
	Prob->status = CPXwriteprob (Prob->env, Prob->lp, file, type);
#else
	char *lfile = strdup (file);
	char *ltype = strdup (type);
	Prob->status = CPXwriteprob (Prob->env, Prob->lp, lfile, ltype);
	free (lfile);
	free (ltype);
#endif
	CPLEXTEST (Prob, "in EGcplexWriteProb");
	return Prob->status;
}

/* ========================================================================= */
/* add a function callback for Cplex incumber solution during MIP note that 
 * the callback is asociated to the CPLEX environment and not to the CPLEX 
 * problem */
/* ========================================================================= */
extern inline int EGcplexAddIncumbentCallBack (EGcplex_t * Prob,
																							 int (CPXPUBLIC *
																										callback) (EGCPXENVptr,
																															 void *,
																															 int,
																															 void *,
																															 double,
																															 double *,
																															 int *,
																															 int *),
																							 void *cinfo)
{
#if CPX_VERSION > 700
	Prob->status =
		CPXsetincumbentcallbackfunc ((CPXENVptr) (Prob->env), callback, cinfo);
	CPLEXTEST (Prob, "in EGcplexAddIncumbentCallBack");
#else
	EXIT (1, "in EGcplexAddIncumbentCallBack, not supported before Cplex 7.1");
#endif
	return Prob->status;
}

/* ========================================================================= */
/* add a function callback for cut generation during MIP note that the 
 * callback is asociated to the CPLEX environment and not to the CPLEX 
 * problem */
/* ========================================================================= */
extern inline int EGcplexAddCutCallBack (EGcplex_t * Prob,
																				 int (CPXPUBLIC *
																							callback) (EGCPXENVptr,
																												 void *,
																												 int,
																												 void *,
																												 int *),
																				 void *cinfo)
{
	Prob->status =
		CPXsetcutcallbackfunc ((CPXENVptr) (Prob->env), callback, cinfo);
	CPLEXTEST (Prob, "in EGcplexAddIncumbentCallBack");
	return Prob->status;
}

/* ========================================================================= */
/* this function is to set any integer parameter of CPLEX, this affects the
 * nefironment */
/* ========================================================================= */
extern inline int EGcplexSetDblParam (CPXENVptr env,
																			int param,
																			double value)
{
	int status;
	status = CPXsetdblparam ((CPXENVptr) env, param, value);
	if (status)
	{
		CPXgeterrorstring (env, status, EGcplexErrMsg);
		fprintf (stderr, "in EGcplexSetDblParam, in file %s, line %d\n", __FILE__,
						 __LINE__);
		fprintf (stderr, "%s\n", EGcplexErrMsg);
	}
	return status;
}

/* ========================================================================= */
/* this function is to set any integer parameter of CPLEX, this affects the
 * nefironment */
/* ========================================================================= */
extern inline int EGcplexSetIntParam (CPXENVptr env,
																			int param,
																			int value)
{
	int status;
	status = CPXsetintparam ((CPXENVptr) env, param, value);
	if (status)
	{
		CPXgeterrorstring (env, status, EGcplexErrMsg);
		fprintf (stderr, "in EGcplexSetIntParam, in file %s, line %d\n", __FILE__,
						 __LINE__);
		fprintf (stderr, "%s\n", EGcplexErrMsg);
	}
	return status;
}

/* ========================================================================= */
/* set the CPLEX environment output off, all problems sharing this environment
 * will not report to screen */
/* ========================================================================= */
extern inline int EGcplexSetScreenOff (EGcplex_t * Prob)
{
	Prob->status =
		CPXsetintparam ((CPXENVptr) (Prob->env), CPX_PARAM_SCRIND, CPX_OFF);
	CPLEXTEST (Prob, "in EGcplexSetScreenOff");
	return Prob->status;
}

/* ========================================================================= */
/* set the CPLEX environment output on, all problems sharing this environment
 * will report to screen */
/* ========================================================================= */
extern inline int EGcplexSetScreenOn (EGcplex_t * Prob)
{
	Prob->status =
		CPXsetintparam ((CPXENVptr) (Prob->env), CPX_PARAM_SCRIND, CPX_ON);
	CPLEXTEST (Prob, "in EGcplexSetScreenOn");
	return Prob->status;
}

/* ========================================================================= */
/* set an upper bound for the LP objetive function, this bound will affect all 
 * problems that share the CPLEX environment */
/* ========================================================================= */
extern inline int EGcplexSetULP (EGcplex_t * Prob,
																 double ubound)
{
	Prob->status =
		CPXsetdblparam ((CPXENVptr) (Prob->env), CPX_PARAM_OBJULIM, ubound);
	CPLEXTEST (Prob, "in EGcplexSetULP");
	return Prob->status;
}

/* ========================================================================= */
/* set an lower bound for the LP objetive function, this bound will affect all 
 * problems that share the CPLEX environment */
/* ========================================================================= */
extern inline int EGcplexSetLLP (EGcplex_t * Prob,
																 double lbound)
{
	Prob->status =
		CPXsetdblparam ((CPXENVptr) Prob->env, CPX_PARAM_OBJLLIM, lbound);
	CPLEXTEST (Prob, "in EGcplexSetLLP");
	return Prob->status;
}

/* ========================================================================= */
/* set an upper bound for the MIP objetive function, this bound will affect 
 * all problems that share the CPLEX environment */
/* ========================================================================= */
extern inline int EGcplexSetUMIP (EGcplex_t * Prob,
																	double ubound)
{
	Prob->status =
		CPXsetdblparam ((CPXENVptr) Prob->env, CPX_PARAM_CUTUP, ubound);
	CPLEXTEST (Prob, "in EGcplexSetUMIP");
	return Prob->status;
}

/* ========================================================================= */
/* set an lower bound for the MIP objetive function, this bound will affect 
 * all problems that share the CPLEX environment */
/* ========================================================================= */
extern inline int EGcplexSetLMIP (EGcplex_t * Prob,
																	double lbound)
{
	Prob->status =
		CPXsetdblparam ((CPXENVptr) Prob->env, CPX_PARAM_CUTLO, lbound);
	CPLEXTEST (Prob, "in EGcplexSetLLP");
	return Prob->status;
}

/* ========================================================================= */
/* this function, as its name says, is a null cut callback, this is used in
 * order to reset or shadow a previous callback asociated to the underlying
 * CPXENV. Note that all calbacks are associated to the environment instead of
 * the problem. */
/* ========================================================================= */
typedef int CPXPUBLIC (*EGcutCallBack_f) (EGCPXENVptr,
																					void *,
																					int,
																					void *,
																					int *);
extern EGcutCallBack_f EGcplexNullCutCallBack;
/*
int CPXPUBLIC EGcplexNullCutCallBack (EGCPXENVptr,
																			void *cbdata,
																			int wherefrom,
																			void *cbhandle,
																			int *usercation__p);
*/
#endif
#endif
