/* EGlib "Efficient General Library" provides some basic structures and
 * algorithms commons in many optimization algorithms.
 *
 * Copyright (C) 2005 Daniel Espinoza and Marcos Goycoolea.
 * 
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but 
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY 
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public 
 * License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA 
 * */
#ifdef HAVE_CPLEX
#include "eg_config.h"
#include "eg_cplex.h"
// if we are debuggin then we try to explain what have happend with cplex, if not, 
// we just don't care, this allows us to eliminate some overhead and also to define 
// SOME cplex operations as macros, that is WAY faster than a function
char EGcplexErrMsg[1024];

/* this function is to set any integer parameter of CPLEX, this affects the
 * nefironment */
int EGcplexSetIntParam (CPXENVptr env,
												int param,
												int value)
{
	int status;
	status = CPXsetintparam (env, param, value);
	if (status)
	{
		CPXgeterrorstring (env, status, EGcplexErrMsg);
		fprintf (stderr, "en EGcplexSetIntParam, in file %s, line %d\n", __FILE__,
						 __LINE__);
		fprintf (stderr, "%s\n", EGcplexErrMsg);
	}
	return status;
}

// create a new CPLEX environment
CPXENVptr EGnewCplexEnv (void)
{

	// local variable
	int status;
	CPXENVptr res;

	// opening it
	res = CPXopenCPLEX (&status);
	EXIT (status, "in EGnewCplexEnv, can't open environment");

	// some default configuration
	EGcplexSetIntParam (res, CPX_PARAM_SIMDISPLAY, 1);
	EGcplexSetIntParam (res, CPX_PARAM_MIPINTERVAL, 1);
	EGcplexSetIntParam (res, CPX_PARAM_MIPDISPLAY, 4);

	// ending
	return res;
}

// return a pointer to an initialized EGcplex object
EGcplex_t *EGnewCplex (CPXENVptr env,
											 EGmemPool_t * mempool)
{
	// local variables
	EGcplex_t *res;
	char name[8] = "EGcplex";

	// look for memory
	res = EGmemPoolSMalloc (mempool, EGcplex_t, 1);

	// we create an environment only if we didn't recive one
	if (env)
	{
		res->env = (CPXENVptr) env;
		res->internal = 0;
	}
	else
	{
		res->env = EGnewCplexEnv ();
		res->internal = 1;
	}

	// now we create the problem
	res->mempool = mempool;
	res->lp = CPXcreateprob (res->env, &(res->status), name);
	CPLEXTEST (res, "in EGnewCplex, can't create problem");

	// ending
	return res;
}

// liberate the memory returned by EGnewCplex, __NEVER__ liberate such a memory space with free
void EGfreeCplex (void *Prob)
{

	// first free the LP pointer data
	CPXENVptr lenv = ((void *) ((EGcplex_t *) Prob)->env);
	((EGcplex_t *) Prob)->status = CPXfreeprob (lenv, &((EGcplex_t *) Prob)->lp);
	CPLEXTEST (((EGcplex_t *) (Prob)), "in EGfreeCplex");

	// if the environment is internal we liberate it
	if (((EGcplex_t *) Prob)->internal)
		((EGcplex_t *) Prob)->status = CPXcloseCPLEX (&lenv);
	EXIT (((EGcplex_t *) Prob)->status,
				"in EGfreeCplex, Can't close CPLEX environment, exiting");

	// ending
	EGmemPoolFree (Prob, sizeof (EGcplex_t), ((EGcplex_t *) Prob)->mempool);
	return;
}

// liberate the memory returned by EGnewCplexEnv, __NEVER__ liberate such a memory space with free
void EGfreeCplexEnv (void *env)
{

	// free the environment
	CPXENVptr lenv = (CPXENVptr) env;
	int status = CPXcloseCPLEX (&lenv);
#if DEBUG > 0
	if (status)
	{
		CPXgeterrorstring (lenv, status, EGcplexErrMsg);
		fprintf (stderr, "%s, in file %s, line %d\n", __func__, __FILE__,
						 __LINE__);
		fprintf (stderr, "%s\n", EGcplexErrMsg);
	}
#endif
	WARNING (status,
					 "in EGfreeCplexEnv, Can't close CPLEX environment, exiting");

	// ending
	return;
}

// set the CPLEX environment output on, all problems sharing this environment
// will report to screen
int EGcplexSetScreenOn (EGcplex_t * Prob)
{
	Prob->status =
		CPXsetintparam ((CPXENVptr) (Prob->env), CPX_PARAM_SCRIND, CPX_ON);
	CPLEXTEST (Prob, "in EGcplexSetScreenOn");
	return Prob->status;
}

// indicates the number of cols (or variables) in the problem
int EGcplexGetNumCols (EGcplex_t * Prob,
											 int *ncolptr)
{
	*ncolptr = CPXgetnumcols (Prob->env, Prob->lp);
	return 0;
}

// Change problem type to MIP.
int EGcplexSetToMip (EGcplex_t * Prob)
{
	CPXgetprobtype (Prob->env, Prob->lp);
	TEST ((Prob->status == -1), "in EGcplexSetToMip, no problem or enviroment");
	if (Prob->status != 1)
		Prob->status = CPXchgprobtype (Prob->env, Prob->lp, EGCPXPROB_MIP);
	else
		Prob->status = 0;
	CPLEXTEST (Prob, "in EGcplexSetToMip");
	return Prob->status;
}

// change the type of the i-th variable to 'type', type must
//  be 'B' for binary, 'I' for integer or 'C' for continue
int EGcplexSetCtype (EGcplex_t * Prob,
										 int nz,
										 int *ind,
										 char *types)
{
	EGcplexSetToMip (Prob);
	TEST ((Prob->status), "in EGcplexSetCtype");
	Prob->status = CPXchgctype (Prob->env, Prob->lp, nz, ind, types);
	CPLEXTEST (Prob, "in EGcplexSetCtype");
	return Prob->status;
}

// set all variables to type binary
int EGcplexSetAllBin (EGcplex_t * Prob)
{
	// local variables
	int ncol,
	  i;
	int *ind;
	char *type;

	// basic set-up
	EGcplexGetNumCols (Prob, &ncol);
	CHECKRVAL (Prob->status);
	if (!ncol)
		return 0;
	ind = EGmemPoolSMalloc (Prob->mempool, int, ncol);
	type = EGmemPoolSMalloc (Prob->mempool, char,
													 ncol);

	// set the information of ind and type
	for (i = ncol; i;)
	{
		i--;
		ind[i] = i;
		type[i] = 'B';
	}

	// call the CPLEX function
	EGcplexSetCtype (Prob, ncol, ind, type);
	TEST (Prob->status, "in EGcplexSetAllBin");

	// ending
	EGmemPoolFree (ind, sizeof (int) * ncol, Prob->mempool);
	EGmemPoolFree (type, sizeof (char) * ncol, Prob->mempool);
	return Prob->status;
}

// set all variables to type integer
int EGcplexSetAllInt (EGcplex_t * Prob)
{
	// local variables
	int ncol,
	  i;
	int *ind;
	char *type;

	// basic set-up
	EGcplexGetNumCols (Prob, &ncol);
	TEST (Prob->status, "in EGcplexSetAllInt");
	ind = EGmemPoolSMalloc (Prob->mempool, int,
													ncol);
	type = EGmemPoolSMalloc (Prob->mempool, char,
													 ncol);

	// set the information of ind and type
	for (i = ncol; i;)
	{
		i--;
		ind[i] = i;
		type[i] = 'I';
	}

	// call the CPLEX function
	EGcplexSetCtype (Prob, ncol, ind, type);
	TEST (Prob->status, "in EGcplexSetAllInt");

	// ending
	EGmemPoolFree (ind, sizeof (int) * ncol, Prob->mempool);
	EGmemPoolFree (type, sizeof (char) * ncol, Prob->mempool);
	return Prob->status;
}

// set all variables to type continue
int EGcplexSetAllCon (EGcplex_t * Prob)
{
	// local variables
	int ncol,
	  i;
	int *ind;
	char *type;

	// basic set-up
	EGcplexGetNumCols (Prob, &ncol);
	TEST (Prob->status, "in EGcplexSetAllCon");
	ind = EGmemPoolSMalloc (Prob->mempool, int,
													ncol);
	type = EGmemPoolSMalloc (Prob->mempool, char,
													 ncol);

	// set the information of ind and type
	for (i = ncol; i;)
	{
		i--;
		ind[i] = i;
		type[i] = 'C';
	}

	// call the CPLEX function
	EGcplexSetCtype (Prob, ncol, ind, type);
	TEST (Prob->status, "in EGcplexSetAllCon");

	// ending
	EGmemPoolFree (ind, sizeof (int) * ncol, Prob->mempool);
	EGmemPoolFree (type, sizeof (char) * ncol, Prob->mempool);
	return Prob->status;
}

// directyly add rows to cplex
int EGcplexAddCut (EGcplex_t * Prob,
									 void *cbdata,
									 int wherefrom,
									 int rnz,
									 double rhs,
									 char sense,
									 int *rmind,
									 double *rmval)
{
	Prob->status =
		CPXcutcallbackadd (Prob->env, cbdata, wherefrom, rnz, rhs, sense, rmind,
											 rmval);
	CPLEXTEST (Prob, "in EGcplexAddCut");
	return Prob->status;
}

// add a callback cut to the LP
int EGcplexAddCallbackCut (EGcplex_t * Prob,
													 void *cbdata,
													 int wherefrom,
													 EGrow_t * row)
{

	/* local variables */
	unsigned int i,
	  rnz = row->list->size;
	EGlistNode_t *It1;
	int *rmind;
	double *rmval;

	/* looking for memory */
	rmind = EGmemPoolSMalloc (Prob->mempool, int,
														rnz);
	rmval = EGmemPoolSMalloc (Prob->mempool, double,
														rnz);

	/* creating rmatind and rmatval */
	for (i = 0, It1 = row->list->begin; It1; It1 = It1->next, i++)
	{
		rmind[i] = ((EGcplexData_t *) (It1->this))->LPind;
		rmval[i] = ((EGcplexData_t *) (It1->this))->coeff;
	}

	/* Adding the cut to CPLEX */
	EGcplexAddCut (Prob, cbdata, wherefrom, (signed) rnz, row->rhs, row->sense,
								 rmind, rmval);
	CPLEXTEST (Prob, "In EGcplexAddCallbackCut");

	EGmemPoolFree (rmind, rnz * sizeof (int), Prob->mempool);
	EGmemPoolFree (rmval, rnz * sizeof (double), Prob->mempool);

	return 0;
}

// indicates the number of rows (or restrictions) in the problem
int EGcplexGetNumRows (EGcplex_t * Prob,
											 int *nrowptr)
{
	*nrowptr = CPXgetnumrows (Prob->env, Prob->lp);
	return 0;
}

/* ========================================================================= */
/* directyly add rows to cplex */
int EGcplexAddRow (EGcplex_t * Prob,
									 int rcnt,
									 int rnz,
									 double *rhs,
									 char *sense,
									 int *rmbeg,
									 int *rmind,
									 double *rmval,
									 char **rname)
{

#if __EG_CPLEX_EXTRADEBUG__
	/* local variables */
	register int row,
	  ind;
	int ncol,
	  nrow;

	/* basic set-up */
	MESSAGE (3, "Performing Sanity tests");
	EGcplexGetNumCols (Prob, &ncol);
	EGcplexGetNumRows (Prob, &nrow);

	/* test non null arrays */
	TEST (!rhs || !rmbeg || (rnz && (!rmind || !rmval)) || !sense || !rname,
				"Unacceptable NULL parameter");
	TEST (rcnt < 0, "Negative Value for rcnt");
	TEST (rnz < 0, "Negative Value for rnz");

	/* loop through every new column added and check if the column is OK */
	for (row = 0; row < rcnt; row++)
	{
		TEST (!(rname[row]), "row[%d]=Null, please select a name", row);
		TEST (strlen (rname[row]) > 256, "row[%d] =%s has a too long name, please "
					"select a better name", row, rname[row]);
		TEST ((sense[row] != 'L') && (sense[row] != 'E') && (sense[row] != 'G'),
					"sense[%d]=%c is not well setted", row, sense[row]);
		TEST (!finite (rhs[row]), "rhs[%d]=%lf is not a number for "
					"row[%d]=%s", row, rhs[row], row, rname[row]);
		TEST ((row + 1 < rcnt) && (rmbeg[row] > rmbeg[row + 1]), "rmbeg is not in "
					"ascending order");
		TEST (rmbeg[row] < 0, "rmbeg[%d] is negative", row);
		for (ind = rmbeg[row]; ind < rnz; ind++)
		{
			if ((row + 1 < rcnt) && (rmbeg[row + 1] <= ind))
				break;
			TEST ((rmind[ind] < 0)
						|| (rmind[ind] >= nrow), "rmind[%d]=%d>%d=ncol for " "row[%d]=%s",
						ind, rmind[ind], ncol, row, rname[row]);
			TEST (!finite (rmval[ind]),
						"rmval[%d]=%lf is not a number for " "row[%d]=%s", ind, rmval[ind],
						row, rname[row]);
		}
	}
#endif

	Prob->status = CPXaddrows (Prob->env, Prob->lp, 0, rcnt, rnz, rhs, sense,
														 rmbeg, rmind, rmval, 0, rname);
	CPLEXTEST (Prob, "in EGcplexAddRow");
	return Prob->status;
}

// add a list of rows to the LP
int EGcplexAddRowList (EGcplex_t * Prob,
											 EGlist_t * rows)
{

	// local variables
	EGlistNode_t *It1;
	EGlistNode_t *It2;
	int rnz,
	  i;
	double *rhs;
	char *sense;
	int *rmbeg;
	int *rmind;
	double *rmval;
	char **rname;
	int nrows;
	char temp_name[24];

	// if there is nothing to do we exit
	if (!rows->size)
		return 0;

	// basic set-up
	EGcplexGetNumRows (Prob, &nrows);

	// computing rnz
	rnz = 0;
	for (It1 = rows->begin; It1; It1 = It1->next)
		rnz += ((EGrow_t *) (It1->this))->list->size;

	// looking for memory
	rhs = EGmemPoolSMalloc (Prob->mempool, double,
													rows->size);
	sense = EGmemPoolSMalloc (Prob->mempool, char,
														rows->size);
	rmbeg = EGmemPoolSMalloc (Prob->mempool, int,
														rows->size);
	rname = EGmemPoolSMalloc (Prob->mempool, char *,
														rows->size);
	if (rnz)
	{
		rmind = EGmemPoolSMalloc (Prob->mempool, int,
															rnz);
		rmval = EGmemPoolSMalloc (Prob->mempool, double,
															rnz);
	}
	else
	{
		rmind = 0;
		rmval = 0;
	}

	// now copy information in the rows list in to the CPLEX arrays
	rnz = 0;
	i = 0;
	for (It1 = rows->begin; It1; It1 = It1->next)
	{
		rmbeg[i] = rnz;
		rhs[i] = ((EGrow_t *) (It1->this))->rhs;
		sense[i] = ((EGrow_t *) (It1->this))->sense;
		if (!(((EGrow_t *) (It1->this))->name))
		{
			EGmvar (temp_name, 1, "R", nrows + i + 1);
			((EGrow_t *) (It1->this))->name = EGmemPoolSMalloc (Prob->mempool, char,
																													strlen (temp_name) +
																													1);
			strcpy (((EGrow_t *) (It1->this))->name, temp_name);
		}
		rname[i] = ((EGrow_t *) (It1->this))->name;
		for (It2 = ((EGrow_t *) (It1->this))->list->begin; It2; It2 = It2->next)
		{
			rmind[rnz] = ((EGcplexData_t *) (It2->this))->LPind;
			rmval[rnz] = ((EGcplexData_t *) (It2->this))->coeff;
			rnz++;
		}
		i++;
	}

	// now we actually add the rows to the CPLEX problem
	Prob->status =
		EGcplexAddRow (Prob, (signed) (rows->size), rnz, rhs, sense, rmbeg, rmind,
									 rmval, rname);
	CPLEXTEST (Prob, "in EGcplexAddRowList");

	// ending
	EGmemPoolFree (rhs, rows->size * sizeof (double), Prob->mempool);
	EGmemPoolFree (sense, rows->size * sizeof (char), Prob->mempool);
	EGmemPoolFree (rmbeg, rows->size * sizeof (int), Prob->mempool);
	if (rnz)
	{
		EGmemPoolFree (rmind, rnz * sizeof (char), Prob->mempool);
		EGmemPoolFree (rmval, rnz * sizeof (char), Prob->mempool);
	}
	EGmemPoolFree (rname, rows->size * sizeof (char *), Prob->mempool);
	return Prob->status;
}

// add an array of rows to the problem
int EGcplexAddRowArray (EGcplex_t * Prob,
												EGrow_t ** Rows,
												unsigned int rsz)
{

	// local variables
	unsigned int It1;
	EGlistNode_t *It2;
	int rnz;
	double *rhs;
	char *sense;
	int *rmbeg;
	int *rmind;
	double *rmval;
	char **rname;
	int nrows;
	char temp_name[24];

	// if there is nothing to do, we return
	if (!rsz)
		return 0;

	// basic set-up
	EGcplexGetNumRows (Prob, &nrows);

	// computing rnz
	rnz = 0;
	for (It1 = 0; It1 < rsz; It1++)
		rnz += Rows[It1]->list->size;

	// looking for memory
	rhs = EGmemPoolSMalloc (Prob->mempool, double,
													rsz);
	sense = EGmemPoolSMalloc (Prob->mempool, char,
														rsz);
	rmbeg = EGmemPoolSMalloc (Prob->mempool, int,
														rsz);
	rname = EGmemPoolSMalloc (Prob->mempool, char *,
														rsz);
	if (rnz)
	{
		rmind = EGmemPoolSMalloc (Prob->mempool, int,
															rnz);
		rmval = EGmemPoolSMalloc (Prob->mempool, double,
															rnz);
	}
	else
	{
		rmind = 0;
		rmval = 0;
	}

	// now copy information in the rows list in to the CPLEX arrays
	rnz = 0;
	for (It1 = 0; It1 < rsz; It1++)
	{
		rmbeg[It1] = rnz;
		rhs[It1] = Rows[It1]->rhs;
		sense[It1] = Rows[It1]->sense;
		if (!(Rows[It1]->name))
		{
			EGmvar (temp_name, 1, "R", nrows + 1 + It1);
			Rows[It1]->name = EGmemPoolSMalloc (Prob->mempool, char,
																					strlen (temp_name) + 1);
			strcpy (Rows[It1]->name, temp_name);
		}
		rname[It1] = Rows[It1]->name;
		for (It2 = Rows[It1]->list->begin; It2; It2 = It2->next)
		{
			rmind[rnz] = ((EGcplexData_t *) (It2->this))->LPind;
			rmval[rnz] = ((EGcplexData_t *) (It2->this))->coeff;
			rnz++;
		}
	}

	// now we actually add the rows to the CPLEX problem
	Prob->status =
		EGcplexAddRow (Prob, (signed) rsz, rnz, rhs, sense, rmbeg, rmind, rmval,
									 rname);
	CPLEXTEST (Prob, "in EGcplexAddRowArray");

	// ending
	EGmemPoolFree (rhs, rsz * sizeof (double), Prob->mempool);
	EGmemPoolFree (sense, rsz * sizeof (char), Prob->mempool);
	EGmemPoolFree (rmbeg, rsz * sizeof (int), Prob->mempool);
	if (rnz)
	{
		EGmemPoolFree (rmind, rnz * sizeof (char), Prob->mempool);
		EGmemPoolFree (rmval, rnz * sizeof (char), Prob->mempool);
	}
	EGmemPoolFree (rname, rsz * sizeof (char *), Prob->mempool);
	return Prob->status;
}

/* ========================================================================= */
/* directly add columns to cplex */
int EGcplexAddCol (EGcplex_t * Prob,
									 int ccnt,
									 int cnz,
									 double *cobj,
									 int *cmbeg,
									 int *cmind,
									 double *cmval,
									 double *clb,
									 double *cub,
									 char **cname)
{

#if __EG_CPLEX_EXTRADEBUG__
	/* local variables */
	register int col,
	  ind;
	int ncol,
	  nrow;
	unsigned int c1 = 0,
	  c2 = 0;

	/* basic set-up */
	MESSAGE (3, "Performing Sanity tests");
	EGcplexGetNumCols (Prob, &ncol);
	EGcplexGetNumRows (Prob, &nrow);

	/* test non null arrays */
	TEST (!cobj || !cmbeg || (cnz && (!cmind || !cmval)) || !clb || !cub
				|| !cname, "Unacceptable NULL parameter");
	TEST (ccnt < 0, "Negative Value for ccnt");
	TEST (cnz < 0, "Negative Value for cnz");

	/* loop through every new column added and check if the column is OK */
	for (col = 0; col < ccnt; col++)
	{
		c1++;
		TEST (!(cname[col]), "col[%d]=Null, please select a name", col);
		TEST (strlen (cname[col]) > 256, "col[%d] =%s has a too long name, please "
					"select a better name", col, cname[col]);
		TEST (clb[col] > cub[col], "ub[%d]=%lf<lb[%d]=%lf col %s", col, cub[col],
					col, clb[col], cname[col]);
		TEST (!finite (cobj[col]),
					"cobj[%d]=%lf is not a number for " "col[%d]=%s", col, cobj[col],
					col, cname[col]);
		TEST ((col + 1 < ccnt)
					&& (cmbeg[col] > cmbeg[col + 1]),
					"cmbeg is not in " "ascending order");
		TEST (cmbeg[col] < 0, "cmbeg[%d] is negative", col);
		for (ind = cmbeg[col]; ind < cnz; ind++)
		{
			if ((col + 1 < ccnt) && (cmbeg[col + 1] <= ind))
				break;
			c2++;
			TEST (!finite (cmval[ind]), "cmval[%d]=%lf is not a number for "
						"col[%d]=%s", ind, cmval[ind], col, cname[col]);
			TEST ((cmind[ind] < 0)
						|| (cmind[ind] >= nrow),
						"cmind[%d]=%d > nrow=%d, " "cmval[%d]=%lf for col[%d]=%s", ind,
						cmind[ind], nrow, ind, cmval[ind], col, cname[col]);
		}
	}
	MESSAGE (3, "tested %u columns and %u column coefficients cnz=%d", c1, c2,
					 cnz);
#endif

	Prob->status = CPXaddcols (Prob->env, Prob->lp, ccnt, cnz, cobj, cmbeg,
														 cmind, cmval, clb, cub, cname);
	CPLEXTEST (Prob, "in EGcplexAddCol");
	return Prob->status;
}


// add a list of columns to the problem
int EGcplexAddColList (EGcplex_t * Prob,
											 EGlist_t * Cols)
{

	// local variables
	EGlistNode_t *It1,
	 *It2;
	int cnz,
	  i;
	double *cobj;
	int *cmbeg;
	int *cmind;
	double *cmval;
	double *cub;
	double *clb;
	char **cname;
	int ncols;
	char temp_name[24];

	/* ED: For datacheck purposes */
	/*
	 * int num_rows = 0, status = 0;
	 * CPXLPptr datacheck_lp;
	 * double *rhs = NULL;
	 * char *sense = NULL;
	 * int *matcnt = NULL;
	 * int j;
	 */

	/* Basic questios */
	if (!Cols->size)
		return 0;

	// basic set-up
	EGcplexGetNumCols (Prob, &ncols);

	// now we compute cnz
	cnz = 0;
	for (It1 = Cols->begin; It1; It1 = It1->next)
		cnz += ((EGcol_t *) (It1->this))->list->size;

	// now we look for memory
	cobj = EGmemPoolSMalloc (Prob->mempool, double,
													 Cols->size);
	cmbeg = EGmemPoolSMalloc (Prob->mempool, int,
														Cols->size);
	cub = EGmemPoolSMalloc (Prob->mempool, double,
													Cols->size);
	clb = EGmemPoolSMalloc (Prob->mempool, double,
													Cols->size);
	cname = EGmemPoolSMalloc (Prob->mempool, char *,
														Cols->size);
	if (cnz)
	{
		cmind = EGmemPoolSMalloc (Prob->mempool, int,
															cnz);
		cmval = EGmemPoolSMalloc (Prob->mempool, double,
															cnz);
	}
	else
	{
		cmind = 0;
		cmval = 0;
	}

	// now we store in those arrays the information
	cnz = i = 0;
	for (It1 = Cols->begin; It1; It1 = It1->next)
	{
		cobj[i] = ((EGcol_t *) (It1->this))->obj;
		cmbeg[i] = cnz;
		cub[i] = ((EGcol_t *) (It1->this))->ub;
		clb[i] = ((EGcol_t *) (It1->this))->lb;
		if (!(((EGcol_t *) (It1->this))->name))
		{
			EGmvar (temp_name, 1, "C", ncols + 1 + i);
			((EGcol_t *) (It1->this))->name = EGmemPoolSMalloc (Prob->mempool, char,
																													strlen (temp_name) +
																													1);
			strcpy (((EGcol_t *) (It1->this))->name, temp_name);
		}
		cname[i] = ((EGcol_t *) (It1->this))->name;
		for (It2 = ((EGcol_t *) (It1->this))->list->begin; It2; It2 = It2->next)
		{
			cmind[cnz] = ((EGcplexData_t *) (It2->this))->LPind;
			cmval[cnz] = ((EGcplexData_t *) (It2->this))->coeff;
			cnz++;
		}
		i++;
	}

	// and finally we add this columns to CPLEX
	Prob->status =
		EGcplexAddCol (Prob, (signed) (Cols->size), cnz, cobj, cmbeg, cmind, cmval,
									 clb, cub, cname);
	TEST (Prob->status, "in EGcplexAddColList");


	/* ED: Check if the data is sound */
	/*
	 * fprintf(stderr, "Begin datacheck....\n");
	 * datacheck_lp = CPXcreateprob (Prob->env, &status, "datacheck");
	 * 
	 * status = EGcplexSetIntParam (Prob->env, CPX_PARAM_DATACHECK, 1);
	 * 
	 * EGcplexGetNumRows (Prob, &num_rows);
	 * 
	 * rhs = EGmemPoolSMalloc(Prob->mempool, double, num_rows);
	 * sense = EGmemPoolSMalloc(Prob->mempool, char, num_rows);
	 * 
	 * status = CPXgetsense (Prob->env, Prob->lp, sense, 0, num_rows-1);
	 * status = CPXgetrhs (Prob->env, Prob->lp, rhs, 0, num_rows-1);
	 * 
	 * matcnt = EGmemPoolSMalloc(Prob->mempool, int, Cols->size);
	 * for (i = 0; i < Cols->size - 1; i++)
	 * matcnt[i] = cmbeg[i+1] - cmbeg[i];
	 * matcnt[Cols->size - 1] = cnz - cmbeg[Cols->size - 1];
	 * 
	 * status = CPXcopylp (Prob->env, datacheck_lp,
	 * Cols->size, num_rows, -1, cobj, rhs, sense, 
	 * cmbeg, matcnt, cmind, cmval, clb, cub,
	 * NULL);
	 * 
	 * 
	 * fprintf(stderr, "Status of copylp is %d\n", status);
	 * 
	 * EGmemPoolFree(rhs, num_rows * sizeof(double), Prob->mempool);
	 * EGmemPoolFree(sense, num_rows * sizeof(char), Prob->mempool);
	 * EGmemPoolFree(matcnt, Cols->size * sizeof(int), Prob->mempool);
	 * 
	 * status = CPXfreeprob (Prob->env, &datacheck_lp);
	 * fprintf(stderr, "End datacheck....\n");
	 */
	/* End datacheck */


	// ending
	EGmemPoolFree (cobj, Cols->size * sizeof (double), Prob->mempool);
	EGmemPoolFree (cmbeg, Cols->size * sizeof (int), Prob->mempool);
	EGmemPoolFree (cub, Cols->size * sizeof (double), Prob->mempool);
	EGmemPoolFree (clb, Cols->size * sizeof (double), Prob->mempool);
	EGmemPoolFree (cname, Cols->size * sizeof (char *), Prob->mempool);
	if (cnz)
	{
		EGmemPoolFree (cmind, cnz * sizeof (int), Prob->mempool);
		EGmemPoolFree (cmval, cnz * sizeof (double), Prob->mempool);
	}
	return Prob->status;
}

// add an array of rows to the problem
int EGcplexAddColArray (EGcplex_t * Prob,
												EGcol_t ** Cols,
												int ccnt)
{

	// local variables
	unsigned int It1;
	EGlistNode_t *It2;
	int cnz;
	double *cobj;
	int *cmbeg;
	int *cmind;
	double *cmval;
	double *cub;
	double *clb;
	char **cname;
	int ncols;
	char temp_name[24];

	/* basic questions */
	if (!ccnt)
		return 0;

	// basic initialization
	EGcplexGetNumCols (Prob, &ncols);

	// now we compute cnz
	cnz = 0;
	for (It1 = 0; It1 < (unsigned) ccnt; It1++)
		cnz += Cols[It1]->list->size;

	// now we look for memory
	cobj = EGmemPoolSMalloc (Prob->mempool, double,
													 ccnt);
	cmbeg = EGmemPoolSMalloc (Prob->mempool, int,
														ccnt);
	cub = EGmemPoolSMalloc (Prob->mempool, double,
													ccnt);
	clb = EGmemPoolSMalloc (Prob->mempool, double,
													ccnt);
	cname = EGmemPoolSMalloc (Prob->mempool, char *,
														ccnt);
	if (cnz)
	{
		cmind = EGmemPoolSMalloc (Prob->mempool, int,
															cnz);
		cmval = EGmemPoolSMalloc (Prob->mempool, double,
															cnz);
	}
	else
	{
		cmind = 0;
		cmval = 0;
	}

	// now we store in those arraays the information
	cnz = 0;
	for (It1 = 0; It1 < (unsigned) ccnt; It1++)
	{
		cobj[It1] = Cols[It1]->obj;
		cmbeg[It1] = cnz;
		cub[It1] = Cols[It1]->ub;
		clb[It1] = Cols[It1]->lb;
		if (!(Cols[It1]->name))
		{
			EGmvar (temp_name, 1, "C", ncols + It1 + 1);
			Cols[It1]->name = EGmemPoolSMalloc (Prob->mempool, char,
																					strlen (temp_name) + 1);
			strcpy (Cols[It1]->name, temp_name);
		}
		cname[It1] = Cols[It1]->name;
		for (It2 = Cols[It1]->list->begin; It2; It2 = It2->next)
		{
			cmind[cnz] = ((EGcplexData_t *) (It2->this))->LPind;
			cmval[cnz] = ((EGcplexData_t *) (It2->this))->coeff;
			cnz++;
		}
	}

	// and finally we add this columns to CPLEX
	Prob->status =
		EGcplexAddCol (Prob, ccnt, cnz, cobj, cmbeg, cmind, cmval, clb, cub,
									 cname);
	TEST (Prob->status, "in EGcplexAddColArray");

	// ending
	EGmemPoolFree (cobj, ccnt * sizeof (double), Prob->mempool);
	EGmemPoolFree (cmbeg, ccnt * sizeof (int), Prob->mempool);
	EGmemPoolFree (cub, ccnt * sizeof (double), Prob->mempool);
	EGmemPoolFree (clb, ccnt * sizeof (double), Prob->mempool);
	EGmemPoolFree (cname, ccnt * sizeof (char *), Prob->mempool);
	if (cnz)
	{
		EGmemPoolFree (cmind, cnz * sizeof (int), Prob->mempool);
		EGmemPoolFree (cmval, cnz * sizeof (double), Prob->mempool);
	}
	return Prob->status;
}

// Change problem type to LP.
int EGcplexSetToLp (EGcplex_t * Prob)
{
	CPXgetprobtype (Prob->env, Prob->lp);
	TEST ((Prob->status == -1), "in EGcplexSetToLp, no problem or enviroment");
	if (Prob->status)
		Prob->status = CPXchgprobtype (Prob->env, Prob->lp, CPXPROB_LP);
	CPLEXTEST (Prob, "in EGcplexSetToLp");
	return Prob->status;
}

// Primal Simplex , store the optimization status in Prob->status
// status 0 means a solution is found (not necesarily
// optimal) (see CPLEX manual for other values).
int EGcplexPrimopt (EGcplex_t * Prob)
{
	EGcplexSetToLp (Prob);
	TEST ((Prob->status), "in EGcplexPrimopt");
	Prob->status = CPXprimopt (Prob->env, Prob->lp);
	CPLEXTEST (Prob, "in EGcplexPrimopt");
	return Prob->status;
}

// hybrid Network Simplex with metod B;
// B = 'p' (primal) or 'd' (dual), return the optimization
// status in Prob->status, 0 means a solution is found (not necesarily optimal)
// see CPLEX manual for other values).
int EGcplexNetopt (EGcplex_t * Prob,
									 const char B)
{
	EGcplexSetToLp (Prob);
	TEST ((Prob->status), "in EGcplexNetopt");
	Prob->status = CPXhybnetopt (Prob->env, Prob->lp, B);
	CPLEXTEST (Prob, "in EGcplexNetopt");
	return Prob->status;
}

// Dual Simplex, return the optimization
// status in Prob->status, 0 means a solution is found (not necesarily
// optimzal) (see CPLEX manual for other values).
int EGcplexDualopt (EGcplex_t * Prob)
{
	EGcplexSetToLp (Prob);
	TEST ((Prob->status), "in EGcplexDualopt");
	Prob->status = CPXdualopt (Prob->env, Prob->lp);
	CPLEXTEST (Prob, "in EGcplexDualopt");
	return Prob->status;
}

// Mixed Integer Methods , return the optimization status
// 0 means a solution is found (not necesarily optimzal)
// (see CPLEX manual for other values). where <B>maxIntSol</B> is
// the maximun number of integer solution to be found before
// stoping. There are other issues as precision, for that you have to 
// configure the environment, see the environment information
int EGcplexMipopt (EGcplex_t * Prob)
{
	EGcplexSetToMip (Prob);
	TEST ((Prob->status), "in EGcplexMipopt");
	Prob->status = CPXmipopt (Prob->env, Prob->lp);
	CPLEXTEST (Prob, "in EGcplexMipopt");
	return Prob->status;
}

// set the problem for minimize (default)
int EGcplexSetToMin (EGcplex_t * Prob)
{
	CPXchgobjsen (Prob->env, Prob->lp, CPX_MIN);
	return 0;
}

// set the problem for maximize
int EGcplexSetToMax (EGcplex_t * Prob)
{
	CPXchgobjsen (Prob->env, Prob->lp, CPX_MAX);
	return 0;
}

// set the right hand side of the rows in ind to val, nz
// indicate the number of entries in the arrays name and val.
// Prob is the EGcplex_t structure
int EGcplexSetRhs (EGcplex_t * Prob,
									 int nz,
									 int *ind,
									 double *val)
{
	Prob->status = CPXchgrhs (Prob->env, Prob->lp, nz, ind, val);
	CPLEXTEST (Prob, "in EGcplexSetRhs");
	return Prob->status;
}

/// set the objetive coefficient of the pos-th variable to val
// Prob is the EGcplex_t structure, nz is the length of the arrays
// ind is the array of indices, type is the array of types of bounds to 
// change and val is the new value of the bound
int EGcplexSetObj (EGcplex_t * Prob,
									 int nz,
									 int *ind,
									 double *val)
{
	Prob->status = CPXchgobj (Prob->env, Prob->lp, nz, ind, val);
	CPLEXTEST (Prob, "in EGcplexSetObj");
	return Prob->status;
}

// set for all the variables in 'bound' the limit 'type' to
// the double stored in the map; type must be one of the
// folowin values : 'U' for the upper limit, 'L' for the
// lower limit and 'B' for both the lower and the upper
// limit.
// Prob is the EGcplex_t structure, nz is the length of the arrays
// ind is the array of indices, type is the array of types of bounds to 
// change and val is the new value of the bound
int EGcplexSetBound (EGcplex_t * Prob,
										 int nz,
										 int *ind,
										 char *type,
										 double *val)
{
	Prob->status = CPXchgbds (Prob->env, Prob->lp, nz, ind, type, val);
	CPLEXTEST (Prob, "in EGcplexSetBound");
	return Prob->status;
}

// deletes rows with index bigger or equal than 'beg' but smaller than 'end'
int EGcplexDelRows (EGcplex_t * Prob,
										int beg,
										int end)
{
	Prob->status = CPXdelrows (Prob->env, Prob->lp, beg, end - 1);
	CPLEXTEST (Prob, "in EGcplexDelRows");
	return Prob->status;
}

// deletes cols with index bigger or equal than 'beg' but smaller than 'end' 
int EGcplexDelCols (EGcplex_t * Prob,
										int beg,
										int end)
{
	Prob->status = CPXdelcols (Prob->env, Prob->lp, beg, end - 1);
	CPLEXTEST (Prob, "in EGcplexDelCols");
	return Prob->status;
}

// return the CPLEX status of the problem, that status
// indicates optimality, infeasibility, etc., see (CPLEX manual
// for further information).
int EGcplexGetStat (EGcplex_t * Prob,
										int *statusptr)
{
	*statusptr = CPXgetstat (Prob->env, Prob->lp);
	return 0;
}

/* ========================================================================= */
/* return the current objetive value for the problem. */
/* ========================================================================= */
int EGcplexGetBestObjval (EGcplex_t * Prob,
													double *valptr)
{
	Prob->status = CPXgetprobtype (Prob->env, Prob->lp);
	if (Prob->status == 0)
		return 1;
	else if (Prob->status == 1)
		Prob->status = CPXgetbestobjval (Prob->env, Prob->lp, valptr);
	else
		TEST (1, "in EGcplexGetBestObjval, unknown problem type");
	CPLEXTEST (Prob, "in EGcplexGetBestObjval");
	return Prob->status;
}


// return the current objetive value for the problem.
int EGcplexGetObjval (EGcplex_t * Prob,
											double *valptr)
{
	Prob->status = CPXgetprobtype (Prob->env, Prob->lp);
	if (Prob->status == 0)
		Prob->status = CPXgetobjval (Prob->env, Prob->lp, valptr);
	else if (Prob->status == 1)
		Prob->status = CPXgetmipobjval (Prob->env, Prob->lp, valptr);
	else
		EXIT (1, "in EGcplexGetObjval, unknown problem type");
	CPLEXTEST (Prob, "in EGcplexGetObjval");
	return Prob->status;
}

// given an array x of size end - begin and a problem, return the current values for the
// variables in the range [begin,end[, i.e. we obtain x[begin],....,x[end-1]
int EGcplexGetX (EGcplex_t * Prob,
								 double *x,
								 int begin,
								 int end)
{
	Prob->status = CPXgetprobtype (Prob->env, Prob->lp);
	if (Prob->status == 0)
		Prob->status = CPXgetx (Prob->env, Prob->lp, x, begin, end - 1);
	else if (Prob->status == 1)
		Prob->status = CPXgetmipx (Prob->env, Prob->lp, x, begin, end - 1);
	else
		EXIT (1, "in EGcplexGetX, unknown problem type");
	CPLEXTEST (Prob, "in EGcplexGetX");
	return Prob->status;
}

// given an array x of size end - begin and a problem, return the current dual values for the
// constraint in the range [begin,end[, i.e. we obtain x[begin],....,x[end-1]
int EGcplexGetPi (EGcplex_t * Prob,
									double *x,
									int begin,
									int end)
{
	Prob->status = CPXgetprobtype (Prob->env, Prob->lp);
	if (Prob->status == 0)
		Prob->status = CPXgetpi (Prob->env, Prob->lp, x, begin, end - 1);
	else
		EXIT (1, "in EGcplexGetPi, unknown problem type or problem is MIP");
	CPLEXTEST (Prob, "in EGcplexGetPi");
	return Prob->status;
}

// given an array x of size end - begin and a problem, return the current slack for the
// constraints in the range [begin,end[, i.e. we obtain x[begin],....,x[end-1]
int EGcplexGetSlack (EGcplex_t * Prob,
										 double *x,
										 int begin,
										 int end)
{
	Prob->status = CPXgetprobtype (Prob->env, Prob->lp);
	if (Prob->status == 0)
		Prob->status = CPXgetslack (Prob->env, Prob->lp, x, begin, end - 1);
	else if (Prob->status == 1)
		Prob->status = CPXgetmipslack (Prob->env, Prob->lp, x, begin, end - 1);
	else
		EXIT (1, "in EGcplexGetSlack, unknown problem type");
	CPLEXTEST (Prob, "in EGcplexGetSlack");
	return Prob->status;
}

// given an array x of size end - begin and a problem, return the current reduced cost values for the
// variables in the range [begin,end[, i.e. we obtain x[begin],....,x[end-1]
int EGcplexGetDj (EGcplex_t * Prob,
									double *x,
									int begin,
									int end)
{
	Prob->status = CPXgetprobtype (Prob->env, Prob->lp);
	if (Prob->status == 0)
		Prob->status = CPXgetdj (Prob->env, Prob->lp, x, begin, end - 1);
	else
		EXIT (1, "in EGcplexGetDj, unknown problem type or problem is MIP");
	CPLEXTEST (Prob, "in EGcplexGetDj");
	return Prob->status;
}

/* return in how many constraints a particular column bellongs */
int EGcplexGetColNZ (EGcplex_t * Prob,
										 int cind,
										 int *cnz)
{

	/* local variables */
	int colnz,
	  matbeg;

	Prob->status =
		CPXgetcols (Prob->env, Prob->lp, &colnz, &matbeg, 0, 0, 0, cnz, cind,
								cind);
	if (Prob->status != 1207)
		CPLEXTEST (Prob, "in EGcplexGetColNZ");
	*cnz *= -1;

	/* ending */
	return Prob->status;
}

/* return a range of rows */
int EGcplexGetRows (EGcplex_t * Prob,
										int *nz,
										int *rbeg,
										int *rind,
										double *rval,
										int size,
										int *surplus,
										int begin,
										int end)
{
	Prob->status =
		CPXgetrows (Prob->env, Prob->lp, nz, rbeg, rind, rval, size, surplus,
								begin, end - 1);
	if (Prob->status == CPXERR_NEGATIVE_SURPLUS)
		return 0;
	CPLEXTEST (Prob, "in EGcplexGetRows");
	return Prob->status;
}

int EGcplexGetCols (EGcplex_t * Prob,
										int *nz,
										int *cbeg,
										int *cind,
										double *cval,
										int size,
										int *surplus,
										int begin,
										int end)
{
	Prob->status =
		CPXgetcols (Prob->env, Prob->lp, nz, cbeg, cind, cval, size, surplus,
								begin, end - 1);
	if (Prob->status == CPXERR_NEGATIVE_SURPLUS)
		return 0;
	CPLEXTEST (Prob, "in EGcplexGetCols");
	return Prob->status;
}

// Return the column names
// Prob is a pointer to a EGcplex_t structure
// begin denote the first column name to take
// end denote the last plus one column name to take
// name is an an array of size end - begin of pointers to char*
// name store is an array of chars of size storesize where the names will be stored
// surplusptr is a pointer to an integer where the size of the non used space in store will be stored
// if *surplusptr is negative, the call had fail
int EGcplexGetColname (EGcplex_t * Prob,
											 char **names,
											 char *store,
											 int storesize,
											 int *splusptr,
											 int begin,
											 int end)
{
	Prob->status =
		CPXgetcolname (Prob->env, Prob->lp, names, store, storesize, splusptr,
									 begin, end - 1);
	if (Prob->status != CPXERR_NEGATIVE_SURPLUS)
		CPLEXTEST (Prob, "in EGcplexGetColname");
	return Prob->status;
}

// Return the row names
// Prob is a pointer to a EGcplex_t structure
// begin denote the first row name to take
// end denote the last plus one row name to take
// name is an an array of size end - begin of pointers to char*
// name store is an array of chars of size storesize where the names will be stored
// surplusptr is a pointer to an integer where the size of the non used space in store will be stored
// if *surplusptr is negative, the call had fail
int EGcplexGetRowname (EGcplex_t * Prob,
											 char **names,
											 char *store,
											 int storesize,
											 int *splusptr,
											 int begin,
											 int end)
{
	Prob->status =
		CPXgetrowname (Prob->env, Prob->lp, names, store, storesize, splusptr,
									 begin, end - 1);
	if (Prob->status != CPXERR_NEGATIVE_SURPLUS)
		CPLEXTEST (Prob, "in EGcplexGetRowname");
	return Prob->status;
}

// given an array x of size end - begin and a problem, return the current upper bounds values for the
// variables in the range [begin,end[, i.e. we obtain x[begin],....,x[end-1]
int EGcplexGetUb (EGcplex_t * Prob,
									double *x,
									int begin,
									int end)
{
	Prob->status = CPXgetub (Prob->env, Prob->lp, x, begin, end - 1);
	CPLEXTEST (Prob, "in EGcplexGetUb");
	return Prob->status;
}

// given an array x of size end - begin and a problem, return the current lower bounds values for the
// variables in the range [begin,end[, i.e. we obtain x[begin],....,x[end-1]
int EGcplexGetLb (EGcplex_t * Prob,
									double *x,
									int begin,
									int end)
{
	Prob->status = CPXgetlb (Prob->env, Prob->lp, x, begin, end - 1);
	CPLEXTEST (Prob, "in EGcplexGetLb");
	return Prob->status;
}

// given an array x of size end - begin and a problem, return the objective values for the
// variables in the range [begin,end[, i.e. we obtain x[begin],....,x[end-1]
int EGcplexGetObjCoeff (EGcplex_t * Prob,
												double *x,
												int begin,
												int end)
{
	Prob->status = CPXgetobj (Prob->env, Prob->lp, x, begin, end - 1);
	CPLEXTEST (Prob, "in EGcplexGetObjCoeff");
	return Prob->status;
}

// given an array x of size end - begin and a problem, return the sense values for the
// constraints in the range [begin,end[, i.e. we obtain x[begin],....,x[end-1]
int EGcplexGetSense (EGcplex_t * Prob,
										 char *x,
										 int begin,
										 int end)
{
	Prob->status =
		CPXgetsense ((CPXENVptr) (Prob->env), Prob->lp, x, begin, end - 1);
	CPLEXTEST (Prob, "in EGcplexGetSense");
	return Prob->status;
}

// given an array x of size end - begin and a problem, return the sense values for the
// constraints in the range [begin,end[, i.e. we obtain x[begin],....,x[end-1]
int EGcplexGetRhs (EGcplex_t * Prob,
									 double *x,
									 int begin,
									 int end)
{
	Prob->status = CPXgetrhs (Prob->env, Prob->lp, x, begin, end - 1);
	CPLEXTEST (Prob, "in EGcplexGetRhs");
	return Prob->status;
}

// save the problem structure in format type, where type must
// be "SAV" for binary matrix and basis file, "MPS" for
// MPS format, "LP" for CPLEX LP format, "REW" or "RMP" for
// MPS format with all names changed to generic names
// or "RLP" for LP format with all names changed to generic
// names.
int EGcplexWriteProb (EGcplex_t * Prob,
											const char *file,
											const char *type)
{
#if CPX_VERSION > 800
	Prob->status = CPXwriteprob (Prob->env, Prob->lp, file, type);
#else
	char *lfile = strdup (file);
	char *ltype = strdup (type);
	Prob->status = CPXwriteprob (Prob->env, Prob->lp, lfile, ltype);
	free (lfile);
	free (ltype);
#endif
	CPLEXTEST (Prob, "in EGcplexWriteProb");
	return Prob->status;
}

// add a function callback for Cplex incumber solution during MIP
// note that the callback is asociated to the CPLEX environment and not to the 
// CPLEX problem
int EGcplexAddIncumbentCallBack (EGcplex_t * Prob,
																 int (CPXPUBLIC * callback) (EGCPXENVptr,
																														 void *,
																														 int,
																														 void *,
																														 double,
																														 double *,
																														 int *,
																														 int *),
																 void *cinfo)
{
#if CPX_VERSION > 700
	Prob->status =
		CPXsetincumbentcallbackfunc ((CPXENVptr) (Prob->env), callback, cinfo);
	CPLEXTEST (Prob, "in EGcplexAddIncumbentCallBack");
#else
	EXIT (1, "in EGcplexAddIncumbentCallBack, not supported before Cplex 7.1");
#endif
	return Prob->status;
}

// add a function callback for cut generation during MIP
// note that the callback is asociated to the CPLEX environment and not to the 
// CPLEX problem
int EGcplexAddCutCallBack (EGcplex_t * Prob,
													 int (CPXPUBLIC * callback) (EGCPXENVptr,
																											 void *,
																											 int,
																											 void *,
																											 int *),
													 void *cinfo)
{
	Prob->status =
		CPXsetcutcallbackfunc ((CPXENVptr) (Prob->env), callback, cinfo);
	CPLEXTEST (Prob, "in EGcplexAddIncumbentCallBack");
	return Prob->status;
}

/* this function is to set any integer parameter of CPLEX, this affects the
 * nefironment */
int EGcplexSetDblParam (CPXENVptr env,
												int param,
												double value)
{
	int status;
	status = CPXsetdblparam (env, param, value);
	if (status)
	{
		CPXgeterrorstring (env, status, EGcplexErrMsg);
		fprintf (stderr, "en EGcplexSetDblParam, in file %s, line %d\n", __FILE__,
						 __LINE__);
		fprintf (stderr, "%s\n", EGcplexErrMsg);
	}
	return status;
}

// set the CPLEX environment output off, all problems sharing this environment
// will not report to screen
int EGcplexSetScreenOff (EGcplex_t * Prob)
{
	Prob->status =
		CPXsetintparam ((CPXENVptr) (Prob->env), CPX_PARAM_SCRIND, CPX_OFF);
	CPLEXTEST (Prob, "in EGcplexSetScreenOff");
	return Prob->status;
}

// set an upper bound for the LP objetive function, this bound will affect all problems
// that share the CPLEX environment
int EGcplexSetULP (EGcplex_t * Prob,
									 double ubound)
{
	Prob->status =
		CPXsetdblparam ((CPXENVptr) (Prob->env), CPX_PARAM_OBJULIM, ubound);
	CPLEXTEST (Prob, "in EGcplexSetULP");
	return Prob->status;
}

// set an lower bound for the LP objetive function, this bound will affect all problems
// that share the CPLEX environment
int EGcplexSetLLP (EGcplex_t * Prob,
									 double lbound)
{
	Prob->status =
		CPXsetdblparam ((CPXENVptr) Prob->env, CPX_PARAM_OBJLLIM, lbound);
	CPLEXTEST (Prob, "in EGcplexSetLLP");
	return Prob->status;
}

// set an upper bound for the MIP objetive function, this bound will affect all problems
// that share the CPLEX environment
int EGcplexSetUMIP (EGcplex_t * Prob,
										double ubound)
{
	Prob->status =
		CPXsetdblparam ((CPXENVptr) Prob->env, CPX_PARAM_CUTUP, ubound);
	CPLEXTEST (Prob, "in EGcplexSetULP");
	return Prob->status;
}

// set an lower bound for the MIP objetive function, this bound will affect all problems
// that share the CPLEX environment
int EGcplexSetLMIP (EGcplex_t * Prob,
										double lbound)
{
	Prob->status =
		CPXsetdblparam ((CPXENVptr) Prob->env, CPX_PARAM_CUTLO, lbound);
	CPLEXTEST (Prob, "in EGcplexSetLLP");
	return Prob->status;
}

/* ========================================================================= */
/* set the sense of a set of constraints */
/* ========================================================================= */
int EGcplexSetSense (EGcplex_t * Prob,
										 int nz,
										 int *ind,
										 char *sense)
{
	Prob->status = CPXchgsense (Prob->env, Prob->lp, nz, ind, sense);
	CPLEXTEST (Prob, "in EGcplexSetSense");
	return Prob->status;
}

/* this function, as its name says, is a null cut callback, this is used in
 * order to reset or shadow a previous callback asociated to the underlying
 * CPXENV. Note that all calbacks are associated to the environment instead of
 * the problem. */
/*
int CPXPUBLIC EGcplexNullCutCallBack (EGCPXENVptr env,
																			void *cbdata,
																			int wherefrom,
																			void *cbhandle,
																			int *usercation__p)
{

	return 0;
}
*/
EGcutCallBack_f EGcplexNullCutCallBack = 0;

// this is for testing purposes only, you should call this function before running just to be 
// sure the structures are OK
#endif
