/* EGlib "Efficient General Library" provides some basic structures and
 * algorithms commons in many optimization algorithms.
 *
 * Copyright (C) 2005 Daniel Espinoza and Marcos Goycoolea.
 * 
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but 
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY 
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public 
 * License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA 
 * */
#include "eg_col.h"
/* return a pointer to an initialized EGcol object */
EGcol_t *EGnewCol (EGmemPool_t * mempool)
{

	/* local variables */
	EGcol_t *res;

	/* looking for memory */
	res = (EGcol_t *) EGmemPoolMalloc (mempool, sizeof (EGcol_t));
	res->list = EGnewList (mempool);
	res->name = 0;

	/* ending */
	return res;
}

/* liberate the memory returned by EGnewCol, __NEVER__ liberate such a memory 
 * space with free */
void EGfreeCol (void *col)
{

	/* local variables */
	EGmemPool_t *mempool = ((EGcol_t *) col)->list->mempool;
	EGlistNode_t *cur,
	 *next;

	/* we free the list, in order to be more inteligent we free everithing from
	 * here, the disadvantage of this is that is __VERY__ implementation dependent */
	cur = ((EGcol_t *) col)->list->begin;
	while (cur)
	{
		next = cur->next;
		EGmemPoolFree (cur->this, sizeof (EGcplexData_t), mempool);
		EGmemPoolFree (cur, sizeof (EGlistNode_t), mempool);
		cur = next;
	}

	/* then we free everyithing else */
	EGmemPoolFree (((EGcol_t *) col)->list, sizeof (EGlist_t), mempool);
	if (((EGcol_t *) col)->name)
		EGmemPoolFree (((EGcol_t *) col)->name,
									 strlen (((EGcol_t *) col)->name) + 1, mempool);
	EGmemPoolFree (col, sizeof (EGcol_t), mempool);

	/* ending */
	return;
}

/* this function just add the pair (int,double) to the List in EGcol, but it does not 
 * check if there is another entry with the same integer, i.e. if you add a constrain 
 * twice you will have two entries for the same constrain, and this will lead to an 
 * error when we add the constraint or column to  */
void EGcolAddCoeff (EGcol_t * col,
										const int LPind,
										const double coeff)
{

	/* local variables */
	EGcplexData_t *data;

	/* looking for memory */
	data =
		(EGcplexData_t *) EGmemPoolMalloc (col->list->mempool,
																			 sizeof (EGcplexData_t));
	data->LPind = LPind;
	data->coeff = coeff;

	/* andd data to the end of the list */
	EGlistPushBack (col->list, data);

	/* ending */
	return;
}

/* change the name, objective lowe and upper bound of a column */
void EGcolInit (EGcol_t * col,
								const char *name,
								const double obj,
								const double lb,
								const double ub)
{

	/* we free the current name */
	if (col->name)
		EGmemPoolFree (col->name, strlen (col->name) + 1, col->list->mempool);

	/* and now we copy all information */
	if (name)
	{
		col->name =
			(char *) EGmemPoolMalloc (col->list->mempool, strlen (name) + 1);
		memcpy (col->name, name, strlen (name) + 1);
	}
	else
		col->name = 0;
	col->obj = obj;
	col->lb = lb;
	col->ub = ub;

	/* ending */
	return;
}

/* this is for testing purposes only, you should call this function before running just to be 
 * sure the structures are OK */
void EGcolTest (void)
{
	/* local variables */
#if DEBUG > 2
	int i,
	  j,
	  size = 10;
	EGcol_t *mycols[10];
#else
	int i,
	  j,
	  size = 1000;
	EGcol_t *mycols[1000];
#endif
	EGmemPool_t *mempool;

	/* creating some columns,  */
	mempool = EGnewMemPool (100, EGmemPoolNewSize, EGmemPoolNewSize (1));
#if DEBUG > 2
	fprintf (stdout, "Creating Cols");
	fflush (stdout);
#endif
	for (i = 0; i < size; i++)
	{
#if DEBUG > 2
		fprintf (stdout, ".");
		fflush (stdout);
#endif
		mycols[i] = EGnewCol (mempool);
		EGcolInit (mycols[i], "X", 1.0, 1.0, 0.0);
		for (j = 0; j < size; j++)
		{
			EGcolAddCoeff (mycols[i], j, drand48 ());
		}
	}
#if DEBUG > 2
	fprintf (stdout, "done\n");
	fflush (stdout);
#endif

	/*erase them,  */
	for (i = 0; i < size; i++)
	{
		EGfreeCol ((void *) mycols[i]);
	}

	/* recreate them and exit */
	/* creating some columns,  */
#if DEBUG > 2
	fprintf (stdout, "Creating Cols");
	fflush (stdout);
#endif
	for (i = 0; i < size; i++)
	{
#if DEBUG > 2
		fprintf (stdout, ".");
		fflush (stdout);
#endif
		mycols[i] = EGnewCol (mempool);
		EGcolInit (mycols[i], "X", 1.0, 1.0, 0.0);
		for (j = 0; j < size; j++)
		{
			EGcolAddCoeff (mycols[i], j, drand48 ());
		}
	}
#if DEBUG > 2
	fprintf (stdout, "done\n");
	fflush (stdout);
#endif

	/*erase them,  */
#if DEBUG > 2
	fprintf (stdout, "Erasing Cols");
	fflush (stdout);
#endif
	for (i = 0; i < size; i++)
	{
#if DEBUG > 2
		fprintf (stdout, ".");
		fflush (stdout);
#endif
		EGfreeCol ((void *) mycols[i]);
	}
#if DEBUG > 2
	fprintf (stdout, "done\n");
	fflush (stdout);
#endif
	EGfreeMemPool (mempool);

	/* ending */
	return;
}

/* this function clear the internal coefficient data */
int EGcolClear (EGcol_t * col)
{

	/* local data */
	EGlistNode_t *cur,
	 *next;
	EGmemPool_t *mempool = ((EGcol_t *) col)->list->mempool;

	/* we clear the list */
	cur = col->list->begin;
	while (cur)
	{
		next = cur->next;
		EGmemPoolFree (cur->this, sizeof (EGcplexData_t), mempool);
		EGmemPoolFree (cur, sizeof (EGlistNode_t), mempool);
		cur = next;
	}
	col->list->size = 0;
	col->list->begin = col->list->end = 0;

	/* ending */
	return 0;
}

/* ========================================================================= */
/* given a column, and LP number and a coefficient, this function will try to
 * find if the LP number already have an entry in the column, if so, it will add
 * the given coefficient to the existing one, if the LP index is not in the 
 * column, it will add it to the column with the given coefficient. */
/* ========================================================================= */
void EGcolAddIncCoeff (EGcol_t * col,
											 const int LPind,
											 const double coeff)
{

	/* local variables */
	EGlistNode_t *it;

	/* first, we try to find an entry with the given LP number */
	for (it = col->list->begin; it; it = it->next)
		if (((EGcplexData_t *) (it->this))->LPind == LPind)
		{
			((EGcplexData_t *) (it->this))->coeff += coeff;
			return;
		}

	/* if we can't find the LP number in the column, we just add it to the end of
	 * the column list of indices */
	EGcolAddCoeff (col, LPind, coeff);

	/* ending */
	return;
}
