/* ========================================================================= */
/* begin eg_kppairs.h */
#ifndef __EG_KP_PAIRS__
#define __EG_KP_PAIRS__
#include "eg_dijkstra.h"
#include "eg_dijkstra_app.h"
#include "eg_dgraph.h"
#include "eg_greedytypes.h"
#include "eg_greedykp.h"

/* ========================================================================= */
/** @brief return in a heap, all pairs in the boundary of the zero domino, 
 * whose total value (i.e. even path + internal path) is less than two.
 * @param data information related to the problem.
 * @param zerodom zero domino in wich we will look for pairs.
 * @param orientation if zero, then compute internal paairs, else, compute
 * external pairs.
 * @param pairs sorted list (up to size max_pairs) storing the best pairs. we
 * assume that the best pair is at the beginning of the list, and the worst
 * pair is at the end of the list.
 * @param max_pairs maximum elements to be stored in the list.
 * @param percentage percentage of the bound of two to use while generating all
 * internal pairs.
 * @param node_mark array of length N_PRIMAL_NODES to be used internally.
 * @param edge_mark array of length N_PRIMAL_EDGES to be used internally.
 * @param dc_nodes array of length N_DUAL_NODES to be used internally.
 * @return zero on success, non-zero otherwise.
 * */
int EGgenerateInternalPairs (EGgreedyData_t*const data,
														 EGdkdomino_t*const zerodom,
														 const unsigned int orientation,
														 EGlist_t*const pairs,
														 const unsigned max_pairs,
														 double const percentage,
														 unsigned char*const node_mark,
														 unsigned char*const edge_mark,
														 EGdGraphNode_t**const dc_nodes);

/* ========================================================================= */
/** @brief destructor for greedytypes */
void EGinternalPairsClear(void*pair,EGmemPool_t*mem);

/* ========================================================================= */
/** @brief destructor for greedytypes */
void EGfreeInternalPairs(void*pair,EGmemPool_t*mem);

/* ========================================================================= */
/** @brief given a cut and an orientation, return the primal nodes in the cut 
 * with that orientation. 
 * @param data structure containing all pertinent data to the problem.
 * @param dualcut cut for wich we want the primal nodes.
 * @param orientation wich side of the cut we want.
 * @param cutsz pointer to an integer where we will return the number of nodes
 * in the cut.
 * @param cutnodes pointer to an array, the function will allocate the array if
 * it is NULL, otherwise will use the given array and assume that is large
 * enough.
 * @param node_mark array of length N_PRIMAL_NODES to be used internally.
 * @param edge_mark array of length N_PRIMAL_EDGES to be used internally.
 * @return zero on success, non-zero otherwise.
 * */ 
int EGgetCutNodes(EGgreedyData_t*const data,
									EGdualCut_t*const dualcut,
									const unsigned int orientation,
									int*const cutsz,
									int**const cutnodes,
									unsigned char*const node_mark,
									unsigned char*const edge_mark);

/* ========================================================================= */
/** @brief given a k-dom and a path, return the primal nodes 
 * in the A halve, the orientation is taken from the k-dom structure.
 * @param data structure containing all pertinent data to the problem.
 * @param kdom k-dom for wich we want the primal A-nodes.
 * @param k for wich path we want the A-nodes.
 * @param cutsz pointer to an integer where we will return the number of nodes
 * in the cut.
 * @param cutnodes pointer to an array, the function will allocate the array if
 * it is NULL, otherwise will use the given array and assume that is large
 * enough.
 * @param node_mark array of length N_PRIMAL_NODES to be used internally.
 * @param edge_mark array of length N_PRIMAL_EDGES to be used internally.
 * @param dc_nodes array of length N_DUAL_NODES to be used internally.
 * @return zero on success, non-zero otherwise.
 * */ 
int EGgetANodes(EGgreedyData_t*const data,
								EGdkdomino_t*const kdom,
								int unsigned const k,
								int*const cutsz,
								int**const cutnodes,
								unsigned char*const node_mark,
								unsigned char*const edge_mark,
								EGdGraphEdge_t**const dc_nodes);

/* ========================================================================= */
/** @brief given a dual cut and an internal path, return the orientation of the
 * path. 
 * @param data structure containing all pertinent data to the problem.
 * @param dualcut cut where we are working.
 * @param pathsz number of edges in the internal path.
 * @param path array of pointer of edges in the internal path.
 * @param orientation pointer where we will retunr the orientation.
 * @param node_mark array of length N_PRIMAL_NODES to be used internally.
 * @param edge_mark array of length N_PRIMAL_EDGES to be used internally.
 * @return zero on success, non-zero otherwise.
 * */ 
int EGgetOrientation( EGgreedyData_t*const data,
											EGdualCut_t*const dualcut,
											int unsigned const pathsz,
											EGdGraphEdge_t**const path,
											unsigned int*const orientation,
											unsigned char*const node_mark,
											unsigned char*const edge_mark);

/* ========================================================================= */
/** @brief Given a primal cut, return a dual cut
 * @param data structure containing all pertinent data to the problem.
 * @param node_mark array of length N_PRIMAL_NODES to be used internally.
 * @param dc_edges array of length N_PRIMAL_EDGES to be used internally.
 * @param pset_sz size of the primal set (number of nodes)
 * @param pset array of length pset_sz containing all nodes in the primal set
 * @param edge_mark array of length N_PRIMAL_EDGES to be used internally.
 * @param dualcut pointer to where to retunr the new dualcut.
 * @return zero on success, non-zero otherwise.
 * */
int EGgetDualCut( EGgreedyData_t*const data,
									EGdualCut_t**const dualcut,
									const int pset_sz,
									const int*const pset,
									unsigned char*const node_mark,
									unsigned char*const edge_mark,
									EGdGraphEdge_t**const dc_edges);

/* ========================================================================= */
/** @bried verbose level */
#define KPPAIRS_VERBOSE 100

/* ========================================================================= */
/* end eg_kppairs.h */
#endif
