#include <stdio.h>
#include <stdlib.h>
#include "dp_config.h"
#include "eg_mempool.h"
#include "eg_2ptighten.h"
#define KPT_ENABLE 1
#if KPT_ENABLE
#include "eg_ugraph.h"
#include "eg_bit.h"
#include "eg_util.h"
#include "eg_bbtree.h"
#include "eg_2pchecker.h"
/* ========================================================================= */
/* various size, note that the maximujm sizes must be of the form 2^n */
#define KPT_MAX_DOM 512U
#define KPT_MAX_DOM_LOG 9U
#define KPT_MAX_NODE 131072U
#define KPT_MAX_NODE_LOG 17U
#define KPT_MAX_DEPTH 1U
/* ========================================================================= */
/* data structure to handle information about the graph */
/* ========================================================================= */

/* ========================================================================= */
/** @brief Structure to hold a movement */
typedef struct
{
	unsigned int n_id:KPT_MAX_NODE_LOG;	/**< store the node ID of this 
																				simple move */
	unsigned int move_id:6;							/**< move number */
	unsigned int dom;										/**< domino asociated with the move */
}
KPTMove_t;

/* ========================================================================= */
/** @brief Full move structure */
typedef struct
{
	KPTMove_t move[KPT_MAX_DEPTH];		/**< single move for each depth */
	double val;												/**< value asociated to the move */
	unsigned int depth;								/**< depth of the stored move */
}
KPTFullMove_t;

/* ========================================================================= */
/** @brief Structure to hold information about the graph */
typedef struct
{
	unsigned int n_AH[2];/**< size of Ahandle(n_H[1]) and co-Ahandle(n_H[0]) */
	unsigned int n_BH[2];/**< size of Bhandle(n_H[1]) and co-Bhandle(n_H[0]) */
	unsigned int n_dominos:KPT_MAX_DOM_LOG;/**< number of dominos */
	unsigned int n_nodes:KPT_MAX_NODE_LOG;/**< nodes in the graph */
	unsigned int n_A[2][KPT_MAX_DOM];		/**< size of every Ac,A set */
	unsigned int n_B[2][KPT_MAX_DOM];		/**< size of every Bc,B set */
	unsigned int n_AdB[2][KPT_MAX_DOM];	/**< size of A\Delta B */
	unsigned int n_Tc[KPT_MAX_DOM];			/**< size of every T^c set */
	EGbitset_t A_partition[EGbitsetSize (KPT_MAX_DOM)];
																										/**< for every domino
																				indicates if it has an A partition. */
	EGbitset_t B_partition[EGbitsetSize (KPT_MAX_DOM)];
																										/**< for every domino
																				indicates if it has a B partition. */
	EGbitset_t Kdom[EGbitsetSize (KPT_MAX_DOM)];
																						 /**< for every domino
																				indicates if it has a B partition. */
}
KPTGdata_t;

/* ========================================================================= */
/** @brief structure holding information regarding every edge */
typedef union
{
	struct
	{
		unsigned int n_AtoAc:KPT_MAX_DOM_LOG;
																				/**< how many E(A:Ac) sets this 
																edge bellong */
		unsigned int n_BtoBc:KPT_MAX_DOM_LOG;
																				/**< how many E(B:Bc) sets this 
																edge bellong */
		unsigned int n_dT:KPT_MAX_DOM_LOG;/**< how many E(delta(T)) 
																sets this edge bellong */
		unsigned int in_AF:1;			/**< one if this edge is in the F set */
		unsigned int in_BF:1;			/**< one if this edge is in the F set */
	}
	num;
	unsigned int used;					/**< If this field is non-zero, it means that the 																	edge has a non-zero coefficient in the 
																	 constraint */
}
KPTEdata_t;

/* ========================================================================= */
/**@ brieg information regarding every node */
struct KPTNdata_t
{
	/* ======================================================================= */
	/* link to the internal information */
	/* ======================================================================= */
	EGbbtreeNode_t *tree_cn;			/**< pointer to the tree conector storing the 
																	best move for this node */
	/* ======================================================================= */
	/* cut description, this HAVE to be keeped wright at every iteration */
	/* ======================================================================= */
	EGbitset_t It[EGbitsetSize (KPT_MAX_DOM)];
																					 /**< indicatriz for T set 
																	for this node */
	EGbitset_t Ia[EGbitsetSize (KPT_MAX_DOM)];
																					 /**< indicatriz for A set 
																	for this node */
	EGbitset_t Ib[EGbitsetSize (KPT_MAX_DOM)];
																					 /**< indicatriz for B set 
																	for this node */
	EGbitset_t Iadb[EGbitsetSize (KPT_MAX_DOM)];
																						 /**< indicatriz for A\Delta B 
																	set for this node */
	EGbitset_t added[EGbitsetSize (KPT_MAX_DOM)];
																							/**< bitset indicating
																	if this node have been added to T_i */
	EGbitset_t flipAc[EGbitsetSize (KPT_MAX_DOM)];
																							 /**< bitset indicating 
																	if this node have been fliped form A to B */
	EGbitset_t flipA[EGbitsetSize (KPT_MAX_DOM)];
																							/**< bitset indicating 
																	if this node have been fliped form A to B */
	EGbitset_t flipBc[EGbitsetSize (KPT_MAX_DOM)];
																							 /**< bitset indicating 
																	if this node have been fliped form B to A */
	EGbitset_t flipB[EGbitsetSize (KPT_MAX_DOM)];
																							/**< bitset indicating 
																	if this node have been fliped form B to A */
	unsigned int n_in_B:KPT_MAX_DOM_LOG;/**< in how many B sets this
																	node bellong */
	unsigned int n_in_A:KPT_MAX_DOM_LOG;/**< in how many A sets this
																	node bellong */
	unsigned int n_in_T:KPT_MAX_DOM_LOG;/**< in how many T sets this
																	node bellong */
	unsigned int in_AH:1;					/**< one if this node in H */
	unsigned int added_AH:1;				/**< one if this node has been aded to H */
	unsigned int in_BH:1;					/**< one if this node in H */
	unsigned int added_BH:1;				/**< one if this node has been aded to H */
	/* ======================================================================= */
	/* move information, this may be changed later on */
	/* ======================================================================= */
	KPTFullMove_t full_move;			/**< Full move for the node */
};
typedef struct KPTNdata_t KPTNdata_t;

/* ========================================================================= */
/* static variables, these variables hold the graphData, node data and edge
 * data, the later two indexed by node->id and edge->id respectivelly. */
static KPTNdata_t *nodeData = 0;
static KPTEdata_t *edgeData = 0;
static KPTGdata_t graphData;
static EGuGraph_t *G = 0;
static EGuGraphNode_t **all_nodes;
static EGuGraphEdge_t **all_edges;
static double const *weight;
static EGbbtree_t *tree;
static EGbitset_t node_update[EGbitsetSize (KPT_MAX_NODE)];

/* ========================================================================= */
/* some macros to make access easy */
/* ========================================================================= */
/* the minimum improvement is for the local search, if negative it means that it
 * will allow movements that may worsen the inequality */
#define KPTMinImprovement (-1e-7)

/* constants for the moves */
enum move
{																/* adding moves: */
	KPT_Tc_AB,										/* move from outside the teeth to A and B */
	KPT_Tc_AcB,										/* move from outside the teeth to Ac and B */
	KPT_Tc_ABc,										/* move from outside the teeth to A and Bc */
	KPT_Tc_AcBc,									/* move from outside the teeth to Ac and Bc */
	KPT_Tc_A,											/* move from Tc to A */
	KPT_Tc_B,											/* move from Tc to B */
	KPT_Tc_Ac,										/* move from Tc to Ac */
	KPT_Tc_Bc,										/* move from Tc to Bc */
	KPT_AHc_AH,										/* move a node from AHc to AH */
	KPT_BHc_BH,										/* move a node from BHc to BH */
	KPT_AHc_AH_BHc_BH,						/* move a node from xHc to xH */
	/* fliping moves */
	KPT_Ac_A,											/* flip within the A set in T */
	KPT_Bc_B,											/* flip within the B set in T */
	KPT_A_Ac,											/* flip within the A set in T */
	KPT_B_Bc,											/* flip within the B set in T */
	KPT_A_Ac_B_Bc,								/* flip both A and B within T */
	KPT_A_Ac_Bc_B,								/* flip both A and B within T */
	KPT_Ac_A_B_Bc,								/* flip both A and B within T */
	KPT_Ac_A_Bc_B,								/* flip both A and B within T */
	KPT_Ac_A_flipAH,							/* flip A and Ahandle */
	KPT_Bc_B_flipBH,							/* flip B and Bhandle */
	KPT_A_Ac_flipAH,							/* flip A and Ahandle */
	KPT_B_Bc_flipBH,							/* flip B and Bhandle */
	KPT_A_Ac_B_Bc_flipAH,					/* flip both A and B within T */
	KPT_A_Ac_Bc_B_flipAH,					/* flip both A and B within T */
	KPT_Ac_A_B_Bc_flipAH,					/* flip both A and B within T */
	KPT_Ac_A_Bc_B_flipAH,					/* flip both A and B within T */
	KPT_A_Ac_B_Bc_flipBH,					/* flip both A and B within T */
	KPT_A_Ac_Bc_B_flipBH,					/* flip both A and B within T */
	KPT_Ac_A_B_Bc_flipBH,					/* flip both A and B within T */
	KPT_Ac_A_Bc_B_flipBH,					/* flip both A and B within T */
	KPT_A_Ac_B_Bc_flipAHBH,				/* flip both A and B, and AH, BH */
	KPT_A_Ac_Bc_B_flipAHBH,				/* flip both A and B within T */
	KPT_Ac_A_B_Bc_flipAHBH,				/* flip both A and B within T */
	KPT_Ac_A_Bc_B_flipAHBH,				/* flip both A and B within T */
	KPT_AH_AHc_BHc_BH,						/* flip both handles */
	KPT_AHc_AH_BH_BHc,						/* flip both handles */
	/* shrinking moves */
	KPT_AcBc_Tc,									/* move from outside the teeth to Ac and Bc */
	KPT_AcB_Tc,										/* move from outside the teeth to Ac and B */
	KPT_ABc_Tc,										/* move from outside the teeth to A and Bc */
	KPT_AB_Tc,										/* move from outside the teeth to A and B */
	KPT_A_Tc,											/* move from Tc to A */
	KPT_B_Tc,											/* move from Tc to B */
	KPT_Ac_Tc,										/* move from Tc to Ac */
	KPT_Bc_Tc,										/* move from Tc to Bc */
	KPT_AH_AHc,										/* move a node from AHc to AH */
	KPT_BH_BHc,										/* move a node from BHc to BH */
	KPT_AH_AHc_BH_BHc,						/* take out the node from both handles */
	KPT_no_move										/* no feasible move found */
};

/* string names for the moves */
static const char move_name[61][40] = {
	[KPT_A_Tc] = "KPT_A_Tc",
	[KPT_B_Tc] = "KPT_B_Tc",
	[KPT_Ac_Tc] = "KPT_Ac_Tc",
	[KPT_Bc_Tc] = "KPT_Bc_Tc",
	[KPT_Tc_A] = "KPT_Tc_A",
	[KPT_Tc_B] = "KPT_Tc_B",
	[KPT_Tc_Ac] = "KPT_Tc_Ac",
	[KPT_Tc_Bc] = "KPT_Tc_Bc",
	[KPT_Tc_AB] = "KPT_Tc_AB",
	[KPT_Tc_AcB] = "KPT_Tc_AcB",
	[KPT_Tc_ABc] = "KPT_Tc_ABc",
	[KPT_Tc_AcBc] = "KPT_Tc_AcBc",
	[KPT_AHc_AH] = "KPT_AHc_AH",
	[KPT_BHc_BH] = "KPT_BHc_BH",
	[KPT_AHc_AH_BHc_BH] = "KPT_AHc_AH_BHc_BH",
	[KPT_Ac_A] = "KPT_Ac_A",
	[KPT_Bc_B] = "KPT_Bc_B",
	[KPT_A_Ac] = "KPT_A_Ac",
	[KPT_B_Bc] = "KPT_B_Bc",
	[KPT_Ac_A_Bc_B] = "KPT_Ac_A_Bc_B",
	[KPT_A_Ac_Bc_B] = "KPT_A_Ac_Bc_B",
	[KPT_Ac_A_B_Bc] = "KPT_Ac_A_B_Bc",
	[KPT_A_Ac_B_Bc] = "KPT_A_Ac_B_Bc",
	[KPT_Ac_A_Bc_B_flipAH] = "KPT_Ac_A_Bc_B_flipAH",
	[KPT_A_Ac_Bc_B_flipAH] = "KPT_A_Ac_Bc_B_flipAH",
	[KPT_Ac_A_B_Bc_flipAH] = "KPT_Ac_A_B_Bc_flipAH",
	[KPT_A_Ac_B_Bc_flipAH] = "KPT_A_Ac_B_Bc_flipAH",
	[KPT_Ac_A_Bc_B_flipBH] = "KPT_Ac_A_Bc_B_flipBH",
	[KPT_A_Ac_Bc_B_flipBH] = "KPT_A_Ac_Bc_B_flipBH",
	[KPT_Ac_A_B_Bc_flipBH] = "KPT_Ac_A_B_Bc_flipBH",
	[KPT_A_Ac_B_Bc_flipBH] = "KPT_A_Ac_B_Bc_flipBH",
	[KPT_Ac_A_Bc_B_flipAHBH] = "KPT_Ac_A_Bc_B_flipAHBH",
	[KPT_A_Ac_Bc_B_flipAHBH] = "KPT_A_Ac_Bc_B_flipAHBH",
	[KPT_Ac_A_B_Bc_flipAHBH] = "KPT_Ac_A_B_Bc_flipAHBH",
	[KPT_A_Ac_B_Bc_flipAHBH] = "KPT_A_Ac_B_Bc_flipAHBH",
	[KPT_AH_AHc_BHc_BH] = "KPT_AH_AHc_BHc_BH",
	[KPT_AHc_AH_BH_BHc] = "KPT_AHc_AH_BH_BHc",
	[KPT_Ac_A_flipAH] = "KPT_Ac_A_flipAH",
	[KPT_Bc_B_flipBH] = "KPT_Bc_B_flipBH",
	[KPT_A_Ac_flipAH] = "KPT_A_Ac_flipAH",
	[KPT_B_Bc_flipBH] = "KPT_B_Bc_flipBH",
	[KPT_AB_Tc] = "KPT_AB_Tc",
	[KPT_AcB_Tc] = "KPT_AcB_Tc",
	[KPT_ABc_Tc] = "KPT_ABc_Tc",
	[KPT_AcBc_Tc] = "KPT_AcBc_Tc",
	[KPT_AHc_AH] = "KPT_AHc_AH",
	[KPT_BH_BHc] = "KPT_BH_BHc",
	[KPT_AH_AHc_BH_BHc] = "KPT_AH_AHc_BH_BHc",
	[KPT_no_move] = "KPT_no_move"
};
/* inverse move for every move */
static unsigned char KPT_inv_move[61] = {
	[KPT_A_Tc] = KPT_Tc_A,
	[KPT_B_Tc] = KPT_Tc_B,
	[KPT_Ac_Tc] = KPT_Tc_Ac,
	[KPT_Bc_Tc] = KPT_Tc_Bc,
	[KPT_Tc_A] = KPT_A_Tc,
	[KPT_Tc_B] = KPT_B_Tc,
	[KPT_Tc_Ac] = KPT_Ac_Tc,
	[KPT_Tc_Bc] = KPT_Bc_Tc,
	[KPT_Tc_AB] = KPT_AB_Tc,
	[KPT_Tc_AcB] = KPT_AcB_Tc,
	[KPT_Tc_ABc] = KPT_ABc_Tc,
	[KPT_Tc_AcBc] = KPT_AcBc_Tc,
	[KPT_AHc_AH] = KPT_AH_AHc,
	[KPT_BHc_BH] = KPT_BH_BHc,
	[KPT_AHc_AH_BHc_BH] = KPT_AH_AHc_BH_BHc,
	[KPT_Ac_A] = KPT_A_Ac,
	[KPT_Bc_B] = KPT_B_Bc,
	[KPT_A_Ac] = KPT_Ac_A,
	[KPT_B_Bc] = KPT_Bc_B,
	[KPT_Ac_A_Bc_B] = KPT_A_Ac_B_Bc,
	[KPT_A_Ac_B_Bc] = KPT_Ac_A_Bc_B,
	[KPT_A_Ac_Bc_B] = KPT_Ac_A_B_Bc,
	[KPT_Ac_A_B_Bc] = KPT_A_Ac_Bc_B,
	[KPT_Ac_A_Bc_B_flipBH] = KPT_A_Ac_B_Bc_flipBH,
	[KPT_A_Ac_Bc_B_flipBH] = KPT_Ac_A_B_Bc_flipBH,
	[KPT_Ac_A_B_Bc_flipBH] = KPT_A_Ac_Bc_B_flipBH,
	[KPT_A_Ac_B_Bc_flipBH] = KPT_Ac_A_Bc_B_flipBH,
	[KPT_Ac_A_Bc_B_flipAH] = KPT_A_Ac_B_Bc_flipAH,
	[KPT_A_Ac_Bc_B_flipAH] = KPT_Ac_A_B_Bc_flipAH,
	[KPT_Ac_A_B_Bc_flipAH] = KPT_A_Ac_Bc_B_flipAH,
	[KPT_A_Ac_B_Bc_flipAH] = KPT_Ac_A_Bc_B_flipAH,
	[KPT_Ac_A_Bc_B_flipAHBH] = KPT_A_Ac_B_Bc_flipAHBH,
	[KPT_A_Ac_Bc_B_flipAHBH] = KPT_Ac_A_B_Bc_flipAHBH,
	[KPT_Ac_A_B_Bc_flipAHBH] = KPT_A_Ac_Bc_B_flipAHBH,
	[KPT_A_Ac_B_Bc_flipAHBH] = KPT_Ac_A_Bc_B_flipAHBH,
	[KPT_AH_AHc_BHc_BH] = KPT_AHc_AH_BH_BHc,
	[KPT_AHc_AH_BH_BHc] = KPT_AH_AHc_BHc_BH,
	[KPT_Ac_A_flipAH] = KPT_A_Ac_flipAH,
	[KPT_Bc_B_flipBH] = KPT_B_Bc_flipBH,
	[KPT_A_Ac_flipAH] = KPT_Ac_A_flipAH,
	[KPT_B_Bc_flipBH] = KPT_Bc_B_flipBH,
	[KPT_AB_Tc] = KPT_Tc_AB,
	[KPT_AcB_Tc] = KPT_Tc_AcB,
	[KPT_ABc_Tc] = KPT_Tc_ABc,
	[KPT_AcBc_Tc] = KPT_Tc_AcBc,
	[KPT_AHc_AH] = KPT_AH_AHc,
	[KPT_BH_BHc] = KPT_BHc_BH,
	[KPT_AH_AHc_BH_BHc] = KPT_AHc_AH_BHc_BH,
	[KPT_no_move] = KPT_no_move
};

/* ========================================================================= */
/* some macros */
#define KPTgetEdge(__e_it) ((EGuGraphEdge_t*)(__e_it->this))
#define KPTgetOtherEndId(__n_id,__e_it) (__n_id == KPTgetEdge(__e_it)->head->id ? KPTgetEdge(__e_it)->tail->id : KPTgetEdge(__e_it)->head->id)
#define KPTgetEdgeId(__e_it) (KPTgetEdge(__e_it)->id)
#define KPTNdata(__this) ((KPTNdata_t*)(__this))

/* ========================================================================= */
/* local static functions */
/* ========================================================================= */

#if KPT_DBG < DEBUG || KPT_EDBG < DEBUG || KPT_VRB < DEBUG
/* ========================================================================= */
/* this function display on the screen what move are we performing */
/* ========================================================================= */
static inline void DPdisplaySingleMove (const KPTMove_t * const move)
{
	/* local variables */
	EGlistNode_t *e_it;
	unsigned no_id = move->n_id,
	  e_id,
	  dom = move->dom;
	/* printing information */
	fprintf (stderr, "\tNode %u:%s:%u", no_id, move_name[move->move_id], dom);
	switch (move->move_id)
	{
	case KPT_AB_Tc:
	case KPT_AcB_Tc:
	case KPT_ABc_Tc:
	case KPT_AcBc_Tc:
	case KPT_Tc_AB:
	case KPT_Tc_AcB:
	case KPT_Tc_ABc:
	case KPT_Tc_AcBc:
	case KPT_Ac_A:
	case KPT_A_Ac:
	case KPT_Bc_B:
	case KPT_B_Bc:
	case KPT_Ac_A_B_Bc:
	case KPT_Ac_A_Bc_B:
	case KPT_A_Ac_B_Bc:
	case KPT_A_Ac_Bc_B:
	case KPT_Ac_A_B_Bc_flipBH:
	case KPT_Ac_A_Bc_B_flipBH:
	case KPT_A_Ac_B_Bc_flipBH:
	case KPT_A_Ac_Bc_B_flipBH:
	case KPT_Ac_A_B_Bc_flipAH:
	case KPT_Ac_A_Bc_B_flipAH:
	case KPT_A_Ac_B_Bc_flipAH:
	case KPT_A_Ac_Bc_B_flipAH:
	case KPT_Ac_A_B_Bc_flipAHBH:
	case KPT_Ac_A_Bc_B_flipAHBH:
	case KPT_A_Ac_B_Bc_flipAHBH:
	case KPT_A_Ac_Bc_B_flipAHBH:
	case KPT_Ac_A_flipAH:
	case KPT_A_Ac_flipAH:
	case KPT_Bc_B_flipBH:
	case KPT_B_Bc_flipBH:
	case KPT_A_Tc:
	case KPT_B_Tc:
	case KPT_Ac_Tc:
	case KPT_Bc_Tc:
	case KPT_Tc_A:
	case KPT_Tc_B:
	case KPT_Tc_Ac:
	case KPT_Tc_Bc:
		fprintf (stderr, ":%s_%u(%u,%u,%u,%u,%u)%s(%u,%u)%s(%u,%u)%s_%s(%u,%u)",
						 !EGbitTest (nodeData[no_id].It, dom) ? "Tc" :
						 EGbitTest (nodeData[no_id].Ia, dom) &&
						 EGbitTest (nodeData[no_id].Ib, dom) ? "TAB" :
						 EGbitTest (nodeData[no_id].Ia, dom) ? "TABc" :
						 EGbitTest (nodeData[no_id].Ib, dom) ? "TAcB" : "TAcBc",
						 dom, graphData.n_A[1][dom], graphData.n_A[0][dom],
						 graphData.n_B[1][dom], graphData.n_B[0][dom],
						 graphData.n_Tc[dom], nodeData[no_id].in_AH ? "aH" : "aHc",
						 graphData.n_AH[1], graphData.n_AH[0],
						 nodeData[no_id].in_BH ? "bH" : "bHc",
						 graphData.n_BH[1], graphData.n_BH[0],
						 EGbitTest (graphData.Kdom, dom) ? "2dom" : "1dom",
						 EGbitTest (nodeData[no_id].Iadb, dom) ? "AdB" : "AdBc",
						 graphData.n_AdB[1][dom], graphData.n_AdB[0][dom]);
		/* printing adjacent edges information */
		for (e_it = all_nodes[no_id]->edges->begin; e_it; e_it = e_it->next)
		{
			no_id = KPTgetOtherEndId (move->n_id, e_it);
			e_id = KPTgetEdgeId (e_it);
			fprintf (stderr, "\n\t\te=(%u,%u:%s)%s:%s[%8.6lf]F[%u,%u]",
							 move->n_id, no_id,
							 !EGbitTest (nodeData[no_id].It, dom) ? "Tc" :
							 EGbitTest (nodeData[no_id].Ia, dom) &&
							 EGbitTest (nodeData[no_id].Ib, dom) ? "TAB" :
							 EGbitTest (nodeData[no_id].Ia, dom) ? "TABc" :
							 EGbitTest (nodeData[no_id].Ib, dom) ? "TAcB" : "TAcBc",
							 nodeData[no_id].in_AH ? "aH" : "aHc",
							 nodeData[no_id].in_BH ? "bH" : "bHc",
							 weight[e_id], edgeData[e_id].num.in_AF,
							 edgeData[e_id].num.in_BF);
		}
		break;
	case KPT_AHc_AH:
	case KPT_BHc_BH:
	case KPT_AH_AHc:
	case KPT_BH_BHc:
	case KPT_AH_AHc_BH_BHc:
	case KPT_AH_AHc_BHc_BH:
	case KPT_AHc_AH_BHc_BH:
	case KPT_AHc_AH_BH_BHc:
		fprintf (stderr, ":%s(%u,%u)%s(%u,%u)",
						 nodeData[no_id].in_AH ? "aH" : "aHc", graphData.n_AH[1],
						 graphData.n_AH[0], nodeData[no_id].in_BH ? "bH" : "bHc",
						 graphData.n_BH[1], graphData.n_BH[0]);
		/* printing adjacent edges information */
		for (e_it = all_nodes[no_id]->edges->begin; e_it; e_it = e_it->next)
		{
			no_id = KPTgetOtherEndId (move->n_id, e_it);
			e_id = KPTgetEdgeId (e_it);
			fprintf (stderr, "\n\t\te=(%u,%u)%s:%s[%8.6lf]F[%u,%u]",
							 move->n_id, no_id, nodeData[no_id].in_AH ? "aH" : "aHc",
							 nodeData[no_id].in_BH ? "bH" : "bHc",
							 weight[e_id], edgeData[e_id].num.in_AF,
							 edgeData[e_id].num.in_BF);
		}
		break;
	}
	fprintf (stderr, "\n");
}

/* ========================================================================= */
/* this function display on the screen what move are we performing */
/* ========================================================================= */
static inline int DPdisplayMove (unsigned int const nc_id)
{
	unsigned depth = nodeData[nc_id].full_move.depth;
	fprintf (stderr, "Move Stored for node %u:%u\n", nc_id,
					 nodeData[nc_id].full_move.depth);
	while (depth--)
		DPdisplaySingleMove (nodeData[nc_id].full_move.move + depth);
	fprintf (stderr, "\tValue %lf\n", nodeData[nc_id].full_move.val);
	fprintf (stderr, "\t");
	__EG_PRINTLOC2__;
	return 1;
}
#endif
/* ========================================================================= */
/* this function compute the violation of the curently stored constraint */
/* ========================================================================= */
static inline void KPTpriceConstraint (double *l_violation)
{
	register unsigned int i;
	/* finally we compute the in_F field for all edges */
	for (i = G->nedges; i--;)
	{
		(*l_violation) += (edgeData[i].num.in_AF + edgeData[i].num.in_BF +
											 edgeData[i].num.n_AtoAc + edgeData[i].num.n_BtoBc +
											 edgeData[i].num.n_dT) * weight[i];
	}
	return;
}

#if (KPT_VRB <= DEBUG-1) || (KPT_EDBG <= DEBUG)
/* ========================================================================= */
/** @brief Display all edges and their membership on all sets */
/* ========================================================================= */
static inline int KPTDisplayEdges (void)
{
	register unsigned int i;
	register unsigned j;
	unsigned h_id,
	  t_id;
	/* finally we compute the in_F field for all edges */
	for (i = G->nedges; i--;)
	{
		h_id = all_edges[i]->head->id;
		t_id = all_edges[i]->tail->id;
		if (weight[i] > 1e-3)
		{
			fprintf (stderr, "e(%u)[%u,%u][%8.6lf] bellongs to: ", i,
							 h_id, t_id, weight[i]);
			for (j = graphData.n_dominos; j--;)
			{
				if (EGbitTest (nodeData[h_id].It, j) !=
						EGbitTest (nodeData[t_id].It, j))
					fprintf (stderr, "delta(T_%u) ", j);
				if (EGbitTest (nodeData[h_id].Ia, j) !=
						EGbitTest (nodeData[t_id].Ia, j))
					fprintf (stderr, "delta(A_%u) ", j);
				if (EGbitTest (nodeData[h_id].Ib, j) !=
						EGbitTest (nodeData[t_id].Ib, j))
					fprintf (stderr, "delta(B_%u) ", j);
			}
			if (nodeData[h_id].in_AH != nodeData[t_id].in_AH)
				fprintf (stderr, "delta(AH) ");
			if (nodeData[h_id].in_BH != nodeData[t_id].in_BH)
				fprintf (stderr, "delta(BH) ");
			if (edgeData[i].num.in_AF)
				fprintf (stderr, "AF ");
			if (edgeData[i].num.in_BF)
				fprintf (stderr, "BF ");
			fprintf (stderr, "\n");
		}
	}
	return 1;
}
#endif

/* ========================================================================= */
/** @brief This function compare two node data information */
/* ========================================================================= */
static int EGdpTightNdataCompare (const void *N1,
																	const void *N2)
{
	/* we first check if the values are near-zero, if so, we sort according to 
	 * the type of the movement */
	if ((fabs (((const KPTNdata_t *) N1)->full_move.val) <
			 fabs (KPTMinImprovement))
			&& (fabs (((const KPTNdata_t *) N2)->full_move.val) <
					fabs (KPTMinImprovement)))
	{
		/* Sort by first move order */
		if (((const KPTNdata_t *) N1)->full_move.move[0].move_id <
				((const KPTNdata_t *) N2)->full_move.move[0].move_id)
			return -1;
		if (((const KPTNdata_t *) N1)->full_move.move[0].move_id >
				((const KPTNdata_t *) N2)->full_move.move[0].move_id)
			return 1;
		/* if they have the same type. choose by simplicity */
		if (((const KPTNdata_t *) N1)->full_move.depth <
				((const KPTNdata_t *) N2)->full_move.depth)
			return -1;
		if (((const KPTNdata_t *) N1)->full_move.depth >
				((const KPTNdata_t *) N2)->full_move.depth)
			return 1;
		if( N1 < N2)
			return -1;
		if(N1 > N2)
			return 1;
		return 0;
	}
	/* otherwise we sort by value */
	if (((const KPTNdata_t *) N1)->full_move.val <
			((const KPTNdata_t *) N2)->full_move.val)
		return -1;
	if (((const KPTNdata_t *) N1)->full_move.val >
			((const KPTNdata_t *) N2)->full_move.val)
		return 1;
	if( N1 < N2)
		return -1;
	if(N1 > N2)
		return 1;
	return 0;
}

/* ######################################################################### */
/* makeMove functions */
/* ######################################################################### */

/* ========================================================================= */
/* macro definitions to move around nodes */
/* ========================================================================= */
#define KPTmoveBH(__n_id) {\
	graphData.n_BH[nodeData[__n_id].in_BH]--;\
	nodeData[__n_id].in_BH++;\
	graphData.n_BH[nodeData[__n_id].in_BH]++;}

#define KPTmoveAH(__n_id) {\
	graphData.n_AH[nodeData[__n_id].in_AH]--;\
	nodeData[__n_id].in_AH++;\
	graphData.n_AH[nodeData[__n_id].in_AH]++;}

#define KPTmoveAdB(__n_id,__dom) {\
	if(EGbitTest(graphData.Kdom,__dom)){\
	graphData.n_AdB[EGbitTest(nodeData[__n_id].Iadb,__dom)?1U:0U][__dom]--;\
	graphData.n_AdB[EGbitTest(nodeData[__n_id].Iadb,__dom)?0U:1U][__dom]++;\
	EGbitTest(nodeData[__n_id].Iadb,__dom) ? \
	EGbitUnset(nodeData[__n_id].Iadb,__dom): \
	EGbitSet(nodeData[__n_id].Iadb,__dom);}}

#define KPTmoveAAc(__n_id,__dom) {\
	graphData.n_A[1][__dom]--;\
	graphData.n_A[0][__dom]++;\
	EGbitUnset(nodeData[__n_id].Ia,__dom);\
	nodeData[__n_id].n_in_A--;}

#define KPTmoveBBc(__n_id,__dom) {\
	graphData.n_B[1][__dom]--;\
	graphData.n_B[0][__dom]++;\
	EGbitUnset(nodeData[__n_id].Ib,__dom);\
	nodeData[__n_id].n_in_B--;}

#define KPTmoveAcA(__n_id,__dom) {\
	graphData.n_A[0][__dom]--;\
	graphData.n_A[1][__dom]++;\
	EGbitSet(nodeData[__n_id].Ia,__dom);\
	nodeData[__n_id].n_in_A++;}

#define KPTmoveBcB(__n_id,__dom) {\
	graphData.n_B[0][__dom]--;\
	graphData.n_B[1][__dom]++;\
	EGbitSet(nodeData[__n_id].Ib,__dom);\
	nodeData[__n_id].n_in_B++;}

/* ========================================================================= */
/** @brief This function make the move BHBHc,assuming the move is feasible */
/* ========================================================================= */
static int KPTmakeMoveBHBHc (KPTMove_t const *const cur_move,
														 const unsigned int update_flags)
{
	/* local variables */
	unsigned int n_id = cur_move->n_id;
	EGlistNode_t *e_it = all_nodes[n_id]->edges->begin;

	/* update flags if necesary */
	if (update_flags && !nodeData[n_id].in_BH)
		nodeData[n_id].added_BH = 1U;
	/* flip the node in the handle */
	KPTmoveBH (n_id);
	/* we have to compute every move */
	for (; e_it; e_it = e_it->next)
	{
		/* we just flip BF */
		edgeData[KPTgetEdgeId (e_it)].num.in_BF++;
	}															/* end loop through all incident edges */
	return 0;
}

/* ========================================================================= */
/** @brief This function make the move BHcBH,assuming the move is feasible */
/* ========================================================================= */
#define KPTmakeMoveBHcBH(__id,__val) KPTmakeMoveBHBHc(__id,__val)

/* ========================================================================= */
/** @brief This function make the move AHAHc,assuming the move is feasible */
/* ========================================================================= */
static int KPTmakeMoveAHAHc (KPTMove_t const *const cur_move,
														 const unsigned int update_flags)
{
	/* local variables */
	unsigned int n_id = cur_move->n_id;
	EGlistNode_t *e_it = all_nodes[n_id]->edges->begin;

	/* update flags if necesary */
	if (update_flags && !nodeData[n_id].in_AH)
		nodeData[n_id].added_AH = 1U;
	/* flip the node in the handle */
	KPTmoveAH (n_id);
	/* we have to compute every move */
	for (; e_it; e_it = e_it->next)
	{
		/* we just flip AF */
		edgeData[KPTgetEdgeId (e_it)].num.in_AF++;
	}															/* end loop through all incident edges */
	return 0;
}

/* ========================================================================= */
/** @brief This function make the move AHcAH,assuming the move is feasible */
/* ========================================================================= */
#define KPTmakeMoveAHcAH(__id,__val) KPTmakeMoveAHAHc(__id,__val)

/* ========================================================================= */
/** @brief This function make the move AHAHcBHBHc,assuming 
 * the move is feasible */
/* ========================================================================= */
static int KPTmakeMoveAHAHcBHBHc (KPTMove_t const *const cur_move,
																	const unsigned int update_flags)
{
	/* local variables */
	unsigned int n_id = cur_move->n_id;
	EGlistNode_t *e_it = all_nodes[n_id]->edges->begin;

	/* update flags if necesary */
	if (update_flags && !nodeData[n_id].in_AH)
		nodeData[n_id].added_AH = 1U;
	if (update_flags && !nodeData[n_id].in_BH)
		nodeData[n_id].added_BH = 1U;
	/* flip the node in the handle */
	KPTmoveBH (n_id);
	KPTmoveAH (n_id);
	/* we have to compute every move */
	for (; e_it; e_it = e_it->next)
	{
		/* we just flip AF */
		edgeData[KPTgetEdgeId (e_it)].num.in_AF++;
		/* we just flip BF */
		edgeData[KPTgetEdgeId (e_it)].num.in_BF++;
	}															/* end loop through all incident edges */
	return 0;
}

/* ========================================================================= */
/** @brief This function make the move AHcAH,assuming the move is feasible */
/* ========================================================================= */
#define KPTmakeMoveAHAHcBHcBH(__id,__val) KPTmakeMoveAHAHcBHBHc(__id,__val)

/* ========================================================================= */
/** @brief This function make the move AHcAH,assuming the move is feasible */
/* ========================================================================= */
#define KPTmakeMoveAHcAHBHBHc(__id,__val) KPTmakeMoveAHAHcBHBHc(__id,__val)

/* ========================================================================= */
/** @brief This function make the move AHcAH,assuming the move is feasible */
/* ========================================================================= */
#define KPTmakeMoveAHcAHBHcBH(__id,__val) KPTmakeMoveAHAHcBHBHc(__id,__val)

/* ========================================================================= */
/** @brief This function make the move AcBTc, assuming the move is feasible*/
/* ========================================================================= */
static int KPTmakeMoveAcBTc (const KPTMove_t * const cur_move)
{
	/* local variables */
	unsigned int e_id,
	  no_id,
	  pos,
	  n_id = cur_move->n_id,
	  dom = cur_move->dom;
	EGlistNode_t *e_it = all_nodes[n_id]->edges->begin;

	/* change the node from AcB to Tc */
	EGbitUnset (nodeData[n_id].Ib, dom);
	EGbitUnset (nodeData[n_id].It, dom);
	graphData.n_A[0][dom]--;
	graphData.n_B[1][dom]--;
	graphData.n_Tc[dom]++;
	nodeData[n_id].n_in_B--;
	nodeData[n_id].n_in_T--;
	graphData.n_AdB[0][dom]--;
	/* we have to compute every move */
	for (; e_it; e_it = e_it->next)
	{
		e_id = KPTgetEdgeId (e_it);
		no_id = KPTgetOtherEndId (n_id, e_it);
		pos = (EGbitTest (nodeData[no_id].It, dom) ? 1U : 0U) |
			(EGbitTest (nodeData[no_id].Ia, dom) ? 2U : 0U) |
			(EGbitTest (nodeData[no_id].Ib, dom) ? 4U : 0U);
		switch (pos)
		{
		case 5U:
			/* in this case, the only change is that we have to pay for this edge in
			 * Delta(T) */
			edgeData[e_id].num.n_dT++;
			break;
		case 3U:
			/* in this case we pay one Delta(T), save one E(A:Ac) and flip AF, save 
			 * one E(B:Bc) and flip BF */
			edgeData[e_id].num.n_dT++;
			edgeData[e_id].num.n_AtoAc--;
			edgeData[e_id].num.in_AF++;
			edgeData[e_id].num.n_BtoBc--;
			edgeData[e_id].num.in_BF++;
			break;
		case 1U:
			/* in this case we pay one Delta(T), save one E(B:Bc) and flip BF */
			edgeData[e_id].num.n_dT++;
			edgeData[e_id].num.n_BtoBc--;
			edgeData[e_id].num.in_BF++;
			break;
		case 7U:
			/* in this case we pay one Delta(T), save one E(A:Ac) and flip AF */
			edgeData[e_id].num.n_dT++;
			edgeData[e_id].num.n_AtoAc--;
			edgeData[e_id].num.in_AF++;
			break;
		case 0U:
			/* in this case we save one Delta(T) */
			edgeData[e_id].num.n_dT--;
			break;
		default:
			EXIT (1, "This should not happen, value %u", pos);
		}														/* end switch */
	}															/* end loop through all incident edges */
	return 0;
}

/* ========================================================================= */
/** @brief This function make the move ABcTc, assuming the move is feasible*/
/* ========================================================================= */
static int KPTmakeMoveABcTc (const KPTMove_t * const cur_move)
{
	/* local variables */
	unsigned int e_id,
	  no_id,
	  pos,
	  n_id = cur_move->n_id,
	  dom = cur_move->dom;
	EGlistNode_t *e_it = all_nodes[n_id]->edges->begin;

	/* change the node from ABc to Tc */
	EGbitUnset (nodeData[n_id].Ia, dom);
	EGbitUnset (nodeData[n_id].It, dom);
	graphData.n_A[1][dom]--;
	graphData.n_B[0][dom]--;
	graphData.n_Tc[dom]++;
	nodeData[n_id].n_in_A--;
	nodeData[n_id].n_in_T--;
	graphData.n_AdB[0][dom]--;
	/* we have to compute every move */
	for (; e_it; e_it = e_it->next)
	{
		e_id = KPTgetEdgeId (e_it);
		no_id = KPTgetOtherEndId (n_id, e_it);
		pos = (EGbitTest (nodeData[no_id].It, dom) ? 1U : 0U) |
			(EGbitTest (nodeData[no_id].Ia, dom) ? 2U : 0U) |
			(EGbitTest (nodeData[no_id].Ib, dom) ? 4U : 0U);
		switch (pos)
		{
		case 3U:
			/* in this case, the only change is that we have to pay for this edge in
			 * Delta(T) */
			edgeData[e_id].num.n_dT++;
			break;
		case 5U:
			/* in this case we pay one Delta(T), save one E(A:Ac) and flip AF, save 
			 * one E(B:Bc) and flip BF */
			edgeData[e_id].num.n_dT++;
			edgeData[e_id].num.n_AtoAc--;
			edgeData[e_id].num.in_AF++;
			edgeData[e_id].num.n_BtoBc--;
			edgeData[e_id].num.in_BF++;
			break;
		case 7U:
			/* in this case we pay one Delta(T), save one E(B:Bc) and flip BF */
			edgeData[e_id].num.n_dT++;
			edgeData[e_id].num.n_BtoBc--;
			edgeData[e_id].num.in_BF++;
			break;
		case 1U:
			/* in this case we pay one Delta(T), save one E(A:Ac) and flip AF */
			edgeData[e_id].num.n_dT++;
			edgeData[e_id].num.n_AtoAc--;
			edgeData[e_id].num.in_AF++;
			break;
		case 0U:
			/* in this case we save one Delta(T) */
			edgeData[e_id].num.n_dT--;
			break;
		default:
			EXIT (1, "This should not happen, value %u", pos);
		}														/* end switch */
	}															/* end loop through all incident edges */
	return 0;
}

/* ========================================================================= */
/** @brief This function make the move AcBcTc, assuming the move is feasible*/
/* ========================================================================= */
static int KPTmakeMoveAcBcTc (const KPTMove_t * const cur_move)
{
	/* local variables */
	unsigned int e_id,
	  no_id,
	  pos,
	  n_id = cur_move->n_id,
	  dom = cur_move->dom;
	EGlistNode_t *e_it = all_nodes[n_id]->edges->begin;

	/* change the node from AcBc to Tc */
	EGbitUnset (nodeData[n_id].It, dom);
	graphData.n_A[0][dom]--;
	graphData.n_B[0][dom]--;
	graphData.n_Tc[dom]++;
	nodeData[n_id].n_in_T--;
	graphData.n_AdB[1][dom]--;
	EGbitUnset (nodeData[n_id].Iadb, dom);
	/* we have to compute every move */
	for (; e_it; e_it = e_it->next)
	{
		e_id = KPTgetEdgeId (e_it);
		no_id = KPTgetOtherEndId (n_id, e_it);
		pos = (EGbitTest (nodeData[no_id].It, dom) ? 1U : 0U) |
			(EGbitTest (nodeData[no_id].Ia, dom) ? 2U : 0U) |
			(EGbitTest (nodeData[no_id].Ib, dom) ? 4U : 0U);
		switch (pos)
		{
		case 1U:
			/* in this case, the only change is that we have to pay for this edge in
			 * Delta(T) */
			edgeData[e_id].num.n_dT++;
			break;
		case 7U:
			/* in this case we pay one Delta(T), save one E(A:Ac) and flip AF, save 
			 * one E(B:Bc) and flip BF */
			edgeData[e_id].num.n_dT++;
			edgeData[e_id].num.n_AtoAc--;
			edgeData[e_id].num.in_AF++;
			edgeData[e_id].num.n_BtoBc--;
			edgeData[e_id].num.in_BF++;
			break;
		case 5U:
			/* in this case we pay one Delta(T), save one E(B:Bc) and flip BF */
			edgeData[e_id].num.n_dT++;
			edgeData[e_id].num.n_BtoBc--;
			edgeData[e_id].num.in_BF++;
			break;
		case 3U:
			/* in this case we pay one Delta(T), save one E(A:Ac) and flip AF */
			edgeData[e_id].num.n_dT++;
			edgeData[e_id].num.n_AtoAc--;
			edgeData[e_id].num.in_AF++;
			break;
		case 0U:
			/* in this case we save one Delta(T) */
			edgeData[e_id].num.n_dT--;
			break;
		default:
			EXIT (1, "This should not happen, value %u", pos);
		}														/* end switch */
	}															/* end loop through all incident edges */
	return 0;
}

/* ========================================================================= */
/** @brief This function make the move BBcflipBH, assuming the move is
 * feasible*/
/* ========================================================================= */
static int KPTmakeMoveBBcflipBH (const KPTMove_t * const cur_move,
																 const unsigned int update_flags)
{
	/* local variables */
	unsigned int e_id,
	  no_id,
	  pos,
	  n_id = cur_move->n_id,
	  dom = cur_move->dom;
	EGlistNode_t *e_it = all_nodes[n_id]->edges->begin;

	/* update flags if necesary */
	if (update_flags)
		EGbitSet (nodeData[n_id].flipB, dom);
	/* flip the handle and change the node */
	KPTmoveBH (n_id);
	KPTmoveBBc (n_id, dom);
	KPTmoveAdB (n_id, dom);
	/* we have to compute every move */
	for (; e_it; e_it = e_it->next)
	{
		e_id = KPTgetEdgeId (e_it);
		no_id = KPTgetOtherEndId (n_id, e_it);
		pos = (EGbitTest (nodeData[no_id].It, dom) ? 1U : 0U) |
			(EGbitTest (nodeData[no_id].Ib, dom) ? 2U : 0U);
		switch (pos)
		{
		case 0U:
			/* in this case, change BH and flip BF  */
			edgeData[e_id].num.in_BF++;
			break;
		case 1U:
			/* in this case we save one E(B:Bc) and double flip BF and change BH */
			edgeData[e_id].num.n_BtoBc--;
			break;
		case 3U:
			/* in this case we pay one E(B:Bc) and double flip BF and change BH */
			edgeData[e_id].num.n_BtoBc++;
			break;
		default:
			EXIT (1, "This should not happen, value %u", pos);
		}														/* end switch */
	}															/* end loop through all incident edges */
	return 0;
}

/* ========================================================================= */
/** @brief This function make the move BcBflipBH, assuming the move is 
 * feasible*/
/* ========================================================================= */
static int KPTmakeMoveBcBflipBH (const KPTMove_t * const cur_move,
																 const unsigned int update_flags)
{
	/* local variables */
	unsigned int e_id,
	  no_id,
	  pos,
	  n_id = cur_move->n_id,
	  dom = cur_move->dom;
	EGlistNode_t *e_it = all_nodes[n_id]->edges->begin;

	/* update flags if necesary */
	if (update_flags)
		EGbitSet (nodeData[n_id].flipBc, dom);
	/* flip the handle and change the node */
	KPTmoveBH (n_id);
	KPTmoveBcB (n_id, dom);
	KPTmoveAdB (n_id, dom);
	/* we have to compute every move */
	for (; e_it; e_it = e_it->next)
	{
		e_id = KPTgetEdgeId (e_it);
		no_id = KPTgetOtherEndId (n_id, e_it);
		pos = (EGbitTest (nodeData[no_id].It, dom) ? 1U : 0U) |
			(EGbitTest (nodeData[no_id].Ib, dom) ? 2U : 0U);
		switch (pos)
		{
		case 0U:
			/* in this case, change BH and flip BF  */
			edgeData[e_id].num.in_BF++;
			break;
		case 3U:
			/* in this case we save one E(B:Bc) and double flip BF and change BH */
			edgeData[e_id].num.n_BtoBc--;
			break;
		case 1U:
			/* in this case we pay one E(B:Bc) and double flip BF and change BH */
			edgeData[e_id].num.n_BtoBc++;
			break;
		default:
			EXIT (1, "This should not happen, value %u", pos);
		}														/* end switch */
	}															/* end loop through all incident edges */
	return 0;
}

/* ========================================================================= */
/** @brief This function make the move AAcflipAH, assuming the move is 
 * feasible*/
/* ========================================================================= */
static int KPTmakeMoveAAcflipAH (const KPTMove_t * const cur_move,
																 const unsigned int update_flags)
{
	/* local variables */
	unsigned int e_id,
	  no_id,
	  pos,
	  n_id = cur_move->n_id,
	  dom = cur_move->dom;
	EGlistNode_t *e_it = all_nodes[n_id]->edges->begin;
	/* update flags if necesary */
	if (update_flags)
		EGbitSet (nodeData[n_id].flipA, dom);
	/* flip the handle and change the node */
	KPTmoveAH (n_id);
	KPTmoveAAc (n_id, dom);
	KPTmoveAdB (n_id, dom);
	/* we have to compute every move */
	for (; e_it; e_it = e_it->next)
	{
		e_id = KPTgetEdgeId (e_it);
		no_id = KPTgetOtherEndId (n_id, e_it);
		pos = (EGbitTest (nodeData[no_id].It, dom) ? 1U : 0U) |
			(EGbitTest (nodeData[no_id].Ia, dom) ? 2U : 0U);
		switch (pos)
		{
		case 0U:
			/* in this case, change AH and flip AF  */
			edgeData[e_id].num.in_AF++;
			break;
		case 1U:
			/* in this case we save one E(A:Ac) and double flip AF and change AH */
			edgeData[e_id].num.n_AtoAc--;
			break;
		case 3U:
			/* in this case we pay one E(A:Ac) and double flip AF and change AH */
			edgeData[e_id].num.n_AtoAc++;
			break;
		default:
			EXIT (1, "This should not happen, value %u", pos);
		}														/* end switch */
	}															/* end loop through all incident edges */
	return 0;
}

/* ========================================================================= */
/** @brief This function make the move AcAflipAH, assuming the move is 
 * feasible*/
/* ========================================================================= */
static int KPTmakeMoveAcAflipAH (const KPTMove_t * const cur_move,
																 const unsigned int update_flags)
{
	/* local variables */
	unsigned int e_id,
	  no_id,
	  pos,
	  n_id = cur_move->n_id,
	  dom = cur_move->dom;
	EGlistNode_t *e_it = all_nodes[n_id]->edges->begin;

	/* update flags if necesary */
	if (update_flags)
		EGbitSet (nodeData[n_id].flipAc, dom);
	/* flip the handle and change the node */
	KPTmoveAH (n_id);
	KPTmoveAcA (n_id, dom);
	KPTmoveAdB (n_id, dom);
	/* we have to compute every move */
	for (; e_it; e_it = e_it->next)
	{
		e_id = KPTgetEdgeId (e_it);
		no_id = KPTgetOtherEndId (n_id, e_it);
		pos = (EGbitTest (nodeData[no_id].It, dom) ? 1U : 0U) |
			(EGbitTest (nodeData[no_id].Ia, dom) ? 2U : 0U);
		switch (pos)
		{
		case 0U:
			/* in this case, change AH and flip AF  */
			edgeData[e_id].num.in_AF++;
			break;
		case 3U:
			/* in this case we save one E(A:Ac) and double flip AF and change AH */
			edgeData[e_id].num.n_AtoAc--;
			break;
		case 1U:
			/* in this case we pay one E(A:Ac) and double flip AF and change AH */
			edgeData[e_id].num.n_AtoAc++;
			break;
		default:
			EXIT (1, "This should not happen, value %u", pos);
		}														/* end switch */
	}															/* end loop through all incident edges */
	return 0;
}

/* ========================================================================= */
/** @brief This function make the move BBc, assuming the move is feasible*/
/* ========================================================================= */
static int KPTmakeMoveBBc (const KPTMove_t * const cur_move,
													 const unsigned int update_flags)
{
	/* local variables */
	unsigned int e_id,
	  no_id,
	  pos,
	  n_id = cur_move->n_id,
	  dom = cur_move->dom;
	EGlistNode_t *e_it = all_nodes[n_id]->edges->begin;

	/* update flags if necesary */
	if (update_flags)
		EGbitSet (nodeData[n_id].flipB, dom);
	/* change the node */
	KPTmoveBBc (n_id, dom);
	KPTmoveAdB (n_id, dom);
	/* we have to compute every move */
	for (; e_it; e_it = e_it->next)
	{
		e_id = KPTgetEdgeId (e_it);
		no_id = KPTgetOtherEndId (n_id, e_it);
		pos = (EGbitTest (nodeData[no_id].It, dom) ? 1U : 0U) |
			(EGbitTest (nodeData[no_id].Ib, dom) ? 2U : 0U);
		switch (pos)
		{
		case 0U:
			/* in this case, nothing happen  */
			break;
		case 1U:
			/* in this case we save one E(B:Bc) and flip AF */
			edgeData[e_id].num.in_BF++;
			edgeData[e_id].num.n_BtoBc--;
			break;
		case 3U:
			/* in this case we pay one E(B:Bc) and flip AF */
			edgeData[e_id].num.in_BF++;
			edgeData[e_id].num.n_BtoBc++;
			break;
		default:
			EXIT (1, "This should not happen, value %u", pos);
		}														/* end switch */
	}															/* end loop through all incident edges */
	return 0;
}

/* ========================================================================= */
/** @brief This function make the move BcB, assuming the move is feasible*/
/* ========================================================================= */
static int KPTmakeMoveBcB (const KPTMove_t * const cur_move,
													 const unsigned int update_flags)
{
	/* local variables */
	unsigned int e_id,
	  no_id,
	  pos,
	  n_id = cur_move->n_id,
	  dom = cur_move->dom;
	EGlistNode_t *e_it = all_nodes[n_id]->edges->begin;

	/* update flags if necesary */
	if (update_flags)
		EGbitSet (nodeData[n_id].flipBc, dom);
	/* flip the handle and change the node */
	KPTmoveBcB (n_id, dom);
	KPTmoveAdB (n_id, dom);
	/* we have to compute every move */
	for (; e_it; e_it = e_it->next)
	{
		e_id = KPTgetEdgeId (e_it);
		no_id = KPTgetOtherEndId (n_id, e_it);
		pos = (EGbitTest (nodeData[no_id].It, dom) ? 1U : 0U) |
			(EGbitTest (nodeData[no_id].Ib, dom) ? 2U : 0U);
		switch (pos)
		{
		case 0U:
			/* in this case, nothing happen  */
			break;
		case 3U:
			/* in this case we save one E(B:Bc) and flip AF */
			edgeData[e_id].num.in_BF++;
			edgeData[e_id].num.n_BtoBc--;
			break;
		case 1U:
			/* in this case we pay one E(B:Bc) and flip AF */
			edgeData[e_id].num.in_BF++;
			edgeData[e_id].num.n_BtoBc++;
			break;
		default:
			EXIT (1, "This should not happen, value %u", pos);
		}														/* end switch */
	}															/* end loop through all incident edges */
	return 0;
}


/* ========================================================================= */
/** @brief This function make the move AAc, assuming the move is feasible*/
/* ========================================================================= */
static int KPTmakeMoveAAc (const KPTMove_t * const cur_move,
													 const unsigned int update_flags)
{
	/* local variables */
	unsigned int e_id,
	  no_id,
	  pos,
	  n_id = cur_move->n_id,
	  dom = cur_move->dom;
	EGlistNode_t *e_it = all_nodes[n_id]->edges->begin;
	/* update flags if necesary */
	if (update_flags)
		EGbitSet (nodeData[n_id].flipA, dom);
	/* flip the handle and change the node */
	KPTmoveAAc (n_id, dom);
	KPTmoveAdB (n_id, dom);
	/* we have to compute every move */
	for (; e_it; e_it = e_it->next)
	{
		e_id = KPTgetEdgeId (e_it);
		no_id = KPTgetOtherEndId (n_id, e_it);
		pos = (EGbitTest (nodeData[no_id].It, dom) ? 1U : 0U) |
			(EGbitTest (nodeData[no_id].Ia, dom) ? 2U : 0U);
		switch (pos)
		{
		case 0U:
			/* in this case, nothing happen  */
			break;
		case 1U:
			/* in this case we save one E(A:Ac) and flip AF */
			edgeData[e_id].num.in_AF++;
			edgeData[e_id].num.n_AtoAc--;
			break;
		case 3U:
			/* in this case we pay one E(A:Ac) and flip AF */
			edgeData[e_id].num.in_AF++;
			edgeData[e_id].num.n_AtoAc++;
			break;
		default:
			EXIT (1, "This should not happen, value %u", pos);
		}														/* end switch */
	}															/* end loop through all incident edges */
	return 0;
}

/* ========================================================================= */
/** @brief This function make the move AcA, assuming the move is feasible*/
/* ========================================================================= */
static int KPTmakeMoveAcA (const KPTMove_t * const cur_move,
													 const unsigned int update_flags)
{
	/* local variables */
	unsigned int e_id,
	  no_id,
	  pos,
	  n_id = cur_move->n_id,
	  dom = cur_move->dom;
	EGlistNode_t *e_it = all_nodes[n_id]->edges->begin;

	/* update flags if necesary */
	if (update_flags)
		EGbitSet (nodeData[n_id].flipAc, dom);
	/* flip the handle and change the node */
	KPTmoveAcA (n_id, dom);
	KPTmoveAdB (n_id, dom);
	/* we have to compute every move */
	for (; e_it; e_it = e_it->next)
	{
		e_id = KPTgetEdgeId (e_it);
		no_id = KPTgetOtherEndId (n_id, e_it);
		pos = (EGbitTest (nodeData[no_id].It, dom) ? 1U : 0U) |
			(EGbitTest (nodeData[no_id].Ia, dom) ? 2U : 0U);
		switch (pos)
		{
		case 0U:
			/* in this case, nothing happen  */
			break;
		case 3U:
			/* in this case we save one E(A:Ac) and flip AF */
			edgeData[e_id].num.in_AF++;
			edgeData[e_id].num.n_AtoAc--;
			break;
		case 1U:
			/* in this case we pay one E(A:Ac) and flip AF */
			edgeData[e_id].num.in_AF++;
			edgeData[e_id].num.n_AtoAc++;
			break;
		default:
			EXIT (1, "This should not happen, value %u", pos);
		}														/* end switch */
	}															/* end loop through all incident edges */
	return 0;
}

/* ========================================================================= */
/** @brief This function make the move AcABBc, assuming the move is 
 * feasible*/
/* ========================================================================= */
static int KPTmakeMoveAcABBc (const KPTMove_t * const cur_move,
															const unsigned int update_flags)
{
	/* local variables */
	unsigned int e_id,
	  no_id,
	  pos,
	  n_id = cur_move->n_id,
	  dom = cur_move->dom;
	EGlistNode_t *e_it = all_nodes[n_id]->edges->begin;

	/* update flags if necesary */
	if (update_flags)
	{
		EGbitSet (nodeData[n_id].flipAc, dom);
		EGbitSet (nodeData[n_id].flipB, dom);
	}
	/* flip the handle and change the node */
	KPTmoveAcA (n_id, dom);
	KPTmoveBBc (n_id, dom);

	/* we have to compute every move */
	for (; e_it; e_it = e_it->next)
	{
		e_id = KPTgetEdgeId (e_it);
		no_id = KPTgetOtherEndId (cur_move->n_id, e_it);
		pos = (EGbitTest (nodeData[no_id].It, dom) ? 1U : 0U) |
			(EGbitTest (nodeData[no_id].Ia, dom) ? 2U : 0U) |
			(EGbitTest (nodeData[no_id].Ib, dom) ? 4U : 0U);
		switch (pos)
		{
		case 0U:
			/* in this case, nothing happen */
			break;
		case 3U:
			/* in this case we save one E(A:Ac) and flip AF, save 
			 * one E(B:Bc) and flip BF */
			edgeData[e_id].num.in_AF++;
			edgeData[e_id].num.in_BF++;
			edgeData[e_id].num.n_AtoAc--;
			edgeData[e_id].num.n_BtoBc--;
			break;
		case 1U:
			/* in this case we pay one E(A:Ac) and flip AF, save 
			 * one E(B:Bc) and flip BF */
			edgeData[e_id].num.in_AF++;
			edgeData[e_id].num.in_BF++;
			edgeData[e_id].num.n_AtoAc++;
			edgeData[e_id].num.n_BtoBc--;
			break;
		case 7U:
			/* in this case we save one E(A:Ac) and flip AF, pay 
			 * one E(B:Bc) and flip BF */
			edgeData[e_id].num.in_AF++;
			edgeData[e_id].num.in_BF++;
			edgeData[e_id].num.n_AtoAc--;
			edgeData[e_id].num.n_BtoBc++;
			break;
		case 5U:
			/* in this case we pay one E(A:Ac) and flip AF, pay 
			 * one E(B:Bc) and flip BF */
			edgeData[e_id].num.in_AF++;
			edgeData[e_id].num.in_BF++;
			edgeData[e_id].num.n_AtoAc++;
			edgeData[e_id].num.n_BtoBc++;
			break;
		default:
			EXIT (1, "This should not happen, value %u", pos);
		}														/* end switch */
	}															/* end loop through all incident edges */
	return 0;
}

/* ========================================================================= */
/** @brief This function makeMoves the move AAcBcB, assuming the move is 
 * feasible*/
/* ========================================================================= */
static int KPTmakeMoveAAcBcB (const KPTMove_t * const cur_move,
															const unsigned int update_flags)
{
	/* local variables */
	unsigned int e_id,
	  no_id,
	  pos,
	  n_id = cur_move->n_id,
	  dom = cur_move->dom;
	EGlistNode_t *e_it = all_nodes[n_id]->edges->begin;

	/* update flags if necesary */
	if (update_flags)
	{
		EGbitSet (nodeData[n_id].flipA, dom);
		EGbitSet (nodeData[n_id].flipBc, dom);
	}
	/* flip the handle and change the node */
	KPTmoveAAc (n_id, dom);
	KPTmoveBcB (n_id, dom);

	/* we have to compute every move */
	for (; e_it; e_it = e_it->next)
	{
		e_id = KPTgetEdgeId (e_it);
		no_id = KPTgetOtherEndId (n_id, e_it);
		pos = (EGbitTest (nodeData[no_id].It, dom) ? 1U : 0U) |
			(EGbitTest (nodeData[no_id].Ia, dom) ? 2U : 0U) |
			(EGbitTest (nodeData[no_id].Ib, dom) ? 4U : 0U);
		switch (pos)
		{
		case 0U:
			/* in this case, nothing happen */
			break;
		case 5U:
			/* in this case we save one E(A:Ac) and flip AF, save 
			 * one E(B:Bc) and flip BF */
			edgeData[e_id].num.in_AF++;
			edgeData[e_id].num.in_BF++;
			edgeData[e_id].num.n_AtoAc--;
			edgeData[e_id].num.n_BtoBc--;
			break;
		case 7U:
			/* in this case we pay one E(A:Ac) and flip AF, save 
			 * one E(B:Bc) and flip BF */
			edgeData[e_id].num.in_AF++;
			edgeData[e_id].num.in_BF++;
			edgeData[e_id].num.n_AtoAc++;
			edgeData[e_id].num.n_BtoBc--;
			break;
		case 1U:
			/* in this case we save one E(A:Ac) and flip AF, pay 
			 * one E(B:Bc) and flip BF */
			edgeData[e_id].num.in_AF++;
			edgeData[e_id].num.in_BF++;
			edgeData[e_id].num.n_AtoAc--;
			edgeData[e_id].num.n_BtoBc++;
			break;
		case 3U:
			/* in this case we pay one E(A:Ac) and flip AF, pay 
			 * one E(B:Bc) and flip BF */
			edgeData[e_id].num.in_AF++;
			edgeData[e_id].num.in_BF++;
			edgeData[e_id].num.n_AtoAc++;
			edgeData[e_id].num.n_BtoBc++;
			break;
		default:
			EXIT (1, "This should not happen, value %u", pos);
		}														/* end switch */
	}															/* end loop through all incident edges */
	return 0;
}

/* ========================================================================= */
/** @brief This function makeMoves the move AcABcB, assuming the move is 
 * feasible*/
/* ========================================================================= */
static int KPTmakeMoveAcABcB (const KPTMove_t * const cur_move,
															const unsigned int update_flags)
{
	/* local variables */
	unsigned int e_id,
	  no_id,
	  pos,
	  n_id = cur_move->n_id,
	  dom = cur_move->dom;
	EGlistNode_t *e_it = all_nodes[n_id]->edges->begin;

	/* update flags if necesary */
	if (update_flags)
	{
		EGbitSet (nodeData[n_id].flipAc, dom);
		EGbitSet (nodeData[n_id].flipBc, dom);
	}
	/* flip the handle and change the node */
	KPTmoveAcA (n_id, dom);
	KPTmoveBcB (n_id, dom);

	/* we have to compute every move */
	for (; e_it; e_it = e_it->next)
	{
		e_id = KPTgetEdgeId (e_it);
		no_id = KPTgetOtherEndId (n_id, e_it);
		pos = (EGbitTest (nodeData[no_id].It, dom) ? 1U : 0U) |
			(EGbitTest (nodeData[no_id].Ia, dom) ? 2U : 0U) |
			(EGbitTest (nodeData[no_id].Ib, dom) ? 4U : 0U);
		switch (pos)
		{
		case 0U:
			/* in this case, nothing happen */
			break;
		case 7U:
			/* in this case we save one E(A:Ac) and flip AF, save 
			 * one E(B:Bc) and flip BF */
			edgeData[e_id].num.in_AF++;
			edgeData[e_id].num.in_BF++;
			edgeData[e_id].num.n_AtoAc--;
			edgeData[e_id].num.n_BtoBc--;
			break;
		case 5U:
			/* in this case we pay one E(A:Ac) and flip AF, save 
			 * one E(B:Bc) and flip BF */
			edgeData[e_id].num.in_AF++;
			edgeData[e_id].num.in_BF++;
			edgeData[e_id].num.n_AtoAc++;
			edgeData[e_id].num.n_BtoBc--;
			break;
		case 3U:
			/* in this case we save one E(A:Ac) and flip AF, pay 
			 * one E(B:Bc) and flip BF */
			edgeData[e_id].num.in_AF++;
			edgeData[e_id].num.in_BF++;
			edgeData[e_id].num.n_AtoAc--;
			edgeData[e_id].num.n_BtoBc++;
			break;
		case 1U:
			/* in this case we pay one E(A:Ac) and flip AF, pay 
			 * one E(B:Bc) and flip BF */
			edgeData[e_id].num.in_AF++;
			edgeData[e_id].num.in_BF++;
			edgeData[e_id].num.n_AtoAc++;
			edgeData[e_id].num.n_BtoBc++;
			break;
		default:
			EXIT (1, "This should not happen, value %u", pos);
		}														/* end switch */
	}															/* end loop through all incident edges */
	return 0;
}

/* ========================================================================= */
/** @brief This function makeMoves the move AAcBBc, assuming the move is 
 * feasible*/
/* ========================================================================= */
static int KPTmakeMoveAAcBBc (const KPTMove_t * const cur_move,
															const unsigned int update_flags)
{
	/* local variables */
	unsigned int e_id,
	  no_id,
	  pos,
	  n_id = cur_move->n_id,
	  dom = cur_move->dom;
	EGlistNode_t *e_it = all_nodes[n_id]->edges->begin;

	/* update flags if necesary */
	if (update_flags)
	{
		EGbitSet (nodeData[n_id].flipA, dom);
		EGbitSet (nodeData[n_id].flipB, dom);
	}
	/* flip the handle and change the node */
	KPTmoveAAc (n_id, dom);
	KPTmoveBBc (n_id, dom);

	/* we have to compute every move */
	for (; e_it; e_it = e_it->next)
	{
		e_id = KPTgetEdgeId (e_it);
		no_id = KPTgetOtherEndId (n_id, e_it);
		pos = (EGbitTest (nodeData[no_id].It, dom) ? 1U : 0U) |
			(EGbitTest (nodeData[no_id].Ia, dom) ? 2U : 0U) |
			(EGbitTest (nodeData[no_id].Ib, dom) ? 4U : 0U);
		switch (pos)
		{
		case 0U:
			/* in this case, nothing happen */
			break;
		case 1U:
			/* in this case we save one E(A:Ac) and flip AF, save 
			 * one E(B:Bc) and flip BF */
			edgeData[e_id].num.in_AF++;
			edgeData[e_id].num.in_BF++;
			edgeData[e_id].num.n_AtoAc--;
			edgeData[e_id].num.n_BtoBc--;
			break;
		case 3U:
			/* in this case we pay one E(A:Ac) and flip AF, save 
			 * one E(B:Bc) and flip BF */
			edgeData[e_id].num.in_AF++;
			edgeData[e_id].num.in_BF++;
			edgeData[e_id].num.n_AtoAc++;
			edgeData[e_id].num.n_BtoBc--;
			break;
		case 5U:
			/* in this case we save one E(A:Ac) and flip AF, pay 
			 * one E(B:Bc) and flip BF */
			edgeData[e_id].num.in_AF++;
			edgeData[e_id].num.in_BF++;
			edgeData[e_id].num.n_AtoAc--;
			edgeData[e_id].num.n_BtoBc++;
			break;
		case 7U:
			/* in this case we pay one E(A:Ac) and flip AF, pay 
			 * one E(B:Bc) and flip BF */
			edgeData[e_id].num.in_AF++;
			edgeData[e_id].num.in_BF++;
			edgeData[e_id].num.n_AtoAc++;
			edgeData[e_id].num.n_BtoBc++;
			break;
		default:
			EXIT (1, "This should not happen, value %u", pos);
		}														/* end switch */
	}															/* end loop through all incident edges */
	return 0;
}

/* ========================================================================= */
/** @brief This function make the move AcABBcflipAH, assuming the move is 
 * feasible*/
/* ========================================================================= */
static int KPTmakeMoveAcABBcflipAH (const KPTMove_t * const cur_move,
																		const unsigned int update_flags)
{
	/* local variables */
	unsigned int e_id,
	  no_id,
	  pos,
	  n_id = cur_move->n_id,
	  dom = cur_move->dom;
	EGlistNode_t *e_it = all_nodes[n_id]->edges->begin;

	/* update flags if necesary */
	if (update_flags)
	{
		EGbitSet (nodeData[n_id].flipAc, dom);
		EGbitSet (nodeData[n_id].flipB, dom);
	}
	/* flip the handle and change the node */
	KPTmoveAcA (n_id, dom);
	KPTmoveBBc (n_id, dom);
	KPTmoveAH (n_id);

	/* we have to compute every move */
	for (; e_it; e_it = e_it->next)
	{
		e_id = KPTgetEdgeId (e_it);
		no_id = KPTgetOtherEndId (cur_move->n_id, e_it);
		pos = (EGbitTest (nodeData[no_id].It, dom) ? 1U : 0U) |
			(EGbitTest (nodeData[no_id].Ia, dom) ? 2U : 0U) |
			(EGbitTest (nodeData[no_id].Ib, dom) ? 4U : 0U);
		switch (pos)
		{
		case 0U:
			/* in this case, flip AH */
			edgeData[e_id].num.in_AF++;
			break;
		case 3U:
			/* in this case we save one E(A:Ac) and double flip AF, save 
			 * one E(B:Bc) and flip BF */
			edgeData[e_id].num.in_BF++;
			edgeData[e_id].num.n_AtoAc--;
			edgeData[e_id].num.n_BtoBc--;
			break;
		case 1U:
			/* in this case we pay one E(A:Ac) and double flip AF, save 
			 * one E(B:Bc) and flip BF */
			edgeData[e_id].num.in_BF++;
			edgeData[e_id].num.n_AtoAc++;
			edgeData[e_id].num.n_BtoBc--;
			break;
		case 7U:
			/* in this case we save one E(A:Ac) and double flip AF, pay 
			 * one E(B:Bc) and flip BF */
			edgeData[e_id].num.in_BF++;
			edgeData[e_id].num.n_AtoAc--;
			edgeData[e_id].num.n_BtoBc++;
			break;
		case 5U:
			/* in this case we pay one E(A:Ac) and double flip AF, pay 
			 * one E(B:Bc) and flip BF */
			edgeData[e_id].num.in_BF++;
			edgeData[e_id].num.n_AtoAc++;
			edgeData[e_id].num.n_BtoBc++;
			break;
		default:
			EXIT (1, "This should not happen, value %u", pos);
		}														/* end switch */
	}															/* end loop through all incident edges */
	return 0;
}

/* ========================================================================= */
/** @brief This function makeMoves the move AAcBcBflipAH, assuming the move is 
 * feasible*/
/* ========================================================================= */
static int KPTmakeMoveAAcBcBflipAH (const KPTMove_t * const cur_move,
																		const unsigned int update_flags)
{
	/* local variables */
	unsigned int e_id,
	  no_id,
	  pos,
	  n_id = cur_move->n_id,
	  dom = cur_move->dom;
	EGlistNode_t *e_it = all_nodes[n_id]->edges->begin;

	/* update flags if necesary */
	if (update_flags)
	{
		EGbitSet (nodeData[n_id].flipA, dom);
		EGbitSet (nodeData[n_id].flipBc, dom);
	}
	/* flip the handle and change the node */
	KPTmoveAAc (n_id, dom);
	KPTmoveBcB (n_id, dom);
	KPTmoveAH (n_id);

	/* we have to compute every move */
	for (; e_it; e_it = e_it->next)
	{
		e_id = KPTgetEdgeId (e_it);
		no_id = KPTgetOtherEndId (n_id, e_it);
		pos = (EGbitTest (nodeData[no_id].It, dom) ? 1U : 0U) |
			(EGbitTest (nodeData[no_id].Ia, dom) ? 2U : 0U) |
			(EGbitTest (nodeData[no_id].Ib, dom) ? 4U : 0U);
		switch (pos)
		{
		case 0U:
			/* in this case, flip AH */
			edgeData[e_id].num.in_AF++;
			break;
		case 5U:
			/* in this case we save one E(A:Ac) and double flip AF, save 
			 * one E(B:Bc) and flip BF */
			edgeData[e_id].num.in_BF++;
			edgeData[e_id].num.n_AtoAc--;
			edgeData[e_id].num.n_BtoBc--;
			break;
		case 7U:
			/* in this case we pay one E(A:Ac) and double flip AF, save 
			 * one E(B:Bc) and flip BF */
			edgeData[e_id].num.in_BF++;
			edgeData[e_id].num.n_AtoAc++;
			edgeData[e_id].num.n_BtoBc--;
			break;
		case 1U:
			/* in this case we save one E(A:Ac) and double flip AF, pay 
			 * one E(B:Bc) and flip BF */
			edgeData[e_id].num.in_BF++;
			edgeData[e_id].num.n_AtoAc--;
			edgeData[e_id].num.n_BtoBc++;
			break;
		case 3U:
			/* in this case we pay one E(A:Ac) and double flip AF, pay 
			 * one E(B:Bc) and flip BF */
			edgeData[e_id].num.in_BF++;
			edgeData[e_id].num.n_AtoAc++;
			edgeData[e_id].num.n_BtoBc++;
			break;
		default:
			EXIT (1, "This should not happen, value %u", pos);
		}														/* end switch */
	}															/* end loop through all incident edges */
	return 0;
}

/* ========================================================================= */
/** @brief This function makeMoves the move AcABcBflipAH, assuming the move is 
 * feasible*/
/* ========================================================================= */
static int KPTmakeMoveAcABcBflipAH (const KPTMove_t * const cur_move,
																		const unsigned int update_flags)
{
	/* local variables */
	unsigned int e_id,
	  no_id,
	  pos,
	  n_id = cur_move->n_id,
	  dom = cur_move->dom;
	EGlistNode_t *e_it = all_nodes[n_id]->edges->begin;

	/* update flags if necesary */
	if (update_flags)
	{
		EGbitSet (nodeData[n_id].flipAc, dom);
		EGbitSet (nodeData[n_id].flipBc, dom);
	}
	/* flip the handle and change the node */
	KPTmoveAcA (n_id, dom);
	KPTmoveBcB (n_id, dom);
	KPTmoveAH (n_id);

	/* we have to compute every move */
	for (; e_it; e_it = e_it->next)
	{
		e_id = KPTgetEdgeId (e_it);
		no_id = KPTgetOtherEndId (n_id, e_it);
		pos = (EGbitTest (nodeData[no_id].It, dom) ? 1U : 0U) |
			(EGbitTest (nodeData[no_id].Ia, dom) ? 2U : 0U) |
			(EGbitTest (nodeData[no_id].Ib, dom) ? 4U : 0U);
		switch (pos)
		{
		case 0U:
			/* in this case, flip AH */
			edgeData[e_id].num.in_AF++;
			break;
		case 7U:
			/* in this case we save one E(A:Ac) and double flip AF, save 
			 * one E(B:Bc) and flip BF */
			edgeData[e_id].num.in_BF++;
			edgeData[e_id].num.n_AtoAc--;
			edgeData[e_id].num.n_BtoBc--;
			break;
		case 5U:
			/* in this case we pay one E(A:Ac) and double flip AF, save 
			 * one E(B:Bc) and flip BF */
			edgeData[e_id].num.in_BF++;
			edgeData[e_id].num.n_AtoAc++;
			edgeData[e_id].num.n_BtoBc--;
			break;
		case 3U:
			/* in this case we save one E(A:Ac) and double flip AF, pay 
			 * one E(B:Bc) and flip BF */
			edgeData[e_id].num.in_BF++;
			edgeData[e_id].num.n_AtoAc--;
			edgeData[e_id].num.n_BtoBc++;
			break;
		case 1U:
			/* in this case we pay one E(A:Ac) and double flip AF, pay 
			 * one E(B:Bc) and flip BF */
			edgeData[e_id].num.in_BF++;
			edgeData[e_id].num.n_AtoAc++;
			edgeData[e_id].num.n_BtoBc++;
			break;
		default:
			EXIT (1, "This should not happen, value %u", pos);
		}														/* end switch */
	}															/* end loop through all incident edges */
	return 0;
}

/* ========================================================================= */
/** @brief This function makeMoves the move AAcBBcflipAH, assuming the move is 
 * feasible*/
/* ========================================================================= */
static int KPTmakeMoveAAcBBcflipAH (const KPTMove_t * const cur_move,
																		const unsigned int update_flags)
{
	/* local variables */
	unsigned int e_id,
	  no_id,
	  pos,
	  n_id = cur_move->n_id,
	  dom = cur_move->dom;
	EGlistNode_t *e_it = all_nodes[n_id]->edges->begin;

	/* update flags if necesary */
	if (update_flags)
	{
		EGbitSet (nodeData[n_id].flipA, dom);
		EGbitSet (nodeData[n_id].flipB, dom);
	}
	/* flip the handle and change the node */
	KPTmoveAAc (n_id, dom);
	KPTmoveBBc (n_id, dom);
	KPTmoveAH (n_id);

	/* we have to compute every move */
	for (; e_it; e_it = e_it->next)
	{
		e_id = KPTgetEdgeId (e_it);
		no_id = KPTgetOtherEndId (n_id, e_it);
		pos = (EGbitTest (nodeData[no_id].It, dom) ? 1U : 0U) |
			(EGbitTest (nodeData[no_id].Ia, dom) ? 2U : 0U) |
			(EGbitTest (nodeData[no_id].Ib, dom) ? 4U : 0U);
		switch (pos)
		{
		case 0U:
			/* in this case, flip AH */
			edgeData[e_id].num.in_AF++;
			break;
		case 1U:
			/* in this case we save one E(A:Ac) and double flip AF, save 
			 * one E(B:Bc) and flip BF */
			edgeData[e_id].num.in_BF++;
			edgeData[e_id].num.n_AtoAc--;
			edgeData[e_id].num.n_BtoBc--;
			break;
		case 3U:
			/* in this case we pay one E(A:Ac) and double flip AF, save 
			 * one E(B:Bc) and flip BF */
			edgeData[e_id].num.in_BF++;
			edgeData[e_id].num.n_AtoAc++;
			edgeData[e_id].num.n_BtoBc--;
			break;
		case 5U:
			/* in this case we save one E(A:Ac) and double flip AF, pay 
			 * one E(B:Bc) and flip BF */
			edgeData[e_id].num.in_BF++;
			edgeData[e_id].num.n_AtoAc--;
			edgeData[e_id].num.n_BtoBc++;
			break;
		case 7U:
			/* in this case we pay one E(A:Ac) and double flip AF, pay 
			 * one E(B:Bc) and flip BF */
			edgeData[e_id].num.in_BF++;
			edgeData[e_id].num.n_AtoAc++;
			edgeData[e_id].num.n_BtoBc++;
			break;
		default:
			EXIT (1, "This should not happen, value %u", pos);
		}														/* end switch */
	}															/* end loop through all incident edges */
	return 0;
}

/* ========================================================================= */
/** @brief This function make the move AcABBcflipBH, assuming the move is 
 * feasible*/
/* ========================================================================= */
static int KPTmakeMoveAcABBcflipBH (const KPTMove_t * const cur_move,
																		const unsigned int update_flags)
{
	/* local variables */
	unsigned int e_id,
	  no_id,
	  pos,
	  n_id = cur_move->n_id,
	  dom = cur_move->dom;
	EGlistNode_t *e_it = all_nodes[n_id]->edges->begin;

	/* update flags if necesary */
	if (update_flags)
	{
		EGbitSet (nodeData[n_id].flipAc, dom);
		EGbitSet (nodeData[n_id].flipB, dom);
	}
	/* flip the handle and change the node */
	KPTmoveAcA (n_id, dom);
	KPTmoveBBc (n_id, dom);
	KPTmoveBH (n_id);

	/* we have to compute every move */
	for (; e_it; e_it = e_it->next)
	{
		e_id = KPTgetEdgeId (e_it);
		no_id = KPTgetOtherEndId (cur_move->n_id, e_it);
		pos = (EGbitTest (nodeData[no_id].It, dom) ? 1U : 0U) |
			(EGbitTest (nodeData[no_id].Ia, dom) ? 2U : 0U) |
			(EGbitTest (nodeData[no_id].Ib, dom) ? 4U : 0U);
		switch (pos)
		{
		case 0U:
			/* in this case, flip BH */
			edgeData[e_id].num.in_BF++;
			break;
		case 3U:
			/* in this case we save one E(A:Ac) and flip AF, save 
			 * one E(B:Bc) and double flip BF */
			edgeData[e_id].num.in_AF++;
			edgeData[e_id].num.n_AtoAc--;
			edgeData[e_id].num.n_BtoBc--;
			break;
		case 1U:
			/* in this case we pay one E(A:Ac) and flip AF, save 
			 * one E(B:Bc) and double flip BF */
			edgeData[e_id].num.in_AF++;
			edgeData[e_id].num.n_AtoAc++;
			edgeData[e_id].num.n_BtoBc--;
			break;
		case 7U:
			/* in this case we save one E(A:Ac) and flip AF, pay 
			 * one E(B:Bc) and double flip BF */
			edgeData[e_id].num.in_AF++;
			edgeData[e_id].num.n_AtoAc--;
			edgeData[e_id].num.n_BtoBc++;
			break;
		case 5U:
			/* in this case we pay one E(A:Ac) and flip AF, pay 
			 * one E(B:Bc) and double flip BF */
			edgeData[e_id].num.in_AF++;
			edgeData[e_id].num.n_AtoAc++;
			edgeData[e_id].num.n_BtoBc++;
			break;
		default:
			EXIT (1, "This should not happen, value %u", pos);
		}														/* end switch */
	}															/* end loop through all incident edges */
	return 0;
}

/* ========================================================================= */
/** @brief This function makeMoves the move AAcBcBflipBH, assuming the move is 
 * feasible*/
/* ========================================================================= */
static int KPTmakeMoveAAcBcBflipBH (const KPTMove_t * const cur_move,
																		const unsigned int update_flags)
{
	/* local variables */
	unsigned int e_id,
	  no_id,
	  pos,
	  n_id = cur_move->n_id,
	  dom = cur_move->dom;
	EGlistNode_t *e_it = all_nodes[n_id]->edges->begin;

	/* update flags if necesary */
	if (update_flags)
	{
		EGbitSet (nodeData[n_id].flipA, dom);
		EGbitSet (nodeData[n_id].flipBc, dom);
	}
	/* flip the handle and change the node */
	KPTmoveAAc (n_id, dom);
	KPTmoveBcB (n_id, dom);
	KPTmoveBH (n_id);

	/* we have to compute every move */
	for (; e_it; e_it = e_it->next)
	{
		e_id = KPTgetEdgeId (e_it);
		no_id = KPTgetOtherEndId (n_id, e_it);
		pos = (EGbitTest (nodeData[no_id].It, dom) ? 1U : 0U) |
			(EGbitTest (nodeData[no_id].Ia, dom) ? 2U : 0U) |
			(EGbitTest (nodeData[no_id].Ib, dom) ? 4U : 0U);
		switch (pos)
		{
		case 0U:
			/* in this case, flip BH */
			edgeData[e_id].num.in_BF++;
			break;
		case 5U:
			/* in this case we save one E(A:Ac) and flip AF, save 
			 * one E(B:Bc) and double flip BF */
			edgeData[e_id].num.in_AF++;
			edgeData[e_id].num.n_AtoAc--;
			edgeData[e_id].num.n_BtoBc--;
			break;
		case 7U:
			/* in this case we pay one E(A:Ac) and flip AF, save 
			 * one E(B:Bc) and double flip BF */
			edgeData[e_id].num.in_AF++;
			edgeData[e_id].num.n_AtoAc++;
			edgeData[e_id].num.n_BtoBc--;
			break;
		case 1U:
			/* in this case we save one E(A:Ac) and flip AF, pay 
			 * one E(B:Bc) and double flip BF */
			edgeData[e_id].num.in_AF++;
			edgeData[e_id].num.n_AtoAc--;
			edgeData[e_id].num.n_BtoBc++;
			break;
		case 3U:
			/* in this case we pay one E(A:Ac) and flip AF, pay 
			 * one E(B:Bc) and double flip BF */
			edgeData[e_id].num.in_AF++;
			edgeData[e_id].num.n_AtoAc++;
			edgeData[e_id].num.n_BtoBc++;
			break;
		default:
			EXIT (1, "This should not happen, value %u", pos);
		}														/* end switch */
	}															/* end loop through all incident edges */
	return 0;
}

/* ========================================================================= */
/** @brief This function makeMoves the move AcABcBflipBH, assuming the move is 
 * feasible*/
/* ========================================================================= */
static int KPTmakeMoveAcABcBflipBH (const KPTMove_t * const cur_move,
																		const unsigned int update_flags)
{
	/* local variables */
	unsigned int e_id,
	  no_id,
	  pos,
	  n_id = cur_move->n_id,
	  dom = cur_move->dom;
	EGlistNode_t *e_it = all_nodes[n_id]->edges->begin;

	/* update flags if necesary */
	if (update_flags)
	{
		EGbitSet (nodeData[n_id].flipAc, dom);
		EGbitSet (nodeData[n_id].flipBc, dom);
	}
	/* flip the handle and change the node */
	KPTmoveAcA (n_id, dom);
	KPTmoveBcB (n_id, dom);
	KPTmoveBH (n_id);

	/* we have to compute every move */
	for (; e_it; e_it = e_it->next)
	{
		e_id = KPTgetEdgeId (e_it);
		no_id = KPTgetOtherEndId (n_id, e_it);
		pos = (EGbitTest (nodeData[no_id].It, dom) ? 1U : 0U) |
			(EGbitTest (nodeData[no_id].Ia, dom) ? 2U : 0U) |
			(EGbitTest (nodeData[no_id].Ib, dom) ? 4U : 0U);
		switch (pos)
		{
		case 0U:
			/* in this case, flip BH */
			edgeData[e_id].num.in_BF++;
			break;
		case 7U:
			/* in this case we save one E(A:Ac) and double flip BF, save 
			 * one E(B:Bc) and flip AF */
			edgeData[e_id].num.in_AF++;
			edgeData[e_id].num.n_AtoAc--;
			edgeData[e_id].num.n_BtoBc--;
			break;
		case 5U:
			/* in this case we pay one E(A:Ac) and double flip BF, save 
			 * one E(B:Bc) and flip AF */
			edgeData[e_id].num.in_AF++;
			edgeData[e_id].num.n_AtoAc++;
			edgeData[e_id].num.n_BtoBc--;
			break;
		case 3U:
			/* in this case we save one E(A:Ac) and double flip BF, pay 
			 * one E(B:Bc) and flip AF */
			edgeData[e_id].num.in_AF++;
			edgeData[e_id].num.n_AtoAc--;
			edgeData[e_id].num.n_BtoBc++;
			break;
		case 1U:
			/* in this case we pay one E(A:Ac) and double flip BF, pay 
			 * one E(B:Bc) and flip AF */
			edgeData[e_id].num.in_AF++;
			edgeData[e_id].num.n_AtoAc++;
			edgeData[e_id].num.n_BtoBc++;
			break;
		default:
			EXIT (1, "This should not happen, value %u", pos);
		}														/* end switch */
	}															/* end loop through all incident edges */
	return 0;
}

/* ========================================================================= */
/** @brief This function makeMoves the move AAcBBcflipBH, assuming the move is 
 * feasible*/
/* ========================================================================= */
static int KPTmakeMoveAAcBBcflipBH (const KPTMove_t * const cur_move,
																		const unsigned int update_flags)
{
	/* local variables */
	unsigned int e_id,
	  no_id,
	  pos,
	  n_id = cur_move->n_id,
	  dom = cur_move->dom;
	EGlistNode_t *e_it = all_nodes[n_id]->edges->begin;

	/* update flags if necesary */
	if (update_flags)
	{
		EGbitSet (nodeData[n_id].flipA, dom);
		EGbitSet (nodeData[n_id].flipB, dom);
	}
	/* flip the handle and change the node */
	KPTmoveAAc (n_id, dom);
	KPTmoveBBc (n_id, dom);
	KPTmoveBH (n_id);

	/* we have to compute every move */
	for (; e_it; e_it = e_it->next)
	{
		e_id = KPTgetEdgeId (e_it);
		no_id = KPTgetOtherEndId (n_id, e_it);
		pos = (EGbitTest (nodeData[no_id].It, dom) ? 1U : 0U) |
			(EGbitTest (nodeData[no_id].Ia, dom) ? 2U : 0U) |
			(EGbitTest (nodeData[no_id].Ib, dom) ? 4U : 0U);
		switch (pos)
		{
		case 0U:
			/* in this case, flip BH */
			edgeData[e_id].num.in_BF++;
			break;
		case 1U:
			/* in this case we save one E(A:Ac) and double flip BF, save 
			 * one E(B:Bc) and flip AF */
			edgeData[e_id].num.in_AF++;
			edgeData[e_id].num.n_AtoAc--;
			edgeData[e_id].num.n_BtoBc--;
			break;
		case 3U:
			/* in this case we pay one E(A:Ac) and double flip BF, save 
			 * one E(B:Bc) and flip AF */
			edgeData[e_id].num.in_AF++;
			edgeData[e_id].num.n_AtoAc++;
			edgeData[e_id].num.n_BtoBc--;
			break;
		case 5U:
			/* in this case we save one E(A:Ac) and double flip BF, pay 
			 * one E(B:Bc) and flip AF */
			edgeData[e_id].num.in_AF++;
			edgeData[e_id].num.n_AtoAc--;
			edgeData[e_id].num.n_BtoBc++;
			break;
		case 7U:
			/* in this case we pay one E(A:Ac) and double flip BF, pay 
			 * one E(B:Bc) and flip AF */
			edgeData[e_id].num.in_AF++;
			edgeData[e_id].num.n_AtoAc++;
			edgeData[e_id].num.n_BtoBc++;
			break;
		default:
			EXIT (1, "This should not happen, value %u", pos);
		}														/* end switch */
	}															/* end loop through all incident edges */
	return 0;
}



/* ========================================================================= */
/** @brief This function make the move AcABBcflipAHBH, assuming the move is 
 * feasible*/
/* ========================================================================= */
static int KPTmakeMoveAcABBcflipAHBH (const KPTMove_t * const cur_move,
																			const unsigned int update_flags)
{
	/* local variables */
	unsigned int e_id,
	  no_id,
	  pos,
	  n_id = cur_move->n_id,
	  dom = cur_move->dom;
	EGlistNode_t *e_it = all_nodes[n_id]->edges->begin;

	/* update flags if necesary */
	if (update_flags)
	{
		EGbitSet (nodeData[n_id].flipAc, dom);
		EGbitSet (nodeData[n_id].flipB, dom);
	}
	/* flip the handle and change the node */
	KPTmoveAcA (n_id, dom);
	KPTmoveBBc (n_id, dom);
	KPTmoveAH (n_id);
	KPTmoveBH (n_id);

	/* we have to compute every move */
	for (; e_it; e_it = e_it->next)
	{
		e_id = KPTgetEdgeId (e_it);
		no_id = KPTgetOtherEndId (cur_move->n_id, e_it);
		pos = (EGbitTest (nodeData[no_id].It, dom) ? 1U : 0U) |
			(EGbitTest (nodeData[no_id].Ia, dom) ? 2U : 0U) |
			(EGbitTest (nodeData[no_id].Ib, dom) ? 4U : 0U);
		switch (pos)
		{
		case 0U:
			/* in this case, flip AF and flip BF */
			edgeData[e_id].num.in_AF++;
			edgeData[e_id].num.in_BF++;
			break;
		case 3U:
			/* in this case we save one E(A:Ac) and double flip AF, save 
			 * one E(B:Bc) and double flip BF */
			edgeData[e_id].num.n_AtoAc--;
			edgeData[e_id].num.n_BtoBc--;
			break;
		case 1U:
			/* in this case we pay one E(A:Ac) and double flip AF, save 
			 * one E(B:Bc) and double flip BF */
			edgeData[e_id].num.n_AtoAc++;
			edgeData[e_id].num.n_BtoBc--;
			break;
		case 7U:
			/* in this case we save one E(A:Ac) and double flip AF, pay 
			 * one E(B:Bc) and double flip BF */
			edgeData[e_id].num.n_AtoAc--;
			edgeData[e_id].num.n_BtoBc++;
			break;
		case 5U:
			/* in this case we pay one E(A:Ac) and double flip AF, pay 
			 * one E(B:Bc) and double flip BF */
			edgeData[e_id].num.n_AtoAc++;
			edgeData[e_id].num.n_BtoBc++;
			break;
		default:
			EXIT (1, "This should not happen, value %u", pos);
		}														/* end switch */
	}															/* end loop through all incident edges */
	return 0;
}

/* ========================================================================= */
/** @brief This function makeMoves the move AAcBcBflipAHBH, assuming the move
 * is feasible*/
/* ========================================================================= */
static int KPTmakeMoveAAcBcBflipAHBH (const KPTMove_t * const cur_move,
																			const unsigned int update_flags)
{
	/* local variables */
	unsigned int e_id,
	  no_id,
	  pos,
	  n_id = cur_move->n_id,
	  dom = cur_move->dom;
	EGlistNode_t *e_it = all_nodes[n_id]->edges->begin;

	/* update flags if necesary */
	if (update_flags)
	{
		EGbitSet (nodeData[n_id].flipA, dom);
		EGbitSet (nodeData[n_id].flipBc, dom);
	}
	/* flip the handle and change the node */
	KPTmoveAAc (n_id, dom);
	KPTmoveBcB (n_id, dom);
	KPTmoveAH (n_id);
	KPTmoveBH (n_id);

	/* we have to compute every move */
	for (; e_it; e_it = e_it->next)
	{
		e_id = KPTgetEdgeId (e_it);
		no_id = KPTgetOtherEndId (n_id, e_it);
		pos = (EGbitTest (nodeData[no_id].It, dom) ? 1U : 0U) |
			(EGbitTest (nodeData[no_id].Ia, dom) ? 2U : 0U) |
			(EGbitTest (nodeData[no_id].Ib, dom) ? 4U : 0U);
		switch (pos)
		{
		case 0U:
			/* in this case, flip AF and flip BF */
			edgeData[e_id].num.in_AF++;
			edgeData[e_id].num.in_BF++;
			break;
		case 5U:
			/* in this case we save one E(A:Ac) and double flip AF, save 
			 * one E(B:Bc) and double flip BF */
			edgeData[e_id].num.n_AtoAc--;
			edgeData[e_id].num.n_BtoBc--;
			break;
		case 7U:
			/* in this case we pay one E(A:Ac) and double flip AF, save 
			 * one E(B:Bc) and double flip BF */
			edgeData[e_id].num.n_AtoAc++;
			edgeData[e_id].num.n_BtoBc--;
			break;
		case 1U:
			/* in this case we save one E(A:Ac) and double flip AF, pay 
			 * one E(B:Bc) and double flip BF */
			edgeData[e_id].num.n_AtoAc--;
			edgeData[e_id].num.n_BtoBc++;
			break;
		case 3U:
			/* in this case we pay one E(A:Ac) and double flip AF, pay 
			 * one E(B:Bc) and double flip BF */
			edgeData[e_id].num.n_AtoAc++;
			edgeData[e_id].num.n_BtoBc++;
			break;
		default:
			EXIT (1, "This should not happen, value %u", pos);
		}														/* end switch */
	}															/* end loop through all incident edges */
	return 0;
}

/* ========================================================================= */
/** @brief This function makeMoves the move AcABcBflipAHBH, assuming the move
 * is feasible*/
/* ========================================================================= */
static int KPTmakeMoveAcABcBflipAHBH (const KPTMove_t * const cur_move,
																			const unsigned int update_flags)
{
	/* local variables */
	unsigned int e_id,
	  no_id,
	  pos,
	  n_id = cur_move->n_id,
	  dom = cur_move->dom;
	EGlistNode_t *e_it = all_nodes[n_id]->edges->begin;

	/* update flags if necesary */
	if (update_flags)
	{
		EGbitSet (nodeData[n_id].flipAc, dom);
		EGbitSet (nodeData[n_id].flipBc, dom);
	}
	/* flip the handle and change the node */
	KPTmoveAcA (n_id, dom);
	KPTmoveBcB (n_id, dom);
	KPTmoveAH (n_id);
	KPTmoveBH (n_id);

	/* we have to compute every move */
	for (; e_it; e_it = e_it->next)
	{
		e_id = KPTgetEdgeId (e_it);
		no_id = KPTgetOtherEndId (n_id, e_it);
		pos = (EGbitTest (nodeData[no_id].It, dom) ? 1U : 0U) |
			(EGbitTest (nodeData[no_id].Ia, dom) ? 2U : 0U) |
			(EGbitTest (nodeData[no_id].Ib, dom) ? 4U : 0U);
		switch (pos)
		{
		case 0U:
			/* in this case, flip AF and flip BF */
			edgeData[e_id].num.in_AF++;
			edgeData[e_id].num.in_BF++;
			break;
		case 7U:
			/* in this case we save one E(A:Ac) and double flip AF, save 
			 * one E(B:Bc) and double flip BF */
			edgeData[e_id].num.n_AtoAc--;
			edgeData[e_id].num.n_BtoBc--;
			break;
		case 5U:
			/* in this case we pay one E(A:Ac) and double flip AF, save 
			 * one E(B:Bc) and double flip BF */
			edgeData[e_id].num.n_AtoAc++;
			edgeData[e_id].num.n_BtoBc--;
			break;
		case 3U:
			/* in this case we save one E(A:Ac) and double flip AF, pay 
			 * one E(B:Bc) and double flip BF */
			edgeData[e_id].num.n_AtoAc--;
			edgeData[e_id].num.n_BtoBc++;
			break;
		case 1U:
			/* in this case we pay one E(A:Ac) and double flip AF, pay 
			 * one E(B:Bc) and double flip BF */
			edgeData[e_id].num.n_AtoAc++;
			edgeData[e_id].num.n_BtoBc++;
			break;
		default:
			EXIT (1, "This should not happen, value %u", pos);
		}														/* end switch */
	}															/* end loop through all incident edges */
	return 0;
}

/* ========================================================================= */
/** @brief This function makeMoves the move AAcBBcflipAHBH, assuming the move is 
 * feasible*/
/* ========================================================================= */
static int KPTmakeMoveAAcBBcflipAHBH (const KPTMove_t * const cur_move,
																			const unsigned int update_flags)
{
	/* local variables */
	unsigned int e_id,
	  no_id,
	  pos,
	  n_id = cur_move->n_id,
	  dom = cur_move->dom;
	EGlistNode_t *e_it = all_nodes[n_id]->edges->begin;

	/* update flags if necesary */
	if (update_flags)
	{
		EGbitSet (nodeData[n_id].flipA, dom);
		EGbitSet (nodeData[n_id].flipB, dom);
	}
	/* flip the handle and change the node */
	KPTmoveAAc (n_id, dom);
	KPTmoveBBc (n_id, dom);
	KPTmoveAH (n_id);
	KPTmoveBH (n_id);

	/* we have to compute every move */
	for (; e_it; e_it = e_it->next)
	{
		e_id = KPTgetEdgeId (e_it);
		no_id = KPTgetOtherEndId (n_id, e_it);
		pos = (EGbitTest (nodeData[no_id].It, dom) ? 1U : 0U) |
			(EGbitTest (nodeData[no_id].Ia, dom) ? 2U : 0U) |
			(EGbitTest (nodeData[no_id].Ib, dom) ? 4U : 0U);
		switch (pos)
		{
		case 0U:
			/* in this case, flip AF and flip BF */
			edgeData[e_id].num.in_AF++;
			edgeData[e_id].num.in_BF++;
			break;
		case 1U:
			/* in this case we save one E(A:Ac) and double flip AF, save 
			 * one E(B:Bc) and double flip BF */
			edgeData[e_id].num.n_AtoAc--;
			edgeData[e_id].num.n_BtoBc--;
			break;
		case 3U:
			/* in this case we pay one E(A:Ac) and double flip AF, save 
			 * one E(B:Bc) and double flip BF */
			edgeData[e_id].num.n_AtoAc++;
			edgeData[e_id].num.n_BtoBc--;
			break;
		case 5U:
			/* in this case we save one E(A:Ac) and double flip AF, pay 
			 * one E(B:Bc) and double flip BF */
			edgeData[e_id].num.n_AtoAc--;
			edgeData[e_id].num.n_BtoBc++;
			break;
		case 7U:
			/* in this case we pay one E(A:Ac) and double flip AF, pay 
			 * one E(B:Bc) and double flip BF */
			edgeData[e_id].num.n_AtoAc++;
			edgeData[e_id].num.n_BtoBc++;
			break;
		default:
			EXIT (1, "This should not happen, value %u", pos);
		}														/* end switch */
	}															/* end loop through all incident edges */
	return 0;
}


/* ========================================================================= */
/** @brief This function makes the move ABTc, assuming the move is feasible*/
/* ========================================================================= */
static int KPTmakeMoveABTc (const KPTMove_t * const cur_move)
{
	/* local variables */
	unsigned int e_id,
	  no_id,
	  pos,
	  dom = cur_move->dom,
	  n_id = cur_move->n_id;
	EGlistNode_t *e_it = all_nodes[n_id]->edges->begin;

	/* now move the nodes around */
	graphData.n_A[1][dom]--;
	graphData.n_B[1][dom]--;
	graphData.n_Tc[dom]++;
	EGbitUnset (nodeData[n_id].It, dom);
	EGbitUnset (nodeData[n_id].Ia, dom);
	EGbitUnset (nodeData[n_id].Ib, dom);
	nodeData[n_id].n_in_A--;
	nodeData[n_id].n_in_B--;
	nodeData[n_id].n_in_T--;
	graphData.n_AdB[1][dom]--;
	EGbitUnset (nodeData[n_id].Iadb, dom);
	/* we have to compute every move */
	for (; e_it; e_it = e_it->next)
	{
		e_id = KPTgetEdgeId (e_it);
		no_id = KPTgetOtherEndId (n_id, e_it);
		pos = (EGbitTest (nodeData[no_id].It, dom) ? 1U : 0U) |
			(EGbitTest (nodeData[no_id].Ia, dom) ? 2U : 0U) |
			(EGbitTest (nodeData[no_id].Ib, dom) ? 4U : 0U);
		switch (pos)
		{
		case 7U:
			/* in this case, the only change is that we have to pay for this edge in
			 * Delta(T) */
			edgeData[e_id].num.n_dT++;
			break;
		case 1U:
			/* in this case we pay one Delta(T), save one E(A:Ac) and flip AF, save 
			 * one E(B:Bc) and flip BF */
			edgeData[e_id].num.n_dT++;
			edgeData[e_id].num.n_AtoAc--;
			edgeData[e_id].num.in_AF++;
			edgeData[e_id].num.n_BtoBc--;
			edgeData[e_id].num.in_BF++;
			break;
		case 3U:
			/* in this case we pay one Delta(T), save one E(B:Bc) and flip BF */
			edgeData[e_id].num.n_dT++;
			edgeData[e_id].num.n_BtoBc--;
			edgeData[e_id].num.in_BF++;
			break;
		case 5U:
			/* in this case we pay one Delta(T), save one E(A:Ac) and flip AF */
			edgeData[e_id].num.n_dT++;
			edgeData[e_id].num.n_AtoAc--;
			edgeData[e_id].num.in_AF++;
			break;
		case 0U:
			/* in this case we save one Delta(T) */
			edgeData[e_id].num.n_dT--;
			break;
		default:
			EXIT (1, "This should not happen, value %u", pos);
		}														/* end switch */
	}															/* end loop through all incident edges */
	return 0;
}

/* ========================================================================= */
/** @brief This function makes the move TcAcBc,assuming the move is feasible*/
/* ========================================================================= */
static int KPTmakeMoveTcAcBc (const KPTMove_t * const cur_move,
															const unsigned int update_flags)
{
	/* local variables */
	unsigned int e_id,
	  no_id,
	  pos,
	  dom = cur_move->dom,
	  n_id = cur_move->n_id;
	EGlistNode_t *e_it = all_nodes[n_id]->edges->begin;

	/* update flags if necesary */
	if (update_flags)
		EGbitSet (nodeData[n_id].added, dom);
	/* nbow move the nodes */
	graphData.n_A[0][dom]++;
	graphData.n_B[0][dom]++;
	graphData.n_Tc[dom]--;
	EGbitSet (nodeData[n_id].It, dom);
	nodeData[n_id].n_in_T++;
	graphData.n_AdB[1][dom]++;
	EGbitSet (nodeData[n_id].Iadb, dom);
	/* we have to compute every move */
	for (; e_it; e_it = e_it->next)
	{
		e_id = KPTgetEdgeId (e_it);
		no_id = KPTgetOtherEndId (n_id, e_it);
		pos = (EGbitTest (nodeData[no_id].It, dom) ? 1U : 0U) |
			(EGbitTest (nodeData[no_id].Ia, dom) ? 2U : 0U) |
			(EGbitTest (nodeData[no_id].Ib, dom) ? 4U : 0U);
		switch (pos)
		{
		case 0U:
			/* in this case, the only change is that we have to pay for this edge in
			 * Delta(T) */
			edgeData[e_id].num.n_dT++;
			break;
		case 7U:
			/* in this case we save one Delta(T), pay one E(A:Ac) and flip AF, pay 
			 * one E(B:Bc) and flip BF */
			edgeData[e_id].num.n_dT--;
			edgeData[e_id].num.n_AtoAc++;
			edgeData[e_id].num.in_AF++;
			edgeData[e_id].num.n_BtoBc++;
			edgeData[e_id].num.in_BF++;
			break;
		case 5U:
			/* in this case we save one Delta(T), pay one E(B:Bc) and flip BF */
			edgeData[e_id].num.n_dT--;
			edgeData[e_id].num.n_BtoBc++;
			edgeData[e_id].num.in_BF++;
			break;
		case 3U:
			/* in this case we save one Delta(T), pay one E(A:Ac) and flip AF */
			edgeData[e_id].num.n_dT--;
			edgeData[e_id].num.n_AtoAc++;
			edgeData[e_id].num.in_AF++;
			break;
		case 1U:
			/* in this case we save one Delta(T) */
			edgeData[e_id].num.n_dT--;
			break;
		default:
			EXIT (1, "This should not happen, value %u", pos);
		}														/* end switch */
	}															/* end loop through all incident edges */
	return 0;
}

/* ========================================================================= */
/** @brief This function makes the move Tc ABc,assuming the move is feasible*/
/* ========================================================================= */
static int KPTmakeMoveTcABc (const KPTMove_t * const cur_move,
														 const unsigned int update_flags)
{
	/* local variables */
	unsigned int e_id,
	  no_id,
	  pos,
	  dom = cur_move->dom,
	  n_id = cur_move->n_id;
	EGlistNode_t *e_it = all_nodes[n_id]->edges->begin;

	/* update flags */
	if (update_flags)
		EGbitSet (nodeData[n_id].added, dom);
	/* move the node */
	graphData.n_A[1][dom]++;
	graphData.n_B[0][dom]++;
	graphData.n_Tc[dom]--;
	EGbitSet (nodeData[n_id].It, dom);
	EGbitSet (nodeData[n_id].Ia, dom);
	nodeData[n_id].n_in_A++;
	nodeData[n_id].n_in_T++;
	graphData.n_AdB[0][dom]++;
	EGbitUnset (nodeData[n_id].Iadb, dom);
	/* we have to compute every move */
	for (; e_it; e_it = e_it->next)
	{
		e_id = KPTgetEdgeId (e_it);
		no_id = KPTgetOtherEndId (n_id, e_it);
		pos = (EGbitTest (nodeData[no_id].It, dom) ? 1U : 0U) |
			(EGbitTest (nodeData[no_id].Ia, dom) ? 2U : 0U) |
			(EGbitTest (nodeData[no_id].Ib, dom) ? 4U : 0U);
		switch (pos)
		{
		case 0U:
			/* in this case, the only change is that we have to pay for this edge in
			 * Delta(T) */
			edgeData[e_id].num.n_dT++;
			break;
		case 5U:
			/* in this case we save one Delta(T), pay one E(A:Ac) and flip AF, pay 
			 * one E(B:Bc) and flip BF */
			edgeData[e_id].num.n_dT--;
			edgeData[e_id].num.n_AtoAc++;
			edgeData[e_id].num.in_AF++;
			edgeData[e_id].num.n_BtoBc++;
			edgeData[e_id].num.in_BF++;
			break;
		case 7U:
			/* in this case we save one Delta(T), pay one E(B:Bc) and flip BF */
			edgeData[e_id].num.n_dT--;
			edgeData[e_id].num.n_BtoBc++;
			edgeData[e_id].num.in_BF++;
			break;
		case 1U:
			/* in this case we save one Delta(T), pay one E(A:Ac) and flip AF */
			edgeData[e_id].num.n_dT--;
			edgeData[e_id].num.n_AtoAc++;
			edgeData[e_id].num.in_AF++;
			break;
		case 3U:
			/* in this case we save one Delta(T) */
			edgeData[e_id].num.n_dT--;
			break;
		default:
			EXIT (1, "This should not happen, value %u", pos);
		}														/* end switch */
	}															/* end loop through all incident edges */
	return 0;
}

/* ========================================================================= */
/** @brief This function makes the move Tc AcB,assuming the move is feasible*/
/* ========================================================================= */
static int KPTmakeMoveTcAcB (const KPTMove_t * const cur_move,
														 const unsigned int update_flags)
{
	/* local variables */
	unsigned int e_id,
	  no_id,
	  pos,
	  dom = cur_move->dom,
	  n_id = cur_move->n_id;
	EGlistNode_t *e_it = all_nodes[n_id]->edges->begin;

	/* update flags if necesary */
	if (update_flags)
		EGbitSet (nodeData[n_id].added, dom);
	/* move the node */
	graphData.n_A[0][dom]++;
	graphData.n_B[1][dom]++;
	graphData.n_Tc[dom]--;
	EGbitSet (nodeData[n_id].It, dom);
	EGbitSet (nodeData[n_id].Ib, dom);
	nodeData[n_id].n_in_T++;
	nodeData[n_id].n_in_B++;
	graphData.n_AdB[0][dom]++;
	EGbitUnset (nodeData[n_id].Iadb, dom);
	/* we have to compute every move */
	for (; e_it; e_it = e_it->next)
	{
		e_id = KPTgetEdgeId (e_it);
		no_id = KPTgetOtherEndId (n_id, e_it);
		pos = (EGbitTest (nodeData[no_id].It, dom) ? 1U : 0U) |
			(EGbitTest (nodeData[no_id].Ia, dom) ? 2U : 0U) |
			(EGbitTest (nodeData[no_id].Ib, dom) ? 4U : 0U);
		switch (pos)
		{
		case 0U:
			/* in this case, the only change is that we have to pay for this edge in
			 * Delta(T) */
			edgeData[e_id].num.n_dT++;
			break;
		case 3U:
			/* in this case we save one Delta(T), pay one E(A:Ac) and flip AF, pay 
			 * one E(B:Bc) and flip BF */
			edgeData[e_id].num.n_dT--;
			edgeData[e_id].num.n_AtoAc++;
			edgeData[e_id].num.in_AF++;
			edgeData[e_id].num.n_BtoBc++;
			edgeData[e_id].num.in_BF++;
			break;
		case 1U:
			/* in this case we save one Delta(T), pay one E(B:Bc) and flip BF */
			edgeData[e_id].num.n_dT--;
			edgeData[e_id].num.n_BtoBc++;
			edgeData[e_id].num.in_BF++;
			break;
		case 7U:
			/* in this case we save one Delta(T), pay one E(A:Ac) and flip AF */
			edgeData[e_id].num.n_dT--;
			edgeData[e_id].num.n_AtoAc++;
			edgeData[e_id].num.in_AF++;
			break;
		case 5U:
			/* in this case we save one Delta(T) */
			edgeData[e_id].num.n_dT--;
			break;
		default:
			EXIT (1, "This should not happen, value %u", pos);
		}														/* end switch */
	}															/* end loop through all incident edges */
	return 0;
}

/* ========================================================================= */
/** @brief This function makes the move Tc AB, assuming the move is feasible*/
/* ========================================================================= */
static int KPTmakeMoveTcAB (const KPTMove_t * const cur_move,
														const unsigned int update_flags)
{
	/* local variables */
	unsigned int e_id,
	  no_id,
	  pos,
	  dom = cur_move->dom,
	  n_id = cur_move->n_id;
	EGlistNode_t *e_it = all_nodes[n_id]->edges->begin;

	/* update flags if needed */
	if (update_flags)
		EGbitSet (nodeData[n_id].added, dom);
	/* move hte node */
	graphData.n_A[1][dom]++;
	graphData.n_B[1][dom]++;
	graphData.n_Tc[dom]--;
	EGbitSet (nodeData[n_id].It, dom);
	EGbitSet (nodeData[n_id].Ia, dom);
	EGbitSet (nodeData[n_id].Ib, dom);
	nodeData[n_id].n_in_T++;
	nodeData[n_id].n_in_A++;
	nodeData[n_id].n_in_B++;
	graphData.n_AdB[1][dom]++;
	EGbitSet (nodeData[n_id].Iadb, dom);
	/* we have to compute every move */
	for (; e_it; e_it = e_it->next)
	{
		e_id = KPTgetEdgeId (e_it);
		no_id = KPTgetOtherEndId (n_id, e_it);
		pos = (EGbitTest (nodeData[no_id].It, dom) ? 1U : 0U) |
			(EGbitTest (nodeData[no_id].Ia, dom) ? 2U : 0U) |
			(EGbitTest (nodeData[no_id].Ib, dom) ? 4U : 0U);
		switch (pos)
		{
		case 0U:
			/* in this case, the only change is that we have to pay for this edge in
			 * Delta(T) */
			edgeData[e_id].num.n_dT++;
			break;
		case 1U:
			/* in this case we save one Delta(T), pay one E(A:Ac) and flip AF, pay 
			 * one E(B:Bc) and flip BF */
			edgeData[e_id].num.n_dT--;
			edgeData[e_id].num.n_AtoAc++;
			edgeData[e_id].num.in_AF++;
			edgeData[e_id].num.n_BtoBc++;
			edgeData[e_id].num.in_BF++;
			break;
		case 3U:
			/* in this case we save one Delta(T), pay one E(B:Bc) and flip BF */
			edgeData[e_id].num.n_dT--;
			edgeData[e_id].num.n_BtoBc++;
			edgeData[e_id].num.in_BF++;
			break;
		case 5U:
			/* in this case we save one Delta(T), pay one E(A:Ac) and flip AF */
			edgeData[e_id].num.n_dT--;
			edgeData[e_id].num.n_AtoAc++;
			edgeData[e_id].num.in_AF++;
			break;
		case 7U:
			/* in this case we save one Delta(T) */
			edgeData[e_id].num.n_dT--;
			break;
		default:
			EXIT (1, "This should not happen, value %u", pos);
		}														/* end switch */
	}															/* end loop through all incident edges */
	return 0;
}

/* ========================================================================= */
/** @brief Perform a KPT_TcA move */
/* ========================================================================= */
static int KPTmakeMoveTcA (KPTMove_t const *const move,
													 const unsigned int update_flags)
{
	/* local vaariables */
	unsigned int e_id,
	  no_id,
	  pos,
	  dom = move->dom,
	  n_id = move->n_id;
	EGlistNode_t *e_it = all_nodes[n_id]->edges->begin;

	/* we have to change all related edges */
	for (; e_it; e_it = e_it->next)
	{
		e_id = KPTgetEdgeId (e_it);
		no_id = KPTgetOtherEndId (n_id, e_it);
		pos = (EGbitTest (nodeData[no_id].Ia, dom) ? 2U : 0U) |
			(EGbitTest (nodeData[no_id].It, dom) ? 1U : 0U);
		switch (pos)
		{
		case 0U:
			/* in this case the only change is that we have to pay for this edge in the
			 * Delta(T) */
			edgeData[e_id].num.n_dT++;
			break;
			/* in this case we save one Delta(T), pay one E(A:B) and flip in_F because
			 * we are changing parity of \sum E(A:B) */
		case 1U:
			edgeData[e_id].num.n_dT--;
			edgeData[e_id].num.n_AtoAc++;
			edgeData[e_id].num.in_AF++;
			break;
			/* in this case we save one Delta(T) */
		case 3U:
			edgeData[e_id].num.n_dT--;
			break;
		default:
			EXIT (1, "This should not happen");
		}														/* end switch */
	}

	/* move the node */
	graphData.n_Tc[dom]--;
	graphData.n_A[1][dom]++;
	EGbitSet (nodeData[n_id].Ia, dom);
	EGbitSet (nodeData[n_id].It, dom);
	nodeData[n_id].n_in_A++;
	nodeData[n_id].n_in_T++;
	if (update_flags)
		EGbitSet (nodeData[n_id].added, dom);
	/* ending */
	return 0;
}

/* ========================================================================= */
/** @brief Perform a KPT_TcAc move */
/* ========================================================================= */
static int KPTmakeMoveTcAc (KPTMove_t const *const move,
														const unsigned int update_flags)
{
	/* local vaariables */
	unsigned int e_id,
	  no_id,
	  pos,
	  dom = move->dom,
	  n_id = move->n_id;
	EGlistNode_t *e_it = all_nodes[n_id]->edges->begin;

	/* we have to change all related edges */
	for (; e_it; e_it = e_it->next)
	{
		e_id = KPTgetEdgeId (e_it);
		no_id = KPTgetOtherEndId (n_id, e_it);
		pos = (EGbitTest (nodeData[no_id].Ia, dom) ? 2U : 0U) |
			(EGbitTest (nodeData[no_id].It, dom) ? 1U : 0U);
		switch (pos)
		{
		case 0U:
			/* in this case the only change is that we have to pay for this edge in the
			 * Delta(T) */
			edgeData[e_id].num.n_dT++;
			break;
			/* in this case we save one Delta(T), pay one E(A:Ac) and flip in_F because
			 * we are changing parity of \sum E(A:B) */
		case 3U:
			edgeData[e_id].num.n_dT--;
			edgeData[e_id].num.n_AtoAc++;
			edgeData[e_id].num.in_AF++;
			break;
			/* in this case we save one Delta(T) */
		case 1U:
			edgeData[e_id].num.n_dT--;
			break;
		default:
			EXIT (1, "This should not happen");
		}														/* end switch */
	}

	/* move the node */
	graphData.n_Tc[dom]--;
	graphData.n_A[0][dom]++;
	EGbitSet (nodeData[n_id].It, dom);
	nodeData[n_id].n_in_T++;
	if (update_flags)
		EGbitSet (nodeData[n_id].added, dom);
	/* ending */
	return 0;
}

/* ========================================================================= */
/** @brief Perform a KPT_TcB move */
/* ========================================================================= */
static int KPTmakeMoveTcB (KPTMove_t const *const move,
													 const unsigned int update_flags)
{
	/* local vaariables */
	unsigned int e_id,
	  no_id,
	  pos,
	  dom = move->dom,
	  n_id = move->n_id;
	EGlistNode_t *e_it = all_nodes[n_id]->edges->begin;

	/* move the node */
	graphData.n_Tc[dom]--;
	graphData.n_B[1][dom]++;
	EGbitSet (nodeData[n_id].Ib, dom);
	EGbitSet (nodeData[n_id].It, dom);
	nodeData[n_id].n_in_T++;
	nodeData[n_id].n_in_B++;
	if (update_flags)
		EGbitSet (nodeData[n_id].added, dom);

	/* we have to change all related edges */
	for (; e_it; e_it = e_it->next)
	{
		e_id = KPTgetEdgeId (e_it);
		no_id = KPTgetOtherEndId (n_id, e_it);
		pos = (EGbitTest (nodeData[no_id].Ib, dom) ? 2U : 0U) |
			(EGbitTest (nodeData[no_id].It, dom) ? 1U : 0U);
		switch (pos)
		{
		case 0U:
			/* in this case the only change is that we have to pay for this edge in the
			 * Delta(T) */
			edgeData[e_id].num.n_dT++;
			break;
			/* in this case we save one Delta(T), pay one E(B:Bc) and flip in_F because
			 * we are changing parity of \sum E(B:Bc) */
		case 1U:
			edgeData[e_id].num.n_dT--;
			edgeData[e_id].num.n_BtoBc++;
			edgeData[e_id].num.in_BF++;
			break;
			/* in this case we save one Delta(T) */
		case 3U:
			edgeData[e_id].num.n_dT--;
			break;
		default:
			EXIT (1, "This should not happen");
		}														/* end switch */
	}
	return 0;
}

/* ========================================================================= */
/** @brief Perform a KPT_TcBc move */
/* ========================================================================= */
static int KPTmakeMoveTcBc (KPTMove_t const *const move,
														const unsigned int update_flags)
{
	/* local vaariables */
	unsigned int e_id,
	  no_id,
	  pos,
	  dom = move->dom,
	  n_id = move->n_id;
	EGlistNode_t *e_it = all_nodes[n_id]->edges->begin;

	/* move the node */
	graphData.n_Tc[dom]--;
	graphData.n_B[0][dom]++;
	EGbitSet (nodeData[n_id].It, dom);
	nodeData[n_id].n_in_T++;
	if (update_flags)
		EGbitSet (nodeData[n_id].added, dom);

	/* we have to change all related edges */
	for (; e_it; e_it = e_it->next)
	{
		e_id = KPTgetEdgeId (e_it);
		no_id = KPTgetOtherEndId (n_id, e_it);
		pos = (EGbitTest (nodeData[no_id].Ib, dom) ? 2U : 0U) |
			(EGbitTest (nodeData[no_id].It, dom) ? 1U : 0U);
		switch (pos)
		{
		case 0U:
			/* in this case the only change is that we have to pay for this edge in the
			 * Delta(T) */
			edgeData[e_id].num.n_dT++;
			break;
			/* in this case we save one Delta(T), pay one E(B:Bc) and flip in_F because
			 * we are changing parity of \sum E(B:Bc) */
		case 3U:
			edgeData[e_id].num.n_dT--;
			edgeData[e_id].num.n_BtoBc++;
			edgeData[e_id].num.in_BF++;
			break;
			/* in this case we save one Delta(T) */
		case 1U:
			edgeData[e_id].num.n_dT--;
			break;
		default:
			EXIT (1, "This should not happen");
		}														/* end switch */
	}
	return 0;
}

/* ========================================================================= */
/** @brief Perform a KPT_ATc move */
/* ========================================================================= */
static int KPTmakeMoveATc (KPTMove_t const *const move)
{
	/* local vaariables */
	unsigned int e_id,
	  no_id,
	  pos,
	  dom = move->dom,
	  n_id = move->n_id;
	EGlistNode_t *e_it = all_nodes[n_id]->edges->begin;

	/* we have to change all related edges */
	for (; e_it; e_it = e_it->next)
	{
		e_id = KPTgetEdgeId (e_it);
		no_id = KPTgetOtherEndId (n_id, e_it);
		pos = (EGbitTest (nodeData[no_id].Ia, dom) ? 2U : 0U) |
			(EGbitTest (nodeData[no_id].It, dom) ? 1U : 0U);
		switch (pos)
		{
		case 0U:
			/* in this case the only change is that we have to save forthis edge in the
			 * Delta(T) */
			edgeData[e_id].num.n_dT--;
			break;
			/* in this case we pay one Delta(T), save one E(A:Ac) flip in_AF because
			 * we are changing parity of \sum E(A:Ac) */
		case 1U:
			edgeData[e_id].num.n_dT++;
			edgeData[e_id].num.n_AtoAc--;
			edgeData[e_id].num.in_AF++;
			break;
			/* in this case we pay one Delta(T) */
		case 3U:
			edgeData[e_id].num.n_dT++;
			break;
		default:
			EXIT (1, "This should not happen");
		}														/* end switch */
	}

	/* move the node */
	graphData.n_Tc[dom]++;
	graphData.n_A[1][dom]--;
	EGbitUnset (nodeData[n_id].Ia, dom);
	EGbitUnset (nodeData[n_id].It, dom);
	nodeData[n_id].n_in_A--;
	nodeData[n_id].n_in_T--;
	/* ending */
	return 0;
}

/* ========================================================================= */
/** @brief Perform a KPT_AcTc move */
/* ========================================================================= */
static int KPTmakeMoveAcTc (KPTMove_t const *const move)
{
	/* local vaariables */
	unsigned int e_id,
	  no_id,
	  pos,
	  dom = move->dom,
	  n_id = move->n_id;
	EGlistNode_t *e_it = all_nodes[n_id]->edges->begin;

	/* we have to change all related edges */
	for (; e_it; e_it = e_it->next)
	{
		e_id = KPTgetEdgeId (e_it);
		no_id = KPTgetOtherEndId (n_id, e_it);
		pos = (EGbitTest (nodeData[no_id].Ia, dom) ? 2U : 0U) |
			(EGbitTest (nodeData[no_id].It, dom) ? 1U : 0U);
		switch (pos)
		{
		case 0U:
			/* in this case the only change is that we have to save this edge in the
			 * Delta(T) */
			edgeData[e_id].num.n_dT--;
			break;
			/* in this case we pay one Delta(T), save one E(A:Ac) flip in_F because
			 * we are changing parity of \sum E(A:B) */
		case 3U:
			edgeData[e_id].num.n_dT++;
			edgeData[e_id].num.n_AtoAc--;
			edgeData[e_id].num.in_AF++;
			break;
			/* in this case we paye one Delta(T) */
		case 1U:
			edgeData[e_id].num.n_dT++;
			break;
		default:
			EXIT (1, "This should not happen");
		}														/* end switch */
	}

	/* move the node */
	graphData.n_Tc[dom]++;
	graphData.n_A[0][dom]--;
	EGbitUnset (nodeData[n_id].It, dom);
	nodeData[n_id].n_in_T--;
	/* ending */
	return 0;
}

/* ========================================================================= */
/** @brief Perform a KPT_BTc move */
/* ========================================================================= */
static int KPTmakeMoveBTc (KPTMove_t const *const move)
{
	/* local vaariables */
	unsigned int e_id,
	  no_id,
	  pos,
	  dom = move->dom,
	  n_id = move->n_id;
	EGlistNode_t *e_it = all_nodes[n_id]->edges->begin;

	/* move the node */
	graphData.n_Tc[dom]++;
	graphData.n_B[1][dom]--;
	EGbitUnset (nodeData[n_id].Ib, dom);
	EGbitUnset (nodeData[n_id].It, dom);
	nodeData[n_id].n_in_T--;
	nodeData[n_id].n_in_B--;

	/* we have to change all related edges */
	for (; e_it; e_it = e_it->next)
	{
		e_id = KPTgetEdgeId (e_it);
		no_id = KPTgetOtherEndId (n_id, e_it);
		pos = (EGbitTest (nodeData[no_id].Ib, dom) ? 2U : 0U) |
			(EGbitTest (nodeData[no_id].It, dom) ? 1U : 0U);
		switch (pos)
		{
		case 0U:
			/* in this case the only change is that we save for this edge in the
			 * Delta(T) */
			edgeData[e_id].num.n_dT--;
			break;
			/* in this case we pay one Delta(T), save one E(B:Bc) and flip in_F because
			 * we are changing parity of \sum E(B:Bc) */
		case 1U:
			edgeData[e_id].num.n_dT++;
			edgeData[e_id].num.n_BtoBc--;
			edgeData[e_id].num.in_BF++;
			break;
			/* in this case we pay one Delta(T) */
		case 3U:
			edgeData[e_id].num.n_dT++;
			break;
		default:
			EXIT (1, "This should not happen");
		}														/* end switch */
	}
	return 0;
}

/* ========================================================================= */
/** @brief Perform a KPT_BcTc move */
/* ========================================================================= */
static int KPTmakeMoveBcTc (KPTMove_t const *const move)
{
	/* local vaariables */
	unsigned int e_id,
	  no_id,
	  pos,
	  dom = move->dom,
	  n_id = move->n_id;
	EGlistNode_t *e_it = all_nodes[n_id]->edges->begin;

	/* move the node */
	graphData.n_Tc[dom]++;
	graphData.n_B[0][dom]--;
	EGbitUnset (nodeData[n_id].It, dom);
	nodeData[n_id].n_in_T--;

	/* we have to change all related edges */
	for (; e_it; e_it = e_it->next)
	{
		e_id = KPTgetEdgeId (e_it);
		no_id = KPTgetOtherEndId (n_id, e_it);
		pos = (EGbitTest (nodeData[no_id].Ib, dom) ? 2U : 0U) |
			(EGbitTest (nodeData[no_id].It, dom) ? 1U : 0U);
		switch (pos)
		{
		case 0U:
			/* in this case the only change is that we save for this edge in the
			 * Delta(T) */
			edgeData[e_id].num.n_dT--;
			break;
			/* in this case we pay one Delta(T), save one E(B:Bc) and flip in_F because
			 * we are changing parity of \sum E(B:Bc) */
		case 3U:
			edgeData[e_id].num.n_dT++;
			edgeData[e_id].num.n_BtoBc--;
			edgeData[e_id].num.in_BF++;
			break;
			/* in this case we pay one Delta(T) */
		case 1U:
			edgeData[e_id].num.n_dT++;
			break;
		default:
			EXIT (1, "This should not happen");
		}														/* end switch */
	}
	return 0;
}

/* ========================================================================= */
/** @brief This function check that the edges got the wright values in their
 * data. */
/* ========================================================================= */
static int KPTTestEdges (void)
{
	/* local variables */
	register unsigned int i = G->nedges,
	  j;
	unsigned int n_AtoAc,
	  n_dT,
	  N1_id,
	  N2_id,
	  N1_inA,
	  N1_inB,
	  N1_inTc,
	  N2_inA,
	  N2_inB,
	  N2_inTc,
	  in_AF,
	  n_BtoBc,
	  in_BF,
	  pos;
	while (i--)
	{
		N1_id = all_edges[i]->head->id;
		N2_id = all_edges[i]->tail->id;
		n_AtoAc = n_BtoBc = n_dT = in_AF = in_BF = 0;
		for (j = graphData.n_dominos; j--;)
		{
			N1_inA = EGbitTest (nodeData[N1_id].Ia, j) ? 1U : 0U;
			N1_inB = EGbitTest (nodeData[N1_id].Ib, j) ? 1U : 0U;
			N1_inTc = EGbitTest (nodeData[N1_id].It, j) ? 0U : 1U;
			N2_inA = EGbitTest (nodeData[N2_id].Ia, j) ? 1U : 0U;
			N2_inB = EGbitTest (nodeData[N2_id].Ib, j) ? 1U : 0U;
			N2_inTc = EGbitTest (nodeData[N2_id].It, j) ? 0U : 1U;
			if ((!N1_inTc && N2_inTc) || (N1_inTc && !N2_inTc))
				n_dT++;
			if (!N1_inTc && !N2_inTc && (N1_inA != N2_inA))
				n_AtoAc++;
			if (!N1_inTc && !N2_inTc && (N1_inB != N2_inB))
				n_BtoBc++;
		}
		pos = 1U & ((nodeData[N1_id].in_AH ^ nodeData[N2_id].in_AH)
								^ edgeData[i].num.n_AtoAc);
		if (pos)
			in_AF = 1U;
		pos = 1U & ((nodeData[N1_id].in_BH ^ nodeData[N2_id].in_BH)
								^ edgeData[i].num.n_BtoBc);
		if (pos)
			in_BF = 1U;
		TEST (in_BF != edgeData[i].num.in_BF || in_AF != edgeData[i].num.in_AF ||
					n_dT != edgeData[i].num.n_dT || n_AtoAc != edgeData[i].num.n_AtoAc ||
					n_BtoBc != edgeData[i].num.n_BtoBc, "in_AF(%u,%u) in_BF(%u,%u)"
					" n_dT(%u,%u) n_AtoAc(%u,%u) n_BtoBc(%u,%u) don't match for edge %u",
					in_AF, edgeData[i].num.in_AF, in_BF, edgeData[i].num.in_BF, n_dT,
					edgeData[i].num.n_dT, n_AtoAc, edgeData[i].num.n_AtoAc, n_BtoBc,
					edgeData[i].num.n_BtoBc, i);
	}
	return 0;
}

/* ========================================================================= */
/** @brief given a node, and a move, make it */
/* ========================================================================= */
#define KPTmakeMove(__move,__flags) __KPTmakeMove(__move,__flags,__LINE__,__FILE__,__func__)
static int __KPTmakeMove (KPTMove_t const *const move,
													const unsigned int update_flags,
													const int line,
													const char *file,
													const char *function)
{
	/* we just switch the correct call */
	MESSAGE (KPT_VRB + 19, "doing move %s for node %u from %s at %s:%d",
					 move_name[move->move_id], move->n_id, function, file, line);
	switch (move->move_id)
	{
	case KPT_AB_Tc:
		KPTmakeMoveABTc (move);
		break;
	case KPT_AcB_Tc:
		KPTmakeMoveAcBTc (move);
		break;
	case KPT_ABc_Tc:
		KPTmakeMoveABcTc (move);
		break;
	case KPT_AcBc_Tc:
		KPTmakeMoveAcBcTc (move);
		break;
	case KPT_Tc_AB:
		KPTmakeMoveTcAB (move, update_flags);
		break;
	case KPT_Tc_AcB:
		KPTmakeMoveTcAcB (move, update_flags);
		break;
	case KPT_Tc_ABc:
		KPTmakeMoveTcABc (move, update_flags);
		break;
	case KPT_Tc_AcBc:
		KPTmakeMoveTcAcBc (move, update_flags);
		break;
	case KPT_Ac_A:
		KPTmakeMoveAcA (move, update_flags);
		break;
	case KPT_A_Ac:
		KPTmakeMoveAAc (move, update_flags);
		break;
	case KPT_Bc_B:
		KPTmakeMoveBcB (move, update_flags);
		break;
	case KPT_B_Bc:
		KPTmakeMoveBBc (move, update_flags);
		break;
	case KPT_Ac_A_B_Bc:
		KPTmakeMoveAcABBc (move, update_flags);
		break;
	case KPT_Ac_A_Bc_B:
		KPTmakeMoveAcABcB (move, update_flags);
		break;
	case KPT_A_Ac_B_Bc:
		KPTmakeMoveAAcBBc (move, update_flags);
		break;
	case KPT_A_Ac_Bc_B:
		KPTmakeMoveAAcBcB (move, update_flags);
		break;
	case KPT_Ac_A_B_Bc_flipAH:
		KPTmakeMoveAcABBcflipAH (move, update_flags);
		break;
	case KPT_Ac_A_Bc_B_flipAH:
		KPTmakeMoveAcABcBflipAH (move, update_flags);
		break;
	case KPT_A_Ac_B_Bc_flipAH:
		KPTmakeMoveAAcBBcflipAH (move, update_flags);
		break;
	case KPT_A_Ac_Bc_B_flipAH:
		KPTmakeMoveAAcBcBflipAH (move, update_flags);
		break;
	case KPT_Ac_A_B_Bc_flipBH:
		KPTmakeMoveAcABBcflipBH (move, update_flags);
		break;
	case KPT_Ac_A_Bc_B_flipBH:
		KPTmakeMoveAcABcBflipBH (move, update_flags);
		break;
	case KPT_A_Ac_B_Bc_flipBH:
		KPTmakeMoveAAcBBcflipBH (move, update_flags);
		break;
	case KPT_A_Ac_Bc_B_flipBH:
		KPTmakeMoveAAcBcBflipBH (move, update_flags);
		break;
	case KPT_Ac_A_B_Bc_flipAHBH:
		KPTmakeMoveAcABBcflipAHBH (move, update_flags);
		break;
	case KPT_Ac_A_Bc_B_flipAHBH:
		KPTmakeMoveAcABcBflipAHBH (move, update_flags);
		break;
	case KPT_A_Ac_B_Bc_flipAHBH:
		KPTmakeMoveAAcBBcflipAHBH (move, update_flags);
		break;
	case KPT_A_Ac_Bc_B_flipAHBH:
		KPTmakeMoveAAcBcBflipAHBH (move, update_flags);
		break;
	case KPT_Ac_A_flipAH:
		KPTmakeMoveAcAflipAH (move, update_flags);
		break;
	case KPT_A_Ac_flipAH:
		KPTmakeMoveAAcflipAH (move, update_flags);
		break;
	case KPT_Bc_B_flipBH:
		KPTmakeMoveBcBflipBH (move, update_flags);
		break;
	case KPT_B_Bc_flipBH:
		KPTmakeMoveBBcflipBH (move, update_flags);
		break;
	case KPT_A_Tc:
		KPTmakeMoveATc (move);
		break;
	case KPT_B_Tc:
		KPTmakeMoveBTc (move);
		break;
	case KPT_Ac_Tc:
		KPTmakeMoveAcTc (move);
		break;
	case KPT_Bc_Tc:
		KPTmakeMoveBcTc (move);
		break;
	case KPT_Tc_A:
		KPTmakeMoveTcA (move, update_flags);
		break;
	case KPT_Tc_B:
		KPTmakeMoveTcB (move, update_flags);
		break;
	case KPT_Tc_Ac:
		KPTmakeMoveTcAc (move, update_flags);
		break;
	case KPT_Tc_Bc:
		KPTmakeMoveTcBc (move, update_flags);
		break;
	case KPT_AHc_AH:
		KPTmakeMoveAHcAH (move, update_flags);
		break;
	case KPT_BHc_BH:
		KPTmakeMoveBHcBH (move, update_flags);
		break;
	case KPT_AH_AHc:
		KPTmakeMoveAHAHc (move, update_flags);
		break;
	case KPT_BH_BHc:
		KPTmakeMoveBHBHc (move, update_flags);
		break;
	case KPT_AH_AHc_BH_BHc:
		KPTmakeMoveAHAHcBHBHc (move, update_flags);
		break;
	case KPT_AH_AHc_BHc_BH:
		KPTmakeMoveAHAHcBHcBH (move, update_flags);
		break;
	case KPT_AHc_AH_BHc_BH:
		KPTmakeMoveAHcAHBHcBH (move, update_flags);
		break;
	case KPT_AHc_AH_BH_BHc:
		KPTmakeMoveAHcAHBHBHc (move, update_flags);
		break;
	default:
		EXIT (1, "unknown move %d", move->move_id);
	}
	/* ending */
#if KPT_EDBG <= DEBUG
	EXIT (KPTTestEdges (), "after move %s", move_name[move->move_id]);
#endif
	return 0;
}

/* ========================================================================= */
/** @brief given a node, and a move, make the inverse move */
/* ========================================================================= */
static int KPTmakeInvMove (KPTMove_t const *const move,
													 const unsigned int update_flags)
{
	/* we just switch the correct call */
	MESSAGE (KPT_VRB + 19, "doing inv move %s for node %u",
					 move_name[move->move_id], move->n_id);
	switch (KPT_inv_move[move->move_id])
	{
	case KPT_AB_Tc:
		KPTmakeMoveABTc (move);
		break;
	case KPT_AcB_Tc:
		KPTmakeMoveAcBTc (move);
		break;
	case KPT_ABc_Tc:
		KPTmakeMoveABcTc (move);
		break;
	case KPT_AcBc_Tc:
		KPTmakeMoveAcBcTc (move);
		break;
	case KPT_Tc_AB:
		KPTmakeMoveTcAB (move, update_flags);
		break;
	case KPT_Tc_AcB:
		KPTmakeMoveTcAcB (move, update_flags);
		break;
	case KPT_Tc_ABc:
		KPTmakeMoveTcABc (move, update_flags);
		break;
	case KPT_Tc_AcBc:
		KPTmakeMoveTcAcBc (move, update_flags);
		break;
	case KPT_Ac_A:
		KPTmakeMoveAcA (move, update_flags);
		break;
	case KPT_A_Ac:
		KPTmakeMoveAAc (move, update_flags);
		break;
	case KPT_Bc_B:
		KPTmakeMoveBcB (move, update_flags);
		break;
	case KPT_B_Bc:
		KPTmakeMoveBBc (move, update_flags);
		break;
	case KPT_Ac_A_B_Bc:
		KPTmakeMoveAcABBc (move, update_flags);
		break;
	case KPT_Ac_A_Bc_B:
		KPTmakeMoveAcABcB (move, update_flags);
		break;
	case KPT_A_Ac_B_Bc:
		KPTmakeMoveAAcBBc (move, update_flags);
		break;
	case KPT_A_Ac_Bc_B:
		KPTmakeMoveAAcBcB (move, update_flags);
		break;
	case KPT_Ac_A_B_Bc_flipAH:
		KPTmakeMoveAcABBcflipAH (move, update_flags);
		break;
	case KPT_Ac_A_Bc_B_flipAH:
		KPTmakeMoveAcABcBflipAH (move, update_flags);
		break;
	case KPT_A_Ac_B_Bc_flipAH:
		KPTmakeMoveAAcBBcflipAH (move, update_flags);
		break;
	case KPT_A_Ac_Bc_B_flipAH:
		KPTmakeMoveAAcBcBflipAH (move, update_flags);
		break;
	case KPT_Ac_A_B_Bc_flipBH:
		KPTmakeMoveAcABBcflipBH (move, update_flags);
		break;
	case KPT_Ac_A_Bc_B_flipBH:
		KPTmakeMoveAcABcBflipBH (move, update_flags);
		break;
	case KPT_A_Ac_B_Bc_flipBH:
		KPTmakeMoveAAcBBcflipBH (move, update_flags);
		break;
	case KPT_A_Ac_Bc_B_flipBH:
		KPTmakeMoveAAcBcBflipBH (move, update_flags);
		break;
	case KPT_Ac_A_B_Bc_flipAHBH:
		KPTmakeMoveAcABBcflipAHBH (move, update_flags);
		break;
	case KPT_Ac_A_Bc_B_flipAHBH:
		KPTmakeMoveAcABcBflipAHBH (move, update_flags);
		break;
	case KPT_A_Ac_B_Bc_flipAHBH:
		KPTmakeMoveAAcBBcflipAHBH (move, update_flags);
		break;
	case KPT_A_Ac_Bc_B_flipAHBH:
		KPTmakeMoveAAcBcBflipAHBH (move, update_flags);
		break;
	case KPT_Ac_A_flipAH:
		KPTmakeMoveAcAflipAH (move, update_flags);
		break;
	case KPT_A_Ac_flipAH:
		KPTmakeMoveAAcflipAH (move, update_flags);
		break;
	case KPT_Bc_B_flipBH:
		KPTmakeMoveBcBflipBH (move, update_flags);
		break;
	case KPT_B_Bc_flipBH:
		KPTmakeMoveBBcflipBH (move, update_flags);
		break;
	case KPT_A_Tc:
		KPTmakeMoveATc (move);
		break;
	case KPT_B_Tc:
		KPTmakeMoveBTc (move);
		break;
	case KPT_Ac_Tc:
		KPTmakeMoveAcTc (move);
		break;
	case KPT_Bc_Tc:
		KPTmakeMoveBcTc (move);
		break;
	case KPT_Tc_A:
		KPTmakeMoveTcA (move, update_flags);
		break;
	case KPT_Tc_B:
		KPTmakeMoveTcB (move, update_flags);
		break;
	case KPT_Tc_Ac:
		KPTmakeMoveTcAc (move, update_flags);
		break;
	case KPT_Tc_Bc:
		KPTmakeMoveTcBc (move, update_flags);
		break;
	case KPT_AHc_AH:
		KPTmakeMoveAHcAH (move, update_flags);
		break;
	case KPT_BHc_BH:
		KPTmakeMoveBHcBH (move, update_flags);
		break;
	case KPT_AH_AHc:
		KPTmakeMoveAHAHc (move, update_flags);
		break;
	case KPT_BH_BHc:
		KPTmakeMoveBHBHc (move, update_flags);
		break;
	case KPT_AH_AHc_BH_BHc:
		KPTmakeMoveAHAHcBHBHc (move, update_flags);
		break;
	case KPT_AH_AHc_BHc_BH:
		KPTmakeMoveAHAHcBHcBH (move, update_flags);
		break;
	case KPT_AHc_AH_BHc_BH:
		KPTmakeMoveAHcAHBHcBH (move, update_flags);
		break;
	case KPT_AHc_AH_BH_BHc:
		KPTmakeMoveAHcAHBHBHc (move, update_flags);
		break;
	default:
		EXIT (1, "unknown move %d", move->move_id);
	}
	/* ending */
#if KPT_EDBG <= DEBUG
	EXITRVAL (KPTTestEdges ());
#endif
	return 0;
}

/* ========================================================================= */
/** @brief given a node, and a move, check if it is feasible, if update_flags 
 * is set to one, then it will check the constrains imposed by them. */
/* ========================================================================= */
static int KPTisMoveFeasible (KPTMove_t const *const move,
															const unsigned int update_flags)
{
	int rval = 0;
	unsigned int type;

	MESSAGE (KPT_VRB, "Entering for node %u, move %s, domino %u",
					 move->n_id, move_name[move->move_id], move->dom);

	/* we just switch the correct call */
	switch (move->move_id)
	{
	case KPT_Tc_Ac:
	case KPT_Tc_A:
		type = (EGbitTest (graphData.A_partition, move->dom) ? 1U : 0U) |
			(EGbitTest (graphData.B_partition, move->dom) ? 2U : 0U);
		rval = !EGbitTest (nodeData[move->n_id].It, move->dom) &&
			(graphData.n_Tc[move->dom] > 1) && (!update_flags ||
																					!EGbitTest (nodeData[move->n_id].
																											added, move->dom))
			&& (type == 1U);
		break;
	case KPT_Tc_Bc:
	case KPT_Tc_B:
		type = (EGbitTest (graphData.A_partition, move->dom) ? 1U : 0U) |
			(EGbitTest (graphData.B_partition, move->dom) ? 2U : 0U);
		rval = !EGbitTest (nodeData[move->n_id].It, move->dom) &&
			(graphData.n_Tc[move->dom] > 1) && (!update_flags ||
																					!EGbitTest (nodeData[move->n_id].
																											added, move->dom))
			&& (type == 2U);
		break;
	case KPT_Tc_AB:
	case KPT_Tc_ABc:
	case KPT_Tc_AcBc:
	case KPT_Tc_AcB:
		type = (EGbitTest (graphData.A_partition, move->dom) ? 1U : 0U) |
			(EGbitTest (graphData.B_partition, move->dom) ? 2U : 0U);
		rval = !EGbitTest (nodeData[move->n_id].It, move->dom) &&
			(graphData.n_Tc[move->dom] > 1) && (!update_flags ||
																					!EGbitTest (nodeData[move->n_id].
																											added, move->dom))
			&& (type == 3U);
		break;
	case KPT_AHc_AH:
		rval = !nodeData[move->n_id].in_AH && (graphData.n_AH[0] > 1) &&
			(!update_flags || !nodeData[move->n_id].added_AH);
		break;
	case KPT_BHc_BH:
		rval = !nodeData[move->n_id].in_BH && (graphData.n_BH[0] > 1) &&
			(!update_flags || !nodeData[move->n_id].added_BH);
		break;
	case KPT_AHc_AH_BHc_BH:
		rval = !nodeData[move->n_id].in_BH && (graphData.n_BH[0] > 1) &&
			(!update_flags || (!nodeData[move->n_id].added_BH &&
												 !nodeData[move->n_id].added_AH)) &&
			!nodeData[move->n_id].in_AH && (graphData.n_AH[0] > 1);
		break;
	case KPT_A_Ac:
		rval = EGbitTest (nodeData[move->n_id].Ia, move->dom) &&
			(graphData.n_A[1][move->dom] > 1) && (!update_flags ||
																						!EGbitTest (nodeData[move->n_id].
																												flipA, move->dom))
			&& (!EGbitTest (graphData.Kdom, move->dom)
					|| (graphData.
							n_AdB[EGbitTest (nodeData[move->n_id].Iadb, move->dom) ? 1U :
										0U][move->dom] > 1));
		break;
	case KPT_A_Ac_flipAH:
		rval = EGbitTest (nodeData[move->n_id].Ia, move->dom) &&
			(graphData.n_A[1][move->dom] > 1) && (!update_flags ||
																						(!EGbitTest
																						 (nodeData[move->n_id].flipA,
																							move->dom)))
			&& (graphData.n_AH[nodeData[move->n_id].in_AH] > 1)
			&& (!EGbitTest (graphData.Kdom, move->dom)
					|| (graphData.
							n_AdB[EGbitTest (nodeData[move->n_id].Iadb, move->dom) ? 1U :
										0U][move->dom] > 1));
		break;
	case KPT_Ac_A:
		rval = !EGbitTest (nodeData[move->n_id].Ia, move->dom) &&
			(graphData.n_A[0][move->dom] > 1) && (!update_flags ||
																						!EGbitTest (nodeData[move->n_id].
																												flipAc, move->dom))
			&& EGbitTest (nodeData[move->n_id].It, move->dom)
			&& (!EGbitTest (graphData.Kdom, move->dom)
					|| (graphData.
							n_AdB[EGbitTest (nodeData[move->n_id].Iadb, move->dom) ? 1U :
										0U][move->dom] > 1));
		break;
	case KPT_Ac_A_flipAH:
		rval = !EGbitTest (nodeData[move->n_id].Ia, move->dom) &&
			(graphData.n_A[0][move->dom] > 1) && (!update_flags ||
																						(!EGbitTest
																						 (nodeData[move->n_id].flipAc,
																							move->dom)))
			&& (graphData.n_AH[nodeData[move->n_id].in_AH] > 1)
			&& EGbitTest (nodeData[move->n_id].It, move->dom)
			&& (!EGbitTest (graphData.Kdom, move->dom)
					|| (graphData.
							n_AdB[EGbitTest (nodeData[move->n_id].Iadb, move->dom) ? 1U :
										0U][move->dom] > 1));
		break;
	case KPT_B_Bc:
		rval = EGbitTest (nodeData[move->n_id].Ib, move->dom) &&
			(graphData.n_B[1][move->dom] > 1) && (!update_flags ||
																						!EGbitTest (nodeData[move->n_id].
																												flipB, move->dom))
			&& (!EGbitTest (graphData.Kdom, move->dom)
					|| (graphData.
							n_AdB[EGbitTest (nodeData[move->n_id].Iadb, move->dom) ? 1U :
										0U][move->dom] > 1));
		break;
	case KPT_B_Bc_flipBH:
		rval = EGbitTest (nodeData[move->n_id].Ib, move->dom) &&
			(graphData.n_B[1][move->dom] > 1) && (!update_flags ||
																						(!EGbitTest
																						 (nodeData[move->n_id].flipB,
																							move->dom)))
			&& (graphData.n_BH[nodeData[move->n_id].in_BH] > 1)
			&& (!EGbitTest (graphData.Kdom, move->dom)
					|| (graphData.
							n_AdB[EGbitTest (nodeData[move->n_id].Iadb, move->dom) ? 1U :
										0U][move->dom] > 1));
		break;
	case KPT_Bc_B:
		rval = !EGbitTest (nodeData[move->n_id].Ib, move->dom) &&
			(graphData.n_B[0][move->dom] > 1) && (!update_flags ||
																						!EGbitTest (nodeData[move->n_id].
																												flipBc, move->dom))
			&& EGbitTest (nodeData[move->n_id].It, move->dom)
			&& (!EGbitTest (graphData.Kdom, move->dom)
					|| (graphData.
							n_AdB[EGbitTest (nodeData[move->n_id].Iadb, move->dom) ? 1U :
										0U][move->dom] > 1));
		break;
	case KPT_Bc_B_flipBH:
		rval = !EGbitTest (nodeData[move->n_id].Ib, move->dom) &&
			(graphData.n_B[0][move->dom] > 1) && (!update_flags ||
																						(!EGbitTest
																						 (nodeData[move->n_id].flipBc,
																							move->dom)))
			&& (graphData.n_BH[nodeData[move->n_id].in_BH] > 1)
			&& EGbitTest (nodeData[move->n_id].It, move->dom)
			&& (!EGbitTest (graphData.Kdom, move->dom)
					|| (graphData.
							n_AdB[EGbitTest (nodeData[move->n_id].Iadb, move->dom) ? 1U :
										0U][move->dom] > 1));
		break;
	case KPT_A_Ac_B_Bc_flipAHBH:
		rval = EGbitTest (nodeData[move->n_id].Ia, move->dom) &&
			(graphData.n_A[1][move->dom] > 1) &&
			EGbitTest (nodeData[move->n_id].Ib, move->dom) &&
			(graphData.n_B[1][move->dom] > 1) &&
			(graphData.n_BH[nodeData[move->n_id].in_BH] > 1) &&
			(graphData.n_AH[nodeData[move->n_id].in_AH] > 1) &&
			(!update_flags || (!EGbitTest (nodeData[move->n_id].flipA, move->dom) &&
												 !EGbitTest (nodeData[move->n_id].flipB, move->dom)));
		break;
	case KPT_A_Ac_Bc_B_flipAHBH:
		rval = EGbitTest (nodeData[move->n_id].Ia, move->dom) &&
			(graphData.n_A[1][move->dom] > 1) &&
			!EGbitTest (nodeData[move->n_id].Ib, move->dom) &&
			(graphData.n_B[0][move->dom] > 1) &&
			EGbitTest (nodeData[move->n_id].It, move->dom) &&
			(graphData.n_BH[nodeData[move->n_id].in_BH] > 1) &&
			(graphData.n_AH[nodeData[move->n_id].in_AH] > 1) &&
			(!update_flags || (!EGbitTest (nodeData[move->n_id].flipA, move->dom) &&
												 !EGbitTest (nodeData[move->n_id].flipBc, move->dom)));
		break;
	case KPT_Ac_A_B_Bc_flipAHBH:
		rval = !EGbitTest (nodeData[move->n_id].Ia, move->dom) &&
			(graphData.n_A[0][move->dom] > 1) &&
			EGbitTest (nodeData[move->n_id].It, move->dom) &&
			EGbitTest (nodeData[move->n_id].Ib, move->dom) &&
			(graphData.n_B[1][move->dom] > 1) &&
			(graphData.n_BH[nodeData[move->n_id].in_BH] > 1) &&
			(graphData.n_AH[nodeData[move->n_id].in_AH] > 1) &&
			(!update_flags || (!EGbitTest (nodeData[move->n_id].flipAc, move->dom) &&
												 !EGbitTest (nodeData[move->n_id].flipB, move->dom)));
		break;
	case KPT_Ac_A_Bc_B_flipAHBH:
		rval = !EGbitTest (nodeData[move->n_id].Ia, move->dom) &&
			(graphData.n_A[0][move->dom] > 1) &&
			EGbitTest (nodeData[move->n_id].It, move->dom) &&
			!EGbitTest (nodeData[move->n_id].Ib, move->dom) &&
			(graphData.n_B[0][move->dom] > 1) &&
			EGbitTest (nodeData[move->n_id].It, move->dom) &&
			(graphData.n_BH[nodeData[move->n_id].in_BH] > 1) &&
			(graphData.n_AH[nodeData[move->n_id].in_AH] > 1) &&
			(!update_flags || (!EGbitTest (nodeData[move->n_id].flipAc, move->dom) &&
												 !EGbitTest (nodeData[move->n_id].flipBc, move->dom)));
		break;
	case KPT_A_Ac_B_Bc_flipBH:
		rval = EGbitTest (nodeData[move->n_id].Ia, move->dom) &&
			(graphData.n_A[1][move->dom] > 1) &&
			EGbitTest (nodeData[move->n_id].Ib, move->dom) &&
			(graphData.n_B[1][move->dom] > 1) &&
			(graphData.n_BH[nodeData[move->n_id].in_BH] > 1) &&
			(!update_flags || (!EGbitTest (nodeData[move->n_id].flipA, move->dom) &&
												 !EGbitTest (nodeData[move->n_id].flipB, move->dom)));
		break;
	case KPT_A_Ac_Bc_B_flipBH:
		rval = EGbitTest (nodeData[move->n_id].Ia, move->dom) &&
			(graphData.n_A[1][move->dom] > 1) &&
			!EGbitTest (nodeData[move->n_id].Ib, move->dom) &&
			(graphData.n_B[0][move->dom] > 1) &&
			EGbitTest (nodeData[move->n_id].It, move->dom) &&
			(graphData.n_BH[nodeData[move->n_id].in_BH] > 1) &&
			(!update_flags || (!EGbitTest (nodeData[move->n_id].flipA, move->dom) &&
												 !EGbitTest (nodeData[move->n_id].flipBc, move->dom)));
		break;
	case KPT_Ac_A_B_Bc_flipBH:
		rval = !EGbitTest (nodeData[move->n_id].Ia, move->dom) &&
			(graphData.n_A[0][move->dom] > 1) &&
			EGbitTest (nodeData[move->n_id].It, move->dom) &&
			EGbitTest (nodeData[move->n_id].Ib, move->dom) &&
			(graphData.n_B[1][move->dom] > 1) &&
			(graphData.n_BH[nodeData[move->n_id].in_BH] > 1) &&
			(!update_flags || (!EGbitTest (nodeData[move->n_id].flipAc, move->dom) &&
												 !EGbitTest (nodeData[move->n_id].flipB, move->dom)));
		break;
	case KPT_Ac_A_Bc_B_flipBH:
		rval = !EGbitTest (nodeData[move->n_id].Ia, move->dom) &&
			(graphData.n_A[0][move->dom] > 1) &&
			EGbitTest (nodeData[move->n_id].It, move->dom) &&
			!EGbitTest (nodeData[move->n_id].Ib, move->dom) &&
			(graphData.n_B[0][move->dom] > 1) &&
			EGbitTest (nodeData[move->n_id].It, move->dom) &&
			(graphData.n_BH[nodeData[move->n_id].in_BH] > 1) &&
			(!update_flags || (!EGbitTest (nodeData[move->n_id].flipAc, move->dom) &&
												 !EGbitTest (nodeData[move->n_id].flipBc, move->dom)));
		break;
	case KPT_A_Ac_B_Bc_flipAH:
		rval = EGbitTest (nodeData[move->n_id].Ia, move->dom) &&
			(graphData.n_A[1][move->dom] > 1) &&
			EGbitTest (nodeData[move->n_id].Ib, move->dom) &&
			(graphData.n_B[1][move->dom] > 1) &&
			(graphData.n_AH[nodeData[move->n_id].in_AH] > 1) &&
			(!update_flags || (!EGbitTest (nodeData[move->n_id].flipA, move->dom) &&
												 !EGbitTest (nodeData[move->n_id].flipB, move->dom)));
		break;
	case KPT_A_Ac_Bc_B_flipAH:
		rval = EGbitTest (nodeData[move->n_id].Ia, move->dom) &&
			(graphData.n_A[1][move->dom] > 1) &&
			!EGbitTest (nodeData[move->n_id].Ib, move->dom) &&
			(graphData.n_B[0][move->dom] > 1) &&
			EGbitTest (nodeData[move->n_id].It, move->dom) &&
			(graphData.n_AH[nodeData[move->n_id].in_AH] > 1) &&
			(!update_flags || (!EGbitTest (nodeData[move->n_id].flipA, move->dom) &&
												 !EGbitTest (nodeData[move->n_id].flipBc, move->dom)));
		break;
	case KPT_Ac_A_B_Bc_flipAH:
		rval = !EGbitTest (nodeData[move->n_id].Ia, move->dom) &&
			(graphData.n_A[0][move->dom] > 1) &&
			EGbitTest (nodeData[move->n_id].It, move->dom) &&
			EGbitTest (nodeData[move->n_id].Ib, move->dom) &&
			(graphData.n_B[1][move->dom] > 1) &&
			(graphData.n_AH[nodeData[move->n_id].in_AH] > 1) &&
			(!update_flags || (!EGbitTest (nodeData[move->n_id].flipAc, move->dom) &&
												 !EGbitTest (nodeData[move->n_id].flipB, move->dom)));
		break;
	case KPT_Ac_A_Bc_B_flipAH:
		rval = !EGbitTest (nodeData[move->n_id].Ia, move->dom) &&
			(graphData.n_A[0][move->dom] > 1) &&
			EGbitTest (nodeData[move->n_id].It, move->dom) &&
			!EGbitTest (nodeData[move->n_id].Ib, move->dom) &&
			(graphData.n_B[0][move->dom] > 1) &&
			EGbitTest (nodeData[move->n_id].It, move->dom) &&
			(graphData.n_AH[nodeData[move->n_id].in_AH] > 1) &&
			(!update_flags || (!EGbitTest (nodeData[move->n_id].flipAc, move->dom) &&
												 !EGbitTest (nodeData[move->n_id].flipBc, move->dom)));
		break;
	case KPT_A_Ac_B_Bc:
		rval = EGbitTest (nodeData[move->n_id].Ia, move->dom) &&
			(graphData.n_A[1][move->dom] > 1) && (!update_flags ||
																						(!EGbitTest
																						 (nodeData[move->n_id].flipA,
																							move->dom)
																						 &&
																						 !EGbitTest (nodeData[move->n_id].
																												 flipB, move->dom)))
			&& EGbitTest (nodeData[move->n_id].Ib, move->dom)
			&& (graphData.n_B[1][move->dom] > 1);
		break;
	case KPT_A_Ac_Bc_B:
		rval = EGbitTest (nodeData[move->n_id].Ia, move->dom) &&
			(graphData.n_A[1][move->dom] > 1) && (!update_flags ||
																						(!EGbitTest
																						 (nodeData[move->n_id].flipA,
																							move->dom)
																						 &&
																						 !EGbitTest (nodeData[move->n_id].
																												 flipBc, move->dom)))
			&& !EGbitTest (nodeData[move->n_id].Ib, move->dom)
			&& (graphData.n_B[0][move->dom] > 1)
			&& EGbitTest (nodeData[move->n_id].It, move->dom);
		break;
	case KPT_Ac_A_B_Bc:
		rval = !EGbitTest (nodeData[move->n_id].Ia, move->dom) &&
			(graphData.n_A[0][move->dom] > 1) && (!update_flags ||
																						(!EGbitTest
																						 (nodeData[move->n_id].flipAc,
																							move->dom)
																						 &&
																						 !EGbitTest (nodeData[move->n_id].
																												 flipB, move->dom)))
			&& EGbitTest (nodeData[move->n_id].It, move->dom)
			&& EGbitTest (nodeData[move->n_id].Ib, move->dom)
			&& (graphData.n_B[1][move->dom] > 1);
		break;
	case KPT_Ac_A_Bc_B:
		rval = !EGbitTest (nodeData[move->n_id].Ia, move->dom) &&
			(graphData.n_A[0][move->dom] > 1) && (!update_flags ||
																						(!EGbitTest
																						 (nodeData[move->n_id].flipAc,
																							move->dom)
																						 &&
																						 !EGbitTest (nodeData[move->n_id].
																												 flipBc, move->dom)))
			&& EGbitTest (nodeData[move->n_id].It, move->dom)
			&& !EGbitTest (nodeData[move->n_id].Ib, move->dom)
			&& (graphData.n_B[0][move->dom] > 1)
			&& EGbitTest (nodeData[move->n_id].It, move->dom);
		break;
	case KPT_AH_AHc_BHc_BH:
		rval = nodeData[move->n_id].in_AH && (graphData.n_AH[1] > 1) &&
			!nodeData[move->n_id].in_BH && (graphData.n_BH[0] > 1) &&
			(!update_flags || !nodeData[move->n_id].added_BH);
		break;
	case KPT_AHc_AH_BH_BHc:
		rval = !nodeData[move->n_id].in_AH && (graphData.n_AH[0] > 1) &&
			nodeData[move->n_id].in_BH && (graphData.n_BH[1] > 1) &&
			(!update_flags || !nodeData[move->n_id].added_AH);
		break;
	case KPT_A_Tc:
		type = (EGbitTest (graphData.A_partition, move->dom) ? 1U : 0U) |
			(EGbitTest (graphData.B_partition, move->dom) ? 2U : 0U);
		rval = (type == 1U) && (graphData.n_A[1][move->dom] > 1) &&
			EGbitTest (nodeData[move->n_id].It, move->dom) &&
			EGbitTest (nodeData[move->n_id].Ia, move->dom);
		break;
	case KPT_Ac_Tc:
		type = (EGbitTest (graphData.A_partition, move->dom) ? 1U : 0U) |
			(EGbitTest (graphData.B_partition, move->dom) ? 2U : 0U);
		rval = (type == 1U) && (graphData.n_A[0][move->dom] > 1) &&
			EGbitTest (nodeData[move->n_id].It, move->dom) &&
			!EGbitTest (nodeData[move->n_id].Ia, move->dom);
		break;
	case KPT_B_Tc:
		type = (EGbitTest (graphData.A_partition, move->dom) ? 1U : 0U) |
			(EGbitTest (graphData.B_partition, move->dom) ? 2U : 0U);
		rval = (type == 2U) && (graphData.n_B[1][move->dom] > 1) &&
			EGbitTest (nodeData[move->n_id].It, move->dom) &&
			EGbitTest (nodeData[move->n_id].Ib, move->dom);
		break;
	case KPT_Bc_Tc:
		type = (EGbitTest (graphData.A_partition, move->dom) ? 1U : 0U) |
			(EGbitTest (graphData.B_partition, move->dom) ? 2U : 0U);
		rval = (type == 2U) && (graphData.n_B[0][move->dom] > 1) &&
			EGbitTest (nodeData[move->n_id].It, move->dom) &&
			!EGbitTest (nodeData[move->n_id].Ib, move->dom);
		break;
	case KPT_AB_Tc:
		type = (EGbitTest (graphData.A_partition, move->dom) ? 1U : 0U) |
			(EGbitTest (graphData.B_partition, move->dom) ? 2U : 0U);
		rval = (type == 3U) && EGbitTest (nodeData[move->n_id].It, move->dom) &&
			EGbitTest (nodeData[move->n_id].Ia, move->dom) &&
			EGbitTest (nodeData[move->n_id].Ib, move->dom) &&
			(graphData.n_A[1][move->dom] > 1) &&
			(graphData.n_B[1][move->dom] > 1) && (graphData.n_AdB[1][move->dom] > 1);
		break;
	case KPT_AcB_Tc:
		type = (EGbitTest (graphData.A_partition, move->dom) ? 1U : 0U) |
			(EGbitTest (graphData.B_partition, move->dom) ? 2U : 0U);
		rval = (type == 3U) && EGbitTest (nodeData[move->n_id].It, move->dom) &&
			!EGbitTest (nodeData[move->n_id].Ia, move->dom) &&
			EGbitTest (nodeData[move->n_id].Ib, move->dom) &&
			(graphData.n_A[0][move->dom] > 1) &&
			(graphData.n_B[1][move->dom] > 1) && (graphData.n_AdB[0][move->dom] > 1);
		break;
	case KPT_ABc_Tc:
		type = (EGbitTest (graphData.A_partition, move->dom) ? 1U : 0U) |
			(EGbitTest (graphData.B_partition, move->dom) ? 2U : 0U);
		rval = (type == 3U) && EGbitTest (nodeData[move->n_id].It, move->dom) &&
			EGbitTest (nodeData[move->n_id].Ia, move->dom) &&
			!EGbitTest (nodeData[move->n_id].Ib, move->dom) &&
			(graphData.n_A[1][move->dom] > 1) &&
			(graphData.n_B[0][move->dom] > 1) && (graphData.n_AdB[0][move->dom] > 1);
		break;
	case KPT_AcBc_Tc:
		type = (EGbitTest (graphData.A_partition, move->dom) ? 1U : 0U) |
			(EGbitTest (graphData.B_partition, move->dom) ? 2U : 0U);
		rval = (type == 3U) && EGbitTest (nodeData[move->n_id].It, move->dom) &&
			!EGbitTest (nodeData[move->n_id].Ia, move->dom) &&
			!EGbitTest (nodeData[move->n_id].Ib, move->dom) &&
			(graphData.n_A[0][move->dom] > 1) &&
			(graphData.n_B[0][move->dom] > 1) && (graphData.n_AdB[1][move->dom] > 1);
		break;
	case KPT_AH_AHc:
		rval = nodeData[move->n_id].in_AH && (graphData.n_AH[1] > 1);
		break;
	case KPT_BH_BHc:
		rval = nodeData[move->n_id].in_BH && (graphData.n_BH[1] > 1);
		break;
	case KPT_AH_AHc_BH_BHc:
		rval = nodeData[move->n_id].in_BH && (graphData.n_BH[1] > 1) &&
			nodeData[move->n_id].in_AH && (graphData.n_AH[1] > 1);
		break;
	case KPT_no_move:
		rval = 1;
		break;
	default:
		EXIT (1, "unknown move %d", move->move_id);
	}
	/* ending */
	MESSAGE (KPT_VRB, "done");
	return rval;
}

/* ========================================================================= */
/** @brief check if a full move is feasible */
static inline int KPTisFullMoveFeasible (KPTFullMove_t const *const full_move)
{
	int rval = 1;
	unsigned int depth = 0;
	MESSAGE (KPT_VRB + 19, "entering, depth %u", full_move->depth);
	/* check if each move is feasible, make it, and check next move */
	while (1)
	{
		rval = KPTisMoveFeasible (full_move->move + depth, 1);
		WARNINGL (KPT_EDBG, !rval, "move level %u/%u is infeasible for node %u",
							depth, full_move->depth, full_move->move[0].n_id);
		if (!rval || (depth + 1 >= full_move->depth))
			break;
		KPTmakeMove (full_move->move + depth, 0);
		depth++;
	}
	/* undo the moves */
	while (depth--)
		KPTmakeInvMove (full_move->move + depth, 0);
	MESSAGE (KPT_VRB + 19, "done, return %d", rval);
	return rval;
}

/* ========================================================================= */
/** @brief, check if we should update a move, if so, do the update */
/* ========================================================================= */
#if KPT_EDBG < DEBUG
#define KPTupdateMove(__cmove,__bmove) __KPTupdateMove(__cmove,__bmove,__FILE__,__LINE__,__func__)
#else
#define KPTupdateMove(__cmove,__bmove) __KPTupdateMove(__cmove,__bmove)
#endif
static int __KPTupdateMove (const KPTFullMove_t * const cur_move,
														KPTFullMove_t * const best_move
#if KPT_EDBG < DEBUG
														,
														const char *file,
														const int line,
														const char *function
#endif
	)
{
	if ((cur_move->val + fabs (KPTMinImprovement) < best_move->val) ||
			(cur_move->move[0].move_id < best_move->move[0].move_id &&
			 (fabs (cur_move->val - best_move->val) < fabs (KPTMinImprovement))))
	{
#if KPT_VRB+1900<DEBUG
		fprintf (stderr, "Storing Move for node %u\n", cur_move->move[0].n_id);
		{
			unsigned l_depth = cur_move->depth;
			fprintf (stderr, "New Move:\n");
			while (l_depth--)
				DPdisplaySingleMove (cur_move->move + l_depth);
			fprintf (stderr, "\tValue %lf\n", cur_move->val);
			l_depth = best_move->depth;
			fprintf (stderr, "Old Move:\n");
			while (l_depth--)
				DPdisplaySingleMove (best_move->move + l_depth);
			fprintf (stderr, "\tValue %lf\n", best_move->val);
		}
#endif
#if KPT_EDBG < DEBUG
		{
			unsigned int l_depth = cur_move->depth;
			while (l_depth--)
				EXIT (cur_move->move[l_depth].move_id == KPT_no_move,
							"Storing KPT_no_move for depth %u, called from %s at %s:%d",
							l_depth, function, file, line);
		}
#endif
		memcpy (best_move, cur_move, sizeof (KPTFullMove_t));
#if KPT_EDBG < DEBUG
		if (!KPTisFullMoveFeasible (best_move))
		{
			unsigned l_depth = best_move->depth;
			while (l_depth--)
				DPdisplaySingleMove (best_move->move + l_depth);
			fprintf (stderr, "\tValue %lf\n", best_move->val);
			WARNING (1, "Baaaaaaaaaad");
		}
#endif
	}
	return 0;
}

/* ========================================================================= */
/** @brief This function prices the move BHBHc,assuming the move is feasible */
/* ========================================================================= */
static inline int KPTpriceBHBHc (KPTMove_t const *const cur_move,
																 double *const move_val)
{
	/* local variables */
	EGlistNode_t *e_it = all_nodes[cur_move->n_id]->edges->begin;
	unsigned e_id;

	/* we have to compute every move */
	for (; e_it; e_it = e_it->next)
	{
		e_id = KPTgetEdgeId (e_it);
		/* we just flip BF */
		if (edgeData[e_id].num.in_BF)
			(*move_val) -= weight[e_id];
		else
			(*move_val) += weight[e_id];
	}															/* end loop through all incident edges */
	return 0;
}

/* ========================================================================= */
/** @brief This function prices the move BHcBH,assuming the move is feasible */
/* ========================================================================= */
#define KPTpriceBHcBH(__move,__val) KPTpriceBHBHc(__move,__val)

/* ========================================================================= */
/** @brief This function prices the move AHAHc,assuming the move is feasible */
/* ========================================================================= */
static inline int KPTpriceAHAHc (KPTMove_t const *const cur_move,
																 double *const move_val)
{
	/* local variables */
	EGlistNode_t *e_it;
	unsigned e_id;

	/* we have to compute every move */
	for (e_it = all_nodes[cur_move->n_id]->edges->begin; e_it; e_it = e_it->next)
	{
		e_id = KPTgetEdgeId (e_it);
		/* we just flip AF */
		if (edgeData[e_id].num.in_AF)
			(*move_val) -= weight[e_id];
		else
			(*move_val) += weight[e_id];
	}															/* end loop through all incident edges */
	return 0;
}

/* ========================================================================= */
/** @brief This function prices the move AHcAH,assuming the move is feasible */
/* ========================================================================= */
#define KPTpriceAHcAH(__id,__val) KPTpriceAHAHc(__id,__val)

/* ========================================================================= */
/** @brief This function prices the move AHAHcBHBHc,assuming 
 * the move is feasible */
/* ========================================================================= */
static inline int KPTpriceAHAHcBHBHc (KPTMove_t const *const cur_move,
																			double *const move_val)
{
	/* local variables */
	EGlistNode_t *e_it;
	unsigned e_id;

	/* we have to compute every move */
	for (e_it = all_nodes[cur_move->n_id]->edges->begin; e_it; e_it = e_it->next)
	{
		e_id = KPTgetEdgeId (e_it);
		/* we just flip AF */
		if (edgeData[e_id].num.in_AF)
			(*move_val) -= weight[e_id];
		else
			(*move_val) += weight[e_id];
		/* we just flip BF */
		if (edgeData[e_id].num.in_BF)
			(*move_val) -= weight[e_id];
		else
			(*move_val) += weight[e_id];
	}															/* end loop through all incident edges */
	return 0;
}

/* ========================================================================= */
/** @brief This function prices the move AHcAH,assuming the move is feasible */
/* ========================================================================= */
#define KPTpriceAHAHcBHcBH(__id,__val) KPTpriceAHAHcBHBHc(__id,__val)

/* ========================================================================= */
/** @brief This function prices the move AHcAH,assuming the move is feasible */
/* ========================================================================= */
#define KPTpriceAHcAHBHBHc(__id,__val) KPTpriceAHAHcBHBHc(__id,__val)

/* ========================================================================= */
/** @brief This function prices the move AHcAH,assuming the move is feasible */
/* ========================================================================= */
#define KPTpriceAHcAHBHcBH(__id,__val) KPTpriceAHAHcBHBHc(__id,__val)

/* ========================================================================= */
/** @brief This function prices the move AcBTc, assuming the move is feasible*/
/* ========================================================================= */
static inline int KPTpriceAcBTc (const KPTMove_t * const cur_move,
																 double *const move_val)
{
	/* local variables */
	EGlistNode_t *e_it = all_nodes[cur_move->n_id]->edges->begin;
	unsigned int e_id,
	  no_id,
	  pos;

	/* we have to compute every move */
	for (; e_it; e_it = e_it->next)
	{
		e_id = KPTgetEdgeId (e_it);
		no_id = KPTgetOtherEndId (cur_move->n_id, e_it);
		pos = (EGbitTest (nodeData[no_id].It, cur_move->dom) ? 1U : 0U) |
			(EGbitTest (nodeData[no_id].Ia, cur_move->dom) ? 2U : 0U) |
			(EGbitTest (nodeData[no_id].Ib, cur_move->dom) ? 4U : 0U);
		switch (pos)
		{
		case 5U:
			/* in this case, the only change is that we have to pay for this edge in
			 * Delta(T) */
			(*move_val) += weight[e_id];
			break;
		case 3U:
			/* in this case we pay one Delta(T), save one E(A:Ac) and flip AF, save 
			 * one E(B:Bc) and flip BF */
			(*move_val) += weight[e_id];
			if (edgeData[e_id].num.in_AF)
				(*move_val) -= 2 * weight[e_id];
			if (edgeData[e_id].num.in_BF)
				(*move_val) -= 2 * weight[e_id];
			break;
		case 1U:
			/* in this case we pay one Delta(T), save one E(B:Bc) and flip BF */
			if (edgeData[e_id].num.in_BF)
				(*move_val) -= weight[e_id];
			else
				(*move_val) += weight[e_id];
			break;
		case 7U:
			/* in this case we pay one Delta(T), save one E(A:Ac) and flip AF */
			if (edgeData[e_id].num.in_AF)
				(*move_val) -= weight[e_id];
			else
				(*move_val) += weight[e_id];
			break;
		case 0U:
			/* in this case we save one Delta(T) */
			(*move_val) -= weight[e_id];
			break;
		default:
			EXIT (1, "This should not happen, value %u", pos);
		}														/* end switch */
	}															/* end loop through all incident edges */
	return 0;
}

/* ========================================================================= */
/** @brief This function prices the move ABcTc, assuming the move is feasible*/
/* ========================================================================= */
static inline int KPTpriceABcTc (const KPTMove_t * const cur_move,
																 double *const move_val)
{
	/* local variables */
	EGlistNode_t *e_it = all_nodes[cur_move->n_id]->edges->begin;
	unsigned int e_id,
	  no_id,
	  pos;

	/* we have to compute every move */
	for (; e_it; e_it = e_it->next)
	{
		e_id = KPTgetEdgeId (e_it);
		no_id = KPTgetOtherEndId (cur_move->n_id, e_it);
		pos = (EGbitTest (nodeData[no_id].It, cur_move->dom) ? 1U : 0U) |
			(EGbitTest (nodeData[no_id].Ia, cur_move->dom) ? 2U : 0U) |
			(EGbitTest (nodeData[no_id].Ib, cur_move->dom) ? 4U : 0U);
		switch (pos)
		{
		case 3U:
			/* in this case, the only change is that we have to pay for this edge in
			 * Delta(T) */
			(*move_val) += weight[e_id];
			break;
		case 5U:
			/* in this case we pay one Delta(T), save one E(A:Ac) and flip AF, save 
			 * one E(B:Bc) and flip BF */
			(*move_val) += weight[e_id];
			if (edgeData[e_id].num.in_AF)
				(*move_val) -= 2 * weight[e_id];
			if (edgeData[e_id].num.in_BF)
				(*move_val) -= 2 * weight[e_id];
			break;
		case 7U:
			/* in this case we pay one Delta(T), save one E(B:Bc) and flip BF */
			if (edgeData[e_id].num.in_BF)
				(*move_val) -= weight[e_id];
			else
				(*move_val) += weight[e_id];
			break;
		case 1U:
			/* in this case we pay one Delta(T), save one E(A:Ac) and flip AF */
			if (edgeData[e_id].num.in_AF)
				(*move_val) -= weight[e_id];
			else
				(*move_val) += weight[e_id];
			break;
		case 0U:
			/* in this case we save one Delta(T) */
			(*move_val) -= weight[e_id];
			break;
		default:
			EXIT (1, "This should not happen, value %u", pos);
		}														/* end switch */
	}															/* end loop through all incident edges */
	return 0;
}

/* ========================================================================= */
/** @brief This function prices the move AcBTc, assuming the move is feasible*/
/* ========================================================================= */
static inline int KPTpriceAcBcTc (const KPTMove_t * const cur_move,
																	double *const move_val)
{
	/* local variables */
	EGlistNode_t *e_it = all_nodes[cur_move->n_id]->edges->begin;
	unsigned int e_id,
	  no_id,
	  pos;

	/* we have to compute every move */
	for (; e_it; e_it = e_it->next)
	{
		e_id = KPTgetEdgeId (e_it);
		no_id = KPTgetOtherEndId (cur_move->n_id, e_it);
		pos = (EGbitTest (nodeData[no_id].It, cur_move->dom) ? 1U : 0U) |
			(EGbitTest (nodeData[no_id].Ia, cur_move->dom) ? 2U : 0U) |
			(EGbitTest (nodeData[no_id].Ib, cur_move->dom) ? 4U : 0U);
		switch (pos)
		{
		case 1U:
			/* in this case, the only change is that we have to pay for this edge in
			 * Delta(T) */
			(*move_val) += weight[e_id];
			break;
		case 7U:
			/* in this case we pay one Delta(T), save one E(A:Ac) and flip AF, save 
			 * one E(B:Bc) and flip BF */
			(*move_val) += weight[e_id];
			if (edgeData[e_id].num.in_AF)
				(*move_val) -= 2 * weight[e_id];
			if (edgeData[e_id].num.in_BF)
				(*move_val) -= 2 * weight[e_id];
			break;
		case 5U:
			/* in this case we pay one Delta(T), save one E(B:Bc) and flip BF */
			if (edgeData[e_id].num.in_BF)
				(*move_val) -= weight[e_id];
			else
				(*move_val) += weight[e_id];
			break;
		case 3U:
			/* in this case we pay one Delta(T), save one E(A:Ac) and flip AF */
			if (edgeData[e_id].num.in_AF)
				(*move_val) -= weight[e_id];
			else
				(*move_val) += weight[e_id];
			break;
		case 0U:
			/* in this case we save one Delta(T) */
			(*move_val) -= weight[e_id];
			break;
		default:
			EXIT (1, "This should not happen, value %u", pos);
		}														/* end switch */
	}															/* end loop through all incident edges */
	return 0;
}

/* ========================================================================= */
/** @brief This function prices the move BBcflipBH, assuming the move is
 * feasible*/
/* ========================================================================= */
static inline int KPTpriceBBcflipBH (const KPTMove_t * const cur_move,
																		 double *const move_val)
{
	/* local variables */
	EGlistNode_t *e_it = all_nodes[cur_move->n_id]->edges->begin;
	unsigned int e_id,
	  no_id,
	  pos;

	/* we have to compute every move */
	for (; e_it; e_it = e_it->next)
	{
		e_id = KPTgetEdgeId (e_it);
		no_id = KPTgetOtherEndId (cur_move->n_id, e_it);
		pos = (EGbitTest (nodeData[no_id].It, cur_move->dom) ? 1U : 0U) |
			(EGbitTest (nodeData[no_id].Ib, cur_move->dom) ? 2U : 0U);
		switch (pos)
		{
		case 0U:
			/* in this case, change BH and flip BF  */
			if (edgeData[e_id].num.in_BF)
				(*move_val) -= weight[e_id];
			else
				(*move_val) += weight[e_id];
			break;
		case 1U:
			/* in this case we save one E(B:Bc) and double flip BF and change BH */
			(*move_val) -= weight[e_id];
			break;
		case 3U:
			/* in this case we pay one E(B:Bc) and double flip BF and change BH */
			(*move_val) += weight[e_id];
			break;
		default:
			EXIT (1, "This should not happen, value %u", pos);
		}														/* end switch */
	}															/* end loop through all incident edges */
	return 0;
}

/* ========================================================================= */
/** @brief This function prices the move BcBflipBH, assuming the move is feasible*/
/* ========================================================================= */
static inline int KPTpriceBcBflipBH (const KPTMove_t * const cur_move,
																		 double *const move_val)
{
	/* local variables */
	EGlistNode_t *e_it = all_nodes[cur_move->n_id]->edges->begin;
	unsigned int e_id,
	  no_id,
	  pos;

	/* we have to compute every move */
	for (; e_it; e_it = e_it->next)
	{
		e_id = KPTgetEdgeId (e_it);
		no_id = KPTgetOtherEndId (cur_move->n_id, e_it);
		pos = (EGbitTest (nodeData[no_id].It, cur_move->dom) ? 1U : 0U) |
			(EGbitTest (nodeData[no_id].Ib, cur_move->dom) ? 2U : 0U);
		switch (pos)
		{
		case 0U:
			/* in this case, change BH and flip BF  */
			if (edgeData[e_id].num.in_BF)
				(*move_val) -= weight[e_id];
			else
				(*move_val) += weight[e_id];
			break;
		case 3U:
			/* in this case we save one E(B:Bc) and double flip BF and change BH */
			(*move_val) -= weight[e_id];
			break;
		case 1U:
			/* in this case we pay one E(B:Bc) and double flip BF and change BH */
			(*move_val) += weight[e_id];
			break;
		default:
			EXIT (1, "This should not happen, value %u", pos);
		}														/* end switch */
	}															/* end loop through all incident edges */
	return 0;
}


/* ========================================================================= */
/** @brief This function prices the move AAcflipAH, assuming the move is feasible*/
/* ========================================================================= */
static inline int KPTpriceAAcflipAH (const KPTMove_t * const cur_move,
																		 double *const move_val)
{
	/* local variables */
	EGlistNode_t *e_it = all_nodes[cur_move->n_id]->edges->begin;
	unsigned int e_id,
	  no_id,
	  pos;

	/* we have to compute every move */
	for (; e_it; e_it = e_it->next)
	{
		e_id = KPTgetEdgeId (e_it);
		no_id = KPTgetOtherEndId (cur_move->n_id, e_it);
		pos = (EGbitTest (nodeData[no_id].It, cur_move->dom) ? 1U : 0U) |
			(EGbitTest (nodeData[no_id].Ia, cur_move->dom) ? 2U : 0U);
		switch (pos)
		{
		case 0U:
			/* in this case, change AH and flip AF  */
			if (edgeData[e_id].num.in_AF)
				(*move_val) -= weight[e_id];
			else
				(*move_val) += weight[e_id];
			break;
		case 1U:
			/* in this case we save one E(A:Ac) and double flip AF and change AH */
			(*move_val) -= weight[e_id];
			break;
		case 3U:
			/* in this case we pay one E(A:Ac) and double flip AF and change AH */
			(*move_val) += weight[e_id];
			break;
		default:
			EXIT (1, "This should not happen, value %u", pos);
		}														/* end switch */
	}															/* end loop through all incident edges */
	return 0;
}

/* ========================================================================= */
/** @brief This function prices the move AcAflipAH, assuming the move is feasible*/
/* ========================================================================= */
static inline int KPTpriceAcAflipAH (const KPTMove_t * const cur_move,
																		 double *const move_val)
{
	/* local variables */
	EGlistNode_t *e_it = all_nodes[cur_move->n_id]->edges->begin;
	unsigned int e_id,
	  no_id,
	  pos;

	/* we have to compute every move */
	for (; e_it; e_it = e_it->next)
	{
		e_id = KPTgetEdgeId (e_it);
		no_id = KPTgetOtherEndId (cur_move->n_id, e_it);
		pos = (EGbitTest (nodeData[no_id].It, cur_move->dom) ? 1U : 0U) |
			(EGbitTest (nodeData[no_id].Ia, cur_move->dom) ? 2U : 0U);
		switch (pos)
		{
		case 0U:
			/* in this case, change AH and flip AF  */
			if (edgeData[e_id].num.in_AF)
				(*move_val) -= weight[e_id];
			else
				(*move_val) += weight[e_id];
			break;
		case 3U:
			/* in this case we save one E(A:Ac) and double flip AF and change AH */
			(*move_val) -= weight[e_id];
			break;
		case 1U:
			/* in this case we pay one E(A:Ac) and double flip AF and change AH */
			(*move_val) += weight[e_id];
			break;
		default:
			EXIT (1, "This should not happen, value %u", pos);
		}														/* end switch */
	}															/* end loop through all incident edges */
	return 0;
}

/* ========================================================================= */
/** @brief This function prices the move BBc, assuming the move is feasible*/
/* ========================================================================= */
static inline int KPTpriceBBc (const KPTMove_t * const cur_move,
															 double *const move_val)
{
	/* local variables */
	EGlistNode_t *e_it = all_nodes[cur_move->n_id]->edges->begin;
	unsigned int e_id,
	  no_id,
	  pos;

	/* we have to compute every move */
	for (; e_it; e_it = e_it->next)
	{
		e_id = KPTgetEdgeId (e_it);
		no_id = KPTgetOtherEndId (cur_move->n_id, e_it);
		pos = (EGbitTest (nodeData[no_id].It, cur_move->dom) ? 1U : 0U) |
			(EGbitTest (nodeData[no_id].Ib, cur_move->dom) ? 2U : 0U);
		switch (pos)
		{
		case 0U:
			/* in this case, nothing happen  */
			break;
		case 1U:
			/* in this case we save one E(B:Bc) and flip AF */
			if (edgeData[e_id].num.in_BF)
				(*move_val) -= 2 * weight[e_id];
			break;
		case 3U:
			/* in this case we pay one E(B:Bc) and flip AF */
			if (!edgeData[e_id].num.in_BF)
				(*move_val) += 2 * weight[e_id];
			break;
		default:
			EXIT (1, "This should not happen, value %u", pos);
		}														/* end switch */
	}															/* end loop through all incident edges */
	return 0;
}

/* ========================================================================= */
/** @brief This function prices the move BcB, assuming the move is feasible*/
/* ========================================================================= */
static inline int KPTpriceBcB (const KPTMove_t * const cur_move,
															 double *const move_val)
{
	/* local variables */
	EGlistNode_t *e_it = all_nodes[cur_move->n_id]->edges->begin;
	unsigned int e_id,
	  no_id,
	  pos;

	/* we have to compute every move */
	for (; e_it; e_it = e_it->next)
	{
		e_id = KPTgetEdgeId (e_it);
		no_id = KPTgetOtherEndId (cur_move->n_id, e_it);
		pos = (EGbitTest (nodeData[no_id].It, cur_move->dom) ? 1U : 0U) |
			(EGbitTest (nodeData[no_id].Ib, cur_move->dom) ? 2U : 0U);
		switch (pos)
		{
		case 0U:
			/* in this case, nothing happen  */
			break;
		case 3U:
			/* in this case we save one E(B:Bc) and flip AF */
			if (edgeData[e_id].num.in_BF)
				(*move_val) -= 2 * weight[e_id];
			break;
		case 1U:
			/* in this case we pay one E(B:Bc) and flip AF */
			if (!edgeData[e_id].num.in_BF)
				(*move_val) += 2 * weight[e_id];
			break;
		default:
			EXIT (1, "This should not happen, value %u", pos);
		}														/* end switch */
	}															/* end loop through all incident edges */
	return 0;
}


/* ========================================================================= */
/** @brief This function prices the move AAc, assuming the move is feasible*/
/* ========================================================================= */
static inline int KPTpriceAAc (const KPTMove_t * const cur_move,
															 double *const move_val)
{
	/* local variables */
	EGlistNode_t *e_it = all_nodes[cur_move->n_id]->edges->begin;
	unsigned int e_id,
	  no_id,
	  pos;

	/* we have to compute every move */
	for (; e_it; e_it = e_it->next)
	{
		e_id = KPTgetEdgeId (e_it);
		no_id = KPTgetOtherEndId (cur_move->n_id, e_it);
		pos = (EGbitTest (nodeData[no_id].It, cur_move->dom) ? 1U : 0U) |
			(EGbitTest (nodeData[no_id].Ia, cur_move->dom) ? 2U : 0U);
		switch (pos)
		{
		case 0U:
			/* in this case, nothing happen  */
			break;
		case 1U:
			/* in this case we save one E(A:Ac) and flip AF */
			if (edgeData[e_id].num.in_AF)
				(*move_val) -= 2 * weight[e_id];
			break;
		case 3U:
			/* in this case we pay one E(A:Ac) and flip AF */
			if (!edgeData[e_id].num.in_AF)
				(*move_val) += 2 * weight[e_id];
			break;
		default:
			EXIT (1, "This should not happen, value %u", pos);
		}														/* end switch */
	}															/* end loop through all incident edges */
	return 0;
}

/* ========================================================================= */
/** @brief This function prices the move AcA, assuming the move is feasible*/
/* ========================================================================= */
static inline int KPTpriceAcA (const KPTMove_t * const cur_move,
															 double *const move_val)
{
	/* local variables */
	EGlistNode_t *e_it = all_nodes[cur_move->n_id]->edges->begin;
	unsigned int e_id,
	  no_id,
	  pos;

	/* we have to compute every move */
	for (; e_it; e_it = e_it->next)
	{
		e_id = KPTgetEdgeId (e_it);
		no_id = KPTgetOtherEndId (cur_move->n_id, e_it);
		pos = (EGbitTest (nodeData[no_id].It, cur_move->dom) ? 1U : 0U) |
			(EGbitTest (nodeData[no_id].Ia, cur_move->dom) ? 2U : 0U);
		switch (pos)
		{
		case 0U:
			/* in this case, nothing happen  */
			break;
		case 3U:
			/* in this case we save one E(A:Ac) and flip AF */
			if (edgeData[e_id].num.in_AF)
				(*move_val) -= 2 * weight[e_id];
			break;
		case 1U:
			/* in this case we pay one E(A:Ac) and flip AF */
			if (!edgeData[e_id].num.in_AF)
				(*move_val) += 2 * weight[e_id];
			break;
		default:
			EXIT (1, "This should not happen, value %u", pos);
		}														/* end switch */
	}															/* end loop through all incident edges */
	return 0;
}

/* ========================================================================= */
/** @brief This function prices the move AcABBc, assuming the move is 
 * feasible*/
/* ========================================================================= */
static inline int KPTpriceAcABBc (const KPTMove_t * const cur_move,
																	double *const move_val)
{
	/* local variables */
	EGlistNode_t *e_it = all_nodes[cur_move->n_id]->edges->begin;
	unsigned int e_id,
	  no_id,
	  pos;

	/* we have to compute every move */
	for (; e_it; e_it = e_it->next)
	{
		e_id = KPTgetEdgeId (e_it);
		no_id = KPTgetOtherEndId (cur_move->n_id, e_it);
		pos = (EGbitTest (nodeData[no_id].It, cur_move->dom) ? 1U : 0U) |
			(EGbitTest (nodeData[no_id].Ia, cur_move->dom) ? 2U : 0U) |
			(EGbitTest (nodeData[no_id].Ib, cur_move->dom) ? 4U : 0U);
		switch (pos)
		{
		case 0U:
			/* in this case, nothing happen */
			break;
		case 3U:
			/* in this case we save one E(A:Ac) and flip AF, save 
			 * one E(B:Bc) and flip BF */
			if (edgeData[e_id].num.in_AF)
				(*move_val) -= 2 * weight[e_id];
			if (edgeData[e_id].num.in_BF)
				(*move_val) -= 2 * weight[e_id];
			break;
		case 1U:
			/* in this case we pay one E(A:Ac) and flip AF, save 
			 * one E(B:Bc) and flip BF */
			if (!edgeData[e_id].num.in_AF)
				(*move_val) += 2 * weight[e_id];
			if (edgeData[e_id].num.in_BF)
				(*move_val) -= 2 * weight[e_id];
			break;
		case 7U:
			/* in this case we save one E(A:Ac) and flip AF, pay 
			 * one E(B:Bc) and flip BF */
			if (edgeData[e_id].num.in_AF)
				(*move_val) -= 2 * weight[e_id];
			if (!edgeData[e_id].num.in_BF)
				(*move_val) += 2 * weight[e_id];
			break;
		case 5U:
			/* in this case we pay one E(A:Ac) and flip AF, pay 
			 * one E(B:Bc) and flip BF */
			if (!edgeData[e_id].num.in_AF)
				(*move_val) += 2 * weight[e_id];
			if (!edgeData[e_id].num.in_BF)
				(*move_val) += 2 * weight[e_id];
			break;
		default:
			EXIT (1, "This should not happen, value %u", pos);
		}														/* end switch */
	}															/* end loop through all incident edges */
	return 0;
}

/* ========================================================================= */
/** @brief This function prices the move AAcBcB, assuming the move is 
 * feasible*/
/* ========================================================================= */
static inline int KPTpriceAAcBcB (const KPTMove_t * const cur_move,
																	double *const move_val)
{
	/* local variables */
	EGlistNode_t *e_it = all_nodes[cur_move->n_id]->edges->begin;
	unsigned int e_id,
	  no_id,
	  pos;

	/* we have to compute every move */
	for (; e_it; e_it = e_it->next)
	{
		e_id = KPTgetEdgeId (e_it);
		no_id = KPTgetOtherEndId (cur_move->n_id, e_it);
		pos = (EGbitTest (nodeData[no_id].It, cur_move->dom) ? 1U : 0U) |
			(EGbitTest (nodeData[no_id].Ia, cur_move->dom) ? 2U : 0U) |
			(EGbitTest (nodeData[no_id].Ib, cur_move->dom) ? 4U : 0U);
		switch (pos)
		{
		case 0U:
			/* in this case, nothing happen */
			break;
		case 5U:
			/* in this case we save one E(A:Ac) and flip AF, save 
			 * one E(B:Bc) and flip BF */
			if (edgeData[e_id].num.in_AF)
				(*move_val) -= 2 * weight[e_id];
			if (edgeData[e_id].num.in_BF)
				(*move_val) -= 2 * weight[e_id];
			break;
		case 7U:
			/* in this case we pay one E(A:Ac) and flip AF, save 
			 * one E(B:Bc) and flip BF */
			if (!edgeData[e_id].num.in_AF)
				(*move_val) += 2 * weight[e_id];
			if (edgeData[e_id].num.in_BF)
				(*move_val) -= 2 * weight[e_id];
			break;
		case 1U:
			/* in this case we save one E(A:Ac) and flip AF, pay 
			 * one E(B:Bc) and flip BF */
			if (edgeData[e_id].num.in_AF)
				(*move_val) -= 2 * weight[e_id];
			if (!edgeData[e_id].num.in_BF)
				(*move_val) += 2 * weight[e_id];
			break;
		case 3U:
			/* in this case we pay one E(A:Ac) and flip AF, pay 
			 * one E(B:Bc) and flip BF */
			if (!edgeData[e_id].num.in_AF)
				(*move_val) += 2 * weight[e_id];
			if (!edgeData[e_id].num.in_BF)
				(*move_val) += 2 * weight[e_id];
			break;
		default:
			EXIT (1, "This should not happen, value %u", pos);
		}														/* end switch */
	}															/* end loop through all incident edges */
	return 0;
}

/* ========================================================================= */
/** @brief This function prices the move AcABcB, assuming the move is 
 * feasible*/
/* ========================================================================= */
static inline int KPTpriceAcABcB (const KPTMove_t * const cur_move,
																	double *const move_val)
{
	/* local variables */
	EGlistNode_t *e_it = all_nodes[cur_move->n_id]->edges->begin;
	unsigned int e_id,
	  no_id,
	  pos;

	/* we have to compute every move */
	for (; e_it; e_it = e_it->next)
	{
		e_id = KPTgetEdgeId (e_it);
		no_id = KPTgetOtherEndId (cur_move->n_id, e_it);
		pos = (EGbitTest (nodeData[no_id].It, cur_move->dom) ? 1U : 0U) |
			(EGbitTest (nodeData[no_id].Ia, cur_move->dom) ? 2U : 0U) |
			(EGbitTest (nodeData[no_id].Ib, cur_move->dom) ? 4U : 0U);
		switch (pos)
		{
		case 0U:
			/* in this case, nothing happen */
			break;
		case 7U:
			/* in this case we save one E(A:Ac) and flip AF, save 
			 * one E(B:Bc) and flip BF */
			if (edgeData[e_id].num.in_AF)
				(*move_val) -= 2 * weight[e_id];
			if (edgeData[e_id].num.in_BF)
				(*move_val) -= 2 * weight[e_id];
			break;
		case 5U:
			/* in this case we pay one E(A:Ac) and flip AF, save 
			 * one E(B:Bc) and flip BF */
			if (!edgeData[e_id].num.in_AF)
				(*move_val) += 2 * weight[e_id];
			if (edgeData[e_id].num.in_BF)
				(*move_val) -= 2 * weight[e_id];
			break;
		case 3U:
			/* in this case we save one E(A:Ac) and flip AF, pay 
			 * one E(B:Bc) and flip BF */
			if (edgeData[e_id].num.in_AF)
				(*move_val) -= 2 * weight[e_id];
			if (!edgeData[e_id].num.in_BF)
				(*move_val) += 2 * weight[e_id];
			break;
		case 1U:
			/* in this case we pay one E(A:Ac) and flip AF, pay 
			 * one E(B:Bc) and flip BF */
			if (!edgeData[e_id].num.in_AF)
				(*move_val) += 2 * weight[e_id];
			if (!edgeData[e_id].num.in_BF)
				(*move_val) += 2 * weight[e_id];
			break;
		default:
			EXIT (1, "This should not happen, value %u", pos);
		}														/* end switch */
	}															/* end loop through all incident edges */
	return 0;
}

/* ========================================================================= */
/** @brief This function prices the move AAcBBc, assuming the move is 
 * feasible*/
/* ========================================================================= */
static inline int KPTpriceAAcBBc (const KPTMove_t * const cur_move,
																	double *const move_val)
{
	/* local variables */
	EGlistNode_t *e_it = all_nodes[cur_move->n_id]->edges->begin;
	unsigned int e_id,
	  no_id,
	  pos;

	/* we have to compute every move */
	for (; e_it; e_it = e_it->next)
	{
		e_id = KPTgetEdgeId (e_it);
		no_id = KPTgetOtherEndId (cur_move->n_id, e_it);
		pos = (EGbitTest (nodeData[no_id].It, cur_move->dom) ? 1U : 0U) |
			(EGbitTest (nodeData[no_id].Ia, cur_move->dom) ? 2U : 0U) |
			(EGbitTest (nodeData[no_id].Ib, cur_move->dom) ? 4U : 0U);
		switch (pos)
		{
		case 0U:
			/* in this case, nothing happen */
			break;
		case 1U:
			/* in this case we save one E(A:Ac) and flip AF, save 
			 * one E(B:Bc) and flip BF */
			if (edgeData[e_id].num.in_AF)
				(*move_val) -= 2 * weight[e_id];
			if (edgeData[e_id].num.in_BF)
				(*move_val) -= 2 * weight[e_id];
			break;
		case 3U:
			/* in this case we pay one E(A:Ac) and flip AF, save 
			 * one E(B:Bc) and flip BF */
			if (!edgeData[e_id].num.in_AF)
				(*move_val) += 2 * weight[e_id];
			if (edgeData[e_id].num.in_BF)
				(*move_val) -= 2 * weight[e_id];
			break;
		case 5U:
			/* in this case we save one E(A:Ac) and flip AF, pay 
			 * one E(B:Bc) and flip BF */
			if (edgeData[e_id].num.in_AF)
				(*move_val) -= 2 * weight[e_id];
			if (!edgeData[e_id].num.in_BF)
				(*move_val) += 2 * weight[e_id];
			break;
		case 7U:
			/* in this case we pay one E(A:Ac) and flip AF, pay 
			 * one E(B:Bc) and flip BF */
			if (!edgeData[e_id].num.in_AF)
				(*move_val) += 2 * weight[e_id];
			if (!edgeData[e_id].num.in_BF)
				(*move_val) += 2 * weight[e_id];
			break;
		default:
			EXIT (1, "This should not happen, value %u", pos);
		}														/* end switch */
	}															/* end loop through all incident edges */
	return 0;
}

/* ========================================================================= */
/** @brief This function prices the move AcABBcBH, assuming the move is 
 * feasible*/
/* ========================================================================= */
static inline int KPTpriceAcABBcBH (const KPTMove_t * const cur_move,
																		double *const move_val)
{
	/* local variables */
	EGlistNode_t *e_it = all_nodes[cur_move->n_id]->edges->begin;
	unsigned int e_id,
	  no_id,
	  pos;

	/* we have to compute every move */
	for (; e_it; e_it = e_it->next)
	{
		e_id = KPTgetEdgeId (e_it);
		no_id = KPTgetOtherEndId (cur_move->n_id, e_it);
		pos = (EGbitTest (nodeData[no_id].It, cur_move->dom) ? 1U : 0U) |
			(EGbitTest (nodeData[no_id].Ia, cur_move->dom) ? 2U : 0U) |
			(EGbitTest (nodeData[no_id].Ib, cur_move->dom) ? 4U : 0U);
		switch (pos)
		{
		case 0U:
			/* in this case, flip BH */
			if (edgeData[e_id].num.in_BF)
				(*move_val) -= weight[e_id];
			else
				(*move_val) += weight[e_id];
			break;
		case 3U:
			/* in this case we save one E(A:Ac) and flip AF, save 
			 * one E(B:Bc) and double flip BF */
			if (edgeData[e_id].num.in_AF)
				(*move_val) -= 2 * weight[e_id];
			(*move_val) -= weight[e_id];
			break;
		case 1U:
			/* in this case we pay one E(A:Ac) and flip AF, save 
			 * one E(B:Bc) and double flip BF */
			if (!edgeData[e_id].num.in_AF)
				(*move_val) += 2 * weight[e_id];
			(*move_val) -= weight[e_id];
			break;
		case 7U:
			/* in this case we save one E(A:Ac) and flip AF, pay 
			 * one E(B:Bc) and double flip BF */
			if (edgeData[e_id].num.in_AF)
				(*move_val) -= 2 * weight[e_id];
			(*move_val) += weight[e_id];
			break;
		case 5U:
			/* in this case we pay one E(A:Ac) and flip AF, pay 
			 * one E(B:Bc) and flip BF */
			if (!edgeData[e_id].num.in_AF)
				(*move_val) += 2 * weight[e_id];
			(*move_val) += weight[e_id];
			break;
		default:
			EXIT (1, "This should not happen, value %u", pos);
		}														/* end switch */
	}															/* end loop through all incident edges */
	return 0;
}

/* ========================================================================= */
/** @brief This function prices the move AAcBcBBH, assuming the move is 
 * feasible*/
/* ========================================================================= */
static inline int KPTpriceAAcBcBBH (const KPTMove_t * const cur_move,
																		double *const move_val)
{
	/* local variables */
	EGlistNode_t *e_it = all_nodes[cur_move->n_id]->edges->begin;
	unsigned int e_id,
	  no_id,
	  pos;

	/* we have to compute every move */
	for (; e_it; e_it = e_it->next)
	{
		e_id = KPTgetEdgeId (e_it);
		no_id = KPTgetOtherEndId (cur_move->n_id, e_it);
		pos = (EGbitTest (nodeData[no_id].It, cur_move->dom) ? 1U : 0U) |
			(EGbitTest (nodeData[no_id].Ia, cur_move->dom) ? 2U : 0U) |
			(EGbitTest (nodeData[no_id].Ib, cur_move->dom) ? 4U : 0U);
		switch (pos)
		{
		case 0U:
			/* in this case, flip BH */
			if (edgeData[e_id].num.in_BF)
				(*move_val) -= weight[e_id];
			else
				(*move_val) += weight[e_id];
			break;
		case 5U:
			/* in this case we save one E(A:Ac) and flip AF, save 
			 * one E(B:Bc) and double flip BF */
			if (edgeData[e_id].num.in_AF)
				(*move_val) -= 2 * weight[e_id];
			(*move_val) -= weight[e_id];
			break;
		case 7U:
			/* in this case we pay one E(A:Ac) and flip AF, save 
			 * one E(B:Bc) and double flip BF */
			if (!edgeData[e_id].num.in_AF)
				(*move_val) += 2 * weight[e_id];
			(*move_val) -= weight[e_id];
			break;
		case 1U:
			/* in this case we save one E(A:Ac) and flip AF, pay 
			 * one E(B:Bc) and double flip BF */
			if (edgeData[e_id].num.in_AF)
				(*move_val) -= 2 * weight[e_id];
			(*move_val) += weight[e_id];
			break;
		case 3U:
			/* in this case we pay one E(A:Ac) and flip AF, pay 
			 * one E(B:Bc) and double flip BF */
			if (!edgeData[e_id].num.in_AF)
				(*move_val) += 2 * weight[e_id];
			(*move_val) += weight[e_id];
			break;
		default:
			EXIT (1, "This should not happen, value %u", pos);
		}														/* end switch */
	}															/* end loop through all incident edges */
	return 0;
}

/* ========================================================================= */
/** @brief This function prices the move AcABcBBH, assuming the move is 
 * feasible*/
/* ========================================================================= */
static inline int KPTpriceAcABcBBH (const KPTMove_t * const cur_move,
																		double *const move_val)
{
	/* local variables */
	EGlistNode_t *e_it = all_nodes[cur_move->n_id]->edges->begin;
	unsigned int e_id,
	  no_id,
	  pos;

	/* we have to compute every move */
	for (; e_it; e_it = e_it->next)
	{
		e_id = KPTgetEdgeId (e_it);
		no_id = KPTgetOtherEndId (cur_move->n_id, e_it);
		pos = (EGbitTest (nodeData[no_id].It, cur_move->dom) ? 1U : 0U) |
			(EGbitTest (nodeData[no_id].Ia, cur_move->dom) ? 2U : 0U) |
			(EGbitTest (nodeData[no_id].Ib, cur_move->dom) ? 4U : 0U);
		switch (pos)
		{
		case 0U:
			/* in this case, flip BH */
			if (edgeData[e_id].num.in_BF)
				(*move_val) -= weight[e_id];
			else
				(*move_val) += weight[e_id];
			break;
		case 7U:
			/* in this case we save one E(A:Ac) and flip AF, save 
			 * one E(B:Bc) and double flip BF */
			if (edgeData[e_id].num.in_AF)
				(*move_val) -= 2 * weight[e_id];
			(*move_val) -= weight[e_id];
			break;
		case 5U:
			/* in this case we pay one E(A:Ac) and flip AF, save 
			 * one E(B:Bc) and double flip BF */
			if (!edgeData[e_id].num.in_AF)
				(*move_val) += 2 * weight[e_id];
			(*move_val) -= weight[e_id];
			break;
		case 3U:
			/* in this case we save one E(A:Ac) and flip AF, pay 
			 * one E(B:Bc) and double flip BF */
			if (edgeData[e_id].num.in_AF)
				(*move_val) -= 2 * weight[e_id];
			(*move_val) += weight[e_id];
			break;
		case 1U:
			/* in this case we pay one E(A:Ac) and flip AF, pay 
			 * one E(B:Bc) and double flip BF */
			if (!edgeData[e_id].num.in_AF)
				(*move_val) += 2 * weight[e_id];
			(*move_val) += weight[e_id];
			break;
		default:
			EXIT (1, "This should not happen, value %u", pos);
		}														/* end switch */
	}															/* end loop through all incident edges */
	return 0;
}

/* ========================================================================= */
/** @brief This function prices the move AAcBBcBH, assuming the move is 
 * feasible*/
/* ========================================================================= */
static inline int KPTpriceAAcBBcBH (const KPTMove_t * const cur_move,
																		double *const move_val)
{
	/* local variables */
	EGlistNode_t *e_it = all_nodes[cur_move->n_id]->edges->begin;
	unsigned int e_id,
	  no_id,
	  pos;

	/* we have to compute every move */
	for (; e_it; e_it = e_it->next)
	{
		e_id = KPTgetEdgeId (e_it);
		no_id = KPTgetOtherEndId (cur_move->n_id, e_it);
		pos = (EGbitTest (nodeData[no_id].It, cur_move->dom) ? 1U : 0U) |
			(EGbitTest (nodeData[no_id].Ia, cur_move->dom) ? 2U : 0U) |
			(EGbitTest (nodeData[no_id].Ib, cur_move->dom) ? 4U : 0U);
		switch (pos)
		{
		case 0U:
			/* in this case, flip BH */
			if (edgeData[e_id].num.in_BF)
				(*move_val) -= weight[e_id];
			else
				(*move_val) += weight[e_id];
			break;
		case 1U:
			/* in this case we save one E(A:Ac) and flip AF, save 
			 * one E(B:Bc) and double flip BF */
			if (edgeData[e_id].num.in_AF)
				(*move_val) -= 2 * weight[e_id];
			(*move_val) -= weight[e_id];
			break;
		case 3U:
			/* in this case we pay one E(A:Ac) and flip AF, save 
			 * one E(B:Bc) and double flip BF */
			if (!edgeData[e_id].num.in_AF)
				(*move_val) += 2 * weight[e_id];
			(*move_val) -= weight[e_id];
			break;
		case 5U:
			/* in this case we save one E(A:Ac) and flip AF, pay 
			 * one E(B:Bc) and double flip BF */
			if (edgeData[e_id].num.in_AF)
				(*move_val) -= 2 * weight[e_id];
			(*move_val) += weight[e_id];
			break;
		case 7U:
			/* in this case we pay one E(A:Ac) and flip AF, pay 
			 * one E(B:Bc) and double flip BF */
			if (!edgeData[e_id].num.in_AF)
				(*move_val) += 2 * weight[e_id];
			(*move_val) += weight[e_id];
			break;
		default:
			EXIT (1, "This should not happen, value %u", pos);
		}														/* end switch */
	}															/* end loop through all incident edges */
	return 0;
}

/* ========================================================================= */
/** @brief This function prices the move AcABBcAH, assuming the move is 
 * feasible*/
/* ========================================================================= */
static inline int KPTpriceAcABBcAH (const KPTMove_t * const cur_move,
																		double *const move_val)
{
	/* local variables */
	EGlistNode_t *e_it = all_nodes[cur_move->n_id]->edges->begin;
	unsigned int e_id,
	  no_id,
	  pos;

	/* we have to compute every move */
	for (; e_it; e_it = e_it->next)
	{
		e_id = KPTgetEdgeId (e_it);
		no_id = KPTgetOtherEndId (cur_move->n_id, e_it);
		pos = (EGbitTest (nodeData[no_id].It, cur_move->dom) ? 1U : 0U) |
			(EGbitTest (nodeData[no_id].Ia, cur_move->dom) ? 2U : 0U) |
			(EGbitTest (nodeData[no_id].Ib, cur_move->dom) ? 4U : 0U);
		switch (pos)
		{
		case 0U:
			/* in this case, flip AH */
			if (edgeData[e_id].num.in_AF)
				(*move_val) -= weight[e_id];
			else
				(*move_val) += weight[e_id];
			break;
		case 3U:
			/* in this case we save one E(A:Ac) and double flip AF, save 
			 * one E(B:Bc) and flip BF */
			(*move_val) -= weight[e_id];
			if (edgeData[e_id].num.in_BF)
				(*move_val) -= 2 * weight[e_id];
			break;
		case 1U:
			/* in this case we pay one E(A:Ac) and double flip AF, save 
			 * one E(B:Bc) and flip BF */
			(*move_val) += weight[e_id];
			if (edgeData[e_id].num.in_BF)
				(*move_val) -= 2 * weight[e_id];
			break;
		case 7U:
			/* in this case we save one E(A:Ac) and double flip AF, pay 
			 * one E(B:Bc) and flip BF */
			(*move_val) -= weight[e_id];
			if (!edgeData[e_id].num.in_BF)
				(*move_val) += 2 * weight[e_id];
			break;
		case 5U:
			/* in this case we pay one E(A:Ac) and double flip AF, pay 
			 * one E(B:Bc) and flip BF */
			(*move_val) += weight[e_id];
			if (!edgeData[e_id].num.in_BF)
				(*move_val) += 2 * weight[e_id];
			break;
		default:
			EXIT (1, "This should not happen, value %u", pos);
		}														/* end switch */
	}															/* end loop through all incident edges */
	return 0;
}

/* ========================================================================= */
/** @brief This function prices the move AAcBcBAH, assuming the move is 
 * feasible*/
/* ========================================================================= */
static inline int KPTpriceAAcBcBAH (const KPTMove_t * const cur_move,
																		double *const move_val)
{
	/* local variables */
	EGlistNode_t *e_it = all_nodes[cur_move->n_id]->edges->begin;
	unsigned int e_id,
	  no_id,
	  pos;

	/* we have to compute every move */
	for (; e_it; e_it = e_it->next)
	{
		e_id = KPTgetEdgeId (e_it);
		no_id = KPTgetOtherEndId (cur_move->n_id, e_it);
		pos = (EGbitTest (nodeData[no_id].It, cur_move->dom) ? 1U : 0U) |
			(EGbitTest (nodeData[no_id].Ia, cur_move->dom) ? 2U : 0U) |
			(EGbitTest (nodeData[no_id].Ib, cur_move->dom) ? 4U : 0U);
		switch (pos)
		{
		case 0U:
			/* in this case, flip AF */
			if (edgeData[e_id].num.in_AF)
				(*move_val) -= weight[e_id];
			else
				(*move_val) += weight[e_id];
			break;
		case 5U:
			/* in this case we save one E(A:Ac) and double flip AF, save 
			 * one E(B:Bc) and flip BF */
			(*move_val) -= weight[e_id];
			if (edgeData[e_id].num.in_BF)
				(*move_val) -= 2 * weight[e_id];
			break;
		case 7U:
			/* in this case we pay one E(A:Ac) and double flip AF, save 
			 * one E(B:Bc) and flip BF */
			(*move_val) += weight[e_id];
			if (edgeData[e_id].num.in_BF)
				(*move_val) -= 2 * weight[e_id];
			break;
		case 1U:
			/* in this case we save one E(A:Ac) and double flip AF, pay 
			 * one E(B:Bc) and flip BF */
			(*move_val) -= weight[e_id];
			if (!edgeData[e_id].num.in_BF)
				(*move_val) += 2 * weight[e_id];
			break;
		case 3U:
			/* in this case we pay one E(A:Ac) and double flip AF, pay 
			 * one E(B:Bc) and flip BF */
			(*move_val) += weight[e_id];
			if (!edgeData[e_id].num.in_BF)
				(*move_val) += 2 * weight[e_id];
			break;
		default:
			EXIT (1, "This should not happen, value %u", pos);
		}														/* end switch */
	}															/* end loop through all incident edges */
	return 0;
}

/* ========================================================================= */
/** @brief This function prices the move AcABcBAH, assuming the move is 
 * feasible*/
/* ========================================================================= */
static inline int KPTpriceAcABcBAH (const KPTMove_t * const cur_move,
																		double *const move_val)
{
	/* local variables */
	EGlistNode_t *e_it = all_nodes[cur_move->n_id]->edges->begin;
	unsigned int e_id,
	  no_id,
	  pos;

	/* we have to compute every move */
	for (; e_it; e_it = e_it->next)
	{
		e_id = KPTgetEdgeId (e_it);
		no_id = KPTgetOtherEndId (cur_move->n_id, e_it);
		pos = (EGbitTest (nodeData[no_id].It, cur_move->dom) ? 1U : 0U) |
			(EGbitTest (nodeData[no_id].Ia, cur_move->dom) ? 2U : 0U) |
			(EGbitTest (nodeData[no_id].Ib, cur_move->dom) ? 4U : 0U);
		switch (pos)
		{
		case 0U:
			/* in this case, flip AH */
			if (edgeData[e_id].num.in_AF)
				(*move_val) -= weight[e_id];
			else
				(*move_val) += weight[e_id];
			break;
		case 7U:
			/* in this case we save one E(A:Ac) and double flip AF, save 
			 * one E(B:Bc) and flip BF */
			(*move_val) -= weight[e_id];
			if (edgeData[e_id].num.in_BF)
				(*move_val) -= 2 * weight[e_id];
			break;
		case 5U:
			/* in this case we pay one E(A:Ac) and double flip AF, save 
			 * one E(B:Bc) and flip BF */
			(*move_val) += weight[e_id];
			if (edgeData[e_id].num.in_BF)
				(*move_val) -= 2 * weight[e_id];
			break;
		case 3U:
			/* in this case we save one E(A:Ac) and double flip AF, pay 
			 * one E(B:Bc) and flip BF */
			(*move_val) -= weight[e_id];
			if (!edgeData[e_id].num.in_BF)
				(*move_val) += 2 * weight[e_id];
			break;
		case 1U:
			/* in this case we pay one E(A:Ac) and double flip AF, pay 
			 * one E(B:Bc) and flip BF */
			(*move_val) += weight[e_id];
			if (!edgeData[e_id].num.in_BF)
				(*move_val) += 2 * weight[e_id];
			break;
		default:
			EXIT (1, "This should not happen, value %u", pos);
		}														/* end switch */
	}															/* end loop through all incident edges */
	return 0;
}

/* ========================================================================= */
/** @brief This function prices the move AAcBBcAH, assuming the move is 
 * feasible*/
/* ========================================================================= */
static inline int KPTpriceAAcBBcAH (const KPTMove_t * const cur_move,
																		double *const move_val)
{
	/* local variables */
	EGlistNode_t *e_it = all_nodes[cur_move->n_id]->edges->begin;
	unsigned int e_id,
	  no_id,
	  pos;

	/* we have to compute every move */
	for (; e_it; e_it = e_it->next)
	{
		e_id = KPTgetEdgeId (e_it);
		no_id = KPTgetOtherEndId (cur_move->n_id, e_it);
		pos = (EGbitTest (nodeData[no_id].It, cur_move->dom) ? 1U : 0U) |
			(EGbitTest (nodeData[no_id].Ia, cur_move->dom) ? 2U : 0U) |
			(EGbitTest (nodeData[no_id].Ib, cur_move->dom) ? 4U : 0U);
		switch (pos)
		{
		case 0U:
			/* in this case, flip AH */
			if (edgeData[e_id].num.in_AF)
				(*move_val) -= weight[e_id];
			else
				(*move_val) += weight[e_id];
			break;
		case 1U:
			/* in this case we save one E(A:Ac) and double flip AF, save 
			 * one E(B:Bc) and flip BF */
			(*move_val) -= weight[e_id];
			if (edgeData[e_id].num.in_BF)
				(*move_val) -= 2 * weight[e_id];
			break;
		case 3U:
			/* in this case we pay one E(A:Ac) and double flip AF, save 
			 * one E(B:Bc) and flip BF */
			(*move_val) += weight[e_id];
			if (edgeData[e_id].num.in_BF)
				(*move_val) -= 2 * weight[e_id];
			break;
		case 5U:
			/* in this case we save one E(A:Ac) and double flip AF, pay 
			 * one E(B:Bc) and flip BF */
			(*move_val) -= weight[e_id];
			if (!edgeData[e_id].num.in_BF)
				(*move_val) += 2 * weight[e_id];
			break;
		case 7U:
			/* in this case we pay one E(A:Ac) and double flip AF, pay 
			 * one E(B:Bc) and flip BF */
			(*move_val) += weight[e_id];
			if (!edgeData[e_id].num.in_BF)
				(*move_val) += 2 * weight[e_id];
			break;
		default:
			EXIT (1, "This should not happen, value %u", pos);
		}														/* end switch */
	}															/* end loop through all incident edges */
	return 0;
}

/* ========================================================================= */
/** @brief This function prices the move AcABBcAHBH, assuming the move is 
 * feasible*/
/* ========================================================================= */
static inline int KPTpriceAcABBcAHBH (const KPTMove_t * const cur_move,
																			double *const move_val)
{
	/* local variables */
	EGlistNode_t *e_it = all_nodes[cur_move->n_id]->edges->begin;
	unsigned int e_id,
	  no_id,
	  pos;

	/* we have to compute every move */
	for (; e_it; e_it = e_it->next)
	{
		e_id = KPTgetEdgeId (e_it);
		no_id = KPTgetOtherEndId (cur_move->n_id, e_it);
		pos = (EGbitTest (nodeData[no_id].It, cur_move->dom) ? 1U : 0U) |
			(EGbitTest (nodeData[no_id].Ia, cur_move->dom) ? 2U : 0U) |
			(EGbitTest (nodeData[no_id].Ib, cur_move->dom) ? 4U : 0U);
		switch (pos)
		{
		case 0U:
			/* in this case, flip AF and flip BF */
			if (edgeData[e_id].num.in_AF)
				(*move_val) -= weight[e_id];
			else
				(*move_val) += weight[e_id];
			if (edgeData[e_id].num.in_BF)
				(*move_val) -= weight[e_id];
			else
				(*move_val) += weight[e_id];
			break;
		case 3U:
			/* in this case we save one E(A:Ac) and double flip AF, save 
			 * one E(B:Bc) and double flip BF */
			(*move_val) -= 2 * weight[e_id];
			break;
		case 1U:
			/* in this case we pay one E(A:Ac) and double flip AF, save 
			 * one E(B:Bc) and double flip BF */
			break;
		case 7U:
			/* in this case we save one E(A:Ac) and double flip AF, pay 
			 * one E(B:Bc) and double flip BF */
			break;
		case 5U:
			/* in this case we pay one E(A:Ac) and double flip AF, pay 
			 * one E(B:Bc) and double flip BF */
			(*move_val) += 2 * weight[e_id];
			break;
		default:
			EXIT (1, "This should not happen, value %u", pos);
		}														/* end switch */
	}															/* end loop through all incident edges */
	return 0;
}

/* ========================================================================= */
/** @brief This function prices the move AAcBcBAHBH, assuming the move is 
 * feasible*/
/* ========================================================================= */
static inline int KPTpriceAAcBcBAHBH (const KPTMove_t * const cur_move,
																			double *const move_val)
{
	/* local variables */
	EGlistNode_t *e_it = all_nodes[cur_move->n_id]->edges->begin;
	unsigned int e_id,
	  no_id,
	  pos;

	/* we have to compute every move */
	for (; e_it; e_it = e_it->next)
	{
		e_id = KPTgetEdgeId (e_it);
		no_id = KPTgetOtherEndId (cur_move->n_id, e_it);
		pos = (EGbitTest (nodeData[no_id].It, cur_move->dom) ? 1U : 0U) |
			(EGbitTest (nodeData[no_id].Ia, cur_move->dom) ? 2U : 0U) |
			(EGbitTest (nodeData[no_id].Ib, cur_move->dom) ? 4U : 0U);
		switch (pos)
		{
		case 0U:
			/* in this case, flip AF and flip BF */
			if (edgeData[e_id].num.in_AF)
				(*move_val) -= weight[e_id];
			else
				(*move_val) += weight[e_id];
			if (edgeData[e_id].num.in_BF)
				(*move_val) -= weight[e_id];
			else
				(*move_val) += weight[e_id];
			break;
		case 5U:
			/* in this case we save one E(A:Ac) and double flip AF, save 
			 * one E(B:Bc) and double flip BF */
			(*move_val) -= 2 * weight[e_id];
			break;
		case 7U:
			/* in this case we pay one E(A:Ac) and double flip AF, save 
			 * one E(B:Bc) and double flip BF */
			break;
		case 1U:
			/* in this case we save one E(A:Ac) and double flip AF, pay 
			 * one E(B:Bc) and double flip BF */
			break;
		case 3U:
			/* in this case we pay one E(A:Ac) and double flip AF, pay 
			 * one E(B:Bc) and double flip BF */
			(*move_val) += 2 * weight[e_id];
			break;
		default:
			EXIT (1, "This should not happen, value %u", pos);
		}														/* end switch */
	}															/* end loop through all incident edges */
	return 0;
}

/* ========================================================================= */
/** @brief This function prices the move AcABcBAHBH, assuming the move is 
 * feasible*/
/* ========================================================================= */
static inline int KPTpriceAcABcBAHBH (const KPTMove_t * const cur_move,
																			double *const move_val)
{
	/* local variables */
	EGlistNode_t *e_it = all_nodes[cur_move->n_id]->edges->begin;
	unsigned int e_id,
	  no_id,
	  pos;

	/* we have to compute every move */
	for (; e_it; e_it = e_it->next)
	{
		e_id = KPTgetEdgeId (e_it);
		no_id = KPTgetOtherEndId (cur_move->n_id, e_it);
		pos = (EGbitTest (nodeData[no_id].It, cur_move->dom) ? 1U : 0U) |
			(EGbitTest (nodeData[no_id].Ia, cur_move->dom) ? 2U : 0U) |
			(EGbitTest (nodeData[no_id].Ib, cur_move->dom) ? 4U : 0U);
		switch (pos)
		{
		case 0U:
			/* in this case, flip AF and flip BF */
			if (edgeData[e_id].num.in_AF)
				(*move_val) -= weight[e_id];
			else
				(*move_val) += weight[e_id];
			if (edgeData[e_id].num.in_BF)
				(*move_val) -= weight[e_id];
			else
				(*move_val) += weight[e_id];
			break;
		case 7U:
			/* in this case we save one E(A:Ac) and double flip AF, save 
			 * one E(B:Bc) and double flip BF */
			(*move_val) -= 2 * weight[e_id];
			break;
		case 5U:
			/* in this case we pay one E(A:Ac) and double flip AF, save 
			 * one E(B:Bc) and double flip BF */
			break;
		case 3U:
			/* in this case we save one E(A:Ac) and double flip AF, pay 
			 * one E(B:Bc) and double flip BF */
			break;
		case 1U:
			/* in this case we pay one E(A:Ac) and double flip AF, pay 
			 * one E(B:Bc) and double flip BF */
			(*move_val) += 2 * weight[e_id];
			break;
		default:
			EXIT (1, "This should not happen, value %u", pos);
		}														/* end switch */
	}															/* end loop through all incident edges */
	return 0;
}

/* ========================================================================= */
/** @brief This function prices the move AAcBBcAHBH, assuming the move is 
 * feasible*/
/* ========================================================================= */
static inline int KPTpriceAAcBBcAHBH (const KPTMove_t * const cur_move,
																			double *const move_val)
{
	/* local variables */
	EGlistNode_t *e_it = all_nodes[cur_move->n_id]->edges->begin;
	unsigned int e_id,
	  no_id,
	  pos;

	/* we have to compute every move */
	for (; e_it; e_it = e_it->next)
	{
		e_id = KPTgetEdgeId (e_it);
		no_id = KPTgetOtherEndId (cur_move->n_id, e_it);
		pos = (EGbitTest (nodeData[no_id].It, cur_move->dom) ? 1U : 0U) |
			(EGbitTest (nodeData[no_id].Ia, cur_move->dom) ? 2U : 0U) |
			(EGbitTest (nodeData[no_id].Ib, cur_move->dom) ? 4U : 0U);
		switch (pos)
		{
		case 0U:
			/* in this case, flip AF and flip BF */
			if (edgeData[e_id].num.in_AF)
				(*move_val) -= weight[e_id];
			else
				(*move_val) += weight[e_id];
			if (edgeData[e_id].num.in_BF)
				(*move_val) -= weight[e_id];
			else
				(*move_val) += weight[e_id];
			break;
		case 1U:
			/* in this case we save one E(A:Ac) and double flip AF, save 
			 * one E(B:Bc) and double flip BF */
			(*move_val) -= 2 * weight[e_id];
			break;
		case 3U:
			/* in this case we pay one E(A:Ac) and double flip AF, save 
			 * one E(B:Bc) and double flip BF */
			break;
		case 5U:
			/* in this case we save one E(A:Ac) and double flip AF, pay 
			 * one E(B:Bc) and double flip BF */
			break;
		case 7U:
			/* in this case we pay one E(A:Ac) and double flip AF, pay 
			 * one E(B:Bc) and double flip BF */
			(*move_val) += 2 * weight[e_id];
			break;
		default:
			EXIT (1, "This should not happen, value %u", pos);
		}														/* end switch */
	}															/* end loop through all incident edges */
	return 0;
}


/* ========================================================================= */
/** @brief This function prices the move ABTc, assuming the move is feasible*/
/* ========================================================================= */
static inline int KPTpriceABTc (const KPTMove_t * const cur_move,
																double *const move_val)
{
	/* local variables */
	EGlistNode_t *e_it = all_nodes[cur_move->n_id]->edges->begin;
	unsigned int e_id,
	  no_id,
	  pos;

	/* we have to compute every move */
	for (; e_it; e_it = e_it->next)
	{
		e_id = KPTgetEdgeId (e_it);
		no_id = KPTgetOtherEndId (cur_move->n_id, e_it);
		pos = (EGbitTest (nodeData[no_id].It, cur_move->dom) ? 1U : 0U) |
			(EGbitTest (nodeData[no_id].Ia, cur_move->dom) ? 2U : 0U) |
			(EGbitTest (nodeData[no_id].Ib, cur_move->dom) ? 4U : 0U);
		switch (pos)
		{
		case 7U:
			/* in this case, the only change is that we have to pay for this edge in
			 * Delta(T) */
			(*move_val) += weight[e_id];
			break;
		case 1U:
			/* in this case we pay one Delta(T), save one E(A:Ac) and flip AF, save 
			 * one E(B:Bc) and flip BF */
			(*move_val) += weight[e_id];
			if (edgeData[e_id].num.in_AF)
				(*move_val) -= 2 * weight[e_id];
			if (edgeData[e_id].num.in_BF)
				(*move_val) -= 2 * weight[e_id];
			break;
		case 3U:
			/* in this case we pay one Delta(T), save one E(B:Bc) and flip BF */
			if (edgeData[e_id].num.in_BF)
				(*move_val) -= weight[e_id];
			else
				(*move_val) += weight[e_id];
			break;
		case 5U:
			/* in this case we pay one Delta(T), save one E(A:Ac) and flip AF */
			if (edgeData[e_id].num.in_AF)
				(*move_val) -= weight[e_id];
			else
				(*move_val) += weight[e_id];
			break;
		case 0U:
			/* in this case we save one Delta(T) */
			(*move_val) -= weight[e_id];
			break;
		default:
			EXIT (1, "This should not happen, value %u", pos);
		}														/* end switch */
	}															/* end loop through all incident edges */
	return 0;
}

/* ========================================================================= */
/** @brief This function prices the move TcAcBc,assuming the move is feasible*/
/* ========================================================================= */
static inline int KPTpriceTcAcBc (const KPTMove_t * const cur_move,
																	double *const move_val)
{
	/* local variables */
	EGlistNode_t *e_it = all_nodes[cur_move->n_id]->edges->begin;
	unsigned int e_id,
	  no_id,
	  pos;

	/* we have to compute every move */
	for (; e_it; e_it = e_it->next)
	{
		e_id = KPTgetEdgeId (e_it);
		no_id = KPTgetOtherEndId (cur_move->n_id, e_it);
		pos = (EGbitTest (nodeData[no_id].It, cur_move->dom) ? 1U : 0U) |
			(EGbitTest (nodeData[no_id].Ia, cur_move->dom) ? 2U : 0U) |
			(EGbitTest (nodeData[no_id].Ib, cur_move->dom) ? 4U : 0U);
		switch (pos)
		{
		case 0U:
			/* in this case, the only change is that we have to pay for this edge in
			 * Delta(T) */
			(*move_val) += weight[e_id];
			break;
		case 7U:
			/* in this case we save one Delta(T), pay one E(A:Ac) and flip AF, pay 
			 * one E(B:Bc) and flip BF */
			(*move_val) -= weight[e_id];
			if (!edgeData[e_id].num.in_AF)
				(*move_val) += 2 * weight[e_id];
			if (!edgeData[e_id].num.in_BF)
				(*move_val) += 2 * weight[e_id];
			break;
		case 5U:
			/* in this case we save one Delta(T), pay one E(B:Bc) and flip BF */
			if (edgeData[e_id].num.in_BF)
				(*move_val) -= weight[e_id];
			else
				(*move_val) += weight[e_id];
			break;
		case 3U:
			/* in this case we save one Delta(T), pay one E(A:Ac) and flip AF */
			if (edgeData[e_id].num.in_AF)
				(*move_val) -= weight[e_id];
			else
				(*move_val) += weight[e_id];
			break;
		case 1U:
			/* in this case we save one Delta(T) */
			(*move_val) -= weight[e_id];
			break;
		default:
			EXIT (1, "This should not happen, value %u", pos);
		}														/* end switch */
	}															/* end loop through all incident edges */
	return 0;
}

/* ========================================================================= */
/** @brief This function prices the move Tc ABc,assuming the move is feasible*/
/* ========================================================================= */
static inline int KPTpriceTcABc (const KPTMove_t * const cur_move,
																 double *const move_val)
{
	/* local variables */
	EGlistNode_t *e_it = all_nodes[cur_move->n_id]->edges->begin;
	unsigned int e_id,
	  no_id,
	  pos;

	/* we have to compute every move */
	for (; e_it; e_it = e_it->next)
	{
		e_id = KPTgetEdgeId (e_it);
		no_id = KPTgetOtherEndId (cur_move->n_id, e_it);
		pos = (EGbitTest (nodeData[no_id].It, cur_move->dom) ? 1U : 0U) |
			(EGbitTest (nodeData[no_id].Ia, cur_move->dom) ? 2U : 0U) |
			(EGbitTest (nodeData[no_id].Ib, cur_move->dom) ? 4U : 0U);
		switch (pos)
		{
		case 0U:
			/* in this case, the only change is that we have to pay for this edge in
			 * Delta(T) */
			(*move_val) += weight[e_id];
			break;
		case 5U:
			/* in this case we save one Delta(T), pay one E(A:Ac) and flip AF, pay 
			 * one E(B:Bc) and flip BF */
			(*move_val) -= weight[e_id];
			if (!edgeData[e_id].num.in_AF)
				(*move_val) += 2 * weight[e_id];
			if (!edgeData[e_id].num.in_BF)
				(*move_val) += 2 * weight[e_id];
			break;
		case 7U:
			/* in this case we save one Delta(T), pay one E(B:Bc) and flip BF */
			if (edgeData[e_id].num.in_BF)
				(*move_val) -= weight[e_id];
			else
				(*move_val) += weight[e_id];
			break;
		case 1U:
			/* in this case we save one Delta(T), pay one E(A:Ac) and flip AF */
			if (edgeData[e_id].num.in_AF)
				(*move_val) -= weight[e_id];
			else
				(*move_val) += weight[e_id];
			break;
		case 3U:
			/* in this case we save one Delta(T) */
			(*move_val) -= weight[e_id];
			break;
		default:
			EXIT (1, "This should not happen, value %u", pos);
		}														/* end switch */
	}															/* end loop through all incident edges */
	return 0;
}

/* ========================================================================= */
/** @brief This function prices the move Tc AcB,assuming the move is feasible*/
/* ========================================================================= */
static inline int KPTpriceTcAcB (const KPTMove_t * const cur_move,
																 double *const move_val)
{
	/* local variables */
	EGlistNode_t *e_it = all_nodes[cur_move->n_id]->edges->begin;
	unsigned int e_id,
	  no_id,
	  pos;

	/* we have to compute every move */
	for (; e_it; e_it = e_it->next)
	{
		e_id = KPTgetEdgeId (e_it);
		no_id = KPTgetOtherEndId (cur_move->n_id, e_it);
		pos = (EGbitTest (nodeData[no_id].It, cur_move->dom) ? 1U : 0U) |
			(EGbitTest (nodeData[no_id].Ia, cur_move->dom) ? 2U : 0U) |
			(EGbitTest (nodeData[no_id].Ib, cur_move->dom) ? 4U : 0U);
		switch (pos)
		{
		case 0U:
			/* in this case, the only change is that we have to pay for this edge in
			 * Delta(T) */
			(*move_val) += weight[e_id];
			break;
		case 3U:
			/* in this case we save one Delta(T), pay one E(A:Ac) and flip AF, pay 
			 * one E(B:Bc) and flip BF */
			(*move_val) -= weight[e_id];
			if (!edgeData[e_id].num.in_AF)
				(*move_val) += 2 * weight[e_id];
			if (!edgeData[e_id].num.in_BF)
				(*move_val) += 2 * weight[e_id];
			break;
		case 1U:
			/* in this case we save one Delta(T), pay one E(B:Bc) and flip BF */
			if (edgeData[e_id].num.in_BF)
				(*move_val) -= weight[e_id];
			else
				(*move_val) += weight[e_id];
			break;
		case 7U:
			/* in this case we save one Delta(T), pay one E(A:Ac) and flip AF */
			if (edgeData[e_id].num.in_AF)
				(*move_val) -= weight[e_id];
			else
				(*move_val) += weight[e_id];
			break;
		case 5U:
			/* in this case we save one Delta(T) */
			(*move_val) -= weight[e_id];
			break;
		default:
			EXIT (1, "This should not happen, value %u", pos);
		}														/* end switch */
	}															/* end loop through all incident edges */
	return 0;
}

/* ========================================================================= */
/** @brief This function prices the move Tc AB, assuming the move is feasible*/
/* ========================================================================= */
static inline int KPTpriceTcAB (const KPTMove_t * const cur_move,
																double *const move_val)
{
	/* local variables */
	EGlistNode_t *e_it = all_nodes[cur_move->n_id]->edges->begin;
	unsigned int e_id,
	  no_id,
	  pos;

	/* we have to compute every move */
	for (; e_it; e_it = e_it->next)
	{
		e_id = KPTgetEdgeId (e_it);
		no_id = KPTgetOtherEndId (cur_move->n_id, e_it);
		pos = (EGbitTest (nodeData[no_id].It, cur_move->dom) ? 1U : 0U) |
			(EGbitTest (nodeData[no_id].Ia, cur_move->dom) ? 2U : 0U) |
			(EGbitTest (nodeData[no_id].Ib, cur_move->dom) ? 4U : 0U);
		switch (pos)
		{
		case 0U:
			/* in this case, the only change is that we have to pay for this edge in
			 * Delta(T) */
			(*move_val) += weight[e_id];
			break;
		case 1U:
			/* in this case we save one Delta(T), pay one E(A:Ac) and flip AF, pay 
			 * one E(B:Bc) and flip BF */
			(*move_val) -= weight[e_id];
			if (!edgeData[e_id].num.in_AF)
				(*move_val) += 2 * weight[e_id];
			if (!edgeData[e_id].num.in_BF)
				(*move_val) += 2 * weight[e_id];
			break;
		case 3U:
			/* in this case we save one Delta(T), pay one E(B:Bc) and flip BF */
			if (edgeData[e_id].num.in_BF)
				(*move_val) -= weight[e_id];
			else
				(*move_val) += weight[e_id];
			break;
		case 5U:
			/* in this case we save one Delta(T), pay one E(A:Ac) and flip AF */
			if (edgeData[e_id].num.in_AF)
				(*move_val) -= weight[e_id];
			else
				(*move_val) += weight[e_id];
			break;
		case 7U:
			/* in this case we save one Delta(T) */
			(*move_val) -= weight[e_id];
			break;
		default:
			EXIT (1, "This should not happen, value %u", pos);
		}														/* end switch */
	}															/* end loop through all incident edges */
	return 0;
}

/* ========================================================================= */
/** @brief This function prices the move TcBc, assuming the move is feasible*/
/* ========================================================================= */
static inline int KPTpriceTcBc (const KPTMove_t * const cur_move,
																double *const move_val)
{
	/* local variables */
	unsigned int e_id,
	  no_id,
	  pos,
	  n_id = cur_move->n_id,
	  dom = cur_move->dom;
	EGlistNode_t *e_it = all_nodes[n_id]->edges->begin;

	/* we have to compute every move */
	for (; e_it; e_it = e_it->next)
	{
		e_id = KPTgetEdgeId (e_it);
		no_id = KPTgetOtherEndId (n_id, e_it);
		pos = (EGbitTest (nodeData[no_id].It, dom) ? 1U : 0U) |
			(EGbitTest (nodeData[no_id].Ib, dom) ? 2U : 0U);
		switch (pos)
		{
		case 0U:
			/* in this case, the only change is that we have to pay for this edge in
			 * Delta(T) */
			(*move_val) += weight[e_id];
			break;
		case 3U:
			/* in this case we save one Delta(T), pay one E(B:Bc) and flip AF */
			(*move_val) -= weight[e_id];
			if (!edgeData[e_id].num.in_BF)
				(*move_val) += 2 * weight[e_id];
			break;
		case 1U:
			/* in this case we save one Delta(T) */
			(*move_val) -= weight[e_id];
			break;
		default:
			EXIT (1, "This should not happen, value %u", pos);
		}														/* end switch */
	}															/* end loop through all incident edges */
	return 0;
}

/* ========================================================================= */
/** @brief This function prices the move Tc B, assuming the move is feasible*/
/* ========================================================================= */
static inline int KPTpriceTcB (const KPTMove_t * const cur_move,
															 double *const move_val)
{
	/* local variables */
	unsigned int e_id,
	  no_id,
	  pos,
	  n_id = cur_move->n_id,
	  dom = cur_move->dom;
	EGlistNode_t *e_it = all_nodes[n_id]->edges->begin;

	/* we have to compute every move */
	for (; e_it; e_it = e_it->next)
	{
		e_id = KPTgetEdgeId (e_it);
		no_id = KPTgetOtherEndId (n_id, e_it);
		pos = (EGbitTest (nodeData[no_id].It, dom) ? 1U : 0U) |
			(EGbitTest (nodeData[no_id].Ib, dom) ? 2U : 0U);
		switch (pos)
		{
		case 0U:
			/* in this case, the only change is that we have to pay for this edge in
			 * Delta(T) */
			(*move_val) += weight[e_id];
			break;
		case 1U:
			/* in this case we save one Delta(T), pay one E(B:Bc) and flip AF */
			(*move_val) -= weight[e_id];
			if (!edgeData[e_id].num.in_BF)
				(*move_val) += 2 * weight[e_id];
			break;
		case 3U:
			/* in this case we save one Delta(T) */
			(*move_val) -= weight[e_id];
			break;
		default:
			EXIT (1, "This should not happen, value %u", pos);
		}														/* end switch */
	}															/* end loop through all incident edges */
	return 0;
}

/* ========================================================================= */
/** @brief This function prices the move TcAc, assuming the move is feasible*/
/* ========================================================================= */
static inline int KPTpriceTcAc (const KPTMove_t * const cur_move,
																double *const move_val)
{
	/* local variables */
	unsigned int e_id,
	  no_id,
	  pos,
	  n_id = cur_move->n_id,
	  dom = cur_move->dom;
	EGlistNode_t *e_it = all_nodes[n_id]->edges->begin;

	/* we have to compute every move */
	for (; e_it; e_it = e_it->next)
	{
		e_id = KPTgetEdgeId (e_it);
		no_id = KPTgetOtherEndId (n_id, e_it);
		pos = (EGbitTest (nodeData[no_id].It, dom) ? 1U : 0U) |
			(EGbitTest (nodeData[no_id].Ia, dom) ? 2U : 0U);
		switch (pos)
		{
		case 0U:
			/* in this case, the only change is that we have to pay for this edge in
			 * Delta(T) */
			(*move_val) += weight[e_id];
			break;
		case 3U:
			/* in this case we save one Delta(T), pay one E(A:Ac) and flip AF */
			(*move_val) -= weight[e_id];
			if (!edgeData[e_id].num.in_AF)
				(*move_val) += 2 * weight[e_id];
			break;
		case 1U:
			/* in this case we save one Delta(T) */
			(*move_val) -= weight[e_id];
			break;
		default:
			EXIT (1, "This should not happen, value %u", pos);
		}														/* end switch */
	}															/* end loop through all incident edges */
	return 0;
}

/* ========================================================================= */
/** @brief This function prices the move Tc A, assuming the move is feasible*/
/* ========================================================================= */
static inline int KPTpriceTcA (const KPTMove_t * const cur_move,
															 double *const move_val)
{
	/* local variables */
	unsigned int e_id,
	  no_id,
	  pos,
	  n_id = cur_move->n_id,
	  dom = cur_move->dom;
	EGlistNode_t *e_it = all_nodes[n_id]->edges->begin;

	/* we have to compute every move */
	for (; e_it; e_it = e_it->next)
	{
		e_id = KPTgetEdgeId (e_it);
		no_id = KPTgetOtherEndId (n_id, e_it);
		pos = (EGbitTest (nodeData[no_id].It, dom) ? 1U : 0U) |
			(EGbitTest (nodeData[no_id].Ia, dom) ? 2U : 0U);
		switch (pos)
		{
		case 0U:
			/* in this case, the only change is that we have to pay for this edge in
			 * Delta(T) */
			(*move_val) += weight[e_id];
			break;
		case 1U:
			/* in this case we save one Delta(T), pay one E(A:Ac) and flip AF */
			(*move_val) -= weight[e_id];
			if (!edgeData[e_id].num.in_AF)
				(*move_val) += 2 * weight[e_id];
			break;
		case 3U:
			/* in this case we save one Delta(T) */
			(*move_val) -= weight[e_id];
			break;
		default:
			EXIT (1, "This should not happen, value %u", pos);
		}														/* end switch */
	}															/* end loop through all incident edges */
	return 0;
}

/* ========================================================================= */
/** @brief This function prices the move BcTc, assuming the move is feasible*/
/* ========================================================================= */
static inline int KPTpriceBcTc (const KPTMove_t * const cur_move,
																double *const move_val)
{
	/* local variables */
	unsigned int e_id,
	  no_id,
	  pos,
	  n_id = cur_move->n_id,
	  dom = cur_move->dom;
	EGlistNode_t *e_it = all_nodes[n_id]->edges->begin;

	/* we have to compute every move */
	for (; e_it; e_it = e_it->next)
	{
		e_id = KPTgetEdgeId (e_it);
		no_id = KPTgetOtherEndId (n_id, e_it);
		pos = (EGbitTest (nodeData[no_id].It, dom) ? 1U : 0U) |
			(EGbitTest (nodeData[no_id].Ib, dom) ? 2U : 0U);
		switch (pos)
		{
		case 0U:
			/* in this case, the only change is that we have to save for this edge in
			 * Delta(T) */
			(*move_val) -= weight[e_id];
			break;
		case 3U:
			/* in this case we pay one Delta(T), save one E(B:Bc) and flip BF */
			(*move_val) += weight[e_id];
			if (edgeData[e_id].num.in_BF)
				(*move_val) -= 2 * weight[e_id];
			break;
		case 1U:
			/* in this case we pay one Delta(T) */
			(*move_val) += weight[e_id];
			break;
		default:
			EXIT (1, "This should not happen, value %u", pos);
		}														/* end switch */
	}															/* end loop through all incident edges */
	return 0;
}

/* ========================================================================= */
/** @brief This function prices the move BTc, assuming the move is feasible*/
/* ========================================================================= */
static inline int KPTpriceBTc (const KPTMove_t * const cur_move,
															 double *const move_val)
{
	/* local variables */
	unsigned int e_id,
	  no_id,
	  pos,
	  n_id = cur_move->n_id,
	  dom = cur_move->dom;
	EGlistNode_t *e_it = all_nodes[n_id]->edges->begin;

	/* we have to compute every move */
	for (; e_it; e_it = e_it->next)
	{
		e_id = KPTgetEdgeId (e_it);
		no_id = KPTgetOtherEndId (n_id, e_it);
		pos = (EGbitTest (nodeData[no_id].It, dom) ? 1U : 0U) |
			(EGbitTest (nodeData[no_id].Ib, dom) ? 2U : 0U);
		switch (pos)
		{
		case 0U:
			/* in this case, the only change is that we have to save for this edge in
			 * Delta(T) */
			(*move_val) -= weight[e_id];
			break;
		case 1U:
			/* in this case we pay one Delta(T), save one E(B:Bc) and flip AF */
			(*move_val) += weight[e_id];
			if (edgeData[e_id].num.in_BF)
				(*move_val) -= 2 * weight[e_id];
			break;
		case 3U:
			/* in this case we pay one Delta(T) */
			(*move_val) += weight[e_id];
			break;
		default:
			EXIT (1, "This should not happen, value %u", pos);
		}														/* end switch */
	}															/* end loop through all incident edges */
	return 0;
}

/* ========================================================================= */
/** @brief This function prices the move AcTc, assuming the move is feasible*/
/* ========================================================================= */
static inline int KPTpriceAcTc (const KPTMove_t * const cur_move,
																double *const move_val)
{
	/* local variables */
	unsigned int e_id,
	  no_id,
	  pos,
	  n_id = cur_move->n_id,
	  dom = cur_move->dom;
	EGlistNode_t *e_it = all_nodes[n_id]->edges->begin;

	/* we have to compute every move */
	for (; e_it; e_it = e_it->next)
	{
		e_id = KPTgetEdgeId (e_it);
		no_id = KPTgetOtherEndId (n_id, e_it);
		pos = (EGbitTest (nodeData[no_id].It, dom) ? 1U : 0U) |
			(EGbitTest (nodeData[no_id].Ia, dom) ? 2U : 0U);
		switch (pos)
		{
		case 0U:
			/* in this case, the only change is that we have to save for this edge in
			 * Delta(T) */
			(*move_val) -= weight[e_id];
			break;
		case 3U:
			/* in this case we save one Delta(T), save one E(A:Ac) and flip AF */
			(*move_val) += weight[e_id];
			if (edgeData[e_id].num.in_AF)
				(*move_val) -= 2 * weight[e_id];
			break;
		case 1U:
			/* in this case we save one Delta(T) */
			(*move_val) += weight[e_id];
			break;
		default:
			EXIT (1, "This should not happen, value %u", pos);
		}														/* end switch */
	}															/* end loop through all incident edges */
	return 0;
}

/* ========================================================================= */
/** @brief This function prices the move ATc, assuming the move is feasible*/
/* ========================================================================= */
static inline int KPTpriceATc (const KPTMove_t * const cur_move,
															 double *const move_val)
{
	/* local variables */
	unsigned int e_id,
	  no_id,
	  pos,
	  n_id = cur_move->n_id,
	  dom = cur_move->dom;
	EGlistNode_t *e_it = all_nodes[n_id]->edges->begin;

	/* we have to compute every move */
	for (; e_it; e_it = e_it->next)
	{
		e_id = KPTgetEdgeId (e_it);
		no_id = KPTgetOtherEndId (n_id, e_it);
		pos = (EGbitTest (nodeData[no_id].It, dom) ? 1U : 0U) |
			(EGbitTest (nodeData[no_id].Ia, dom) ? 2U : 0U);
		switch (pos)
		{
		case 0U:
			/* in this case, the only change is that we have to save for this edge in
			 * Delta(T) */
			(*move_val) -= weight[e_id];
			break;
		case 1U:
			/* in this case we pay one Delta(T), save one E(A:Ac) and flip AF */
			(*move_val) += weight[e_id];
			if (edgeData[e_id].num.in_AF)
				(*move_val) -= 2 * weight[e_id];
			break;
		case 3U:
			/* in this case we pay one Delta(T) */
			(*move_val) += weight[e_id];
			break;
		default:
			EXIT (1, "This should not happen, value %u", pos);
		}														/* end switch */
	}															/* end loop through all incident edges */
	return 0;
}

/* ========================================================================= */
/** @brief Make a full move (in his whole depth ) */
/* ========================================================================= */
static inline int KPTmakeFullMove (KPTFullMove_t const *const full_move)
{
	unsigned int depth = 0;
	for (; depth < full_move->depth; depth++)
		KPTmakeMove (full_move->move + depth, 1);
	return 0;
}

/* ========================================================================= */
/** @ brief, call the best move on a deeper base */
/* ========================================================================= */
#define KPTgoDeeper(__depth) if(__depth+1<KPT_MAX_DEPTH){\
	/* now we call deeper moves */\
	KPTmakeMove(base_move->move+__depth,0);\
	for( e_it = all_nodes[nc_id]->edges->begin; e_it ; e_it=e_it->next){\
		/*if(KPTgetOtherEndId(nc_id,e_it)<nc_id)*/\
		KPTSetBestMove( KPTgetOtherEndId( nc_id, e_it), best_move, \
										base_move, __depth + 1);}\
	/* undo the last move */\
	KPTmakeInvMove(base_move->move+__depth,0);\
	}


/* ========================================================================= */
/** @brief, given a node, compute the best possible move for it, taking into
 * acount if the node has been added to the T set before or not, remember that
 * the priority is to add, and then to substract, note that this change is not
 * reflected in the tree of best moves, you have to do it outside this
 * function. the best move and value is stored in the given field, the actual
 * best move stored inside the node data is not changed. */
/* ========================================================================= */
static int KPTSetBestMove (const unsigned int nc_id,
													 KPTFullMove_t * const best_move,
													 KPTFullMove_t * const base_move,
													 const unsigned int depth)
{
	/* local variables */
	unsigned nc_inA,
	  nc_inB,
	  nc_inTc;
	KPTMove_t *c_movep = base_move->move + depth;
	unsigned type,
	  pos,
	  nc_inAdB,
	 *domp = &(c_movep->dom);
	const double move_val = base_move->val;
	EGlistNode_t *e_it;
	MESSAGE (KPT_VRB + 19, "entering depth %u", depth);
	TESTL (KPT_EDBG, nc_id > graphData.n_nodes,
				 "nc_id %u out of range (n_nodes %u)", nc_id, graphData.n_nodes);
	/* check that the depth is OK, if no, we return back */
	if (depth >= KPT_MAX_DEPTH)
		return 0;
	/* basic set-up */
	c_movep->n_id = nc_id;
	c_movep->move_id = KPT_no_move;
	base_move->depth = depth + 1;
	/* now we check the adding moves if they can be considered */
	(*domp) = graphData.n_dominos;
	for (; (*domp)--;)
	{
		nc_inA = EGbitTest (nodeData[nc_id].Ia, (*domp)) ? 1U : 0U;
		nc_inB = EGbitTest (nodeData[nc_id].Ib, (*domp)) ? 1U : 0U;
		nc_inTc = EGbitTest (nodeData[nc_id].It, (*domp)) ? 0U : 1U;
		type = (EGbitTest (graphData.A_partition, (*domp)) ? 1U : 0U) |
			(EGbitTest (graphData.B_partition, (*domp)) ? 2U : 0U);
		/* first check addind moves, but only if they have not been done */
		if (nc_inTc && !EGbitTest (nodeData[nc_id].added, (*domp)) &&
				(graphData.n_Tc[(*domp)] > 1))
		{
			switch (type)
			{
			case 1U:
				/* ----------------------------------------------------------------- */
				/* Check KPT_Tc_Ac */
				base_move->val = move_val;
				c_movep->move_id = KPT_Tc_Ac;
				KPTpriceTcAc (c_movep, &(base_move->val));
				KPTupdateMove (base_move, best_move);
				/* now we call deeper moves */
				KPTgoDeeper (depth);
				/* ----------------------------------------------------------------- */
				/* Check KPT_Tc_A */
				base_move->val = move_val;
				c_movep->move_id = KPT_Tc_A;
				KPTpriceTcA (c_movep, &(base_move->val));
				KPTupdateMove (base_move, best_move);
				/* now we call deeper moves */
				KPTgoDeeper (depth);
				break;
			case 2U:
				/* ----------------------------------------------------------------- */
				/* Check KPT_Tc_B */
				base_move->val = move_val;
				c_movep->move_id = KPT_Tc_B;
				KPTpriceTcB (c_movep, &(base_move->val));
				KPTupdateMove (base_move, best_move);
				/* now we call deeper moves */
				KPTgoDeeper (depth);
				/* ----------------------------------------------------------------- */
				/* Check KPT_Tc_Bc */
				base_move->val = move_val;
				c_movep->move_id = KPT_Tc_Bc;
				KPTpriceTcBc (c_movep, &(base_move->val));
				KPTupdateMove (base_move, best_move);
				/* now we call deeper moves */
				KPTgoDeeper (depth);
				break;
			case 3U:
				/* ----------------------------------------------------------------- */
				/* Check KPT_Tc_AB */
				base_move->val = move_val;
				c_movep->move_id = KPT_Tc_AB;
				KPTpriceTcAB (c_movep, &(base_move->val));
				KPTupdateMove (base_move, best_move);
				/* now we call deeper moves */
				KPTgoDeeper (depth);
				/* ----------------------------------------------------------------- */
				/* Check KPT_Tc_AcB */
				base_move->val = move_val;
				c_movep->move_id = KPT_Tc_AcB;
				KPTpriceTcAcB (c_movep, &(base_move->val));
				KPTupdateMove (base_move, best_move);
				/* now we call deeper moves */
				KPTgoDeeper (depth);
				/* ----------------------------------------------------------------- */
				/* Check KPT_Tc_ABc */
				base_move->val = move_val;
				c_movep->move_id = KPT_Tc_ABc;
				KPTpriceTcABc (c_movep, &(base_move->val));
				KPTupdateMove (base_move, best_move);
				/* now we call deeper moves */
				KPTgoDeeper (depth);
				/* ----------------------------------------------------------------- */
				/* Check KPT_Tc_AcBc */
				base_move->val = move_val;
				c_movep->move_id = KPT_Tc_AcBc;
				KPTpriceTcAcBc (c_movep, &(base_move->val));
				KPTupdateMove (base_move, best_move);
				/* now we call deeper moves */
				KPTgoDeeper (depth);
				break;
			default:
				EXIT (1, "domino type %u unknown", type);
				/* ----------------------------------------------------------------- */
			}													/* end checking adding moves */
		}														/* end switch */
	}															/* end loop through all dominos */
	/* now we check if this node can be added to the handle */
	type = ((!nodeData[nc_id].in_BH && !nodeData[nc_id].added_BH &&
					 (graphData.n_BH[0] > 1)) ? 2U : 0U) | ((!nodeData[nc_id].in_AH &&
																									 !nodeData[nc_id].added_AH
																									 && (graphData.n_AH[0] >
																											 1)) ? 1U : 0U);
	switch (type)
	{
	case 3U:
	case 1U:
		/* --------------------------------------------------------------------- */
		/* Check KPT_AHc_AH */
		base_move->val = move_val;
		c_movep->move_id = KPT_AHc_AH;
		KPTpriceAHcAH (c_movep, &(base_move->val));
		KPTupdateMove (base_move, best_move);
		/* now we call deeper moves */
		KPTgoDeeper (depth);
		if (type == 1U)
			break;
	case 2U:
		/* --------------------------------------------------------------------- */
		/* Check KPT_BHc_BH */
		base_move->val = move_val;
		c_movep->move_id = KPT_BHc_BH;
		KPTpriceBHcBH (c_movep, &(base_move->val));
		KPTupdateMove (base_move, best_move);
		/* now we call deeper moves */
		KPTgoDeeper (depth);
		if (type == 2U)
			break;
		/* --------------------------------------------------------------------- */
		/* Check KPT_AHc_AH_BHc_BH */
		base_move->val = move_val;
		c_movep->move_id = KPT_AHc_AH_BHc_BH;
		KPTpriceAHcAHBHcBH (c_movep, &(base_move->val));
		KPTupdateMove (base_move, best_move);
		/* now we call deeper moves */
		KPTgoDeeper (depth);
		break;
	case 0:
		break;
		/* --------------------------------------------------------------------- */
	default:
		EXIT (1, "Unknown type %u", type);
	}
	/* now we check moves that flip this node from A to B */
	(*domp) = graphData.n_dominos;
	for (; (*domp)--;)
	{
		nc_inA = EGbitTest (nodeData[nc_id].Ia, (*domp)) ? 1U : 0U;
		nc_inB = EGbitTest (nodeData[nc_id].Ib, (*domp)) ? 1U : 0U;
		nc_inTc = EGbitTest (nodeData[nc_id].It, (*domp)) ? 0U : 1U;
		nc_inAdB = EGbitTest (nodeData[nc_id].Iadb, (*domp)) ? 1U : 0U;
		type = (EGbitTest (graphData.A_partition, (*domp)) ? 1U : 0U) |
			(EGbitTest (graphData.B_partition, (*domp)) ? 2U : 0U);
		if (!nc_inTc)
		{
			switch (type)
			{
			case 3U:
			case 1U:
				/* ----------------------------------------------------------------- */
				/* check KPT_A_Ac */
				if (nc_inA && (graphData.n_A[1][(*domp)] > 1) &&
						!EGbitTest (nodeData[nc_id].flipA, (*domp)) &&
						(!EGbitTest (graphData.Kdom, (*domp)) ||
						 (graphData.n_AdB[nc_inAdB][(*domp)] > 1)))
				{
					base_move->val = move_val;
					c_movep->move_id = KPT_A_Ac;
					KPTpriceAAc (c_movep, &(base_move->val));
					KPTupdateMove (base_move, best_move);
					/* now we call deeper moves */
					KPTgoDeeper (depth);
					/* ----------------------------------------------------------------- */
					/* check KPT_A_Ac_flipAH */
					if (graphData.n_AH[nodeData[nc_id].in_AH] > 1)
					{
						base_move->val = move_val;
						c_movep->move_id = KPT_A_Ac_flipAH;
						KPTpriceAAcflipAH (c_movep, &(base_move->val));
						KPTupdateMove (base_move, best_move);
						/* now we call deeper moves */
						KPTgoDeeper (depth);
					}
				}
				/* ----------------------------------------------------------------- */
				/* check KPT_Ac_A */
				else if (!nc_inA && (graphData.n_A[0][(*domp)] > 1) &&
								 !EGbitTest (nodeData[nc_id].flipAc, (*domp)) &&
								 (!EGbitTest (graphData.Kdom, (*domp)) ||
									(graphData.n_AdB[nc_inAdB][(*domp)] > 1)))
				{
					base_move->val = move_val;
					c_movep->move_id = KPT_Ac_A;
					KPTpriceAcA (c_movep, &(base_move->val));
					KPTupdateMove (base_move, best_move);
					/* now we call deeper moves */
					KPTgoDeeper (depth);
					/* ----------------------------------------------------------------- */
					/* check KPT_Ac_A_flipH */
					if (graphData.n_AH[nodeData[nc_id].in_AH] > 1)
					{
						base_move->val = move_val;
						c_movep->move_id = KPT_Ac_A_flipAH;
						KPTpriceAcAflipAH (c_movep, &(base_move->val));
						KPTupdateMove (base_move, best_move);
						/* now we call deeper moves */
						KPTgoDeeper (depth);
					}
				}
				/* ----------------------------------------------------------------- */
				if (type == 1U)
					break;
			case 2U:
				/* ----------------------------------------------------------------- */
				/* check KPT_B_Bc */
				if (nc_inB && (graphData.n_B[1][(*domp)] > 1) &&
						!EGbitTest (nodeData[nc_id].flipB, (*domp)) &&
						(!EGbitTest (graphData.Kdom, (*domp)) ||
						 (graphData.n_AdB[nc_inAdB][(*domp)] > 1)))
				{
					base_move->val = move_val;
					c_movep->move_id = KPT_B_Bc;
					KPTpriceBBc (c_movep, &(base_move->val));
					KPTupdateMove (base_move, best_move);
					/* now we call deeper moves */
					KPTgoDeeper (depth);
					/* ----------------------------------------------------------------- */
					/* check KPT_B_Bc_flipBH */
					if (graphData.n_BH[nodeData[nc_id].in_BH] > 1)
					{
						base_move->val = move_val;
						c_movep->move_id = KPT_B_Bc_flipBH;
						KPTpriceBBcflipBH (c_movep, &(base_move->val));
						KPTupdateMove (base_move, best_move);
						/* now we call deeper moves */
						KPTgoDeeper (depth);
					}
				}
				/* ----------------------------------------------------------------- */
				/* check KPT_Bc_B */
				else if (!nc_inB && (graphData.n_B[0][(*domp)] > 1) &&
								 !EGbitTest (nodeData[nc_id].flipBc, (*domp)) &&
								 (!EGbitTest (graphData.Kdom, (*domp)) ||
									(graphData.n_AdB[nc_inAdB][(*domp)] > 1)))
				{
					base_move->val = move_val;
					c_movep->move_id = KPT_Bc_B;
					KPTpriceBcB (c_movep, &(base_move->val));
					KPTupdateMove (base_move, best_move);
					/* now we call deeper moves */
					KPTgoDeeper (depth);
					/* ----------------------------------------------------------------- */
					/* check KPT_Bc_B_flipH */
					if (graphData.n_BH[nodeData[nc_id].in_BH] > 1)
					{
						base_move->val = move_val;
						c_movep->move_id = KPT_Bc_B_flipBH;
						KPTpriceBcBflipBH (c_movep, &(base_move->val));
						KPTupdateMove (base_move, best_move);
						/* now we call deeper moves */
						KPTgoDeeper (depth);
					}
				}
				/* ----------------------------------------------------------------- */
				if (type == 2U)
					break;
				/* ----------------------------------------------------------------- */
				/* check KPT_A_Ac_B_Bc */
				if (nc_inA && (graphData.n_A[1][(*domp)] > 1) &&
						!EGbitTest (nodeData[nc_id].flipA, (*domp)) &&
						nc_inB && (graphData.n_B[1][(*domp)] > 1) &&
						!EGbitTest (nodeData[nc_id].flipB, (*domp)))
				{
					base_move->val = move_val;
					c_movep->move_id = KPT_A_Ac_B_Bc;
					KPTpriceAAcBBc (c_movep, &(base_move->val));
					KPTupdateMove (base_move, best_move);
					/* now we call deeper moves */
					KPTgoDeeper (depth);
					/* ----------------------------------------------------------------- */
					/* check KPT_A_Ac_B_Bc_flipAH */
					if ((graphData.n_AH[nodeData[nc_id].in_AH] > 1))
					{
						base_move->val = move_val;
						c_movep->move_id = KPT_A_Ac_B_Bc_flipAH;
						KPTpriceAAcBBcAH (c_movep, &(base_move->val));
						KPTupdateMove (base_move, best_move);
						/* now we call deeper moves */
						KPTgoDeeper (depth);
					}
					/* ----------------------------------------------------------------- */
					/* check KPT_A_Ac_B_Bc_flipBH */
					if ((graphData.n_BH[nodeData[nc_id].in_BH] > 1))
					{
						base_move->val = move_val;
						c_movep->move_id = KPT_A_Ac_B_Bc_flipBH;
						KPTpriceAAcBBcBH (c_movep, &(base_move->val));
						KPTupdateMove (base_move, best_move);
						/* now we call deeper moves */
						KPTgoDeeper (depth);
					}
					/* ----------------------------------------------------------------- */
					/* check KPT_A_Ac_B_Bc_flipAHBH */
					if ((graphData.n_BH[nodeData[nc_id].in_BH] > 1) &&
							(graphData.n_AH[nodeData[nc_id].in_AH] > 1))
					{
						base_move->val = move_val;
						c_movep->move_id = KPT_A_Ac_B_Bc_flipAHBH;
						KPTpriceAAcBBcAHBH (c_movep, &(base_move->val));
						KPTupdateMove (base_move, best_move);
						/* now we call deeper moves */
						KPTgoDeeper (depth);
					}
				}
				/* ----------------------------------------------------------------- */
				/* check KPT_Ac_A_B_Bc */
				else if (!nc_inA && (graphData.n_A[0][(*domp)] > 1) &&
								 !EGbitTest (nodeData[nc_id].flipAc, (*domp)) &&
								 nc_inB && (graphData.n_B[1][(*domp)] > 1) &&
								 !EGbitTest (nodeData[nc_id].flipB, (*domp)))
				{
					base_move->val = move_val;
					c_movep->move_id = KPT_Ac_A_B_Bc;
					KPTpriceAcABBc (c_movep, &(base_move->val));
					KPTupdateMove (base_move, best_move);
					/* now we call deeper moves */
					KPTgoDeeper (depth);
					/* ----------------------------------------------------------------- */
					/* check KPT_Ac_A_B_Bc_flipAH */
					if ((graphData.n_AH[nodeData[nc_id].in_AH] > 1))
					{
						base_move->val = move_val;
						c_movep->move_id = KPT_Ac_A_B_Bc_flipAH;
						KPTpriceAcABBcAH (c_movep, &(base_move->val));
						KPTupdateMove (base_move, best_move);
						/* now we call deeper moves */
						KPTgoDeeper (depth);
					}
					/* ----------------------------------------------------------------- */
					/* check KPT_Ac_A_B_Bc_flipBH */
					if ((graphData.n_BH[nodeData[nc_id].in_BH] > 1))
					{
						base_move->val = move_val;
						c_movep->move_id = KPT_Ac_A_B_Bc_flipBH;
						KPTpriceAcABBcBH (c_movep, &(base_move->val));
						KPTupdateMove (base_move, best_move);
						/* now we call deeper moves */
						KPTgoDeeper (depth);
					}
					/* ----------------------------------------------------------------- */
					/* check KPT_Ac_A_B_Bc_flipAHBH */
					if ((graphData.n_BH[nodeData[nc_id].in_BH] > 1) &&
							(graphData.n_AH[nodeData[nc_id].in_AH] > 1))
					{
						base_move->val = move_val;
						c_movep->move_id = KPT_Ac_A_B_Bc_flipAHBH;
						KPTpriceAcABBcAHBH (c_movep, &(base_move->val));
						KPTupdateMove (base_move, best_move);
						/* now we call deeper moves */
						KPTgoDeeper (depth);
					}
				}
				/* ----------------------------------------------------------------- */
				/* check KPT_A_Ac_Bc_B */
				else if (nc_inA && (graphData.n_A[1][(*domp)] > 1) &&
								 !EGbitTest (nodeData[nc_id].flipA, (*domp)) &&
								 !nc_inB && (graphData.n_B[0][(*domp)] > 1) &&
								 !EGbitTest (nodeData[nc_id].flipBc, (*domp)))
				{
					base_move->val = move_val;
					c_movep->move_id = KPT_A_Ac_Bc_B;
					KPTpriceAAcBcB (c_movep, &(base_move->val));
					KPTupdateMove (base_move, best_move);
					/* now we call deeper moves */
					KPTgoDeeper (depth);
					/* ----------------------------------------------------------------- */
					/* check KPT_A_Ac_Bc_B_flipAH */
					if ((graphData.n_AH[nodeData[nc_id].in_AH] > 1))
					{
						base_move->val = move_val;
						c_movep->move_id = KPT_A_Ac_Bc_B_flipAH;
						KPTpriceAAcBcBAH (c_movep, &(base_move->val));
						KPTupdateMove (base_move, best_move);
						/* now we call deeper moves */
						KPTgoDeeper (depth);
					}
					/* ----------------------------------------------------------------- */
					/* check KPT_A_Ac_Bc_B_flipBH */
					if ((graphData.n_BH[nodeData[nc_id].in_BH] > 1))
					{
						base_move->val = move_val;
						c_movep->move_id = KPT_A_Ac_Bc_B_flipBH;
						KPTpriceAAcBcBBH (c_movep, &(base_move->val));
						KPTupdateMove (base_move, best_move);
						/* now we call deeper moves */
						KPTgoDeeper (depth);
					}
					/* ----------------------------------------------------------------- */
					/* check KPT_A_Ac_Bc_B_flipAHBH */
					if ((graphData.n_BH[nodeData[nc_id].in_BH] > 1) &&
							(graphData.n_AH[nodeData[nc_id].in_AH] > 1))
					{
						base_move->val = move_val;
						c_movep->move_id = KPT_A_Ac_Bc_B_flipAHBH;
						KPTpriceAAcBcBAHBH (c_movep, &(base_move->val));
						KPTupdateMove (base_move, best_move);
						/* now we call deeper moves */
						KPTgoDeeper (depth);
					}
				}
				/* ----------------------------------------------------------------- */
				/* check KPT_Ac_A_Bc_B */
				else if (!nc_inA && (graphData.n_A[0][(*domp)] > 1) &&
								 !EGbitTest (nodeData[nc_id].flipAc, (*domp)) &&
								 !nc_inB && (graphData.n_B[0][(*domp)] > 1) &&
								 !EGbitTest (nodeData[nc_id].flipBc, (*domp)))
				{
					base_move->val = move_val;
					c_movep->move_id = KPT_Ac_A_Bc_B;
					KPTpriceAcABcB (c_movep, &(base_move->val));
					KPTupdateMove (base_move, best_move);
					/* now we call deeper moves */
					KPTgoDeeper (depth);
					/* ----------------------------------------------------------------- */
					/* check KPT_Ac_A_Bc_B_flipAH */
					if ((graphData.n_AH[nodeData[nc_id].in_AH] > 1))
					{
						base_move->val = move_val;
						c_movep->move_id = KPT_Ac_A_Bc_B_flipAH;
						KPTpriceAcABcBAH (c_movep, &(base_move->val));
						KPTupdateMove (base_move, best_move);
						/* now we call deeper moves */
						KPTgoDeeper (depth);
					}
					/* ----------------------------------------------------------------- */
					/* check KPT_Ac_A_Bc_B_flipBH */
					if ((graphData.n_BH[nodeData[nc_id].in_BH] > 1))
					{
						base_move->val = move_val;
						c_movep->move_id = KPT_Ac_A_Bc_B_flipBH;
						KPTpriceAcABcBBH (c_movep, &(base_move->val));
						KPTupdateMove (base_move, best_move);
						/* now we call deeper moves */
						KPTgoDeeper (depth);
					}
					/* ----------------------------------------------------------------- */
					/* check KPT_Ac_A_Bc_B_flipAHBH */
					if ((graphData.n_BH[nodeData[nc_id].in_BH] > 1) &&
							(graphData.n_AH[nodeData[nc_id].in_AH] > 1))
					{
						base_move->val = move_val;
						c_movep->move_id = KPT_Ac_A_Bc_B_flipAHBH;
						KPTpriceAcABcBAHBH (c_movep, &(base_move->val));
						KPTupdateMove (base_move, best_move);
						/* now we call deeper moves */
						KPTgoDeeper (depth);
					}
				}
				break;
			default:
				EXIT (1, "Unknown type %u", type);
				/* ----------------------------------------------------------------- */
			}													/* end case nc in T */
		}														/* end switch type */
	}															/* end checking non-growing moves for dominos */
	/* now we check if this node can shufled between handles */
	/* ----------------------------------------------------------------------- */
	/* check KPT_AHc_AH_BH_BHc */
	if (!nodeData[nc_id].in_AH && (graphData.n_AH[0] > 1) &&
			nodeData[nc_id].in_BH && (graphData.n_BH[1] > 1) &&
			!nodeData[nc_id].added_AH)
	{
		base_move->val = move_val;
		c_movep->move_id = KPT_AHc_AH_BH_BHc;
		KPTpriceAHcAHBHBHc (c_movep, &(base_move->val));
		KPTupdateMove (base_move, best_move);
		/* now we call deeper moves */
		KPTgoDeeper (depth);
	}
	/* ----------------------------------------------------------------------- */
	/* check KPT_AH_AHc_BHc_BH */
	if (nodeData[nc_id].in_AH && (graphData.n_AH[1] > 1) &&
			!nodeData[nc_id].in_BH && (graphData.n_BH[0] > 1) &&
			!nodeData[nc_id].added_BH)
	{
		base_move->val = move_val;
		c_movep->move_id = KPT_AH_AHc_BHc_BH;
		KPTpriceAHAHcBHcBH (c_movep, &(base_move->val));
		KPTupdateMove (base_move, best_move);
		/* now we call deeper moves */
		KPTgoDeeper (depth);
	}
	/* ----------------------------------------------------------------------- */
	/* now we check moves that may get this node out of a teeth */
	(*domp) = graphData.n_dominos;
	for (; (*domp)--;)
	{
		nc_inA = EGbitTest (nodeData[nc_id].Ia, (*domp)) ? 1U : 0U;
		nc_inB = EGbitTest (nodeData[nc_id].Ib, (*domp)) ? 1U : 0U;
		nc_inTc = EGbitTest (nodeData[nc_id].It, (*domp)) ? 0U : 1U;
		nc_inAdB = EGbitTest (nodeData[nc_id].Iadb, (*domp)) ? 1U : 0U;
		type = (EGbitTest (graphData.A_partition, (*domp)) ? 1U : 0U) |
			(EGbitTest (graphData.B_partition, (*domp)) ? 2U : 0U);
		if (!nc_inTc)
		{
			switch (type)
			{
			case 1U:
				if (graphData.n_A[nc_inA][(*domp)] > 1)
				{
					/* --------------------------------------------------------------- */
					/* Check KPT_A_Tc */
					if (nc_inA)
					{
						base_move->val = move_val;
						c_movep->move_id = KPT_A_Tc;
						KPTpriceATc (c_movep, &(base_move->val));
						KPTupdateMove (base_move, best_move);
						/* now we call deeper moves */
						KPTgoDeeper (depth);
					}
					/* --------------------------------------------------------------- */
					/* Check KPT_Ac_Tc */
					else
					{
						base_move->val = move_val;
						c_movep->move_id = KPT_Ac_Tc;
						KPTpriceAcTc (c_movep, &(base_move->val));
						KPTupdateMove (base_move, best_move);
						/* now we call deeper moves */
						KPTgoDeeper (depth);
					}
				}												/*end case a A-domino */
				break;
			case 2U:
				if (graphData.n_B[nc_inB][(*domp)] > 1)
				{
					/* --------------------------------------------------------------- */
					/* Check KPT_B_Tc */
					if (nc_inB)
					{
						base_move->val = move_val;
						c_movep->move_id = KPT_B_Tc;
						KPTpriceBTc (c_movep, &(base_move->val));
						KPTupdateMove (base_move, best_move);
						/* now we call deeper moves */
						KPTgoDeeper (depth);
					}
					/* --------------------------------------------------------------- */
					/* Check KPT_Bc_Tc */
					else
					{
						base_move->val = move_val;
						c_movep->move_id = KPT_Bc_Tc;
						KPTpriceBcTc (c_movep, &(base_move->val));
						KPTupdateMove (base_move, best_move);
						/* now we call deeper moves */
						KPTgoDeeper (depth);
					}
					/* --------------------------------------------------------------- */
				}												/*end case a B-domino */
				break;
			case 3U:
				if ((graphData.n_B[nc_inB][(*domp)] > 1) &&
						(graphData.n_A[nc_inA][(*domp)] > 1) &&
						(graphData.n_AdB[nc_inAdB][(*domp)] > 1))
				{
					pos = (nc_inA ? 1U : 0U) | (nc_inB ? 2U : 0U);
					switch (pos)
					{
					case 0:
						/* ------------------------------------------------------------- */
						/* Check KPT_AcBcTc */
						base_move->val = move_val;
						c_movep->move_id = KPT_AcBc_Tc;
						KPTpriceAcBcTc (c_movep, &(base_move->val));
						KPTupdateMove (base_move, best_move);
						/* now we call deeper moves */
						KPTgoDeeper (depth);
						break;
					case 1:
						/* ------------------------------------------------------------- */
						/* Check KPT_ABcTc */
						base_move->val = move_val;
						c_movep->move_id = KPT_ABc_Tc;
						KPTpriceABcTc (c_movep, &(base_move->val));
						KPTupdateMove (base_move, best_move);
						/* now we call deeper moves */
						KPTgoDeeper (depth);
						break;
					case 2:
						/* ------------------------------------------------------------- */
						/* Check KPT_AcBTc */
						base_move->val = move_val;
						c_movep->move_id = KPT_AcB_Tc;
						KPTpriceAcBTc (c_movep, &(base_move->val));
						KPTupdateMove (base_move, best_move);
						/* now we call deeper moves */
						KPTgoDeeper (depth);
						break;
					case 3:
						/* ------------------------------------------------------------- */
						/* Check KPT_ABTc */
						base_move->val = move_val;
						c_movep->move_id = KPT_AB_Tc;
						KPTpriceABTc (c_movep, &(base_move->val));
						KPTupdateMove (base_move, best_move);
						/* now we call deeper moves */
						KPTgoDeeper (depth);
						break;
						/* ------------------------------------------------------------- */
					default:
						EXIT (1, "unknown pos %u", pos);
					}
				}												/* end case 2-domino */
				break;
			default:
				EXIT (1, "unknown type %u", type);
			}													/* end switch */
		}														/* end checking shrinking moves for dominoes */
	}															/*end loop through dominoes */
	/* now we check if this node can be shrinked from the handle */
	type = ((nodeData[nc_id].in_BH && (graphData.n_BH[1] > 1)) ? 2U : 0U) |
		((nodeData[nc_id].in_AH && (graphData.n_AH[1] > 1)) ? 1U : 0U);
	switch (type)
	{
	case 3U:
	case 1U:
		/* --------------------------------------------------------------------- */
		/* Check KPT_AH_AHc */
		base_move->val = move_val;
		c_movep->move_id = KPT_AH_AHc;
		KPTpriceAHAHc (c_movep, &(base_move->val));
		KPTupdateMove (base_move, best_move);
		/* now we call deeper moves */
		KPTgoDeeper (depth);
		if (type == 1U)
			break;
	case 2U:
		/* --------------------------------------------------------------------- */
		/* Check KPT_BH_BHc */
		base_move->val = move_val;
		c_movep->move_id = KPT_BH_BHc;
		KPTpriceBHBHc (c_movep, &(base_move->val));
		KPTupdateMove (base_move, best_move);
		/* now we call deeper moves */
		KPTgoDeeper (depth);
		if (type == 2U)
			break;
		/* --------------------------------------------------------------------- */
		/* Check KPT_AH_AHc_BH_BHc */
		base_move->val = move_val;
		c_movep->move_id = KPT_AH_AHc_BH_BHc;
		KPTpriceAHAHcBHBHc (c_movep, &(base_move->val));
		KPTupdateMove (base_move, best_move);
		/* now we call deeper moves */
		KPTgoDeeper (depth);
		break;
	case 0:
		break;
		/* --------------------------------------------------------------------- */
	default:
		EXIT (1, "Unknown type %u", type);
	}
	/* ending setting best move */
	base_move->val = move_val;
	base_move->depth = depth;
	MESSAGE (KPT_VRB + 19, "done");
	return 0;
}

/* ========================================================================= */
/** @brief This function reset all flags to zero, we do this every time that we
 * make a substancial improvement */
/* ========================================================================= */
static inline int KPTresetFlags (void)
{
	/* local variables */
	register unsigned int i = G->nodes->size;
	/* loop through all nodes */
	while (i--)
	{
		memset (nodeData[i].added, 0, KPT_MAX_DOM >> 3);
		memset (nodeData[i].flipA, 0, KPT_MAX_DOM >> 3);
		memset (nodeData[i].flipAc, 0, KPT_MAX_DOM >> 3);
		memset (nodeData[i].flipB, 0, KPT_MAX_DOM >> 3);
		memset (nodeData[i].flipBc, 0, KPT_MAX_DOM >> 3);
		nodeData[i].added_AH = 0;
		nodeData[i].added_BH = 0;
	}
	return 0;
}

/* ========================================================================= */
/** @brief This function stores a move in a node */
/* ========================================================================= */
static inline int KPTStoreFullMove (const unsigned int nc_id,
																		KPTFullMove_t const *const full_move)
{
	int rval = 0;
	unsigned int l_depth = full_move->depth;
	memcpy (&(nodeData[nc_id].full_move), full_move, sizeof (KPTFullMove_t));
	while (l_depth--)
	{
#if KPT_EDBG < DEBUG
		EXIT (full_move->move[l_depth].move_id == KPT_no_move, "Storing "
					"KPT_no_move for node %u, depth %u", nc_id, l_depth);
#endif
	}
	rval = EGbbtreeRemove (tree, nodeData[nc_id].tree_cn);
	CHECKRVAL(rval);
	rval = !(nodeData[nc_id].tree_cn = EGbbtreeAdd (tree, nodeData + nc_id));
	return rval;
}

/* ========================================================================= */
/** @brief reset a full move to all no-move */
/* ========================================================================= */
static inline int KPTResetMove (KPTFullMove_t * move)
{
	memset (move, 0, sizeof (KPTFullMove_t));
	move->move[0].move_id = KPT_no_move;
	return 0;
}

/* ========================================================================= */
/** @brief This function update the moves of all neighbours in a full move */
/* ========================================================================= */
static inline int KPTupdateNeighMove (KPTFullMove_t * full_move)
{
	int rval;
	unsigned int const max_depth = full_move->depth + KPT_MAX_DEPTH;
	unsigned int no_id[2 * KPT_MAX_DEPTH + 1],
	  n_depth = 0;
	EGlistNode_t *e_it[KPT_MAX_DEPTH * 2];
	KPTFullMove_t base_move,
	  cur_move;
	memset (&base_move, 0, sizeof (base_move));
	memset (&cur_move, 0, sizeof (base_move));
	KPTResetMove (&cur_move);
	cur_move.val = DBL_MAX;
	base_move.val = 0;
	no_id[0] = full_move->move[0].n_id;
	KPTSetBestMove (no_id[0], &cur_move, &base_move, 0);
	rval = KPTStoreFullMove (no_id[0], &cur_move);
	CHECKRVAL(rval);
	EGbitSet (node_update, no_id[0]);
	e_it[0] = all_nodes[no_id[0]]->edges->begin;
	n_depth = 0;
	/* for each depth move, we update the neighbours up to depth
	 * KPT_MAX_DEPTH+full_move->depth */
#if KPT_VRB+19<DEBUG
	DPdisplayMove (no_id[0]);
#endif
	while (e_it[0])
	{
		no_id[n_depth + 1] = KPTgetOtherEndId (no_id[n_depth], e_it[n_depth]);
		if (!EGbitTest (node_update, no_id[n_depth + 1]))
		{
			KPTResetMove (&cur_move);
			cur_move.val = DBL_MAX;
			base_move.val = 0;
			KPTSetBestMove (no_id[n_depth + 1], &cur_move, &base_move, 0);
			KPTStoreFullMove (no_id[n_depth + 1], &cur_move);
			EGbitSet (node_update, no_id[n_depth + 1]);
		}
		/* now we look at the next neighbour, if we are not at the end of the
		 * recursion, we go down one more level */
		if (n_depth + 1 < max_depth)
		{
			n_depth++;
			e_it[n_depth] = all_nodes[no_id[n_depth]]->edges->begin;
		}
		/* otherwise we move horizontally in the current level */
		else
		{
			e_it[n_depth] = e_it[n_depth]->next;
			/* if the next e_it is null, we have to move up one level, 
			 * if posible */
			while (n_depth && !e_it[n_depth])
			{
				n_depth--;
				e_it[n_depth] = e_it[n_depth]->next;
			}
		}														/* end looking at next neighbour */
	}															/* end updating for the given simple move */
#if KPT_VRB+19<DEBUG
	DPdisplayMove (no_id[0]);
#endif
/* ========================================================================= */
#if 0
#warning Note that here we update a node too many times, we could \
	decrease this by adding a flag for each node and update only if they \
	havent been updated so far. of course, this would need to reset the \
	flags after we are done.
#endif
/* ========================================================================= */
	memset (node_update, 0, sizeof (node_update));
	return rval;
}

/* ========================================================================= */
/* given a residual graph G^*, and a valid DP inequality, it will try to find a
 * 'more' violated DP-inequality constraint; the function will not touch the
 * data abour the graph nor about the original inequality, but it will alloc
 * memory for all data in the new returned inequality. Note too that the number
 * of dominos of the new inequality is the same as in the original inequality.
 * Finaly, the violation (if it is positive) of the new inequality is returned
 * in (*violation). If an error occurs the function will return 1, zero on
 * success (this doesn't imply that we were able to find a violated constraint
 * from the original onw */
/* ========================================================================= */
int KPtighten (									/* graph G* data */
								int const n_nodes,
								int const n_edges,
								int const *const edges,
								double const *const external_weight,
								/* original constrain to tighten */
								int const n2dominos,
								int const *const naset,
								int const *const nbset,
								int const *const ntset,
								int const nahandle,
								int const nbhandle,
								int **const aset,
								int **const bset,
								int **const tset,
								int const *const ahandle,
								int const *const bhandle,
								/* new generated constraint */
								int **const new_naset,
								int **const new_nbset,
								int **const new_ntset,
								int *const new_nahandle,
								int *const new_nbhandle,
								int ***const new_aset,
								int ***const new_bset,
								int ***const new_tset,
								int **const new_ahandle,
								int **const new_bhandle,
								double *const violation)
{
	/* --------------------------------------------------------------------- */
	/* local variables */
	EGmemPool_t *mem;
	int rval = 0;
	unsigned pos,
	  max_iter = n2dominos * n_nodes * 61,
	  in_AH = 0,
	  N1_inA,
	  N1_inB,
	  N2_inA,
	  N2_inB,
	  N1_id,
	  N2_id,
	  N1_inTc,
	  N2_inTc,
	  nadom = 0,
	  nbdom = 0,
	  in_BH = 0;
	double cost_tmp,
	  l_violation,
	  o_violation;
	register unsigned int i,
	  j;
	KPTNdata_t *nc_data = 0,
	 *n_data;
	KPTFullMove_t cur_move,
	  base_move;

	/* --------------------------------------------------------------------- */
	/* we test that the necesary input is not NULL */
	MESSAGE (KPT_VRB + 19, "Entering with n_nodes %d n_edges %d n2dominos %d",
					 n_nodes, n_edges, n2dominos);
	EXIT ((unsigned) n2dominos > KPT_MAX_DOM,
				"n2dominos (%d) exced max_n_dom (%u)"
				"change the value of KPT_MAX_NODE to an apropiate value", n2dominos,
				KPT_MAX_DOM);
	EXIT ((unsigned) n_nodes > KPT_MAX_NODE,
				"n_nodes (%d) exced max_n_nodes (%u)"
				"change the value of KPT_MAX_NODE to an apropiate value", n_nodes,
				KPT_MAX_NODE);
	TEST (!n_nodes, "graph has no nodes");
	TEST (!n_edges, "graph has no edges");
	TEST (!edges, "edges is NULL");
	TEST (!external_weight, "weight is NULL");
	TEST (!n2dominos, "inequality has no dominos");
	TEST (!naset, "n_aset is NULL");
	TEST (!nbset, "n_bset is NULL");
	TEST (!ntset, "n_bset is NULL");
	TEST (!nahandle, "inequality has handle");
	TEST (!nbhandle, "inequality has handle");
	TEST (!aset, "aset is NULL");
	TEST (!bset, "bset is NULL");
	TEST (!tset, "bset is NULL");
	TEST (!ahandle, "handle is NULL");
	TEST (!bhandle, "handle is NULL");
	TEST (!new_naset, "new_n_aset is NULL");
	TEST (!new_nbset, "new_n_bset is NULL");
	TEST (!new_ntset, "new_n_bset is NULL");
	TEST (!new_nahandle, "new_n_handle is NULL");
	TEST (!new_nbhandle, "new_n_handle is NULL");
	TEST (!new_aset, "new_aset is NULL");
	TEST (!new_bset, "new_bset is NULL");
	TEST (!new_tset, "new_bset is NULL");
	TEST (!new_ahandle, "new_handle is NULL");
	TEST (!new_bhandle, "new_handle is NULL");
	TEST (!violation, "violation is NULL");
#if LOG_MODE
	/* we save the graph if necesary */
	saveGraph ("graph_2tighten.x", n_nodes, n_edges, edges, external_weight);
#endif
	/* basic tests */
	ADVTESTL (KPT_DBG, (n_nodes < 2) || (n_edges < 1) || !(n2dominos) ||
						(nahandle < 1) || (nbhandle < 1) || (n_nodes - nahandle < 1) ||
						(n_nodes - nbhandle < 1), 1, "either n_nodes (%d), n_edges (%d),"
						" n2dominos (%d), n_ahandle (%d) or n_bhandle (%d) has wrong value",
						n_nodes, n_edges, n2dominos, nahandle, nbhandle);
	for (i = n2dominos; i--;)
	{
		ADVTESTL (KPT_EDBG, ((naset[i] < 1) && (nbset[i] < 1)) ||
							(naset[i] > ntset[i]) || (nbset[i] > ntset[i]) ||
							(ntset[i] > n_nodes), 1, "either naset[%d] (%d) or nbset[%d] "
							"(%d) or ntset[%d] (%d) has wrong value, n_nodes %d",
							i, naset[i], i, nbset[i], i, ntset[i], n_nodes);
		if (naset[i])
			nadom++;
		if (nbset[i])
			nbdom++;
	}
	ADVTESTL (KPT_EDBG, !(nadom & 1U) || !(nbdom & 1U), 1, "number of Adom %u, "
						"number of Bdom %u, they are not odd-odd", nadom, nbdom);

	/* --------------------------------------------------------------------- */
	/* basic set-up */
	weight = external_weight;
	memset (node_update, 0, sizeof (node_update));
	mem = EGnewMemPool (512, EGmemPoolNewSize, 4096);
	tree = EGnewBbtree (mem, EGdpTightNdataCompare);
	nodeData = EGsMalloc (KPTNdata_t, n_nodes);
	edgeData = EGsMalloc (KPTEdata_t, n_edges);
	all_nodes = EGsMalloc (EGuGraphNode_t *, n_nodes);
	all_edges = EGsMalloc (EGuGraphEdge_t *, n_edges);
	memset (all_nodes, 0, sizeof (EGuGraphNode_t *) * n_nodes);
	memset (nodeData, 0, sizeof (KPTNdata_t) * n_nodes);
	memset (edgeData, 0, sizeof (KPTEdata_t) * n_edges);
	memset (all_edges, 0, sizeof (EGuGraphEdge_t *) * n_nodes);
	memset (&cur_move, 0, sizeof (KPTFullMove_t));
	memset (&base_move, 0, sizeof (KPTFullMove_t));
	l_violation = (2 * n2dominos + 2 + nadom + nbdom);
	(*new_naset) = 0;
	(*new_nbset) = 0;
	(*new_ntset) = 0;
	(*new_nahandle) = 0;
	(*new_nbhandle) = 0;
	(*new_aset) = 0;
	(*new_bset) = 0;
	(*new_tset) = 0;
	(*new_ahandle) = 0;
	(*new_bhandle) = 0;

	/* first we build the graph with its internal data */
	memset (&graphData, 0, sizeof (KPTGdata_t));
	graphData.n_dominos = n2dominos;
	graphData.n_nodes = n_nodes;
	G = EGnewUGraph (mem);
	G->data = &graphData;

	/* now we initialize all the data structures */
	MESSAGE (KPT_VRB + 19, "Initializing");
	/* add all nodes */
	for (i = 0; i < (unsigned) n_nodes; i++)
	{
		nodeData[i].full_move.val = DBL_MAX;
		all_nodes[i] = EGuGraphAddNode (G, nodeData + i);
		rval = !(nodeData[i].tree_cn = EGbbtreeAdd (tree, nodeData + i));
		CHECKRVALG(rval,CLEANUP);
	}
	/* add all edges */
	for (i = 0; i < (unsigned) n_edges; i++)
	{
		all_edges[i] = EGuGraphAddEdge (G, edgeData + i,
																		all_nodes[edges[i << 1]],
																		all_nodes[edges[(i << 1) | 1]]);
	}
	/* compute handle stuff */
	for (i = nahandle; i--;)
	{
		nodeData[ahandle[i]].in_AH = 1U;
		graphData.n_AH[1]++;
	}
	graphData.n_AH[0] = n_nodes - graphData.n_AH[1];
	for (i = nbhandle; i--;)
	{
		nodeData[bhandle[i]].in_BH = 1U;
		graphData.n_BH[1]++;
	}
	graphData.n_BH[0] = n_nodes - graphData.n_BH[1];
	/* now we update information related to the dominoes */
	for (i = n2dominos; i--;)
	{
		if ((graphData.n_A[1][i] = naset[i]))
		{
			EGbitSet (graphData.A_partition, i);
			graphData.n_A[0][i] = ntset[i] - naset[i];
		}
		if ((graphData.n_B[1][i] = nbset[i]))
		{
			EGbitSet (graphData.B_partition, i);
			graphData.n_B[0][i] = ntset[i] - nbset[i];
		}
		if (EGbitTest (graphData.A_partition, i)
				&& EGbitTest (graphData.B_partition, i))
			EGbitSet (graphData.Kdom, i);
		graphData.n_Tc[i] = n_nodes - ntset[i];
		for (j = ntset[i]; j--;)
		{
			pos = tset[i][j];
			EGbitSet (nodeData[pos].It, i);
			nodeData[pos].n_in_T++;
		}
		for (j = naset[i]; j--;)
		{
			pos = aset[i][j];
			EGbitSet (nodeData[pos].Ia, i);
			nodeData[pos].n_in_A++;
			ADVTESTL (KPT_EDBG, !EGbitTest (nodeData[pos].It, i), 1, "a node %d"
								"is in A_%d but not in T", pos, i);
		}
		for (j = nbset[i]; j--;)
		{
			pos = bset[i][j];
			EGbitSet (nodeData[pos].Ib, i);
			nodeData[pos].n_in_B++;
			ADVTESTL (KPT_EDBG, !EGbitTest (nodeData[pos].It, i), 1, "a node %d"
								"is in A_%d but not in T", pos, i);
		}
		for (j = ntset[i]; j--;)
		{
			pos = tset[i][j];
			if (!EGbitTest (nodeData[pos].Ib, i) == !EGbitTest (nodeData[pos].Ia, i))
			{
				graphData.n_AdB[1][i]++;
				EGbitSet (nodeData[pos].Iadb, i);
			}
		}
		graphData.n_AdB[0][i] = ntset[i] - graphData.n_AdB[1][i];
		ADVTESTL (KPT_EDBG, (!graphData.n_AdB[1][i] || !graphData.n_AdB[0][i]) &&
							EGbitTest (graphData.Kdom, i), 1,
							"domino %u has no 3-partition", i);
	}
	/* now we update the information for all edges */
	for (i = n_edges; i--;)
	{
		N1_id = all_edges[i]->head->id;
		N2_id = all_edges[i]->tail->id;
		for (j = n2dominos; j--;)
		{
			N1_inA = EGbitTest (nodeData[N1_id].Ia, j) ? 1U : 0U;
			N1_inB = EGbitTest (nodeData[N1_id].Ib, j) ? 1U : 0U;
			N1_inTc = EGbitTest (nodeData[N1_id].It, j) ? 0U : 1U;
			N2_inA = EGbitTest (nodeData[N2_id].Ia, j) ? 1U : 0U;
			N2_inB = EGbitTest (nodeData[N2_id].Ib, j) ? 1U : 0U;
			N2_inTc = EGbitTest (nodeData[N2_id].It, j) ? 0U : 1U;
			if ((!N1_inTc && N2_inTc) || (N1_inTc && !N2_inTc))
				edgeData[i].num.n_dT++;
			if (N1_inTc == N2_inTc)
			{
				if (N1_inA != N2_inA)
					edgeData[i].num.n_AtoAc++;
				if (N1_inB != N2_inB)
					edgeData[i].num.n_BtoBc++;
			}
		}
		pos = 1U & ((nodeData[N1_id].in_AH ^ nodeData[N2_id].in_AH) ^
								edgeData[i].num.n_AtoAc);
		if (pos)
			edgeData[i].num.in_AF = 1U;
		pos = 1U & ((nodeData[N1_id].in_BH ^ nodeData[N2_id].in_BH) ^
								edgeData[i].num.n_BtoBc);
		if (pos)
			edgeData[i].num.in_BF = 1U;
	}
#if KPT_VRB <= DEBUG-1
	KPTDisplayEdges ();
#endif
	/* --------------------------------------------------------------------- */
	/* now we need to get the best move for each node */
	MESSAGE (KPT_VRB + 19, "Setting up best move for all nodes");
	/* note that we only need to update moves for nodes whose edges has an edge
	 * with coefficient of non-zero */
	for (i = n_edges; i--;)
	{
		/* if the edge has a non-zero coefficient, we update it's both ends if they
		 * haven't been update yet. */
		if (edgeData[i].used)
		{
			if (!EGbitTest (node_update, all_edges[i]->tail->id))
			{
				KPTResetMove (&cur_move);
				cur_move.val = DBL_MAX;
				base_move.val = 0;
				KPTSetBestMove (all_edges[i]->tail->id, &cur_move, &base_move, 0);
				KPTStoreFullMove (all_edges[i]->tail->id, &cur_move);
				EGbitSet (node_update, all_edges[i]->tail->id);
#if KPT_VRB+19<DEBUG
				DPdisplayMove (all_edges[i]->tail->id);
#endif
			}
			if (!EGbitTest (node_update, all_edges[i]->head->id))
			{
				KPTResetMove (&cur_move);
				cur_move.val = DBL_MAX;
				base_move.val = 0;
				KPTSetBestMove (all_edges[i]->head->id, &cur_move, &base_move, 0);
				KPTStoreFullMove (all_edges[i]->head->id, &cur_move);
				EGbitSet (node_update, all_edges[i]->head->id);
#if KPT_VRB+19<DEBUG
				DPdisplayMove (all_edges[i]->head->id);
#endif
			}
		}
	}															/* end checking all edges */
	/* reset the node update flags */
	memset (node_update, 0, sizeof (node_update));
	MESSAGE (KPT_VRB + 19, "Best move set for all nodes");

	/* --------------------------------------------------------------------- */
	/* now we compute the original violation */
	o_violation = 2 * n2dominos + 2 + nadom + nbdom;
	o_violation = -o_violation;
	KPTpriceConstraint (&o_violation);
	MESSAGE (KPT_VRB + 19, "Computing Original violation %lg", o_violation);
	l_violation = o_violation;

	/* --------------------------------------------------------------------- */
	/* now we check if the original constraint is correct */
#if KPT_EDBG < DEBUG
	{
		double t_violation;
		int *e_coeff = EGsMalloc (int, n_edges);
		rval = EG2pChecker (n_nodes, n_edges, edges, external_weight, 1,
												&n2dominos, &naset, &nbset, &ntset, &nahandle,
												&nbhandle, &aset, &bset, &tset, &ahandle, &bhandle,
												&t_violation, &e_coeff);
		EXIT (rval, "inequality os not wright");
		EXIT (fabs (t_violation - o_violation) > 1e-3,
					"wrong violation, computed %lf," " checked %lf difference %lf",
					o_violation, t_violation, fabs (o_violation - t_violation));
		EGfree (e_coeff);
	}
#endif

	/* --------------------------------------------------------------------- */
	/* now we enter our main loop, while there is a negative move, we keep going
	 * */
	while (((cost_tmp = (nc_data =
											 KPTNdata (EGbbtreeMin (tree)->this))->full_move.val)
					< -KPTMinImprovement) && max_iter--)
	{
		MESSAGE (KPT_VRB + 19, "Try to find best move (current best %8.6lf) "
						 "Iteration: %u", cost_tmp, max_iter);
		memcpy (&cur_move, &(nc_data->full_move), sizeof (KPTFullMove_t));
#if KPT_VRB+19<DEBUG
		i = cur_move.move[0].n_id;
		DPdisplayMove (i);
#endif
		/* now check if the movement is infeasible, if so, update the best move and
		 * move-on */
		if (!KPTisFullMoveFeasible (&cur_move))
		{
			i = cur_move.move[0].n_id;
#if KPT_VRB+0<DEBUG
			DPdisplayMove (i);
#endif
			MESSAGE (KPT_VRB + 1, "Move for node %u dom %u infeasible", i,
							 cur_move.move[0].dom);
			KPTResetMove (&cur_move);
			cur_move.val = DBL_MAX;
			base_move.val = 0;
			KPTSetBestMove (i, &cur_move, &base_move, 0);
			KPTStoreFullMove (i, &cur_move);
#if KPT_VRB+0<DEBUG
			DPdisplayMove (i);
#endif
			continue;
		}
		/* now, if the move is good (i.e. non-zero) we reset the flags */
		if (cost_tmp < KPTMinImprovement)
			KPTresetFlags ();
		/* now we perform the move */
		MESSAGE (KPT_VRB + 0, "Imp: %8.6lf Tot: %8.6lf node %u move %s depth %u",
						 cost_tmp, l_violation, cur_move.move[0].n_id,
						 move_name[cur_move.move[0].move_id], nc_data->full_move.depth);
		KPTmakeFullMove (&cur_move);
		l_violation += cost_tmp;
		/* this check is to be sure if the violation predicted is OK */
#if KPT_EDBG <= DEBUG
		EXITRVAL (KPTTestEdges ());
		cost_tmp = 2 * n2dominos + 2 + nadom + nbdom;
		cost_tmp = -cost_tmp;
		KPTpriceConstraint (&cost_tmp);
		EXIT ((fabs (cost_tmp - l_violation) > 1e-3) &&
					DPdisplayMove ((i =
													cur_move.move[0].n_id)),
					"Error computing violation"
					", predicted %8.6lf computed %8.6lf diference %8.6lf", l_violation,
					cost_tmp, l_violation - cost_tmp);
#endif
		/* now update best move for the node and all its neighbours */
		/* update moves */
		KPTupdateNeighMove (&cur_move);
	}
	/* --------------------------------------------------------------------- */
	cost_tmp = ((KPTNdata_t *) (EGbbtreeMin (tree)->this))->full_move.val;
	MESSAGE (KPT_VRB + 19, "No good moves left (current best %8.6lf)",
					 (cost_tmp));

	/* --------------------------------------------------------------------- */
	if (o_violation > l_violation + 1e-4)
		/* now, if we have a violated and better inequality, we save it and send it
		 * back to the calling function. */
	{
		MESSAGE (0, "Original: %8.6lf New: %8.6lf Diff: %lg", o_violation,
						 l_violation, fabs (o_violation - l_violation));
		/* look for memory */
		(*new_naset) = EGsMalloc (int,
															n2dominos);
		(*new_nbset) = EGsMalloc (int,
															n2dominos);
		(*new_ntset) = EGsMalloc (int,
															n2dominos);
		(*new_aset) = EGsMalloc (int *,
														 n2dominos);
		(*new_bset) = EGsMalloc (int *,
														 n2dominos);
		(*new_tset) = EGsMalloc (int *,
														 n2dominos);
		(*new_nahandle) = 0;
		(*new_nbhandle) = 0;
		in_AH = graphData.n_AH[1] < graphData.n_AH[0] ? 1 : 0;
		in_BH = graphData.n_BH[1] < graphData.n_BH[0] ? 1 : 0;
		TESTL (KPT_EDBG,
					 graphData.n_AH[0] + graphData.n_AH[1] != (unsigned) n_nodes,
					 "wrong A handle");
		TESTL (KPT_EDBG,
					 graphData.n_BH[0] + graphData.n_BH[1] != (unsigned) n_nodes,
					 "wrong B handle");
		TESTL (KPT_EDBG, !(graphData.n_AH[in_AH]), "handle is empty!");
		TESTL (KPT_EDBG, !(graphData.n_BH[in_BH]), "handle is empty!");
		(*new_ahandle) = EGsMalloc (int,
																graphData.n_AH[in_AH]);
		(*new_bhandle) = EGsMalloc (int,
																graphData.n_BH[in_BH]);
		for (i = n2dominos; i--;)
		{
			TESTL (KPT_EDBG,
						 (!(graphData.n_A[0][i]) && EGbitTest (graphData.A_partition, i)) ||
						 (!(graphData.n_B[0][i]) && EGbitTest (graphData.B_partition, i)) ||
						 (!(graphData.n_A[1][i]) && EGbitTest (graphData.A_partition, i)) ||
						 (!(graphData.n_B[1][i]) && EGbitTest (graphData.B_partition, i)) ||
						 !(graphData.n_Tc[i]), "domino(%u) with empty component A=(%u,%u)"
						 ", B=(%u,%u), T^c=%u", i, graphData.n_A[1][i], graphData.n_A[0][i],
						 graphData.n_B[1][i], graphData.n_B[0][i], graphData.n_Tc[i]);
			TESTL (KPT_EDBG, EGbitTest (graphData.Kdom, i)
						 && (!graphData.n_AdB[0][i]
								 || !graphData.n_AdB[1][i]),
						 "the 2-domino_%d is not a 3-partition", i);
			if (graphData.n_A[1][i])
				(*new_aset)[i] = EGsMalloc (int,
																		graphData.n_A[1][i]);
			else
				(*new_aset)[i] = 0;
			if (graphData.n_B[1][i])
				(*new_bset)[i] = EGsMalloc (int,
																		graphData.n_B[1][i]);
			else
				(*new_bset)[i] = 0;
			(*new_tset)[i] = EGsMalloc (int,
																	n_nodes - graphData.n_Tc[i]);
			((*new_naset)[i]) = 0;
			((*new_nbset)[i]) = 0;
			((*new_ntset)[i]) = 0;
		}

		/* we loop through all nodes and put them where they should go */
		for (i = n_nodes; i--;)
		{
			n_data = nodeData + i;
			if ((n_data->in_AH & 1U) == in_AH)
				(*new_ahandle)[(*new_nahandle)++] = i;
			if ((n_data->in_BH & 1U) == in_BH)
				(*new_bhandle)[(*new_nbhandle)++] = i;
			/* loop through all dominos to check to which one the node belong */
			for (j = n2dominos; j--;)
			{
				if (EGbitTest (n_data->It, j))
					(*new_tset)[j][((*new_ntset)[j])++] = i;
				if (EGbitTest (n_data->Ia, j))
					(*new_aset)[j][((*new_naset)[j])++] = i;
				if (EGbitTest (n_data->Ib, j))
					(*new_bset)[j][((*new_nbset)[j])++] = i;
			}													/* end loop through dominos for each node */
		}														/* end loop every node */

		TESTL (KPT_EDBG, (in_AH ? ((*new_nahandle) != (signed) graphData.n_AH[1]) :
											((*new_nahandle) != (signed) graphData.n_AH[0])),
					 "Ahandle size " "doesn't agree with what we have found");
		TESTL (KPT_EDBG,
					 (in_BH ? ((*new_nbhandle) != (signed) graphData.n_BH[1])
						: ((*new_nbhandle) != (signed) graphData.n_BH[0])),
					 "Bhandle size " "doesn't agree with what we have found");
		for (i = n2dominos; i--;)
		{
			TESTL (KPT_EDBG, (*new_naset)[i] != (signed) graphData.n_A[1][i],
						 "size of A_[%u] doesn't match (%d,%u)", i, (*new_naset)[i],
						 graphData.n_A[1][i]);
			TESTL (KPT_EDBG, (*new_nbset)[i] != (signed) graphData.n_B[1][i],
						 "size of B_[%u] doesn't match (%d,%u)", i, (*new_nbset)[i],
						 graphData.n_B[1][i]);
		}
		/* --------------------------------------------------------------------- */
		/* now we check that the inequality os wroight, we use the 2pchecker . */
#if KPT_EDBG < DEBUG
		{
			double t_violation;
			int *e_coeff = EGsMalloc (int, n_edges);
			rval = EG2pChecker (n_nodes, n_edges, edges, external_weight, 1,
													&n2dominos, new_naset, new_nbset, new_ntset,
													new_nahandle, new_nbhandle, new_aset, new_bset,
													new_tset, new_ahandle, new_bhandle, &t_violation,
													&e_coeff);
			EXIT (rval, "Inequality is wrong");
			EXIT (fabs (t_violation - l_violation) > 1e-3,
						"violation is wrongly computed"
						", given %lf, computed %lf, diff %lf", l_violation, t_violation,
						fabs (l_violation - t_violation));
			EGfree (e_coeff);
		}
#endif
	}															/* end saving newly found constraint */
	/* ending */
	CLEANUP:
	MESSAGE (KPT_VRB + 19, "Clearing graph");
	EGuGraphClearMP (G, nullFreeMP, nullFreeMP, nullFreeMP, mem, mem, mem);
	MESSAGE (KPT_VRB + 19, "Freing graph");
	EGfreeUGraph (G);
	MESSAGE (KPT_VRB + 19, "Freing all_nodes");
	EGfree (all_nodes);
	MESSAGE (KPT_VRB + 19, "Freing all_edges");
	EGfree (all_edges);
	MESSAGE (KPT_VRB + 19, "Freing tree");
	EGfreeBbtree (tree);
	MESSAGE (KPT_VRB + 19, "Freing nodeData");
	EGfree (nodeData);
	MESSAGE (KPT_VRB + 19, "Freing edgeData");
	EGfree (edgeData);
	MESSAGE (KPT_VRB + 19, "Freing mempool");
	EGfreeMemPool (mem);
	*violation = l_violation;
	MESSAGE (KPT_VRB + 19, "Ending with violation %8.6lf", *violation);
	return rval;
}
#else
int KPtighten (									/* graph G* data */
								int const n_nodes,
								int const n_edges,
								int const *const edges,
								double const *const weight,
								/* original constrain to tighten */
								int const n2dominos,
								int const *const naset,
								int const *const nbset,
								int const *const ntset,
								int const nahandle,
								int const nbhandle,
								int **const aset,
								int **const bset,
								int **const tset,
								int const *const ahandle,
								int const *const bhandle,
								/* new generated constraint */
								int **const new_naset,
								int **const new_nbset,
								int **const new_ntset,
								int *const new_nahandle,
								int *const new_nbhandle,
								int ***const new_aset,
								int ***const new_bset,
								int ***const new_tset,
								int **const new_ahandle,
								int **const new_bhandle,
								double *const violation)
{
	(*new_naset) = 0;
	(*new_nbset) = 0;
	(*new_ntset) = 0;
	(*new_nahandle) = 0;
	(*new_nbhandle) = 0;
	(*new_aset) = 0;
	(*new_bset) = 0;
	(*new_tset) = 0;
	(*new_ahandle) = 0;
	(*new_bhandle) = 0;
	(*violation) = 0;
	return 0;
}
#endif
