#include <stdio.h>

#include "dp_config.h"

#include "eg_ddomino.h"
#include "eg_nettype.h"
#include "eg_mempool.h"
#include "eg_list.h"
#include "eg_dgraph.h"
#include "eg_util.h"

#include "bc_util.h"

#define DOM_TASK_WAIT_SECONDS  (5)
#define DOM_GRUNT_VERBOSE 5
#define DOM_GRUNT_BASIC 0

#ifndef CCtsp_DOMINO_PORT
#define CCtsp_DOMINO_PORT ((unsigned short) 24869)
#endif

#ifndef CCtsp_DOMINO_WORK
#define CCtsp_DOMINO_WORK        'A'
#endif
#ifndef CCtsp_DOMINO_GRAPH
#define CCtsp_DOMINO_GRAPH       'G'
#endif
#ifndef CCtsp_DOMINO_NO
#define CCtsp_DOMINO_NO          'N'
#endif
#ifndef CCtsp_DOMINO_RECEIVE
#define CCtsp_DOMINO_RECEIVE     'R'
#endif
#ifndef CCtsp_DOMINO_SEND
#define CCtsp_DOMINO_SEND        'S'
#endif
#ifndef CCtsp_DOMINO_WAIT
#define CCtsp_DOMINO_WAIT        'W'
#endif
#ifndef CCtsp_DOMINO_YES
#define CCtsp_DOMINO_YES         'Y'
#endif
#ifndef CCtsp_DOMINO_EXIT
#define CCtsp_DOMINO_EXIT        'X'
#endif

int main (int ac,
					char **av)
{
	char *bosshost = (char *) NULL;
	double rtime = 0.0;
	int id = -1;
	int rval = 0;
	CC_SFILE *s = (CC_SFILE *) NULL;
	double szeit;
	char task,
	  need,
	  ready;

	int current_g_id = -1,
	  domino_k = -1;

	EGdGraph_t *G = 0;
	EGdGraphNode_t *source,
	**node_array = 0;
	EGlist_t *D = 0;
	EGlistNode_t *l_it;
	EGddomino_t *ddom;
	EGmemPool_t *mem;
	double ddp_heuristic_maxtime;

	mem = EGnewMemPool (8192, EGmemPoolNewSize, EGmemPoolNewSize (1));
	D = EGnewList (mem);

	if (ac != 2)
	{
		fprintf (stderr, "Usage: %s boss\n", av[0]);
		rval = 1;
		goto CLEANUP;
	}

	CCutil_printlabel ();

	bosshost = av[1];

	while (1)
	{

		/* open port                                                          */

		MESSAGE (DOM_GRUNT_VERBOSE, "  Domgrunt: Opening Port (%d).",
						 CCtsp_DOMINO_PORT);
		s = CCutil_snet_open (bosshost, CCtsp_DOMINO_PORT);
		MESSAGE (DOM_GRUNT_VERBOSE, "open fd %d", s->desc);
		if (!s)
		{
			fprintf (stderr, "CCutil_snet_open failed\n");
			rval = 1;
			goto CLEANUP;
		}

		/* establish communication with the boss.                             */

		MESSAGE (DOM_GRUNT_VERBOSE, "  Domgrunt: Communicating with boss.");
		rval = CCutil_swrite_char (s, CCtsp_DOMINO_RECEIVE);
		CCcheck_rval (rval, "CCutil_swrite_char failed (RECEIVE)");

		/* Is the boss ready to work with us?                                 */

		MESSAGE (DOM_GRUNT_VERBOSE, "  Domgrunt: Awaiting instructions.");
		rval = CCutil_sread_char (s, &ready);
		CCcheck_rval (rval, "CCutil_sread_char failed (ready)");
		MESSAGE (DOM_GRUNT_VERBOSE, "  Domgrunt: Instruction: %c.", ready);

		/* if the answer is yes.                                              */

		if (ready == CCtsp_DOMINO_YES)
		{

			/* Tell the boss which graph id we have in memory                 */

			MESSAGE (DOM_GRUNT_VERBOSE, "  Domgrunt: Current graph id = %d.",
							 current_g_id);

			rval = CCutil_swrite_int (s, current_g_id);
			CCcheck_rval (rval, "CCutil_swrite_int failed (gid");


			/* Tell it which problem we are working on (-1 if we are have not
			 * been working on anything, or if we are just starting.          */

			MESSAGE (DOM_GRUNT_VERBOSE, "  Domgrunt: Current id = %d.", id);
			rval = CCutil_swrite_int (s, id);
			CCcheck_rval (rval, "CCutil_swrite_int failed (id)");


			/* If we have been working on a problem, we offer to send the 
			 * dominoes we have found to the boss.                            */

			if (id != -1)
			{

				rval = CCutil_swrite_double (s, rtime);
				CCcheck_rval (rval, "CCutil_swrite_double failed (rtime)");

				/* Does the boss still need the dominoes?                     */

				rval = CCutil_sread_char (s, &need);
				CCcheck_rval (rval, "CCutil_sread_char failed (need)");

				/* If so, send them to him.                                   */

				if (need == CCtsp_DOMINO_YES)
				{

					fprintf (stderr, "\r  Domgrunt: Sending dominoes for id %5d.", id);

					rval = CCutil_swrite_int (s, (int) (D->size));
					CCcheck_rval (rval, "send ndominoes failed");

					for (l_it = D->begin; l_it; l_it = l_it->next)
					{
						ddom = (EGddomino_t *) (l_it->this);
						rval = EGsendRealDomino (s, ddom);
						CCcheck_rval (rval, "send_dominos failed");
					}

				}

				/* Free up the memory used to hold the dominoes               */


				MESSAGE (DOM_GRUNT_VERBOSE,
								 "  Domgrunt: Clearing current domino list.");

				EGlistClearMP (D, EGfreeDdomino, mem);
				id = -1;

				rtime = 0;

				MESSAGE (DOM_GRUNT_VERBOSE, "  Domgrunt: Done.");
			}

			/* What does the boss want us to do?                              */

			MESSAGE (DOM_GRUNT_VERBOSE,
							 "  Domgrunt: Waiting for instructions from boss.");

			rval = CCutil_sread_char (s, &task);
			CCcheck_rval (rval, "CCutil_sread_char failed (task)");

			MESSAGE (DOM_GRUNT_VERBOSE, "  Domgrunt: Instructions: %c.", task);

			switch (task)
			{

				/* Wait.                                                          */

			case CCtsp_DOMINO_WAIT:
				break;

				/* Receive a graph.                                               */

			case CCtsp_DOMINO_GRAPH:

				MESSAGE (DOM_GRUNT_VERBOSE, "  Domgrunt: Receiving new graph.");


				if (G)
				{
					EGmemPoolFree (node_array,
												 sizeof (EGdGraphNode_t *) * (G->nodes->size), mem);
					EGdGraphClearMP (G, EGfreeMengerEdgeDataMP, EGfreeMengerNodeDataMP, 0,
													 mem, mem, 0);
					G = 0;
				}

				MESSAGE (DOM_GRUNT_VERBOSE, "  Domgrunt: Receiving graph ID.");

				rval = CCutil_sread_int (s, &current_g_id);
				CCcheck_rval (rval, "receive_g_id failed");

				MESSAGE (DOM_GRUNT_BASIC, "  Domgrunt: graph ID = %d.", current_g_id);


				rval = EGreceiveRealGraph (s, &G, mem);
				CCcheck_rval (rval, "receive_graph failed");

				rval = CCutil_sread_int (s, &domino_k);
				CCcheck_rval (rval, "receive_domino_k failed");

				rval = CCutil_sread_double (s, &ddp_heuristic_maxtime);
				CCcheck_rval (rval, "receive_ddp_heuristic_maxtime failed");
				DDP_HEURISTIC_MAXTIME = ddp_heuristic_maxtime;

				MESSAGE (0, "  Domgrunt: nnodes= %u. nedges = %u. "
								 "domino_k = %d.", G->nodes->size, G->nedges, domino_k);

				node_array =
					(EGdGraphNode_t **) EGmemPoolMalloc (mem,
																							 sizeof (EGdGraphNode_t *) *
																							 G->nodes->size);

				for (l_it = G->nodes->begin; l_it; l_it = l_it->next)
				{
					source = (EGdGraphNode_t *) (l_it->this);
					node_array[source->id] = source;
				}
				break;



				/* Start solving for dominoes on the current graph on a source.   */

			case CCtsp_DOMINO_WORK:
				rval = CCutil_sread_int (s, &id);
				CCcheck_rval (rval, "CCutil_sread_int failed (id)");
				break;

				/* Quit.                                                          */

			case CCtsp_DOMINO_EXIT:
				MESSAGE (DOM_GRUNT_VERBOSE, "Shutting down the domino grunt");

				goto CLEANUP;
			}

			if (s)
				CCutil_sclose (s);
			s = (CC_SFILE *) NULL;

			/* Now we work: We call EGddominoProcessS                         */

			if (id != -1)
			{

				MESSAGE (DOM_GRUNT_VERBOSE, "PROCESSING node %d, graph %d", id,
								 current_g_id);

				szeit = CCutil_zeit ();

				source = node_array[id];

				rval = EGddominoComputeS (mem, source, G, D, domino_k, 1.0);
				CHECKRVAL (rval);

				rtime = CCutil_zeit () - szeit;
			}

			/* If the boss does not want to work with us ...                      */

		}
		else
		{
			EGlistClearMP (D, EGfreeDdomino, mem);
			id = -1;
			if (s)
				CCutil_sclose (s);
			s = 0;
			sleep (DOM_TASK_WAIT_SECONDS);
		}
	}

CLEANUP:

	if (s != (CC_SFILE *) NULL)
	{
		CCutil_sclose (s);
	}

	EGmemPoolFree (node_array, sizeof (EGdGraphNode_t *) * G->nodes->size, mem);
	EGdGraphClearMP (G, EGfreeMengerEdgeDataMP, EGfreeMengerNodeDataMP, 0, mem,
									 mem, 0);
	EGfreeDGraph (G);
	EGfreeList (D);
	EGfreeMemPool (mem);

	return rval;
}
