/* EGlib "Efficient General Library" provides some basic structures and
 * algorithms commons in many optimization algorithms.
 *
 * Copyright (C) 2005 Daniel Espinoza and Marcos Goycoolea.
 * 
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but 
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY 
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public 
 * License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA 
 * */
/* ========================================================================= */
/** Implementation of basic uGraph support, the idea is that this structure
 * should be flexible enough to support fast uGraph algorithm.
 *
 * 2003-06-13 - First implementation (Marcos).
 * 2004-02-04 - Added support for importing graphs from Cook format (Marcos)
 *
 * */
/* ========================================================================= */

#ifndef _EGuGraph
#define _EGuGraph
#include <stdio.h>
#include <limits.h>
#include <stddef.h>
#include "eg_config.h"
#include "eg_macros.h"
#include "eg_io.h"
#include "eg_list.h"
#include "eg_listint.h"
#include "eg_mempool.h"
#ifndef __EG_UGRAPH_CHECK__
#define __EG_UGRAPH_CHECK__ 0
#endif
#define UG_LEVEL 9
/* ========================================================================= */
/* struct to hold nodes */
/* ========================================================================= */
typedef struct
{
	EGlist_t *edges;
	EGlistNode_t *cn;
	void *data;
	unsigned int id;
}
EGuGraphNode_t;

/* ========================================================================= */
/* struct to hold edges, note that an edge may be a loop, in such a case
 * head=tail and tail_cn=head_cn */
/* ========================================================================= */
typedef struct
{
	EGuGraphNode_t *head;
	EGuGraphNode_t *tail;
	EGlistNode_t *head_cn;
	EGlistNode_t *tail_cn;
	void *data;
	unsigned int id;
}
EGuGraphEdge_t;

/* ========================================================================= */
/* struct to hold uGraphs */
/* ========================================================================= */
typedef struct
{
	EGlist_t *nodes;
	void *data;
	EGmemPool_t *mem;
	unsigned int id;
	unsigned int nedges;
	unsigned int node_id_max;
	unsigned int edge_id_max;
}
EGuGraph_t;

EGuGraphNode_t *EGuGraphOtherEnd (EGuGraphEdge_t * e,
																	EGuGraphNode_t * n);
EGlistNode_t *EGuGraphOtherEit (EGuGraphEdge_t * e,
																EGuGraphNode_t * n);
int EGuGraphSetOtherEit (EGuGraphEdge_t * e,
												 EGuGraphNode_t * n,
												 EGlistNode_t * other_eit);

/* attach/unattach does not change id's, but it does mantain the number of nodes
 * and edges. */
int EGuGraphAttachNode (EGuGraph_t * G,
												EGuGraphNode_t * n);
int EGuGraphAttachEdge (EGuGraph_t * G,
												EGuGraphEdge_t * e);
int EGuGraphUnattachNode (EGuGraph_t * G,
													EGuGraphNode_t * n);
int EGuGraphUnattachEdge (EGuGraph_t * G,
													EGuGraphEdge_t * e);

EGuGraph_t *EGnewUGraph (EGmemPool_t * mem);
EGuGraphNode_t *EGnewUGraphNode (EGmemPool_t * mem);
EGuGraphEdge_t *EGnewUGraphEdge (EGmemPool_t * mem);

void EGfreeUGraph (void *v);
void EGfreeUGraphNode (void *v);
void EGfreeUGraphEdge (void *v,
											 EGmemPool_t * mem);

/* clear functions leave the structure 'as new' */
void EGuGraphClear (EGuGraph_t * G,
										EGfree_f userFreeEdgeData,
										EGfree_f userFreeNodeData,
										EGfree_f userFreeUGraphData);

void EGuGraphClearMP (EGuGraph_t * G,
											EGfreeMP_f userFreeEdgeDataMP,
											EGfreeMP_f userFreeNodeDataMP,
											EGfreeMP_f userFreeUGraphDataMP,
											EGmemPool_t * edge_data_mem,
											EGmemPool_t * node_data_mem,
											EGmemPool_t * uGraph_data_mem);

/* erase functions clear all memory for the structure and it's sub-structure, it
 * has the ability to call 'frees' for the internal 'data' pointers if wanted */
void EGuGraphEraseEdge (EGuGraph_t * G,
												EGuGraphEdge_t * e,
												EGfree_f userFreeEdgeData);

void EGuGraphEraseEdgeMP (EGuGraph_t * G,
													EGuGraphEdge_t * e,
													EGfreeMP_f userFreeEdgeDataMP,
													EGmemPool_t * edge_data_mem);

void EGuGraphEraseNode (EGuGraph_t * G,
												EGuGraphNode_t * v,
												EGfree_f userFreeEdgeData,
												EGfree_f userFreeNodeData);

void EGuGraphEraseNodeMP (EGuGraph_t * G,
													EGuGraphNode_t * v,
													EGfreeMP_f userFreeEdgeDataMP,
													EGfreeMP_f userFreeNodeDataMP,
													EGmemPool_t * node_data_mem,
													EGmemPool_t * edge_data_mem);

/* this create a new structure and update internal data values */
extern inline EGuGraphNode_t *EGuGraphAddNode (EGuGraph_t * G,
																							 void *ndata)
{

	/* local variable */
	EGuGraphNode_t *v;

	/* BASIC SET-UP */
	v = EGnewUGraphNode (G->mem);
	EGlistPushBack (G->nodes, v);
	v->cn = G->nodes->end;
	v->data = ndata;
	v->id = G->node_id_max++;

	/* ending */
	return v;
}

extern inline EGuGraphEdge_t *EGuGraphAddEdge (EGuGraph_t * G,
																							 void *edata,
																							 EGuGraphNode_t * head,
																							 EGuGraphNode_t * tail)
{

	/* local variables */
	EGuGraphEdge_t *e;

	/* basic set-up */
	e = EGnewUGraphEdge (G->mem);
	e->head = head;
	e->tail = tail;
	EGlistPushBack (head->edges, e);
	e->head_cn = head->edges->end;
	if (head != tail)
		EGlistPushBack (tail->edges, e);
	e->tail_cn = tail->edges->end;
	e->data = edata;
	e->id = G->edge_id_max++;
	G->nedges++;

	/* ending */
	return e;
}

/* this functions compress the Id's in use */
int EGuGraphResetNodeId (EGuGraph_t * G);
int EGuGraphResetEdgeId (EGuGraph_t * G);

/* print in some format the graph and its internal data */
int EGuGraphDisplay (EGuGraph_t * G,
										 EGdisplay_f userDisplayUGraphData,
										 EGdisplay_f userDisplayNodeData,
										 EGdisplay_f userDisplayEdgeData,
										 FILE * file);


/* this function resets the node ids so they are numbered from 0 .. nnodes -1 */
int EGuGraphRenumberNodeId (EGuGraph_t * G);

/* ========================================================================= */
/* exports graph from Cook format. If no gdata, edata, or ndata, use 0. */
/* assumes that the number of nodes is G->node_id_max!! */
/* assumes that edges initialized appropriately */
/* assumes if the data != 0, then initialized as above */
int EGuGraphExport (EGuGraph_t * G,
										int *edges,
										void **gdata,
										void **edata,
										void **ndata);

/* import graph from Cook format. If no gdata, edata, or ndata, use 0.  */
EGuGraph_t *EGuGraphImport (int nnodes,
														int nedges,
														int *edges,
														void *gdata,
														void **edata,
														void **ndata,
														EGmemPool_t * mem);

int EGuGraphBreakUpComponents (EGuGraph_t * G,
															 EGlist_t * G_list,
															 EGmemPool_t * mem);
int EGuGraphIsConnected (EGuGraph_t * G,
												 EGmemPool_t * mem);

/* ========================================================================= */
/* get graph components */
/* assumes that nodes is an array of at least G->node_id_max EGuGraphNode_t* */
/* assumes that comps is an array of at least G->node_id_max ints */
int EGuGraphComponents (EGuGraph_t * G,
												int *num_components,
												EGuGraphNode_t ** nodes,
												int *comps,
												EGmemPool_t * mem);

#endif
