/* EGlib "Efficient General Library" provides some basic structures and
 * algorithms commons in many optimization algorithms.
 *
 * Copyright (C) 2005 Daniel Espinoza and Marcos Goycoolea.
 * 
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but 
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY 
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public 
 * License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA 
 * */
/* ========================================================================= */
/** @defgroup EGtimer EGtimer 
 *
 * Here we implement types and functions for timer functions
 *
 * @version 1.0.1 
 * @par History:
 * - 2005-05-31
 * 						- Eliminate the definition of #EGwallClockTimer_t and replace it by
 * 						a macro definition that replace it by #EGtimer_t.
 * - 2004-01-20
 * 						- Add a 'wall clock' timer (Renan-Marcos) type and functions
 * - 2003-05-08
 * 						- First Implementation
 * @note Up to now, this code will only work on linux machines, and maybe on
 * unix/posix systems.
 * */
/** @file 
 * @ingroup EGtimer */
/** @addtogroup EGtimer */
/** @{ */
/* ========================================================================= */

#ifndef __EG_TIMER_H__
#define __EG_TIMER_H__

#include <stdlib.h>
#include <stdio.h>
#include <sys/resource.h>
#include <sys/time.h>
#include <unistd.h>
#include <time.h>
#include <errno.h>
#include "eg_macros.h"

/* ========================================================================= */
/** @brief this structure holds a timer structure */
typedef struct
{
	double time;	/**< hold the accumulated time */
	double stime;	/**< hols the last time when we start counting, this is only 
										 for internal purposes, the user should only use the 
										 field 'time' */
}
EGtimer_t;

/* ========================================================================= */
/** @brief This is done for backward compability, we used to define
 * EGwallClockTimer_t just as the normal timer, so in reality we don't need
 * another type, but keep the name so that older code depending on this still
 * compiles. */
#define EGwallClockTimer_t EGtimer_t

/* ========================================================================= */
/** @brief Get system time.
 *
 * This function is for internal purposes only and should not be called from the
 * user space, it ask the (user) time from the system. 
 * @return the time (in seconds) stored as a double. */
#define __EGzeit() ({\
	struct rusage __EGzeit_ru;\
	int __EGzeit_st = getrusage(RUSAGE_SELF,&__EGzeit_ru);\
	EXIT(__EGzeit_st,"getrusage failed with code error %d (%s)", errno, \
	strerror(errno));\
	(((double)__EGzeit_ru.ru_utime.tv_sec) + \
	((double)__EGzeit_ru.ru_utime.tv_usec)/1000000);})

/* ========================================================================= */
/** @brief Set a new starting time for the timer.
 * @param timer pointer to a EGtimer_t structure.
 * @return starting time (in seconds), and the type is a double. */
#define EGtimerStart(timer) ({(timer)->stime = __EGzeit();})

/* ========================================================================= */
/** @brief Stop a 'running' timer and accumulate the run time.
 * @return the time elapsed since the last 'start' call (in seconds). */
#define EGtimerStop(timer) ({(timer)->time += __EGzeit() - (timer)->stime;})

/* ========================================================================= */
/** @brief this function reset the accumulated time to zero */
#define EGtimerReset(timer) ({(timer)->time = 0;})

/* ========================================================================= */
/** @brief Set the starting time the current (wall) time.
 * @return the current wall time. */
#define EGwallClockTimerStart(timer) ({(timer)->stime = time(0);})

/* ========================================================================= */
/** @brief Stop a 'running' timer and accumulate the (wall) runing time.
 * @return the wall time elapsed since the last initialization. */
#define EGwallClockTimerStop(timer) ({\
	(timer)->time += difftime(time(0),(timer)->stime);})

/* ========================================================================= */
/** @brief Reset the accumulated time to zero */
#define EGwallClockTimerReset(timer) EGtimerReset(timer)

/* ========================================================================= */
/** @}
 * end of eg_timer.h */
#endif
