/* EGlib "Efficient General Library" provides some basic structures and
 * algorithms commons in many optimization algorithms.
 *
 * Copyright (C) 2005 Daniel Espinoza and Marcos Goycoolea.
 * 
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but 
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY 
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public 
 * License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA 
 * */
#include "eg_spanning.h"

/* ========================================================================= */
/* internal name for the cost */
typedef EGspanningTreeCost_t EGcost_t;
#define EGgetParamPtr(pointer,offset,type) ((type*)(((char*)(pointer))+offset))
#define EGgetCost(element) *(EGgetParamPtr(((EGuGraphEdge_t*)(element->this))->data,costOS,EGcost_t))

/* ========================================================================= */
EGuGraph_t *EGnewSpanningTreeUGraph (int nnodes,
																		 int nedges,
																		 int *edges,
																		 EGspanningTreeCost_t * weight,
																		 EGmemPool_t * mem)
{
	/* local variables */
	EGuGraph_t *G;
	EGuGraphNode_t **n;
	EGuGraphEdge_t *e;
	register int i;

	/* create the uGraph */
	G = EGnewUGraph (mem);
	G->id = 0;
	void *ptr;

	/* populate it with the nodes */
	n =
		(EGuGraphNode_t **) EGmemPoolMalloc (mem,
																				 sizeof (EGuGraphNode_t *) * nnodes);
	for (i = 0; i < nnodes; i++)
	{
		ptr = EGmemPoolMalloc (mem, sizeof (void *));
		n[i] = EGuGraphAddNode (G, ptr);
	}

	/* populate it with the edges */
	for (i = 0; i < nedges; i++)
	{
		e =
			EGuGraphAddEdge (G, weight + i, n[edges[i << 1]],
											 n[edges[(i << 1) + 1]]);
		MESSAGE (9, "Edge %u:(%u,%u)[%lf]\n", e->id, e->head->id, e->tail->id,
						 weight[i]);
	}

	/* ending */
	EGmemPoolFree (n, sizeof (EGuGraphNode_t *) * nnodes, mem);
	return G;
}

void EGfreeSTNodeData (void *data,
											 EGmemPool_t * mem)
{
	EGmemPoolFree (data, sizeof (void *), mem);
}

/* ========================================================================= */
int EGspanningTree (EGuGraph_t * G,
										size_t costOS,
										size_t voidOS,
										int sense,
										EGuGraphEdge_t ** treeEdges,
										EGspanningTreeCost_t * value,
										EGmemPool_t * mem)
{

	/* local variables */
	void **intArray;;
	int status;

	/* basic set-up */
	intArray = (void **) EGmemPoolMalloc (mem, sizeof (void *) * G->nedges);

	/* call the advanced function */
	status = EGspanningTreeADV (G, costOS, voidOS, sense, treeEdges, intArray,
															value, mem);

	/* clear internal memory */
	EGmemPoolFree (intArray, sizeof (void *) * G->nedges, mem);
	CHECKRVAL (status);

	return status;
}

#define SWAP(a,b,tmp) {(tmp)=(a);(a)=(b);(b)=(tmp);}
static void EGspanningTreeQsort (void **intArray,
																 size_t costOS,
																 unsigned int l,
																 unsigned int u)
{

	/* local variable */
	void *tmp;
	register unsigned int i,
	  j;
	EGspanningTreeCost_t val;

	/* ending condition */
	if (l >= u)
		return;

	/* pick the pivot and put the mid value in the middle to avoid counting
	 * problems */
	SWAP (intArray[l], intArray[(l + u) / 2], tmp);
	i = l;
	j = u + 1;
	val =
		EGosGetData (((EGuGraphEdge_t *) intArray[l])->data, costOS,
								 EGspanningTreeCost_t);

	/* sort all values < pivot to the left and values > pivot to the wright */
	while (1)
	{
		do
			i++;
		while (i <= u
					 && EGosGetData (((EGuGraphEdge_t *) intArray[i])->data, costOS,
													 EGspanningTreeCost_t) < val);
		do
			j--;
		while (EGosGetData
					 (((EGuGraphEdge_t *) intArray[j])->data, costOS,
						EGspanningTreeCost_t) > val);
		if (j < i)
			break;
		SWAP (intArray[i], intArray[j], tmp);
	}

	/* reposition the middle value */
	SWAP (intArray[l], intArray[j], tmp);
	/* and call recursivelly the function */
	EGspanningTreeQsort (intArray, costOS, l, j - 1);
	EGspanningTreeQsort (intArray, costOS, i, u);

	/* ending */
	return;
}

/* ========================================================================= */
/* Main Function */
int EGspanningTreeADV (EGuGraph_t * G,
											 size_t costOS,
											 size_t voidOS,
											 int sense,
											 EGuGraphEdge_t ** treeEdges,
											 void **intArray,
											 EGspanningTreeCost_t * value,
											 EGmemPool_t * mem)
{
	/* local variables */
	register unsigned int i;
	int pos = 0;
	EGlistNode_t *nIt1,
	 *nIt2;
	EGuGraphEdge_t *e;

	/* we put in the intArtray all edges, and make the sets */
	*value = 0;
	i = 0;
	for (nIt1 = G->nodes->begin; nIt1; nIt1 = nIt1->next)
	{
		EGosSetData (((EGuGraphNode_t *) nIt1->this)->data, voidOS,
								 EGequiSetElem_t *, EGnewEquiSet (mem, nIt1->this));
		for (nIt2 = ((EGuGraphNode_t *) nIt1->this)->edges->begin; nIt2;
				 nIt2 = nIt2->next)
		{
			intArray[((EGuGraphEdge_t *) nIt2->this)->id] = nIt2->this;
		}
	}
	i = G->nedges;

	/* now we sort the array */
	EGspanningTreeQsort (intArray, costOS, 0, i - 1);

	/* now we create the tree */
	/* if we want max we do this */
	if (sense > 0)
	{
		for (i = G->nedges, pos = G->nodes->size - 1; i-- && pos;)
		{
			e = (EGuGraphEdge_t *) (intArray[i]);
			if (EGequiSetFind
					(EGosGetData (e->head->data, voidOS, EGequiSetElem_t *)) !=
					EGequiSetFind (EGosGetData
												 (e->tail->data, voidOS, EGequiSetElem_t *)))
			{
				EGequiSetLink (EGequiSetFind (EGosGetData (e->head->data, voidOS,
																									 EGequiSetElem_t *)),
											 EGequiSetFind (EGosGetData
																			(e->tail->data, voidOS,
																			 EGequiSetElem_t *)));
				treeEdges[pos--] = e;
				*value += EGosGetData (e->data, costOS, EGspanningTreeCost_t);
			}													/* end if */
		}														/* end for */
	}
	else
	{
		for (i = 0, pos = G->nodes->size - 1; i < G->nedges && pos; i++)
		{
			e = (EGuGraphEdge_t *) (intArray[i]);
			if (EGequiSetFind
					(EGosGetData (e->head->data, voidOS, EGequiSetElem_t *)) !=
					EGequiSetFind (EGosGetData
												 (e->tail->data, voidOS, EGequiSetElem_t *)))
			{
				EGequiSetLink (EGequiSetFind (EGosGetData (e->head->data, voidOS,
																									 EGequiSetElem_t *)),
											 EGequiSetFind (EGosGetData
																			(e->tail->data, voidOS,
																			 EGequiSetElem_t *)));
				treeEdges[pos--] = e;
				*value += EGosGetData (e->data, costOS, EGspanningTreeCost_t);
			}													/* end if */
		}														/* end for */
	}
	TEST (pos, "Graph is not Connected");

	/* ending, free internal memory */
	for (nIt1 = G->nodes->begin; nIt1; nIt1 = nIt1->next)
	{
		EGfreeEquiSetElem (EGosGetData
											 (((EGuGraphNode_t *) nIt1->this)->data, voidOS,
												EGequiSetElem_t *), mem);
	}
	return 0;
}																/* end EGminSpanningTree */
