/* EGlib "Efficient General Library" provides some basic structures and
 * algorithms commons in many optimization algorithms.
 *
 * Copyright (C) 2005 Daniel Espinoza and Marcos Goycoolea.
 * 
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but 
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY 
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public 
 * License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA 
 * */
#include "eg_row.h"

/* return a pointer to an initialized EGrow object */
EGrow_t *EGnewRow (EGmemPool_t * mempool)
{

	/* local variables */
	EGrow_t *res;

	/* looking for memory */
	res = (EGrow_t *) EGmemPoolMalloc (mempool, sizeof (EGrow_t));
	res->list = EGnewList (mempool);
	res->name = 0;

	/* ending */
	return res;
}

/* liberate the memory returned by EGnewRow, __NEVER__ liberate such a 
 * memory space with free */
void EGfreeRow (void *row)
{

	/* local variables */
	EGmemPool_t *mempool = ((EGrow_t *) row)->list->mempool;
	EGlistNode_t *cur,
	 *next;

	/* we free the list, in order to be more inteligent we free everything from
	 * here. the disadvantage of this is that it is __VERY__ implementation
	 * dependent */
	cur = ((EGrow_t *) row)->list->begin;
	while (cur)
	{
		next = cur->next;
		EGmemPoolFree (cur->this, sizeof (EGcplexData_t), mempool);
		EGmemPoolFree (cur, sizeof (EGlistNode_t), mempool);
		cur = next;
	}

	/* then we free everything else */
	EGmemPoolFree (((EGrow_t *) row)->list, sizeof (EGlist_t), mempool);
	if (((EGrow_t *) row)->name)
		EGmemPoolFree (((EGrow_t *) row)->name,
									 strlen (((EGrow_t *) row)->name) + 1, mempool);
	EGmemPoolFree (row, sizeof (EGrow_t), mempool);

	/* ending */
	return;
}

/* this function just add the pair (int,double) to the List in EGrow, but 
 * it does not check if there is another entry with the same integer, i.e. 
 * if you add a constrain twice you will have two entries for the same 
 * constrain, and this will lead to an error when we add the constraint or 
 * column to  */
void EGrowAddCoeff (EGrow_t * col,
										const int LPind,
										const double coeff)
{

	/* local variables */
	EGcplexData_t *data;

	/* looking for memory */
	data =
		(EGcplexData_t *) EGmemPoolMalloc (col->list->mempool,
																			 sizeof (EGcplexData_t));
	data->LPind = LPind;
	data->coeff = coeff;

	/* andd data to the end of the list */
	EGlistPushBack (col->list, data);

	/* ending */
	return;
}

/* change the name, objective lowe and upper bound of a column */
void EGrowInit (EGrow_t * row,
								const char *name,
								const int sense,
								const double rhs)
{

	/* we free the current name */
	if (row->name)
		EGmemPoolFree (row->name, strlen (row->name) + 1, row->list->mempool);

	/* and now we copy all information */
	if (name)
	{
		row->name =
			(char *) EGmemPoolMalloc (row->list->mempool, strlen (name) + 1);
		memcpy (row->name, name, strlen (name) + 1);
	}
	else
	{
		row->name = 0;
	}
	row->sense = (char) sense;
	row->rhs = rhs;

	/* ending */
	return;
}

/* this is for testing purposes only, you should call this function before 
 * running just to be sure the structures are OK */
void EGrowTest (void)
{
	/* local variables */
#if DEBUG > 2
	int i,
	  j,
	  size = 10;
#else
	int i,
	  j,
	  size = 1000;
#endif
	EGrow_t **myrows;
	EGmemPool_t *mempool;

	/* creating some columns,  */
	mempool = EGnewMemPool (100, EGmemPoolNewSize, EGmemPoolNewSize (1));
	myrows = (EGrow_t **) EGmemPoolMalloc (mempool, size * sizeof (EGrow_t *));
#if DEBUG > 2
	fprintf (stdout, "Creating Rows");
	fflush (stdout);
#endif
	for (i = 0; i < size; i++)
	{
#if DEBUG > 2
		fprintf (stdout, ".");
		fflush (stdout);
#endif
		myrows[i] = EGnewRow (mempool);
		EGrowInit (myrows[i], "X", 'E', 1.0);
		for (j = 0; j < size; j++)
		{
			EGrowAddCoeff (myrows[i], j, drand48 ());
		}
	}
#if DEBUG > 2
	fprintf (stdout, "done\n");
	fflush (stdout);
#endif

	/*erase them,  */
	for (i = 0; i < size; i++)
	{
		EGfreeRow ((void *) myrows[i]);
	}

	/* recreate them and exit creating some columns,  */
#if DEBUG > 2
	fprintf (stdout, "Creating Rows");
	fflush (stdout);
#endif
	for (i = 0; i < size; i++)
	{
#if DEBUG > 2
		fprintf (stdout, ".");
		fflush (stdout);
#endif
		myrows[i] = EGnewRow (mempool);
		EGrowInit (myrows[i], "X", 'E', 1.0);
		for (j = 0; j < size; j++)
		{
			EGrowAddCoeff (myrows[i], j, drand48 ());
		}
	}
#if DEBUG > 2
	fprintf (stdout, "done\n");
	fflush (stdout);
#endif

	/*erase them,  */
#if DEBUG > 2
	fprintf (stdout, "Erasing Rows");
	fflush (stdout);
#endif
	for (i = 0; i < size; i++)
	{
#if DEBUG > 2
		fprintf (stdout, ".");
		fflush (stdout);
#endif
		EGfreeRow ((void *) myrows[i]);
	}
#if DEBUG > 2
	fprintf (stdout, "done\n");
	fflush (stdout);
#endif

	/* ending */
	EGmemPoolFree (myrows, size * sizeof (EGrow_t *), mempool);
	EGfreeMemPool (mempool);
	return;
}

/* this function reset a row */
int EGrowClear (EGrow_t * row)
{

	/* local data */
	EGlistNode_t *cur,
	 *next;
	EGmemPool_t *mempool = row->list->mempool;

	/* we clear the list */
	cur = row->list->begin;
	while (cur)
	{
		next = cur->next;
		EGmemPoolFree (cur->this, sizeof (EGcplexData_t), mempool);
		EGmemPoolFree (cur, sizeof (EGlistNode_t), mempool);
		cur = next;
	}
	row->list->size = 0;
	row->list->begin = row->list->end = 0;

	/* ending */
	return 0;
}
