/* EGlib "Efficient General Library" provides some basic structures and
 * algorithms commons in many optimization algorithms.
 *
 * Copyright (C) 2005 Daniel Espinoza and Marcos Goycoolea.
 * 
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but 
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY 
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public 
 * License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA 
 * */
/** @file
 * @ingroup EGnet */
/** @addtogroup EGnet */
/** @{ */
/* ========================================================================= */
#ifndef __EGNET_C__
#define __EGNET_C__
#include "eg_net.h"

/* ========================================================================= */
EGsocket_t *EGnewSocket (EGmemPool_t * mem)
{
	EGsocket_t *skt = EGmemPoolSMalloc (mem, EGsocket_t, 1);
	EGnetInitSocket (skt);
	return skt;
}

/* ========================================================================= */
void EGfreeSocket (void *skt,
									 EGmemPool_t * mem)
{
	if (skt)
		EGnetClearSocket (skt);
	EGmemPoolSFree (skt, EGsocket_t, 1, mem);
}

/* ========================================================================= */
int EGnetListen (EGsocket_t * const skt,
								 unsigned short p)
{
	/* local variables */
	struct sockaddr_in addr;
	int rval;

	/* test if the socket exist */
	TESTL (EGNET_DBGL, !skt, "socket pointer is null");

	/* if the socket is already open, we close it and reset it */
	if (skt->s_fd)
	{
		close (skt->s_fd);
		memset (skt, 0, sizeof (EGsocket_t));
	}

	/* store the port number (in machine byte order) */
	skt->port = p;

	/* open the socket, AF_INET indicate that the connection is IP, the
	 * SOCK_STREAM */
	skt->s_fd = socket (AF_INET, SOCK_STREAM, 0);
	if (skt->s_fd < 0)
	{
		perror ("socket");
		memset (skt, 0, sizeof (EGsocket_t));
		TESTL (EGNET_DBGL, 1, "can't open socket");
		return 1;
	}

	/* bind it to an address, according to the ip(7) manual page, this operation
	 * links the socket to a phisical interface, the INADDR_ANY link the socket 
	 * to all phisical (and logical) interefaces. the family AF_INET is to 
	 * specify that the connection is through TCP/IP protocol. p is the port of
	 * connection, but it must be stored in network byte order (thus the call to
	 * htons). */
	memset (&addr, 0, sizeof (addr));
	addr.sin_addr.s_addr = INADDR_ANY;
	addr.sin_family = AF_INET;
	addr.sin_port = htons (p);
	rval = bind (skt->s_fd, (struct sockaddr *) &addr, sizeof (addr));
	if (rval)
	{
		perror ("bind");
		close (skt->s_fd);
		memset (skt, 0, sizeof (EGsocket_t));
		TESTL (EGNET_DBGL, 1, "Can't bind socket to IP address");
		return 1;
	}

	/* now we set the socket to be listening for incomming connections. here we
	 * set the size of the queue of incomming connections (second parameter). */
	rval = listen (skt->s_fd, EG_NET_LISTEN_QUEUE_SIZE);
	if (rval)
	{
		perror ("listen");
		close (skt->s_fd);
		memset (skt, 0, sizeof (EGsocket_t));
		TESTL (EGNET_DBGL, 1, "Can't lisent in socket");
		return 1;
	}

	/* ending */
	return 0;
}

/* ========================================================================= */
int EGnetConnect (EGsocket_t * const skt,
									const char *host_name,
									unsigned short port)
{
	struct hostent *host;
	struct sockaddr_in addr;
	int rval = 0;
	size_t max_buff_size = EG_NET_DATA_QUEUE_SIZE;

	/* check the socket and the FD */
	TESTL (EGNET_DBGL, !host_name, "no host name given");
	TESTL (EGNET_DBGL, !skt, "socket is NULL");
	TESTL (EGNET_DBGL, skt->s_fd, "socket is open, can't reconnect");
	TESTL (EGNET_DBGL, skt->f_fd, "socket is open, can't reconnect");

	/* open the socket, AF_INET indicate that the connection is IP, the
	 * SOCK_STREAM */
	skt->s_fd = socket (AF_INET, SOCK_STREAM, 0);
	if (skt->s_fd < 0)
	{
		perror ("socket");
		memset (skt, 0, sizeof (EGsocket_t));
		TESTL (EGNET_DBGL, 1, "can't open socket");
		return 1;
	}

	/* first we get the host information */
	host = gethostbyname (host_name);
	TESTL (EGNET_DBGL, !host, "Can't get host information for %s", host_name);

	/* now we set the addr structure to connect to the remote host */
	memcpy (&addr.sin_addr, host->h_addr, (unsigned) host->h_length);
	addr.sin_family = AF_INET;
	addr.sin_port = htons (port);

	/* now we connect to the remote host */
	rval = connect (skt->s_fd, (struct sockaddr *) &addr, sizeof (addr));
	if (rval)
	{
		perror ("connect");
		TESTL (EGNET_DBGL, 1, "Unable to Connect to %s", host_name);
		return 1;
	}

	/* set the send and receive buffer to max_buff_size */
	rval =
		setsockopt (skt->s_fd, SOL_SOCKET, SO_SNDBUF, &max_buff_size,
								sizeof (size_t));
	if (rval)
	{
		perror ("setsockopt");
		TESTL (EGNET_DBGL, 1, "Unable to Connect to %s", host_name);
		return 1;
	}
	rval =
		setsockopt (skt->s_fd, SOL_SOCKET, SO_RCVBUF, &max_buff_size,
								sizeof (size_t));
	if (rval)
	{
		perror ("setsockopt");
		TESTL (EGNET_DBGL, 1, "Unable to Connect to %s", host_name);
		return 1;
	}

	/* now the F_FD is set to S_FD */
	skt->f_fd = skt->s_fd;

	/* ending */
	return 0;
}

/* ========================================================================= */
int EGnetDisconnect (EGsocket_t * const skt)
{
	TESTL (EGNET_DBGL, !skt, "Null socket");
	TESTL (EGNET_DBGL, !skt->s_fd, "socket not ready");
	TESTL (EGNET_DBGL, !skt->f_fd, "socket not connected");
	TESTL (EGNET_DBGL, skt->f_fd != skt->s_fd, "socket not connected");
	close (skt->f_fd);
	skt->s_fd = skt->f_fd = 0;
#if EG_NET_SYNCHRONIZE
	skt->in_sz = skt->out_sz = 0;
#endif
	return 0;
}

/* ========================================================================= */
int EGnetStartReadNB (EGsocket_t * const skt)
{
	struct sockaddr_in addr;
	unsigned int addr_size = sizeof (addr);
	size_t max_buff_size = EG_NET_DATA_QUEUE_SIZE;
#if EG_NET_ALLOW_NON_BLOCKING
	long fd_mode;
#endif
	int rval = 0;
	TESTL (EGNET_DBGL, !skt, "Null socket");
	TESTL (EGNET_DBGL, skt->f_fd, "socket already ready to read/write");
	TESTL (EGNET_DBGL, !skt->s_fd, "socket not initialized by listen");

	/* set the state of the listening socket to blocking mode, i.e. it block the
	 * execution of the program until we get an incomming connection */
#if EG_NET_ALLOW_NON_BLOCKING
	fcntl (skt->s_fd, F_GETFL, &fd_mode);
	fd_mode = fd_mode | O_NONBLOCK;
	fcntl (skt->s_fd, F_SETFL, &fd_mode);
#endif

	/* init the addr structure, AF_INET indicate that isa IP connection,
	 * INADDR_ANY indicate that all interfaces are good (or allowed to recive 
	 * connections) */
	addr.sin_family = AF_INET;
	addr.sin_addr.s_addr = INADDR_ANY;
	addr.sin_port = 0;
	skt->f_fd = accept (skt->s_fd, ((__SOCKADDR_ARG) (&addr)), &addr_size);

	/* if accept return with EAGAIN, it means that the incomming connections 
	 * queue was empty, and thus there is no message to read. In this case we 
	 * return EAGAIN to the calling function and let the socket in listening 
	 * state */
	if (skt->f_fd == EAGAIN)
	{
		skt->f_fd = 0;
		ADVTESTL (EGNET_DBGL, 1, EAGAIN, "no incomming connection to accept");
		return EAGAIN;
	}
	if (skt->f_fd < 0)
	{
		perror ("accept");
		memset (skt, 0, sizeof (EGsocket_t));
		TESTL (EGNET_DBGL, 1, "can't stablish connection with incomming request");
		return 1;
	}

	/* set the send and receive buffer to max_buff_size */
	rval =
		setsockopt (skt->f_fd, SOL_SOCKET, SO_SNDBUF, &max_buff_size,
								sizeof (size_t));
	if (rval)
	{
		perror ("setsockopt");
		TESTL (EGNET_DBGL, 1, "Can't start reading");
		return 1;
	}
	rval =
		setsockopt (skt->f_fd, SOL_SOCKET, SO_RCVBUF, &max_buff_size,
								sizeof (size_t));
	if (rval)
	{
		perror ("setsockopt");
		TESTL (EGNET_DBGL, 1, "Can't start reading");
		return 1;
	}

	MESSAGE (EGNET_DBGL, "connected (%d)", addr_size);

	/* return 0 */
	return 0;
}

/* ========================================================================= */
int EGnetStartRead (EGsocket_t * const skt)
{
	size_t max_buff_size = EG_NET_DATA_QUEUE_SIZE;
	struct sockaddr_in addr;
	unsigned int addr_size = sizeof (addr);
#if EG_NET_ALLOW_NON_BLOCKING
	long fd_mode;
#endif
	int rval = 0;
	TESTL (EGNET_DBGL, !skt, "Null socket");
	TESTL (EGNET_DBGL, skt->f_fd, "socket already ready to read/write");
	TESTL (EGNET_DBGL, !skt->s_fd, "socket not initialized by listen");

	/* set the state of the listening socket to blocking mode, i.e. it block the
	 * execution of the program until we get an incomming connection */
#if EG_NET_ALLOW_NON_BLOCKING
	fcntl (skt->s_fd, F_GETFL, &fd_mode);
	fd_mode = fd_mode & (~O_NONBLOCK);
	fcntl (skt->s_fd, F_SETFL, &fd_mode);
#endif

	/* init the addr structure, AF_INET indicate that isa IP connection,
	 * INADDR_ANY indicate that all interfaces are good (or allowed to recive 
	 * connections) */
	addr.sin_family = AF_INET;
	addr.sin_addr.s_addr = INADDR_ANY;
	addr.sin_port = 0;
	skt->f_fd = accept (skt->s_fd, ((__SOCKADDR_ARG) (&addr)), &addr_size);
	if (skt->f_fd < 0)
	{
		perror ("accept");
		memset (skt, 0, sizeof (EGsocket_t));
		TESTL (EGNET_DBGL, 1, "can't stablish connection with incomming request");
		return 1;
	}

	/* set the send and receive buffer to max_buff_size */
	rval =
		setsockopt (skt->f_fd, SOL_SOCKET, SO_SNDBUF, &max_buff_size,
								sizeof (size_t));
	if (rval)
	{
		perror ("setsockopt");
		TESTL (EGNET_DBGL, 1, "Can't start reading");
		return 1;
	}
	rval =
		setsockopt (skt->f_fd, SOL_SOCKET, SO_RCVBUF, &max_buff_size,
								sizeof (size_t));
	if (rval)
	{
		perror ("setsockopt");
		TESTL (EGNET_DBGL, 1, "Can't start reading");
		return 1;
	}

	MESSAGE (EGNET_DBGL, "connected (%d)", addr_size);

	/* return 0 */
	return 0;
}

/* ========================================================================= */
int EGnetStopRead (EGsocket_t * const skt)
{
	TESTL (EGNET_DBGL, !skt, "Null socket");
	TESTL (EGNET_DBGL, !skt->s_fd, "socket is not open");
	TESTL (EGNET_DBGL, !skt->f_fd, "socket have not been put to read before");
	TESTL (EGNET_DBGL, skt->s_fd == skt->f_fd,
				 "socket is not a lisening socket");
	close (skt->f_fd);
	skt->f_fd = 0;
#if EG_NET_SYNCHRONIZE
	skt->in_sz = skt->out_sz = 0;
#endif

	/* ending */
	return 0;
}

#define EG_NET_SEND 0
#define EG_NET_RECEIVE 1
#define EG_NET_MARKER '5'

/* ========================================================================= */
/** @brief Assure that the data queue don't overflow by using syncronizations
 * calls between both ends of the connections.
 * @param skt socket where we are working.
 * @param type either EG_NET_SEND or EG_NET RECEIVE.
 * @param length bytes being transfered.
 * */
#if EG_NET_SYNCHRONIZE
void static inline EGnetSynchronize (EGsocket_t * const skt,
																		 int const type,
																		 const size_t length)
{
	unsigned char marker = EG_NET_MARKER;
	switch (type)
	{
	case EG_NET_SEND:
		skt->in_sz = EG_NET_TCPIP_OVERHEAD;
		if (skt->out_sz + EG_NET_TCPIP_OVERHEAD + length >= EG_NET_DATA_QUEUE_SIZE)
		{
			MESSAGE (EGNET_DBGL + 10, "Syncronizing %u %u", skt->in_sz, skt->out_sz);
			EXIT (1 != recv (skt->f_fd, &marker, 1, 0), "Error while synchronizing");
			EXIT (marker != EG_NET_MARKER, "Error while synchronizing, expecting "
						"to receive %c(%d) and received %c(%d)", EG_NET_MARKER,
						(int) EG_NET_MARKER, marker, (int) marker);
			skt->out_sz = EG_NET_TCPIP_OVERHEAD;
		}
		else
			skt->out_sz += EG_NET_TCPIP_OVERHEAD + length;
		break;
	case EG_NET_RECEIVE:
		skt->out_sz = EG_NET_TCPIP_OVERHEAD;
		if (skt->in_sz + EG_NET_TCPIP_OVERHEAD + length >= EG_NET_DATA_QUEUE_SIZE)
		{
			MESSAGE (EGNET_DBGL + 10, "Syncronizing %u %u", skt->in_sz, skt->out_sz);
			EXIT (1 != send (skt->f_fd, &marker, 1, 0), "Error while synchronizing");
			skt->in_sz = EG_NET_TCPIP_OVERHEAD;
		}
		else
			skt->in_sz += EG_NET_TCPIP_OVERHEAD + length;
		break;
	default:
		EXIT (1, "Unknown communication type %d", type);
		break;
	}
}
#else
#define EGnetSynchronize(skt,type,length)
#endif

/* ========================================================================= */
int EGnetSendChar (EGsocket_t * const skt,
									 const int c)
{
	int rval;
	unsigned char uc = ((unsigned char) c) & 0xff;
	TESTL (EGNET_DBGL, !skt, "Null socket");
	TESTL (EGNET_DBGL, !skt->f_fd, "socket not ready to I/O operations");
	EGnetSynchronize (skt, EG_NET_SEND, 1);
	rval = send (skt->f_fd, &uc, 1, 0);
	TESTL (EGNET_DBGL, rval != 1, "can't send char");
#if EG_NET_CONFIRM
	rval = recv (skt->f_fd, &uc, 1, 0);
	TESTL (EGNET_DBGL, rval != 1, "can't recieve char");
	TESTL (EGNET_DBGL, uc != (((unsigned char) c) & 0xff), "wrong comunication");
#endif
	return 0;
}

/* ========================================================================= */
int EGnetRecvChar (EGsocket_t * const skt,
									 char *const c)
{
	int rval;
	TESTL (EGNET_DBGL, !skt, "Null socket");
	TESTL (EGNET_DBGL, !skt->f_fd, "socket not ready to I/O operations");
	EGnetSynchronize (skt, EG_NET_RECEIVE, 1);
	rval = recv (skt->f_fd, c, 1, 0);
	TESTL (EGNET_DBGL, rval != 1, "can't recieve char");
#if EG_NET_CONFIRM
	rval = send (skt->f_fd, c, 1, 0);
	TESTL (EGNET_DBGL, rval != 1, "can't send char");
#endif
	return 0;
}

/* ========================================================================= */
int EGnetSendUshort (EGsocket_t * const skt,
										 unsigned short n)
{
	unsigned char buff[2];
#if EG_NET_CONFIRM
	unsigned char buffcp[2];
#endif
	int rval;
	TESTL (EGNET_DBGL, !skt, "Null socket");
	TESTL (EGNET_DBGL, !skt->f_fd, "socket not ready to I/O operations");
	buff[0] = (n >> 8) & 0xff;
	buff[1] = (n) & 0xff;
	EGnetSynchronize (skt, EG_NET_SEND, 2);
	rval = send (skt->f_fd, buff, 2, 0);
	TESTL (EGNET_DBGL, rval != 2, "can't send ushort");
#if EG_NET_CONFIRM
	rval = recv (skt->f_fd, buffcp, 2, 0);
	TESTL (EGNET_DBGL, rval != 2, "can't recieve ushort");
	TESTL (EGNET_DBGL, memcmp (buff, buffcp, sizeof (buff)),
				 " wrong comunication");
#endif
	return 0;
}

/* ========================================================================= */
int EGnetRecvUshort (EGsocket_t * const skt,
										 unsigned short *n)
{
	unsigned char buff[2];
	int rval;
	TESTL (EGNET_DBGL, !skt, "Null socket");
	TESTL (EGNET_DBGL, !skt->f_fd, "socket not ready to I/O operations");
	EGnetSynchronize (skt, EG_NET_RECEIVE, 2);
	rval = recv (skt->f_fd, buff, 2, 0);
	TESTL (EGNET_DBGL, rval != 2, "can't recieve ushort");
#if EG_NET_CONFIRM
	rval = send (skt->f_fd, buff, 2, 0);
	TESTL (EGNET_DBGL, rval != 2, "can't send ushort");
#endif
	*n = ((unsigned) buff[0]) << 8;
	*n |= ((unsigned) buff[1]);
	return 0;
}

/* ========================================================================= */
#define EGnetSendShort(skt,n) EGnetSendUshort(skt,(unsigned)n)

/* ========================================================================= */
int EGnetRecvShort (EGsocket_t * const skt,
										short *n)
{
	unsigned short un;
	int rval;
	rval = EGnetRecvUshort (skt, &un);
	CHECKRVAL (rval);
	*n = (short) un;
	return 0;
}

/* ========================================================================= */
int EGnetSendUint (EGsocket_t * const skt,
									 unsigned int n)
{
	unsigned char buff[4];
#if EG_NET_CONFIRM
	unsigned char buffcp[4];
#endif
	int rval = 0;
	TESTL (EGNET_DBGL, !skt, "Null socket");
	TESTL (EGNET_DBGL, !skt->f_fd, "socket not ready to I/O operations");
	buff[0] = (n >> 24) & 0xff;
	buff[1] = (n >> 16) & 0xff;
	buff[2] = (n >> 8) & 0xff;
	buff[3] = (n) & 0xff;
	EGnetSynchronize (skt, EG_NET_SEND, 4);
	rval = send (skt->f_fd, buff, 4, 0);
	TESTL (EGNET_DBGL, rval != 4, "can't send uint");
#if EG_NET_CONFIRM
	rval = recv (skt->f_fd, buffcp, 4, 0);
	TESTL (EGNET_DBGL, rval != 4, "can't recieve uint");
	TESTL (EGNET_DBGL, memcmp (buff, buffcp, sizeof (buff)),
				 "wrong comunication");
#endif
	return 0;
}

/* ========================================================================= */
int EGnetRecvUint (EGsocket_t * const skt,
									 unsigned int *n)
{
	unsigned char buff[4];
	int rval;
	TESTL (EGNET_DBGL, !skt, "Null socket");
	TESTL (EGNET_DBGL, !skt->f_fd, "socket not ready to I/O operations");
	EGnetSynchronize (skt, EG_NET_RECEIVE, 4);
	rval = recv (skt->f_fd, buff, 4, 0);
	TESTL (EGNET_DBGL, rval != 4, "can't recieve uint");
#if EG_NET_CONFIRM
	rval = send (skt->f_fd, buff, 4, 0);
	TESTL (EGNET_DBGL, rval != 4, "can't send uint");
#endif
	*n = ((unsigned) buff[0]) << 24;
	*n |= ((unsigned) buff[1]) << 16;
	*n |= ((unsigned) buff[2]) << 8;
	*n |= ((unsigned) buff[3]);
	return 0;
}

/* ========================================================================= */
#define EGnetSendInt(skt,n) EGnetSendUint(skt,(unsigned) n)

/* ========================================================================= */
int EGnetRecvInt (EGsocket_t * const skt,
									int *n)
{
	unsigned un;
	int rval;
	rval = EGnetRecvUint (skt, &un);
	CHECKRVAL (rval);
	*n = (int) un;
	return 0;
}

/* ========================================================================= */
int EGnetSendDouble (EGsocket_t * const skt,
										 const double exd)
{
	/* the idea is that we store in buff[9] the sign, in buff[8] we store the
	 * exponent, and in the first 8 bytes we store the value. the only drawback 
	 * is that this only allow us to send/recieve values whose exponents are 
	 * at most 128 and at least -128 */
	double d = exd;
	int rval;
	unsigned char buff[10] = { 0, 0, 0, 0, 0, 0, 0, 0, 128, 0 };
#if EG_NET_CONFIRM
	unsigned char buffcp[10];
#endif
	if (d < 0)
	{
		buff[9] = 1;
		d = -d;
	}
	if (d >= 1)
	{
#define __HI_EXP(x,e,v,lv) {if( x >=v ){ e = e + lv; x /= v;}}
		__HI_EXP (d, buff[8], 18446744073709551616.0, 64);
		__HI_EXP (d, buff[8], 4294967296.0, 32);
		__HI_EXP (d, buff[8], 65536.0, 16);
		__HI_EXP (d, buff[8], 256.0, 8);
		__HI_EXP (d, buff[8], 16.0, 4);
		__HI_EXP (d, buff[8], 4.0, 2);
		__HI_EXP (d, buff[8], 2.0, 1);
#undef __HI_EXP
		d /= 2;
		buff[8] += 1;
	}
	else if (d < 0.5)
	{
#define __LO_EXP(x,e,v,lv) {if( x < 1/v ) { e = e - lv; x *= v;}}
		__LO_EXP (d, buff[8], 18446744073709551616.0, 64);
		__LO_EXP (d, buff[8], 4294967296.0, 32);
		__LO_EXP (d, buff[8], 65536.0, 16);
		__LO_EXP (d, buff[8], 256.0, 8);
		__LO_EXP (d, buff[8], 16.0, 4);
		__LO_EXP (d, buff[8], 4.0, 2);
		__LO_EXP (d, buff[8], 2.0, 1);
#undef __LO_EXP
	}
	d *= 256;
	buff[0] = (unsigned char) d;
	d -= buff[0];
	d *= 256;
	buff[1] = (unsigned char) d;
	d -= buff[1];
	d *= 256;
	buff[2] = (unsigned char) d;
	d -= buff[2];
	d *= 256;
	buff[3] = (unsigned char) d;
	d -= buff[3];
	d *= 256;
	buff[4] = (unsigned char) d;
	d -= buff[4];
	d *= 256;
	buff[5] = (unsigned char) d;
	d -= buff[5];
	d *= 256;
	buff[6] = (unsigned char) d;
	d -= buff[6];
	d *= 256;
	buff[7] = (unsigned char) d;
	EGnetSynchronize (skt, EG_NET_SEND, 10);
	rval = send (skt->f_fd, buff, 10, 0);
	TESTL (EGNET_DBGL, rval != 10, "can't send double");
#if EG_NET_CONFIRM
	rval = recv (skt->f_fd, buffcp, 10, 0);
	TESTL (EGNET_DBGL, rval != 10, "can't receive double");
	TESTL (EGNET_DBGL, memcmp (buff, buffcp, sizeof (buff)),
				 "wrong comunication");
#endif
	return 0;
}

/* ========================================================================= */
int EGnetRecvDouble (EGsocket_t * const skt,
										 double *const d)
{
	unsigned char buff[10];
	int rval;
	unsigned int ui1,
	  ui2;
	TESTL (EGNET_DBGL, !skt->f_fd, "Uninitialized socket");
	EGnetSynchronize (skt, EG_NET_RECEIVE, 10);
	rval = recv (skt->f_fd, buff, 10, 0);
	if (rval == -1)
	{
		perror ("recv");
		TESTL (EGNET_DBGL, 1, "can't read from socket");
		return 1;
	}
	TESTL (EGNET_DBGL, rval != 10,
				 "can't read double, we have recived only %d bytes of information",
				 rval);
#if EG_NET_CONFIRM
	rval = send (skt->f_fd, buff, 10, 0);
	TESTL (EGNET_DBGL, rval != 10, "can't send double");
#endif
	ui1 = (buff[0] << 24) | (buff[1] << 16) | (buff[2] << 8) | buff[3];
	ui2 = (buff[4] << 24) | (buff[5] << 16) | (buff[6] << 8) | buff[7];
	*d = ((ui2 / 4294967296.0) + ui1) / 4294967296.0;
	if (buff[8] > 128)
	{
#define __HI_EXP(x,e,v,lv) {if( e >= lv + 128){ e = e - lv; x *= v;}}
		__HI_EXP (*d, buff[8], 18446744073709551616.0, 64);
		__HI_EXP (*d, buff[8], 4294967296.0, 32);
		__HI_EXP (*d, buff[8], 65536.0, 16);
		__HI_EXP (*d, buff[8], 256.0, 8);
		__HI_EXP (*d, buff[8], 16.0, 4);
		__HI_EXP (*d, buff[8], 4.0, 2);
		__HI_EXP (*d, buff[8], 2.0, 1);
#undef __HI_EXP
	}
	else if (buff[8] < 128)
	{
#define __LO_EXP(x,e,v,lv) {if( e <= 128 - lv){ e = e + lv; x /= v;}}
		__LO_EXP (*d, buff[8], 18446744073709551616.0, 64);
		__LO_EXP (*d, buff[8], 4294967296.0, 32);
		__LO_EXP (*d, buff[8], 65536.0, 16);
		__LO_EXP (*d, buff[8], 256.0, 8);
		__LO_EXP (*d, buff[8], 16.0, 4);
		__LO_EXP (*d, buff[8], 4.0, 2);
		__LO_EXP (*d, buff[8], 2.0, 1);
#undef __LO_EXP
	}
	if (buff[9])
		*d = -(*d);
	return 0;
}

/* ========================================================================= */
int EGnetSendString (EGsocket_t * const skt,
										 const char *const const exstr)
{
	int rval;
	const char *str = exstr;
	while (*str)
	{
		rval = EGnetSendChar (skt, *str);
		CHECKRVAL (rval);
		str++;
	}
	rval = EGnetSendChar (skt, 0);
	CHECKRVAL (rval);
	return 0;
}

/* ========================================================================= */
int EGnetRecvString (EGsocket_t * const skt,
										 char *const const exstr,
										 size_t max_size)
{
	int rval;
	char *str = exstr;
	size_t count = max_size - 1;
	do
	{
		rval = EGnetRecvChar (skt, str);
		CHECKRVAL (rval);
	} while ((*str) && (str++, --count));
	*str = 0;
	return 0;
}

/* ========================================================================= */
/** @} */
#endif
