/* EGlib "Efficient General Library" provides some basic structures and
 * algorithms commons in many optimization algorithms.
 *
 * Copyright (C) 2005 Daniel Espinoza and Marcos Goycoolea.
 * 
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but 
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY 
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public 
 * License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA 
 * */
/* ========================================================================= */
/** @defgroup EGmemPool EGmemPool
 *
 * This header contains the definitions of memory pools, this memory 
 * pools are used by some sub-libraries to ask for memory.
 * These memory pools try to alloc big chunks of memory and give it
 * to the system as you ask for it; it also try to re-utilize 'freed'
 * memory by keeping the address of the 'freed' memory.
 * 
 * Once you free all memory inside the memory pool, all the 
 * structures that alloc'ed memory from this pool will be free
 * so __BE__ __CAREFUL__.
 * 
 * Also, the memory pool can't alloc sizes under sizeof(void*),
 * if you call memory from the pool of a smaller size you will 
 * recive a block of size sizeof(void*) anyway.
 * 
 * Finally you have to alloc and free memory pools by the functions
 * EGnewMempool and EGfreeMempool
 *
 * Another point is that the EGmemPoolFree can't deal with null pointers, you
 * have to check if the pointer to free is null or not before calling it.
 * 
 * @version 0.0.1 
 * @par History:
 * - 2005-08-20
 * 						- Update memory algiment to new definitions.
 * 						- Move options definitions here from eg_config.h
 * - 2005-08-01
 * 						- Fix some printing flags.
 * - 2005-07-30
 * 						- Split this header into eg_mempool.h and eg_mem.h for clarity,
 * 							but we keep backward compability, this is done so that we can
 * 							use other allocators and also to separate general-use
 * 							alloc/free functions from the original mempool functions.
 * - 2005-07-12
 * 						- Add EGmemPoolInit macro.
 * 						- Add EGmemPoolClear macro.
 * - 2004-03-25
 * 						- Add big chunk profile
 * - 2003-12-01
 * 						- Add EGcopyMP_f definition.
 * - 2003-11-20
 * 						- Improve EGfree functionality
 * - 2003-11-07 
 * 						- Add EGsMalloc macro based on EGmalloc to allocate memory of
 * 							certain type without doing the cast.
 * - 2003-09-17 
 * 						- Adding suport for malloc/free tracking information
 * - 2003-05-22 
 * 						- Redefine nullFree and nullFreeMP as pointer to zero, it is
 * 							important not to call those functions, becouse you will generate
 * 							an exception or segmentation fault. The functions EGlistClear
 * 							and EGfreeListNode handle that correctly.
 * - 2003-05-20 
 * 						- Add nullFreeMP support.
 * - 2003-05-19 
 * 						- Fix a condition on EGmemPoolFree to support free of size 0 of a
 * 							null memory adress.
 * 						- Improve support for malloc of size zero, if the debug level is
 * 							at least 2, we throw a warning each time that we alloc zero
 * 							bytes.
 * - 2003-05-15 
 * 						- Remove EGmemPoolClear from support, it wasn't coded anyway.
 * - 2003-05-14 
 * 						- Add EGmalloc and EGfree, they just replace malloc and free (with
 * 							test of existance of memory. The idea is to add memory leack
 * 							checkers for later versions and some extra tests (as free null
 * 							pointers and so on)
 * - 2003-05-08 
 * 						- Add checks on the free function, if we free a null memory we
 * 							just return, but if the memory is null and the size is not (or
 * 							vicevesa ) we report an error and display who called the
 * 							function.
 * - 2003-05-05 
 * 						- fix bug in the count of total memory used per size
 * - 2003-04-30 
 * 						- fix a bug in the tail memory recover code
 * - 2003-04-29 
 * 						- recover when posible the tail memory when we do resize, this 
 * 							means	that when we do the memory resize, we check if the 
 * 							freeMemHead still has unused memory, and if it is so, and its 
 * 							size is larger than sizeof(void*), we put it in the freedMemory 
 * 							array.
 * - 2003-04-11 
 * 						- add suport to __REDUCE_TO_MALLOC__ 
 *              move nullFree here
 *            - define a default memory size function
 * - 2003-04-10
 *						- First Implementation
 *
 * */
 /** @file
	* @ingroup EGmemPool */
 /** @addtogroup EGmemPool 
	* @{ */
/* ========================================================================= */

#ifndef __EG_MEMPOOL_H__
#define __EG_MEMPOOL_H__
#include <string.h>
#include <unistd.h>
#include "eg_macros.h"
#include "eg_mem.h"

/* ========================================================================= */
/** @brief if we want to reduce all this to malloc/free we can do it at 
 * compile time by setting this constant to 1 */
#ifndef __EG_MEMPOOL_REDUCE_TO_MALLOC__
#define __EG_MEMPOOL_REDUCE_TO_MALLOC__ 0
#endif

/* ========================================================================= */
/** @brief this is used for debugg and profile purposes only,
 * when setted it keeps profiling information about the 
 * memory pool usage */
#ifndef __EG_MEMPOOL_PROFILE__
#define __EG_MEMPOOL_PROFILE__ (2 && (__EG_MEMPOOL_REDUCE_TO_MALLOC__ || !DEBUG ? 0 : 7))
#endif

/* ========================================================================= */
/** @brief this is used to enable malloc/free tracking and extra debugging */
#ifndef __EG_MEMPOOL_MALLOC_FREE_CHECK__
#define __EG_MEMPOOL_MALLOC_FREE_CHECK__ (!__EG_MEMPOOL_REDUCE_TO_MALLOC__ && 1 && DEBUG)
#endif


/* ========================================================================= */
/** @brief debug level for this set of functions. If the debug used at 
 * compile time was at least this level, we will generate tons of messages to 
 * keep track of what was going on */
#ifndef __EG_MEMPOOL_DEBUGL__
#define __EG_MEMPOOL_DEBUGL__ 200
#endif

/* ========================================================================= */
/** @brief if we enable malloc checking, we will add to every piece of 
 * memory four void*, one for size, other for where we alloc'it, line, and a 
 * status */
#if __EG_MEMPOOL_MALLOC_FREE_CHECK__
#define __EG_MEMPOOL_OVERHEAD__ (8*(EG_MEM_ALIGNMENT))
#define __EG_MEMPOOL_WORD__ EG_MEM_ALIGNMENT
#define __EG_MEMPOOL_SET_FILE(A,B) \
	(*((const char **)((char*)A+4*__EG_MEMPOOL_WORD__))=B)
#define __EG_MEMPOOL_SET_SIZE(A,B) \
	(*((void**)(((char*)A)+1*__EG_MEMPOOL_WORD__))=(void*)((size_t)B))
#define __EG_MEMPOOL_SET_LINE(A,B) \
	(*((void**)(((char*)A)+2*__EG_MEMPOOL_WORD__))=(void*)((size_t)B))
#define __EG_MEMPOOL_SET_ALLOCED(A) \
	(*((void**)(((char*)A)+3*__EG_MEMPOOL_WORD__))=0)
#define __EG_MEMPOOL_SET_FREEED(A) \
	(*((void**)(((char*)A)+3*__EG_MEMPOOL_WORD__))=(void*)((size_t)1))
#define __EG_MEMPOOL_SET_FFILE(A,B) \
	(*((const char **)((char*)A+5*__EG_MEMPOOL_WORD__))=B)
#define __EG_MEMPOOL_SET_FLINE(A,B) \
	(*((void**)(((char*)A)+6*__EG_MEMPOOL_WORD__))=(void*)((size_t)B))
#define __EG_MEMPOOL_STATUS_ALLOCED__ 0
#define __EG_MEMPOOL_STATUS_FREEED__ 1
#define __EG_MEMPOOL_GET_FILE(A) \
	(*((const char**)(((char*)A)+4*__EG_MEMPOOL_WORD__)))
#define __EG_MEMPOOL_GET_SIZE(A) \
	((size_t)(*((void**)(((char*)A)+1*__EG_MEMPOOL_WORD__))))
#define __EG_MEMPOOL_GET_LINE(A) \
	((size_t)(*((void**)(((char*)A)+2*__EG_MEMPOOL_WORD__))))
#define __EG_MEMPOOL_GET_STATUS(A) \
	((size_t)(*((void**)(((char*)A)+3*__EG_MEMPOOL_WORD__))))
#define __EG_MEMPOOL_GET_FFILE(A) \
	(*((const char**)(((char*)A)+5*__EG_MEMPOOL_WORD__)))
#define __EG_MEMPOOL_GET_FLINE(A) \
	((size_t)(*((void**)(((char*)A)+6*__EG_MEMPOOL_WORD__))))
#endif
#if __EG_MEMPOOL_PROFILE__
#define __EG_MEMPOOL_N_BIG_CHUNKS__ 50
#define __EG_MEMPOOL_MEM_SHIFT__ 8
#define __EG_MEMPOOL_BIGBUCK__(size) (\
		(size>>__EG_MEMPOOL_MEM_SHIFT__ > __EG_MEMPOOL_N_BIG_CHUNKS__ - 1) ? \
		__EG_MEMPOOL_N_BIG_CHUNKS__ - 1 : (size>>__EG_MEMPOOL_MEM_SHIFT__))
#endif

/* ========================================================================= */
/** @brief this structure holds a memory pool */
typedef struct
{

	/** and array of size MemoryListSize that contains a pointer
	 * to each memory chunk that we have alloc'ed */
	void **MemoryList;

	/** store the size of MemoryList */
	size_t MemoryListSize;

	/** the memory is stored in two ways, we have an array of memory
	 * from were we return to the system, but if we run out of 
	 * these memory we look to the 'freed' memory and if that is 
	 * empty we alloc another array of memory. This pointer points
	 * to the begining of the 'freeArray' */
	void *freeArrayHead;

	/** how many bytes are free in the freeArray */
	size_t freeArraySize;

	/** this array store list of 'freed' memory, the lists
	 * are indexed acording to the size of the memory freed
	 * (discretized by the size of (void*) ) */
	void **freedMemory;

	/** this define up to what size of memory we manage through
	 * the pool, if we ask the pool for memory over this size
	 * the pool will only call malloc/free */
	size_t manageLimit;

	/** a function that return the size (in bytes) of the
	 * i-th memory block */
	  size_t (*newsize) (size_t);

	/* this is defined only for debug and profile purposes */
#if __EG_MEMPOOL_PROFILE__
	/* identification of the memory pool */
	const char *memAllocationFile;/**< where (which file) the pool was created */
	int memAllocationLine;				/**< where (which line) the pool was created */

	/** maximum non managed block of memory asked to the pool */
	size_t maxBlock;
	const char *maxBlockFile;
	int maxBlockLine;
	unsigned int nBigBlocks[__EG_MEMPOOL_N_BIG_CHUNKS__];
	unsigned int sBigBlocks[__EG_MEMPOOL_N_BIG_CHUNKS__];
	unsigned int cBigBlocks[__EG_MEMPOOL_N_BIG_CHUNKS__];

	/** null alloc information */
	size_t nNullAlloc;
	const char *nullAllocFile;
	int nullAllocLine;

	/** total amount of memory alloc'ed from the pool */
	size_t allocMem;

	/** memory really alloc'ed by the pool */
	size_t totalMem;

	/** memory alloc'ed by the pool but not managed */
	size_t noManageMem;

	/** maximum memory alloc'ed at the same time per size */
	size_t *maxUsedMem;

	/** memory currently in use per size */
	size_t *curUsedMem;
#endif
}
EGmemPool_t;

/* ========================================================================= */
/** @brief type of the free functions that recive a pointer to the memory to
 * be freed, and a pointer to the memory pool where store the 'freed' memory. */
typedef void (*EGfreeMP_f) (void *,
														EGmemPool_t *);

/* ========================================================================= */
/** @brief this is the the data free that does nothing, use it when you don't 
 * want/need to free the internal list data becouse you will do it 
 * elsewere */
#define nullFreeMP ((EGfreeMP_f)0)

/* ========================================================================= */
/** @brief this fucntion allocate a new memory pool, with resize function
 * 'newsize' with manageLimit 'manageLimit' and with initial memory
 * of size 'initSize'. note that a memory pool with 'manageLimit == 0' 
 * is equivalent to malloc/free */
EGmemPool_t *__EGnewMemPool (const size_t manageLimit,
														 size_t (*newsize) (size_t),
														 const size_t initSize);
#define EGnewMemPool(A,B,C) __EGnewMemPool(A,B,C)

/* ========================================================================= */
/** @brief this function free a memory pool */
void EGfreeMemPool (EGmemPool_t *);

/* ========================================================================= */
/** @brief  this function liberate the memory stored in the memory pool */
void EGmemPoolClear (EGmemPool_t *);

/* ========================================================================= */
/** @brief this function ask to a memory pool a memory block of size 'size', 
 * in fact this is a macro that calls the real malloc function but that keeps 
 * track of who and where was it call */
#if DEBUG && (__EG_MEMPOOL_REDUCE_TO_MALLOC__==0)
#define EGmemPoolMalloc(A,B) __EGmemPoolMalloc(A,B,__FILE__,__LINE__)
#else
#define EGmemPoolMalloc(A,B) __EGmemPoolMalloc(A,B)
#endif
void *__EGmemPoolMalloc (EGmemPool_t * const,
												 size_t
#if DEBUG && (__EG_MEMPOOL_REDUCE_TO_MALLOC__==0)
												 ,
												 const char *,
												 const int
#endif
	);

/* ========================================================================= */
/** @brief this macro define a 'safe' malloc for the memory pool, it recives 
 * the type and how many elements of that type we want */
#define EGmemPoolSMalloc(mem,type,count) \
		(type*)EGmemPoolMalloc(mem,sizeof(type)*(count))

/* ========================================================================= */
/** @brief this function free to the memory pool a memory block of size 'size; 
 * It is __VERY__ important to free a block memory this way and to the 
 * apropiate memory pool, any error in this __WILL__ cause serious memory 
 * errors */
#if DEBUG && (__EG_MEMPOOL_REDUCE_TO_MALLOC__==0)
#define EGmemPoolFree(A,B,C) __EGmemPoolFree(A,B,C,__FILE__,__LINE__)
#else
#define EGmemPoolFree(A,B,C) __EGmemPoolFree(A,B,C)
#endif
void __EGmemPoolFree (void *mem,
											size_t size,
											EGmemPool_t * mypool
#if DEBUG && (__EG_MEMPOOL_REDUCE_TO_MALLOC__==0)
											,
											const char *file,
											const int line
#endif
	);

/* ========================================================================= */
/** @brief this macro define a 'safe' free for the memory pool, it recives the 
 * type and how many elements of that type we want */
#define EGmemPoolSFree(mem,type,count,pool) EGmemPoolFree(mem,sizeof(type)*(count),pool)

/* ========================================================================= */
/** @brief this definition is intended to identify copy functions, these 
 * functions return copy of objects but with independent storage space, there 
 * are two versions, one that require a memory pool from where to look for 
 * memory, and another where we don't care about that.... the place from where 
 * the memory was asked for depend on the function, se the function definition 
 * for details. Note that if the is no more memory available the function 
 * should call exit(1).
 */
typedef void *(*EGcopyMP_f) (void *p,
														 EGmemPool_t * mem);

/* ========================================================================= */
/** @brief NULL copy function */
#define nullCopyMP ((EGcopyMP_f)0)

/* ========================================================================= */
/** @brief this function offer a default memory resizer strategy, this function 
 * return 1 << (i+10) */
size_t EGmemPoolNewSize (size_t);

/* ========================================================================= */
/** @brief If profiling is enabled, initiaslize the profiling information of the
 * memory pool.
 * @param lpool memory pool to initialize profiling information. */
# if ( (__EG_MEMPOOL_PROFILE__ > 0) && (DEBUG > 0 ) && (__EG_MEMPOOL_REDUCE_TO_MALLOC__==0) )
#define __EGmemPoolInitProfile(lpool) {\
	(lpool)->memAllocationFile = __FILE__;\
	(lpool)->memAllocationLine = __LINE__;\
	(lpool)->totalMem = (lpool)->freeArraySize;\
	if((lpool)->manageLimit) (lpool)->maxUsedMem = EGsMalloc(size_t,((lpool)->manageLimit >> EG_MEM_ALIGNMENT_SHIFT)+1);\
	else (lpool)->maxUsedMem = 0;\
	if((lpool)->manageLimit) (lpool)->curUsedMem = EGsMalloc(size_t,((lpool)->manageLimit >> EG_MEM_ALIGNMENT_SHIFT)+1);\
	else (lpool)->curUsedMem = 0;}
#else
#define __EGmemPoolInitProfile(lpool)
#endif

/* ========================================================================= */
/** @brief Initialize a memory pool structure with the given parameters.
 * @param pool memory pool to initialize.
 * @param manage_limit maximum size of blocks to pool.
 * @param resize_fn function that dictaminates the grow of the pool.
 * @param first_size initial size of the memory allocated in the pool.
 * @return zero on success, non-zero otherwise.
 * */
#define EGmemPoolInit(pool,manage_limit,resize_fn,first_size) ({\
	EGmemPool_t *const __EGmp = (pool);\
	(__EGmp)->manageLimit = (manage_limit);\
	(__EGmp)->newsize = (resize_fn);\
	if((__EGmp)->manageLimit) (__EGmp)->freedMemory = EGsMalloc(void*,((__EGmp)->manageLimit >> EG_MEM_ALIGNMENT_SHIFT)+1);\
	else (__EGmp)->freedMemory = 0;\
	(__EGmp)->freeArraySize = (first_size);\
	if((signed)((__EGmp)->freeArraySize) < sysconf(_SC_PAGESIZE)) (__EGmp)->freeArraySize = sysconf(_SC_PAGESIZE);\
	(__EGmp)->MemoryListSize = 1;\
	(__EGmp)->MemoryList = EGsMalloc(void*,1);\
	(__EGmp)->freeArrayHead = (__EGmp)->MemoryList[0] = (void*)EGmalloc((__EGmp)->freeArraySize);\
	__EGmemPoolInitProfile(__EGmp);})

/* ========================================================================= */
/** @brief Free internal memory non-allocated by the user in this structure.
 * @param pool memory pool to be cleared
 * @{ */
# if ( (__EG_MEMPOOL_PROFILE__ > 0) && (DEBUG > 0 ) && (__EG_MEMPOOL_REDUCE_TO_MALLOC__==0) )
#define __EGmemPoolClearProfile(pool) {\
	unsigned int __EGmp_i;\
	fprintf (stderr, "Memory Profile for EGmemPool %p (%s:%d)\n",\
					 (void *) (pool), (pool)->memAllocationFile,\
					 (pool)->memAllocationLine);\
	fprintf (stderr, "\tInternal Memory    : %u \n",\
					 (unsigned) ((pool)->totalMem));\
	fprintf (stderr, "\tUsed Memory        : %u \n",\
					 (unsigned) ((pool)->allocMem));\
	fprintf (stderr, "\tNon-Managed Memory : %u \n",\
					 (unsigned) ((pool)->noManageMem));\
	fprintf (stderr, "\tMaximum Alloc Block: %u (%s:%d)\n",\
					 (unsigned) ((pool)->maxBlock), (pool)->maxBlockFile,\
					 (unsigned) ((pool)->maxBlockLine));\
	fprintf (stderr, "\tNull Allocs        : %u \n",\
					 (unsigned) ((pool)->nNullAlloc));\
	if ((pool)->nNullAlloc)\
		fprintf (stderr, "\tLast Null Alloc (%s:%d)\n",\
						 (pool)->nullAllocFile, (pool)->nullAllocLine);\
	fprintf (stderr, "\tMax Memory Used    :\n");\
	for (__EGmp_i = sizeof (void *); __EGmp_i < (pool)->manageLimit; __EGmp_i += EG_MEM_ALIGNMENT)\
	{\
		if ((pool)->maxUsedMem[__EGmp_i >> EG_MEM_ALIGNMENT_SHIFT])\
			fprintf (stderr, "\tSize %3u           : %u\n", __EGmp_i,\
							 (pool)->maxUsedMem[__EGmp_i >> EG_MEM_ALIGNMENT_SHIFT] / (__EGmp_i));\
	}\
	fprintf (stderr, "\tBig Blocks:\n");\
	for (__EGmp_i = 0; __EGmp_i < __EG_MEMPOOL_N_BIG_CHUNKS__; __EGmp_i++)\
	{\
		if ((pool)->nBigBlocks[__EGmp_i])\
		{\
			fprintf (stderr, "\tSize %u: n_allloc %u s_alloc %u\n",\
							 (__EGmp_i + 1U) << __EG_MEMPOOL_MEM_SHIFT__,\
							 (pool)->nBigBlocks[__EGmp_i], (pool)->cBigBlocks[__EGmp_i]);\
		}\
	}\
	EGfree ((pool)->maxUsedMem);\
	EGfree ((pool)->curUsedMem);}

#else
#define __EGmemPoolClearProfile(pool)
#endif

/* ========================================================================= */
#define __EGmemPoolClear(pool) ({\
	EGmemPool_t*const __EGmp = (pool);\
	__EGmemPoolClearProfile(__EGmp);\
	while(__EGmp->MemoryListSize--) EGfree(__EGmp->MemoryList[__EGmp->MemoryListSize]);\
	EGfree(__EGmp->MemoryList);\
	if(__EGmp->manageLimit) EGfree(__EGmp->freedMemory);})

/* ========================================================================= */
#define EGmemPoolClear(pool) if(pool) __EGmemPoolClear(pool)
/** @} */

/* ========================================================================= */
/** @} */
/* end of eg_mempool.h */
#endif
