/* EGlib "Efficient General Library" provides some basic structures and
 * algorithms commons in many optimization algorithms.
 *
 * Copyright (C) 2005 Daniel Espinoza and Marcos Goycoolea.
 * 
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but 
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY 
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public 
 * License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA 
 * */
#ifndef __EG_LIST_INT_H__
#define __EG_LIST_INT_H__
#include "eg_macros.h"
#include "eg_mempool.h"

/* ========================================================================= */
/* Implementation of doubled linked list with posible satelite data inside, we
 * assume that the node structure used at least contain the fields next, prev
 * and this; while the list structure itself must contain at least the fields
 * begin and end.
 *
 * - 2003-07-02
 * 					- Add support for attach/unattach
 * - 2003-06-10 
 * 					- Version 0.0.1
 * */
/* ========================================================================= */

/* ========================================================================= */
/* this macro return a pointer to listNode (not initialized), 
 * 
 * mempool : is a pointer to a memory pool manager.
 * type    : is the structure type of the node of this list.
 * */
#define __EGlistIntNewListNode(mypool,type) \
	((type*)(EGmemPoolMalloc(mypool,sizeof(type))))

/* ========================================================================= */
/* this function return to the pool an unused EGlistNode, this function _MUST_ 
 * be used for every EGlistNode allocated by 'EGnewListNode'.
 *
 * datafree : is a pointer to a function (void*)(void*) that act as a 'free'
 * 						function for the internal data of the list.
 * node : is a pointer to the actual node to be freed.
 * mypool : is a pointer to the memory manager of the node memory. 
 * type : is the type of the node of the list. 
 * */
#define __EGlistIntFreeListNode(datafree,node,mypool,type) {\
	/* if we have a data free function, we use it */\
	if(datafree) datafree(((type*)(node))->this);\
	/* then we free the node */\
	EGmemPoolFree(node,sizeof(type),mypool);}

/* ========================================================================= */
/* this function return to the pool an unused EGlistNode, this function _MUST_ 
 * be used for every EGlistNode allocated by 'EGnewListNode'.
 * 
 * datafree : is a pointer to a function (void*)(void*,EGmemPool_t*) that act 
 * 						as a 'free' function for the internal data of the list, but that
 * 						return its memory to the 'datamem' memory manager.
 * datamem : is a pointer to the memory manager of the data memory (it might be
 * 					the same pointer as  mypool ).
 * node : is a pointer to the actual node to be freed.
 * mypool : is a pointer to the memory manager of the node memory. 
 * type : is the type of the node of the list. 
 * */
#define __EGlistIntFreeListNodeMP(datafree,datamem,node,mypool,type) {\
	/* if we have a data free function, we use it */\
	if(datafree) datafree(((type*)(node))->this,datamem);\
	/* then we free the node */\
	EGmemPoolFree(node,sizeof(type),mypool);}

/* ========================================================================= */
/* this function return a pointer to an unitialized list structure 
 *
 * mypool : pointer to a memory manager from where we will ask the memory to be
 * 					used as node.
 * type : is the type of the node to be used.
 * */
#define __EGlistIntNewList(mypool,type) \
	((type*)(EGmemPoolMalloc(mypool,sizeof(type))))

/* ========================================================================= */
/* this function return to the pool an unused EGlist, this function _MUST_ 
 * be used for every EGlist allocated by 'EGnewList' 
 *
 * list : pointer to an actual (initialized) list.
 * mempool : pointer to the memory manager that control the list internal
 * 					 memory.
 * ltype : type of the list.
 * ntype : type of the nodes of the list.
 * */
#define __EGlistIntFreeList(list,mempool,ltype,ntype) {\
	/* local variables */\
	ntype *__nextEGlist__,*__curEGlist__;\
	/* we free starting from the head */\
	__curEGlist__ = ((ltype*)(list))->begin;\
	while(__curEGlist__){\
		__nextEGlist__ = __curEGlist__->next;\
		EGmemPoolFree(__curEGlist__,sizeof(ntype),mempool);\
		__curEGlist__ = __nextEGlist__;}\
	/* at this point all node list are freed */\
	EGmemPoolFree(list,sizeof(ltype),mempool);}

/* ========================================================================= */
/* given an initialized list this will insert a new node at the tail of the */
/* list with 'this' pointed by void* 
 *
 * list : pointer to an initialized list.
 * data : pointer to the satelite data of the node to be inserted.
 * listInit: macro called once (without parameters) that actualize
 * 					 any internal list data.
 * mempool : pointer to a memory manager from where we obtain the new node
 * 					 memory.
 * ntype : type of the node of the list.
 * */
#define __EGlistIntPushBack(list,data,listInit,mempool,ntype) {\
	/* if the list is not empty, we just put it at the tail */\
	if((list)->begin){\
		(list)->end->next = __EGlistIntNewListNode(mempool,ntype);\
		(list)->end->next->prev = (list)->end;\
		(list)->end = (list)->end->next;\
		(list)->end->next = 0;\
		}\
	/* otherwise the list is empty, we set both head and tail */\
	else{\
		(list)->end = (list)->begin = __EGlistIntNewListNode(mempool,ntype);\
		(list)->end->next = (list)->end->prev = 0;}\
	/* save the data */\
	(list)->end->this = data;\
	listInit;}

/* ========================================================================= */
/* given an initialized list this will insert a new node before the given node 
 * in the list.
 *
 * list : pointer to an initialized list
 * node : pointer to the node that will succeed the new node.
 * data : pointer to the satelite data of the node to be inserted
 * listInit: macro called once (without parameters) that actualize
 * 					 any internal list data.
 * mempool : pointer to a memory manager from where we obtain the new node
 * 					 memory.
 * ntype : type of the node of the list.
 * */
#define __EGlistIntInsertBefore(list,node,data,listInit,mempool,ntype) {\
	ntype* __n_tmp = __EGlistIntNewListNode(mempool,ntype);\
	__n_tmp->this = data;\
	__n_tmp->next = (node);\
	__n_tmp->prev = (node)->prev;\
	(node)->prev = __n_tmp;\
	if((node)==(list)->begin)\
	{\
		(list)->begin = __n_tmp;\
	}\
	else\
		__n_tmp->prev->next = __n_tmp;\
	listInit;}

/* ========================================================================= */
/* given an initialized list this will insert a new node after the given node in
 * the list.
 *
 * list : pointer to an initialized list
 * node : pointer to the node that will precede the new node.
 * data : pointer to the satelite data of the node to be inserted
 * listInit: macro called once (without parameters) that actualize
 * 					 any internal list data.
 * mempool : pointer to a memory manager from where we obtain the new node
 * 					 memory.
 * ntype : type of the node of the list.
 * */
#define __EGlistIntInsertAfter(list,node,data,listInit,mempool,ntype) {\
	ntype* __n_tmp = __EGlistIntNewListNode(mempool,ntype);\
	__n_tmp->this = data;\
	__n_tmp->prev = (node);\
	__n_tmp->next = (node)->next;\
	(node)->next = __n_tmp;\
	if((node)==(list)->end)\
	{\
		(list)->end = __n_tmp;\
	}\
	else\
		__n_tmp->next->prev = __n_tmp;\
	listInit;}

/* ========================================================================= */
/* given an initialized list this will insert a new node at the head of the */
/* list with 'this' pointed by void*
 *
 * list : pointer to an initialized list.
 * data : pointer to the satelite data of the node to be inserted.
 * listInit: macro called once (without parameters) that actualize
 * 					 any internal list data.
 * mempool : pointer to a memory manager from where we obtain the new node
 * 					 memory.
 * ntype : type of the node of the list.
 * */
#define __EGlistIntPushHead(list,data,listInit,mempool,ntype) {\
	/* if the list is not empty, we just put it at the beginning */\
	if((list)->begin){\
		(list)->begin->prev = __EGlistIntNewListNode(mempool,ntype);\
		(list)->begin->prev->next = (list)->begin;\
		(list)->begin = (list)->begin->prev;\
		(list)->begin->prev = 0;}\
	/* otherwise the list is empty, we set both head and tail */\
	else{\
		(list)->end = (list)->begin = __EGlistIntNewListNode(mempool,ntype);\
		(list)->end->next = (list)->end->prev = 0;}\
	/* save the data */\
	(list)->begin->this = data;\
	listInit;}

/* ========================================================================= */
/* this is to erase an element pointed by an EGlistNode from the list where 
 * this node bellongs note that this function __WILL_NOT__ check if the node 
 * really is in the list, that is up to you, use with precuation, 
 *
 * list : pointer to the list that contain the node.
 * node : pointer to the node to be erased.
 * datafree : pointer to a function (void*)(void*,EGmemPool_t*) that free the
 * 						satelite data to the memory pool datamem.
 * listInit : macro that update internal list data.
 * mempool : pointer to the memory manager of the list internal structure.
 * datamem : pointer to the memory manager of the satelite data.
 * ntype : type of the node.
 * */
#define __EGlistIntEraseMP(list,node,datafree,listInit,mempool,datamem,ntype) {\
	/* case by case erasing first, when the list has only one element in it */\
	if(( (node) == (list)->begin )&&( (node) == (list)->end )){\
		(list)->begin = (list)->end = 0;}\
	/* if the node is the begin node */\
	else if( (node) == (list)->begin ){\
		(list)->begin = (node)->next;\
		(list)->begin->prev = 0;}\
	/* if the node is the end node */\
	else if( (node) == (list)->end ){\
		(list)->end = (node)->prev;\
		(list)->end->next = 0;}\
	/* this is the normal case, where the node is in the midle of the list */\
	else{\
		(node)->next->prev = (node)->prev;\
		(node)->prev->next = (node)->next;}\
	/* now we free the node and update the list */\
	__EGlistIntFreeListNodeMP(datafree,datamem,node,mempool,ntype);\
	listInit;}

/* ========================================================================= */
/* this is to erase an element pointed by an EGlistNode from the list where 
 * this node bellongs note that this function __WILL_NOT__ check if the node 
 * really is in the list, that is up to you, use with precuation, also, once 
 * the function return, the listNode pointer will be NULL. 
 *
 * list : pointer to the list that contain the node.
 * node : pointer to the node to be erased.
 * datafree : pointer to a function (void*)(void*) that free the satelite 
 * 						data.
 * listInit : macro that update internal list data.
 * mempool : pointer to the memory manager of the list internal structure.
 * datamem : pointer to the memory manager of the satelite data.
 * ntype : type of the node.
 * */
#define __EGlistIntErase(list,node,datafree,listInit,mempool,type) {\
	/* case by case erasing first, when the list has only one element in it */\
	if(( (node) == (list)->begin )&&( (node) == (list)->end )){\
		(list)->begin = (list)->end = 0;}\
	/* if the node is the begin node */\
	else if( (node) == (list)->begin ){\
		(list)->begin = (node)->next;\
		(list)->begin->prev = 0;}\
	/* if the node is the end node */\
	else if( (node) == (list)->end ){\
		(list)->end = (node)->prev;\
		(list)->end->next = 0;}\
	/* this is the normal case, where the node is in the midle of the list */\
	else{\
		(node)->next->prev = (node)->prev;\
		(node)->prev->next = (node)->next;}\
	/* now we free the node and update the list */\
	__EGlistIntFreeListNode(datafree,node,mempool,type);\
	listInit;}

/* ========================================================================= */
/* this function left the list empty
 *
 * list : pointer to an initialized list.
 * datafree : pointer to a function (void*)(void*) that free the satelite data
 * 						of the nodes.
 * listInit : macro called once to reset any internal value of the list
 * 						structure.
 * mempool : pointer to the memory manager of the internal list structures.
 * ntype : type of the node.
 * */
#define __EGlistIntClear(list,datafree,listInit,mempool,ntype) {\
	/* local variables */\
	ntype *__nextEGlist__,*__curEGlist__;\
	/* we free starting from the head */\
	__curEGlist__ = (list)->begin;\
	while(__curEGlist__){\
		__nextEGlist__ = __curEGlist__->next;\
		__EGlistIntFreeListNode(datafree,__curEGlist__,mempool,ntype);\
		__curEGlist__ = __nextEGlist__;}\
	/* at this point all node list are freed */\
	(list)->begin = 0;\
	(list)->end = 0;\
	listInit;}

/* ========================================================================= */
/* this function left the list empty
 *
 * list : pointer to an initialized list.
 * datafree : pointer to a function (void*)(void*,EGmemPool_t*) that free the 
 * 						satelite data	of the nodes to the memory pool datamem.
 * listInit : macro called once to reset any internal value of the list
 * 						structure.
 * mempool : pointer to the memory manager of the internal list structures.
 * mempool : pointer to the memory manager of the satelite data.
 * ntype : type of the node.
 * */
#define __EGlistIntClearMP(list,datafree,listInit,mempool,datamem,ntype) {\
	/* local variables */\
	ntype *__nextEGlist__,*__curEGlist__;\
	/* we free starting from the head */\
	__curEGlist__ = (list)->begin;\
	while(__curEGlist__){\
		__nextEGlist__ = __curEGlist__->next;\
		__EGlistIntFreeListNodeMP(datafree,datamem,__curEGlist__,mempool,ntype);\
		__curEGlist__ = __nextEGlist__;}\
	/* at this point all node list are freed */\
	(list)->begin = 0;\
	(list)->end = 0;\
	listInit;}

/* ========================================================================= */
/* this function look for something in the list, if the search is succesfull 
 * node will be pointing to the node in the list containing mem, otherwise. 
 * node will be zero.
 *
 * list : pointer to an initialized list.
 * node : pointer to a node.
 * mem : pattern to be found in the list.
 * */
#define __EGlistIntFind(list,node,mem) {\
	/* loop through all the list */\
	(node) = (list)->begin;\
	while( (node) && ((node)->this != mem) ) (node)=(node)->next;}

/* ========================================================================= */
/* this function unattach a list_node from a list, and make an update for the
 * internal datas of the list */
#define __EGlistIntUnattach(list,node,listUpdate) {\
	/* case by moving, first, when the list has only one element */\
	if( ((list)->begin == (node)) && ((list)->end == (node)) ){\
		(list)->begin = 0;\
		(list)->end = 0;}\
	/* if the node is at the beginning of the list */\
	else if ( (list)->begin == (node) ){\
		(list)->begin = (node)->next;\
		(list)->begin->prev = 0;}\
	/* if the node is at the end of the list */\
	else if( (list)->end == (node) ){\
		(list)->end = (node)->prev;\
		(list)->end->next = 0;}\
	/* the normal case when node is in the middle of the list */\
	else{\
		(node)->prev->next = (node)->next;\
		(node)->next->prev = (node)->prev;}\
	/* update the source list */\
	listUpdate;}

/* ========================================================================= */
/* this function attach to a list a new node at the end of the list, and make an
 * update for the internal structures */
#define __EGlistIntAttach(list,node,listUpdate)	{\
	/* put the node in the list */\
	if((list)->begin){\
		(list)->end->next = (node);\
		(node)->prev = (list)->end;\
		(list)->end = (node);\
		(node)->next = 0;}\
	else{\
		(list)->end = (list)->begin = (node);\
		(node)->next = (node)->prev = 0;}\
	/* update the list */\
	listUpdate;}

/* ========================================================================= */
/* this function moves a node from one list to another list, and that the 
 * pointer to the listNode to be valid a belong to the source list. The node is
 * left at the end of the dlist.
 *
 * slist : pointer to an initialized list that contain the node 'node'.
 * dlist : pointer to an initialized list where we want to put the node 'node',
 * 				 it can be the same as slist.
 * node : pointer to a node of slist.
 * slistInit : macro that update the slist other fields (if any).
 * dlistInit : macro that update the dlist other fields (if any).
 * */
#define __EGlistIntMoveNode(slist,dlist,node,slistInit,dlistInit) {\
	/* case by moving, first, when the list has only one element */\
	if( ((slist)->begin == (node)) && ((slist)->end == (node)) ){\
		(slist)->begin = 0;\
		(slist)->end = 0;}\
	/* if the node is at the beginning of the list */\
	else if ( (slist)->begin == (node) ){\
		(slist)->begin = (node)->next;\
		(slist)->begin->prev = 0;}\
	/* if the node is at the end of the list */\
	else if( (slist)->end == (node) ){\
		(slist)->end = (node)->prev;\
		(slist)->end->next = 0;}\
	/* the normal case when node is in the middle of the list */\
	else{\
		(node)->prev->next = (node)->next;\
		(node)->next->prev = (node)->prev;}\
	/* update the source list */\
	slistInit;\
	/* put the node in the dlist list */\
	if((dlist)->begin){\
		(dlist)->end->next = (node);\
		(node)->prev = (dlist)->end;\
		(dlist)->end = (node);\
		(node)->next = 0;}\
	else{\
		(dlist)->end = (dlist)->begin = (node);\
		(node)->next = (node)->prev = 0;}\
	/* update the destine list */\
	dlistInit;}

/* ========================================================================= */
/* this function ties together two list, moving the whole second list to the
 * tail of the first list. the destination list initialization command will be
 * executed BEFORE the origin list comand */
#define __EGlistIntLink(dlist,olist,dlistInc,olistInc) {\
	if((olist)->begin){\
		if((dlist)->begin){\
			(dlist)->end->next=(olist)->begin;\
			(olist)->begin->prev=(dlist)->end;\
			(dlist)->end = (olist)->end;\
			(olist)->begin=(olist)->end=0;\
			}\
		else{\
			(dlist)->begin=(olist)->begin;\
			(dlist)->end=(olist)->end;\
			(olist)->begin=(olist)->end=0;\
			}\
		dlistInc;\
		olistInc;\
		}\
	}
/* end eg_listInt.h */
#endif
