/* EGlib "Efficient General Library" provides some basic structures and
 * algorithms commons in many optimization algorithms.
 *
 * Copyright (C) 2005 Daniel Espinoza and Marcos Goycoolea.
 * 
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but 
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY 
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public 
 * License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA 
 * */
#ifndef __EG_LIST_H__
#define __EG_LIST_H__
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include "eg_macros.h"
#include "eg_mempool.h"
#include "eg_listint.h"

/* ========================================================================= */
/** @defgroup EGlist EGlist
 *
 * Here are deffined the basic interface for a double linked list with external
 * data asociated to each node in the list.
 *
 * @par History:
 * Revision 0.0.2
 * - 2003-07-02
 * 						- Add attach/unattach interface.
 * - 2003-06-11 
 * 						- Modified to use the general list code eg_listint.h
 * 						- Add inline support for most of the functions.
 * - 2003-05-22 
 * 						- Modified EGfreeListNode and others to handle nullFree correctly
 * - 2003-05-05 
 * 						- (Marcos) Added the MP (memPool) versions of EGlistErase, 
 * 							EGlistClear, and EGfreeListNode. These versions allow 
 * 							the destructor passed to the function to use a memory 
 * 							pool manager as well.
 * 						- Added function EGlistMoveNode, to move a node (with its
 * 							internal data) from one list to another different list.
 * Revision 0.0.1
 * - 2003-04-10
 *
 * @note In general, the functions described bellow don't perform consistency
 * checks. It is asumed that the user does know what is he doing.
 *
 * @note If you want to have some debugging control try changing the debug level
 * at compile time, and lowering the debug level asociated to the list function
 * as defined in eg_configure.h.
 *
 * */
/*@{*/
/** @file 
 * */
/* ========================================================================= */

/* ========================================================================= */
/** 
 * @brief List Node Structure.
 * @par Description:
 * This structure is to store a general node of the list. It is composed by
 * three members, one to store external or satelite data, and the others to to
 * point to the next and previous structures in the list.  */
typedef struct EGlistNode_t__
{
	void *this;									/**< Pointer to the external/satelite data */
	struct EGlistNode_t__ *next;/**< Pointer to the next EGlistNode_t__ structure in the list */
	struct EGlistNode_t__ *prev;/**< Pointer to the previous EGlistNode_t__ structure in the list */
}
EGlistNode_t;

/* ========================================================================= */
/** 
 * @brief Given a list and a node_iterator return the next node iterator of the
 * list but in a circular way.
 * @param list EGlist_t* pointer to the list.
 * @param node EGlistNode_t* pointer to the current iterator in the list.
 * @return EGlistNode_t* pointer to the next iterator in the (circular) list.
 * @par Description:
 * Given a list and a node_iterator return the next node iterator of the
 * list but in a circular way. thus the result of the next of the tail, is the
 * head of the list.
 * */
#define EGclistGetNextIt(list,node) ((node)->next ? (node)->next : (list)->begin)
#define EGclistRoot(list) ((list)->begin)

/* ========================================================================= */
/**
 * @brief List Basic Structure
 * @par Description:
 * This structure holds a general doubled linked list, it mantains the size of
 * the list (i.e. how many members does the list contain), also has a reference
 * to the memory pool manager from where we alloc memory, and a pointer to the
 * end and the beginning of the list. If the list is empty, then begin and end
 * point to the NULL pointer */
typedef struct
{
	EGlistNode_t *begin;	/**< Pointer to the first element of the list.*/
	EGlistNode_t *end;		/**< Pointer to the last element of the list.*/
	EGmemPool_t *mempool;	/**< Pointer to the local memory pool manager */
	unsigned int size;		/**< Actual size of the list */
}
EGlist_t;

/* ========================================================================= */
/** 
 * @brief alloc a new initialized list.
 * @param mypool Memory manager from where all the list structures
 * will be allocated.
 * @return EGlist_t* A pointer to an initialized new list structure.
 * @par Description:
 * This function return a pointer to an initialized EGlist alloc'ed by
 * mypool, also, all internal memory alloc'ed for the list will be
 * generated from the pool. */
extern inline EGlist_t *EGnewList (EGmemPool_t * mypool)
{

	/* local vars */
	EGlist_t *res;

	/* looking for memory */
	res = __EGlistIntNewList (mypool, EGlist_t);

	/* initializate answer */
	res->mempool = mypool;
	res->begin = 0;
	res->end = 0;
	res->size = 0;

	/* ending */
	return res;
}

/* ========================================================================= */
/**
 * @brief Return the list's memory to the memory pool.
 * @param mylist Pointer to an initialized list to be returned to the mempool.
 * @par Description:
 * This function return to the construction pool all memory used by the list, 
 * this function _MUST_ be used for every EGlist allocated by 'EGnewList'.
 * Note that the memory bointed by the internall data will not be freed, to 
 * free internal memory, first use EGlistClear(). */
extern inline void EGfreeList (void *mylist)
{

	/* call the internal function */
	__EGlistIntFreeList (mylist, ((EGlist_t *) (mylist))->mempool, EGlist_t,
											 EGlistNode_t);
	/* ending */
	return;
}

/* ========================================================================= */
/**
 * @brief Return the list node memory to the memory pool. 
 * @param localNode Node to be freed (returned to the memory pool).
 * @param mypool Pointer of the memory manager where we will return the
 * memory of the node.
 * @par Description:
 * This function return to the pool an unused EGlistNode, this function _MUST_
 * be used for every EGlistNode allocated by 'EGnewListNode', you should not use
 * this function, if you use it, is at your own risk. */
extern inline void EGfreeListNode (void *localNode,
																	 EGmemPool_t * mypool)
{

	/* call the internal function */
	__EGlistIntFreeListNode (nullFree, localNode, mypool, EGlistNode_t);
	/* ending */
	return;
}

/* ========================================================================= */
/** 
 * @brief Eliminate one node form the list.
 * @param mylist List from where we will eliminate the node 'mynode'.
 * @param mynode Node to be eliminate from the list.
 * @param dataFree Pointer to a function that free the internal data memory 
 * stored in the node, it can be NUNLL, in whose case it is not executed and the
 * the internal data is not touched at all.
 * @return zero on success, nonzero otherwise.
 * @par Description:
 * This function erase an element pointed by an EGlistNode_t from the list where
 * this node bellongs.
 * @warning Note that this function __WILL_NOT__ check if the node 
 * really is in the list, that is up to you, use with precuation. */
extern inline int EGlistErase (EGlist_t * mylist,
															 EGlistNode_t * mynode,
															 EGfree_f dataFree)
{

	/* call the internal function */
	__EGlistIntErase (mylist, mynode, dataFree, mylist->size--, mylist->mempool,
										EGlistNode_t);

	/* ending */
	return 0;
}

/* ========================================================================= */
/** 
 * @brief given an initialized list, left it at its initial state (just after
 * EGnewList).
 * @param mylist Pointer to an initialized list to be cleared.
 * @param dataFree Pointer to a function that free the internal data stored in
 * the nodes of the list. It can be the null function (nullFree), in which case
 * we don't touch the internal data at all.
 * @return zero on success, nonzero otherwise.
 * @par Description:
 * This function left the list empty, and call 'dataFree' to liberate the
 * external data stored in the node of the list */
extern inline int EGlistClear (EGlist_t * mylist,
															 EGfree_f dataFree)
{

	/* call the internal function */
	__EGlistIntClear (mylist, dataFree, mylist->size =
										0, mylist->mempool, EGlistNode_t);

	/* ending */
	return 0;
}

/* ========================================================================= */
/** 
 * @brief Erase a node from the list and retunr internal data to a specified
 * memory pool.
 * @param mylist Pointer to the list from where we will eliminate a node.
 * @param mynode Pointer to the node to be eliminated from the specified list.
 * @param dataFree Pointer to a function that free the internal (or satelite)
 * data of the node to the memory pool specified. This function can be
 * nullFreeMP, in which case we don't touch the internal data at all.
 * @param datamem Pointer to the memory pool where we will return the memory
 * used by the internal data, it is passed as the second argument for dataFree.
 * @return zero on success, nonzero otherwise.
 * @par Description:
 * this is to erase an element pointed by an EGlistNode from the list where
 * this node bellongs. Note that this function __WILL_NOT__ check if the node 
 * really is in the list, that is up to you, use with precuation. The list
 * structures will be liberate to the list memory manager, and the internal data
 * will be liberated to the 'datamem' memory pool. */
extern inline int EGlistEraseMP (EGlist_t * mylist,
																 EGlistNode_t * mynode,
																 EGfreeMP_f dataFree,
																 EGmemPool_t * datamem)
{

	/* call the internal function */
	__EGlistIntEraseMP (mylist, mynode, dataFree, mylist->size--,
											mylist->mempool, datamem, EGlistNode_t);

	/* ending */
	return 0;
}

/* ========================================================================= */
/**
 * @brief Free one node and return the internal data to the specified memory 
 * pool. 
 * @param localNode Node to be freed, the memory used by it will be returned to
 * the 'structmem' memory pool.
 * @param dataFree Pointer to the function that free the internal data of the
 * node to the 'datamem' memory pool.
 * @param structmem Pointer to the memory pool where the list structure memory
 * will be returned.
 * @param datamem Pointer to the memory pool where the internal data memory will
 * be returned.
 * @par Description:
 * This function return to the pool an unused EGlistNode, this function _MUST_
 * be used for every EGlistNode allocated by 'EGnewListNode'. The list
 * structures will be liberate to the list memory manager, and the internal data
 * will be liberated to the 'datamem' memory pool.*/
extern inline void EGfreeListNodeMP (void *localNode,
																		 EGfreeMP_f dataFree,
																		 EGmemPool_t * structmem,
																		 EGmemPool_t * datamem)
{

	/* call the internal function */
	__EGlistIntFreeListNodeMP (dataFree, datamem, localNode, structmem,
														 EGlistNode_t);
	/* ending */
	return;
}

/* ========================================================================= */
/**
 * @brief Clean out the list and leave it in its original state, the internal
 * memory is returned to the provided memory pool.
 * @param mylist Pointer to the list that is going to be cleaned.
 * @param dataFree Pointer to the function that free the internal data
 * structured to the provided memory pool.
 * @param datamem Pointer to the memory manager where we will return the
 * internal memory.
 * @par Description:
 * This function left the list empty The list
 * structures will be liberate to the list memory manager, and the internal data
 * will be liberated to the 'datamem' memory pool.*/
extern inline int EGlistClearMP (EGlist_t * mylist,
																 EGfreeMP_f dataFree,
																 EGmemPool_t * datamem)
{

	/* call the internal function */
	__EGlistIntClearMP (mylist, dataFree, mylist->size = 0, mylist->mempool,
											datamem, EGlistNode_t);

	/* ending */
	return 0;
}

/* ========================================================================= */
/** 
 * @brief move anode from one list to the end of another.
 * @param sourceList Pointer to an initialized list from where we will take out
 * a node.
 * @param destList Pointer to an initialized list where we will add the node.
 * @param localNode Pointer to the node to be moved.
 * @par Description: 
 * This function moves a node from one list to another list, both list don't 
 * need to be differents, and that the pointer to the listNode to be
 * valid an belong to the source list. The node will be placed at the end of 
 * the destine list. */
extern inline int EGlistMoveNode (EGlist_t * sourceList,
																	EGlist_t * destList,
																	EGlistNode_t * localNode)
{

	/* call the internal function */
	__EGlistIntMoveNode (sourceList, destList, localNode, sourceList->size--,
											 destList->size++);

	/* ending */
	return 0;
}

/* ========================================================================= */
/**
 * @brief Add to the (end of the) list a new node pointing to the provided data.
 * @param mylist Pointer to an initialized list where we will add the user data.
 * @param data Pointer to the user data to be stored in the list.
 * @par Description:
 * @return zero in success, nonzero otherwise.
 * Given an initialized list this will insert a new node at the end of the 
 * list with 'this' pointing to 'data', note that we will not copy the data, but
 * rather mantain a pointer to the user data. */
extern inline int EGlistPushBack (EGlist_t * mylist,
																	void *data)
{

	/* call the internal function */
	__EGlistIntPushBack (mylist, data, mylist->size++, mylist->mempool,
											 EGlistNode_t);

	/* ending */
	return 0;
}

/* ========================================================================= */
/**
 * @brief Add to the (begining of the) list a new node pointing to the provided 
 * data.
 * @param mylist Pointer to an initialized list where we will add the user data.
 * @param data Pointer to the user data to be stored in the list.
 * @par Description:
 * @return zero in success, nonzero otherwise.
 * Given an initialized list this will insert a new node at the begin of the 
 * list with 'this' pointing to 'data', note that we will not copy the data, but
 * rather mantain a pointer to the user data. */
extern inline int EGlistPushHead (EGlist_t * mylist,
																	void *data)
{

	/* call the internal function */
	__EGlistIntPushHead (mylist, data, mylist->size++, mylist->mempool,
											 EGlistNode_t);

	/* ending */
	return 0;
}

/* ========================================================================= */
/**
 * @brief Add to the list a new node pointing to the provided 
 * data that is before the given node in the list.
 * @param mylist Pointer to an initialized list where we will add the user data.
 * @param node Pointer to the node that will precede the inserted node.
 * @param data Pointer to the user data to be stored in the list.
 * @par Description:
 * @return zero in success, nonzero otherwise.
 * Given an initialized list this will insert a new node at the begin of the 
 * list with 'this' pointing to 'data', note that we will not copy the data, but
 * rather mantain a pointer to the user data. */
extern inline int EGlistInsertBefore (EGlist_t * mylist,
																			EGlistNode_t * node,
																			void *data)
{

	/* call the internal function */
	__EGlistIntInsertBefore (mylist, node, data, mylist->size++, mylist->mempool,
													 EGlistNode_t);

	/* ending */
	return 0;
}

/* ========================================================================= */
/**
 * @brief Add to the list a new node pointing to the provided 
 * data that is after the given node in the list.
 * @param mylist Pointer to an initialized list where we will add the user data.
 * @param node Pointer to the node that will precede the inserted node.
 * @param data Pointer to the user data to be stored in the list.
 * @par Description:
 * @return zero in success, nonzero otherwise.
 * Given an initialized list this will insert a new node at the begin of the 
 * list with 'this' pointing to 'data', note that we will not copy the data, but
 * rather mantain a pointer to the user data. */
extern inline int EGlistInsertAfter (EGlist_t * mylist,
																		 EGlistNode_t * node,
																		 void *data)
{

	/* call the internal function */
	__EGlistIntInsertAfter (mylist, node, data, mylist->size++, mylist->mempool,
													EGlistNode_t);

	/* ending */
	return 0;
}

/* ========================================================================= */
/**
 * @brief unattach a node from the list, without erasing it 
 * @param list List from where we will remove the node.
 * @param eg_node node to be eliminated from the list.
 * @return zero on success non zero otherwise.
 * @par Description:
 * This function takes a list and a node and eliminate the node
 * from the list without erasing it. */
extern inline int EGlistUnattach (EGlist_t * list,
																	EGlistNode_t * eg_node)
{
	/* call the internal function */
	__EGlistIntUnattach (list, eg_node, list->size--);

	/* ending */
	return 0;
}

/* ========================================================================= */
/**
 * @brief attach a node to the list at the end of the list 
 * @param list List where we will add the node.
 * @param eg_node node to be added to the tail of the list.
 * @return zero on success, non zero otherwise.
 * @par Description:
 * Given a list and an initialized node, it put at the end of the list the 
 * node.*/
extern inline int EGlistAttach (EGlist_t * list,
																EGlistNode_t * eg_node)
{
	/* call the internal function */
	__EGlistIntAttach (list, eg_node, list->size++);

	/* ending */
	return 0;
}

/* ========================================================================= */
/**
 * @brief Look for a user data pointer in the provided list.
 * @param mylist Pointer to an initialized list where we will look for the user
 * data pointer.
 * @param mem Pointer to be looked for in the list.
 * @return pointer to the EGlistNode_t that contains the desired element in the
 * list, otherwise it returns zero.
 * @par Description:
 * This function look for something in the list, return the pointer to the
 * EGlistNode_t* if the search is succesfull, zero otherwise. If it find the 
 * memory address, it store in the node the pointer to the node containing that
 * address */
extern inline EGlistNode_t *EGlistFind (EGlist_t * mylist,
																				const void *mem)
{
	/* local variables */
	EGlistNode_t *mynode;

	/* call the internal function */
	__EGlistIntFind (mylist, mynode, mem);

	/* once we get out that loop 
	 * we check if we have the answer and return */
	return mynode;
}

/* ========================================================================= */
/**
 * @brief Create a copy of the given list using as memory pool the memory pool
 * of the given list.
 * @param source Pointer to the list to be copied, note that this copier don't
 * create copies of the pointed elements, but just copy the pointers to them in
 * the new list.
 * @return Pointer to the copy list.
 * @par Description:
 * Given a list, it creates a copy of such list but does not create copies of
 * the 'this' fields, but just copy its values. The memory pool used is the same
 * as the memory pool of the source list. The function complies with the
 * EGcopy_f prototype so it can be called in other copiers. */
extern inline void *EGlistCopy (void *source)
{
	/* local variables */
	EGlist_t *src = (EGlist_t *) source,
	 *dst;
	if (!src)
		return 0;
	EGlistNode_t *it;

	dst = EGnewList (src->mempool);
	for (it = src->begin; it; it = it->next)
		EGlistPushBack (dst, it->this);

	/* ending */
	return dst;
}

/* ========================================================================= */
/**
 * @brief Move the contents of the second list into the tail of the first list.
 * @param dlist list where all elements will be put together.
 * @param olist list from where elements will be taken out.
 * @return zero on success and nonzero if an error occurs.
 * @par Description:
 * Given a destination list (dlist) and an origin list (olist) this function
 * moves (without altering the original ordering) the elements of olist to the
 * tail of dlist, leaving olist as an empty (and valid) list.
 * */
extern inline int EGlistLink (EGlist_t * dlist,
															EGlist_t * olist)
{
	__EGlistIntLink (dlist, olist, dlist->size += olist->size, olist->size = 0);
	return 0;
}

/*@}*/
/* ========================================================================= */
/** this is for testing purposes */
void EGlistTest (void);

/* end of eg_list.h */
#endif
