/* EGlib "Efficient General Library" provides some basic structures and
 * algorithms commons in many optimization algorithms.
 *
 * Copyright (C) 2005 Daniel Espinoza and Marcos Goycoolea.
 * 
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but 
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY 
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public 
 * License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA 
 * */
#include "eg_list.h"
/** 
 * @addtogroup EGlist
 * */
/** @file
 * */
/*@{*/

/* ========================================================================= */
/* this function return to the pool an unused EGlistNode, this function _MUST_ */
/* be used for every EGlistNode allocated by 'EGnewListNode' */
inline void EGfreeListNode (void *node,
														EGmemPool_t * mypool)
{

	/* call the internal function */
	__EGlistIntFreeListNode (nullFree, node, mypool, EGlistNode_t);
	/* ending */
	return;
}

/* ========================================================================= */
/* this function return to the pool an unused EGlistNode, this function _MUST_ */
/* be used for every EGlistNode allocated by 'EGnewListNode' */
inline void EGfreeListNodeMP (void *node,
															EGfreeMP_f dataFree,
															EGmemPool_t * mypool,
															EGmemPool_t * nodemem)
{

	/* call the internal function */
	__EGlistIntFreeListNodeMP (dataFree, nodemem, node, mypool, EGlistNode_t);
	/* ending */
	return;
}

/* ========================================================================= */
/* this function return a pointer to an initialized EGlist */
EGlist_t *EGnewList (EGmemPool_t * mypool)
{

	/* local vars */
	EGlist_t *res;

	/* looking for memory */
	res = __EGlistIntNewList (mypool, EGlist_t);

	/* initializate answer */
	res->mempool = mypool;
	res->begin = 0;
	res->end = 0;
	res->size = 0;

	/* ending */
	return res;
}

/* ========================================================================= */
/* this function return to the pool an unused EGlist, this function _MUST_ */
/* be used for every EGlist allocated by 'EGnewList' */
void EGfreeList (void *list)
{

	/* call the internal function */
	__EGlistIntFreeList (list, ((EGlist_t *) (list))->mempool, EGlist_t,
											 EGlistNode_t);
	/* ending */
	return;
}

/* ========================================================================= */
/**
 * @brief Add to the list a new node pointing to the provided 
 * data that is before the given node in the list.
 * @param mylist Pointer to an initialized list where we will add the user data.
 * @param node Pointer to the node that will precede the inserted node.
 * @param data Pointer to the user data to be stored in the list.
 * @par Description:
 * @return zero in success, nonzero otherwise.
 * Given an initialized list this will insert a new node at the begin of the 
 * list with 'this' pointing to 'data', note that we will not copy the data, but
 * rather mantain a pointer to the user data. */
inline int EGlistInsertBefore (EGlist_t * mylist,
															 EGlistNode_t * node,
															 void *data)
{

	/* call the internal function */
	__EGlistIntInsertBefore (mylist, node, data, mylist->size++, mylist->mempool,
													 EGlistNode_t);

	/* ending */
	return 0;
}

/* ========================================================================= */
/**
 * @brief Add to the list a new node pointing to the provided 
 * data that is after the given node in the list.
 * @param mylist Pointer to an initialized list where we will add the user data.
 * @param node Pointer to the node that will precede the inserted node.
 * @param data Pointer to the user data to be stored in the list.
 * @par Description:
 * @return zero in success, nonzero otherwise.
 * Given an initialized list this will insert a new node at the begin of the 
 * list with 'this' pointing to 'data', note that we will not copy the data, but
 * rather mantain a pointer to the user data. */
inline int EGlistInsertAfter (EGlist_t * mylist,
															EGlistNode_t * node,
															void *data)
{

	/* call the internal function */
	__EGlistIntInsertAfter (mylist, node, data, mylist->size++, mylist->mempool,
													EGlistNode_t);

	/* ending */
	return 0;
}


/* ========================================================================= */
/* given an initialized list this will insert a new node at the tail of the */
/* list with 'this' pointed by void* */
int EGlistPushBack (EGlist_t * list,
										void *data)
{

	/* call the internal function */
	__EGlistIntPushBack (list, data, list->size++, list->mempool, EGlistNode_t);

	/* ending */
	return 0;
}

/* ========================================================================= */
/* given an initialized list this will insert a new node at the head of the */
/* list with 'this' pointed by void* */
int EGlistPushHead (EGlist_t * list,
										void *data)
{

	/* call the internal function */
	__EGlistIntPushHead (list, data, list->size++, list->mempool, EGlistNode_t);

	/* ending */
	return 0;
}

/* ========================================================================= */
/* this is to erase an element pointed by an EGlistNode from the list where this node bellongs */
/* note that this function __WILL_NOT__ check if the node really is in the list, that is up to */
/* you, use with precuation, also, once the function return, the listNode pointer will be NULL. */
int EGlistEraseMP (EGlist_t * list,
									 EGlistNode_t * node,
									 EGfreeMP_f dataFree,
									 EGmemPool_t * nodemem)
{

	/* call the internal function */
	__EGlistIntEraseMP (list, node, dataFree, list->size--, list->mempool,
											nodemem, EGlistNode_t);

	/* ending */
	return 0;
}

/* ========================================================================= */
/* this is to erase an element pointed by an EGlistNode from the list where this node bellongs */
/* note that this function __WILL_NOT__ check if the node really is in the list, that is up to */
/* you, use with precuation, also, once the function return, the listNode pointer will be NULL. */
int EGlistErase (EGlist_t * list,
								 EGlistNode_t * node,
								 EGfree_f dataFree)
{

	MESSAGE (200, "Entering node %p list %p", (void *) node, (void *) list);
	/* call the internal function */
	__EGlistIntErase (list, node, dataFree, list->size--, list->mempool,
										EGlistNode_t);

	/* ending */
	return 0;
}

/* ========================================================================= */
/* this function left the list empty */
int EGlistClear (EGlist_t * list,
								 EGfree_f dataFree)
{

	/* call the internal function */
	__EGlistIntClear (list, dataFree, list->size =
										0, list->mempool, EGlistNode_t);

	/* ending */
	return 0;
}

/* ========================================================================= */
/* this function left the list empty */
int EGlistClearMP (EGlist_t * list,
									 EGfreeMP_f dataFree,
									 EGmemPool_t * nodemem)
{

	/* call the internal function */
	__EGlistIntClearMP (list, dataFree, list->size = 0, list->mempool, nodemem,
											EGlistNode_t);

	/* ending */
	return 0;
}

/* ========================================================================= */
/* this function look for something in the list, return 1 
 * if the search is succesfull, zero otherwise. If it 
 * find the memory address, it store in the node the 
 * pojnter to the node containing that address */
EGlistNode_t *EGlistFind (EGlist_t * mylist,
													const void *mem)
{

	/* local variables */
	EGlistNode_t *mynode;

	/* call the internal function */
	__EGlistIntFind (mylist, mynode, mem);

	/* once we get out that loop 
	 * we check if we have the answer and return */
	return mynode;
}

/* ========================================================================= */
/* this function moves a node from one list to another list, the user must make
 * sure that both list are different, and that the pointer to the listNode to be
 * valid a belong to the source list */
int EGlistMoveNode (EGlist_t * sourceList,
										EGlist_t * destList,
										EGlistNode_t * node)
{

	/* call the internal function */
	__EGlistIntMoveNode (sourceList, destList, node, sourceList->size--,
											 destList->size++);

	/* ending */
	return 0;
}

/* ========================================================================= */
/* unattach a node from the list, without erasing it */
int EGlistUnattach (EGlist_t * list,
										EGlistNode_t * node)
{

	/* call the internal function */
	__EGlistIntUnattach (list, node, list->size--);

	/* ending */
	return 0;
}

/* ========================================================================= */
/* attach a node from the list, without erasing it */
int EGlistAttach (EGlist_t * list,
									EGlistNode_t * node)
{

	/* call the internal function */
	__EGlistIntAttach (list, node, list->size++);

	/* ending */
	return 0;
}

/* ========================================================================= */
/* unit tho lists into one */
int EGlistLink (EGlist_t * dlist,
								EGlist_t * olist)
{
	__EGlistIntLink (dlist, olist, dlist->size += olist->size, olist->size = 0);
	return 0;
}

/* ========================================================================= */
/* make a copy of the list */
void *EGlistCopy (void *source)
{
	/* local variables */
	EGlist_t *src = (EGlist_t *) source,
	 *dst;
	if (!src)
		return 0;
	EGlistNode_t *it;

	dst = EGnewList (src->mempool);
	for (it = src->begin; it; it = it->next)
		EGlistPushBack (dst, it->this);

	/* ending */
	return dst;
}


/*@}*/

/* ========================================================================= */
/** tester for the list */
void EGlistTest (void)
{
	/* local variables */
#if DEBUG > 3
	int size = 10;
	EGlist_t *mylist[10];
	int mydata[100];
#else
	int size = 100;
	EGlist_t *mylist[100];
	int mydata[10000];
#endif
	int i,
	  j,
	  k;
	EGmemPool_t *mempool;

	mempool =
		EGnewMemPool ((size_t) 100, EGmemPoolNewSize,
									EGmemPoolNewSize ((size_t) 1));
	for (k = 0; k < 100; k++)
	{
		MESSAGE (2, "Setting up data");
		/* setting up mydata */
		for (i = 0; i < size * size; i++)
			mydata[i] = i;

		MESSAGE (2, "Populating lists");
		/* add all values */
		for (i = 0; i < size; i++)
		{
			mylist[i] = EGnewList (mempool);
			for (j = 0; j < size; j++)
			{
				EGlistPushBack (mylist[i], mydata + size * i + j);
			}
		}

		MESSAGE (2, "Depopulating lists");
		/* erase the list */
		for (i = 0; i < size; i++)
		{
			EGfreeList ((void *) mylist[i]);
		}

		MESSAGE (2, "Populating lists");
		/* and create it again */
		/* add all values */
		for (i = 0; i < size; i++)
		{
			mylist[i] = EGnewList (mempool);
			for (j = 0; j < size; j++)
			{
				EGlistPushBack (mylist[i], mydata + size * i + j);
			}
		}

		MESSAGE (2, "Depopulating lists");
		/* erase the list */
		for (i = 0; i < size; i++)
		{
			MESSAGE (2, "Depopulating list %d", i);
			EGfreeList ((void *) mylist[i]);
		}
	}

	/* ending */
	EGfreeMemPool (mempool);
	return;
}
