/* EGlib "Efficient General Library" provides some basic structures and
 * algorithms commons in many optimization algorithms.
 *
 * Copyright (C) 2005 Daniel Espinoza and Marcos Goycoolea.
 * 
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but 
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY 
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public 
 * License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA 
 * */
#include "eg_io.h"
void EGmvar (char *str,
						 int nind,
						 const char *header,
						 ...)
{

	/* local variables */
	char lstr[100];
	va_list largs;

	/* first we put the header in the destination */
	str[0] = '\0';
	strcat (str, header);
	strcat (str, "(");

	/* now we initialize the va_list */
	va_start (largs, nind);
	while (--nind)
	{
		sprintf (lstr, "%d", va_arg (largs, int));
		strcat (str, lstr);
		strcat (str, ",");
	}
	sprintf (lstr, "%d", va_arg (largs, int));
	strcat (str, lstr);
	strcat (str, ")");


	/*end */
	va_end (largs);
	return;
}

/* Given a string 'input' this function uses EGioParse to separate
 * up to N words in it, we assume that argc is an array of pointers to strings
 * of size N, and note that the input array will be changed. */
void EGioNParse (char *input,
								 int max_argc,
								 const char *delim,
								 const char *comment,
								 int *argc,
								 char **argv)
{

	char *next;

	for ((*argc) = 0, next = input; ((*argc) < max_argc) && next; (*argc)++)
	{
		EGioParse (&next, &input, delim, comment);
		if (next == 0)
			break;
		argv[(*argc)] = next;
	}

	return;
}

/* given two *pointers 'next' and 'current' and a constant set  
 * of strings (parse delimiters), it store in 'next the next 
 * meaningfull string, and in current the rest of the secuence, 
 * the idea is to iterate over 'next' while it is not true; 
 * you have to store the original pointer to the string stream 
 * elsewere; also, we assume that the original stream is 
 * terminated with '\0'; also, it will discaard any sub-string 
 * that start with #, that is inteded for discard comments.
 * NOTE: this function WILL change the original string!!!!!
 * There is no guarantee on how it will be changed. */
void EGioParse (char **next,
								char **current,
								const char *delim,
								const char *comment)
{

	/* local variables */
	int i;
	char *ctmp;
	char *curchar = *current;

	/* if the current line is NULL we stop */
	if (*current == 0)
	{
		*next = 0;
		return;
	}

	/* first try to find a meaningful caracter */
	while ((((*curchar < 32) || (*curchar > 126))) && (*curchar != '\0'))
		curchar++;

	/* we check for sequences of delimiters */
	while (index (delim, *curchar) && (*curchar != '\0'))
		curchar++;

	/* if we are at the end, we stop */
	if ((*curchar) == '\0')
	{
		*next = *current = 0;
		return;
	}

	/* now we see if it is a comment, and if it is we discard everything */
	if (index (comment, *curchar))
	{
		*next = *current = 0;
		return;
	}

	/* now we know that we have a meaningful token, we only need to find 
	 * the next delimiter */
	*next = curchar;
	/* we look at the end of this stream */
	*current = index (curchar, 0);
	i = strlen (delim);
	while (i)
	{
		i--;
		ctmp = index (curchar, delim[i]);
		if (ctmp && (ctmp < *current) && (ctmp > *next))
			*current = ctmp;
	}

	/* now we assume that all non-printable characters are also delimiters */
	i = 32;
	while (i)
	{
		i--;
		ctmp = index (curchar, i);
		if (ctmp && (ctmp < *current) && (ctmp > *next))
			*current = ctmp;
	}


	/* at this stage *current point to the next delimiter or \0 */
	if (index (curchar, 0) == *current)
	{
		*current = 0;
		return;
	}

	/* otherwise there is going to be a non null current and we have to 
	 * add the \0 to the end of next */
	**current = 0;
	(*current)++;

	/* ending */
	return;
}

/* this discard all lines starting with comments and stores the next 
 * leading line in current and the next token in next, we assume that next 
 * does not contain data, and that current store the remainings of the
 * current line */
void EGioDisCom (char **next,
								 char **current,
								 const char *delim,
								 const char *comment,
								 char *store,
								 unsigned int storeSize,
								 FILE * in)
{

	/* local variables */
	int status = 1;

	/* if no current line we read it from the file */
	if (!(*current))
	{
		status = (store == fgets (store, (int) storeSize, in));
		*current = store;
	}
	/* we process the current line, and while the line does
	 * not have a token we look the next line */
	EGioParse (next, current, delim, comment);
	while (!(*next) && status)
	{
		status = (store == fgets (store, (int) storeSize, in));
		*current = store;
		EGioParse (next, current, delim, comment);
	}
	/* ending */
	return;
}

/* ========================================================================= */
/** @brief display function for strings
 * @param str pointer to a null terminated string of chars.
 * @param file pointer to a stream where we write the string.
 * @par Description:
 * This function just print the string on the file, it won't add a '\n' at the
 * end. */
void EGdisplayString (void *str,
											FILE * file)
{
	fprintf (file, "%s", (char *) str);
}
