/* EGlib "Efficient General Library" provides some basic structures and
 * algorithms commons in many optimization algorithms.
 *
 * Copyright (C) 2005 Daniel Espinoza and Marcos Goycoolea.
 * 
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but 
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY 
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public 
 * License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA 
 * */
#include "eg_heap.h"
EGheapCost_t EGheapGetMinVal (EGheap_t * h)
{

	EXITL (9, !h->size, "Empty heap");
	return (h->heap[0]->val);

}

void *EGheapGetMinThis (EGheap_t * h)
{

	EXITL (9, !h->size, "Empty heap");
	return (h->heap[0]->this);

}

EGheapConnector_t *EGheapGetMin (EGheap_t * h)
{

	EXITL (9, !h->size, "Empty heap");
	return (h->heap[0]);

}

EGheapConnector_t *EGnewHeapConnector (EGmemPool_t * mem,
																			 void *obj,
																			 EGheapCost_t val,
																			 unsigned int pos)
{

	EGheapConnector_t *hc;

	hc = (EGheapConnector_t *) EGmemPoolMalloc (mem, sizeof (EGheapConnector_t));

	hc->this = obj;
	hc->val = val;
	hc->pos = pos;

	return (hc);

}

int EGheapSiftup (EGheap_t * h,
									unsigned int current_pos)
{

	unsigned int father;
	EGheapConnector_t *swap;

	TESTL (9, current_pos >= h->size, "Position out of range");

	father = ((current_pos + 1) >> 1) - 1;

	while ((current_pos != 0)
				 && EGheapCostIsLess (h->heap[current_pos]->val, h->heap[father]->val))
	{

		swap = h->heap[current_pos];
		h->heap[current_pos] = h->heap[father];
		h->heap[father] = swap;

		h->heap[father]->pos = father;
		h->heap[current_pos]->pos = current_pos;

		current_pos = father;
		father = ((current_pos + 1) >> 1) - 1;

	}

	return 0;

}

EGheapConnector_t *EGheapInsert (EGheap_t * h,
																 void *obj,
																 EGheapCost_t val,
																 EGmemPool_t * mem)
{

	EGheapConnector_t *h_con;

	EXITL (9, h->size >= h->max_size, "Heap full");

	h_con = EGnewHeapConnector (mem, obj, val, h->size);
	h->heap[h->size] = h_con;
	h->size++;

	EGheapSiftup (h, h->size - 1);

	return h_con;

}

int EGheapDecreaseVal (EGheap_t * h,
											 EGheapConnector_t * hc,
											 EGheapCost_t val)
{

	unsigned int rval;

	hc->val = val;
	rval = EGheapSiftup (h, hc->pos);
	CHECKRVAL (rval);

	return 0;

}

unsigned int EGheapGetMinChild (EGheap_t * h,
																unsigned int father_pos)
{

	unsigned int first_pos,
	  last_pos;

	if (!h->size)
		return UINT_MAX;
	first_pos = (father_pos << 1) + 1;
	last_pos = h->size - 1;

	/* case both chindrend exists */
	if (first_pos < last_pos)
	{
		last_pos = first_pos + 1;
		if (EGheapCostIsLess (h->heap[last_pos]->val, h->heap[first_pos]->val))
			return last_pos;
		return first_pos;
	}

	/* case of childs out of range */
	if (first_pos > last_pos)
		return UINT_MAX;

	/* case there is only one child */
	return first_pos;
}

int EGheapSiftDown (EGheap_t * h,
										unsigned int pos)
{
	unsigned int first_pos,
	  last_pos;
	unsigned int child_pos;
	EGheapConnector_t *swap;

	/* get min child */
	if (!h->size)
	{
		return 0;
	}
	first_pos = (pos << 1) + 1;
	last_pos = h->size - 1;

	/* case both chindren exists */
	if (first_pos < last_pos)
	{
		last_pos = first_pos + 1;
		if (EGheapCostIsLess (h->heap[last_pos]->val, h->heap[first_pos]->val))
			child_pos = last_pos;
		else
			child_pos = first_pos;
		goto pos1;
	}
	/* case of children out of range */
	if (first_pos > last_pos)
	{
		return 0;
	}
	/* case there is only one child */
	child_pos = first_pos;

	/* end getminchild */
pos1:


	while (child_pos < h->size
				 && EGheapCostIsLess (h->heap[child_pos]->val, h->heap[pos]->val))
	{
		swap = h->heap[pos];
		h->heap[pos] = h->heap[child_pos];
		h->heap[child_pos] = swap;

		h->heap[pos]->pos = pos;
		h->heap[child_pos]->pos = child_pos;

		pos = child_pos;
		/* get min child */
		first_pos = (pos << 1) + 1;
		last_pos = h->size - 1;

		/* case both children exists */
		if (first_pos < last_pos)
		{
			last_pos = first_pos + 1;
			if (EGheapCostIsLess (h->heap[last_pos]->val, h->heap[first_pos]->val))
				child_pos = last_pos;
			else
				child_pos = first_pos;
		}
		/* case of children out of range */
		else if (first_pos > last_pos)
		{
			return 0;
		}
		/* case there is only one child */
		else
		{
			child_pos = first_pos;
		}
		/* end getminchild */

	}

	return 0;

}

EGheap_t *EGnewHeap (EGmemPool_t * mem,
										 unsigned int d,
										 unsigned int n)
{

	EGheap_t *h;

	h = (EGheap_t *) EGmemPoolMalloc (mem, sizeof (EGheap_t));

	h->heap =
		(EGheapConnector_t **) EGmemPoolMalloc (mem,
																						sizeof (EGheapConnector_t *) * n);
	h->size = 0;
	h->max_size = n;
	h->d = d = 2;

	return (h);

}

void EGfreeHeapConnector (void *v,
													EGmemPool_t * mem)
{

	EGmemPoolFree (v, sizeof (EGheapConnector_t), mem);
	return;

}

void EGheapClear (EGheap_t * h,
									EGmemPool_t * mem)
{

	unsigned int i;

	for (i = 0; i < h->size; i++)
		EGfreeHeapConnector (h->heap[i], mem);

	h->size = 0;

	return;

}



void EGfreeHeap (void *v,
								 EGmemPool_t * mem)
{

	EGheap_t *h;

	h = (EGheap_t *) (v);

	EGheapClear (h, mem);

	EGmemPoolFree (h->heap, sizeof (EGheapConnector_t *) * (h->max_size), mem);
	EGmemPoolFree (h, sizeof (EGheap_t), mem);

	return;

}

int EGheapDeleteMin (EGheap_t * h,
										 EGmemPool_t * mem)
{

	EXITL (9, !h || !h->size, "Empty heap");

	EGmemPoolFree (h->heap[0], sizeof (EGheapConnector_t), mem);

	h->heap[0] = h->heap[h->size - 1];
	if (h->size - 1)
		h->heap[0]->pos = 0;

	h->heap[h->size - 1] = 0;

	h->size--;

	EGheapSiftDown (h, 0);

	return 0;

}
