/* EGlib "Efficient General Library" provides some basic structures and
 * algorithms commons in many optimization algorithms.
 *
 * Copyright (C) 2005 Daniel Espinoza and Marcos Goycoolea.
 * 
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but 
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY 
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public 
 * License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA 
 * */
#include "eg_hash.h"
/* ========================================================================= */
/* this is for the hash function */
/* prime number, this is one of the 1024 biggest prime for unsigned int */
EGkey_t EGhashPrime = 4294959751LLU;
/* other prime number */
EGkey_t EGhashB = 109180541LLU;
/* and yet other prime number */
EGkey_t EGhashA = 179422291LLU;

/* ========================================================================= */
/* we only need to create hash tables, liberate them, insert, find and delete */
/* this return a pointer to an initialized hash table of maxsize 'size' */
static inline EGhashList_t *EGnewHashList (EGhash_t * myhash)
{

	/* local vars */
	EGhashList_t *res;

	/* resize memory if necesary */
	res = __EGlistIntNewList (myhash->mempool, EGhashList_t);
	res->begin = res->end = 0;

	/* ending */
	return res;
}

/* ========================================================================= */
/* this return a pointer to an initialized hash table of maxsize 'size' */
EGhash_t *__EGnewHash (size_t maxsize,
											 EGmemPool_t * mempool
#if __EG_HASH_PROFILE__
											 ,
											 const char *file,
											 const int line
#endif
	)
{

	/* local vars */
	EGhash_t *res;

	/* check that the hashsize is bigger than the EGhashPrime */
	EXIT ((maxsize > EGhashPrime), "the maxsize = %zu specified is bigger than "
				"the internal prime p = %ju\nYou should recompile the library with a "
				"bigger EGhashPrime ", maxsize, EGhashPrime);

	/* allocating memory if necesary */
	res = EGmemPoolMalloc (mempool, sizeof (EGhash_t));
	res->mempool = mempool;

	/* initializate answer */
	res->maxsize = maxsize;
	res->entry =
		(EGhashList_t **) EGmemPoolMalloc (mempool,
																			 maxsize * sizeof (EGhashList_t *));
	memset (res->entry, 0, maxsize * sizeof (EGhashList_t *));
	res->size = 0;
#if __EG_HASH_PROFILE__
	res->hashAllocationFile = file;
	res->hashAllocationLine = line;
#endif

	/* ending */
	return res;
}

/* ========================================================================= */
/* this free a hash table  */
void EGfreeHash (void *hash)
{

	/* local data */
	int i;
	EGmemPool_t *mempool = ((EGhash_t *) hash)->mempool;
#if __EG_HASH_PROFILE__
	unsigned int rusage = 0;
#endif

	/* we first free all list we can doit here faster if instead of calling
	 * the functions we re-write them here*/
	for (i = ((EGhash_t *) hash)->maxsize; i;)
	{
		i--;
		if (((EGhash_t *) hash)->entry[i])
		{
#if __EG_HASH_PROFILE__
			rusage++;
#endif
			__EGlistIntFreeList (((EGhash_t *) hash)->entry[i], mempool,
													 EGhashList_t, EGhashListNode_t);
		}														/* end if */
	}															/* end for i */

	/* now we free the internal data */
	EGmemPoolFree (((EGhash_t *) hash)->entry, ((EGhash_t *) hash)->maxsize *
								 sizeof (EGhashList_t *), mempool);

	/* now we log our findings */
#if __EG_HASH_PROFILE__
	fprintf (stderr, "Profile for EGhash %p (%s:%d)\n", hash,
					 ((EGhash_t *) hash)->hashAllocationFile,
					 ((EGhash_t *) hash)->hashAllocationLine);
	fprintf (stderr, "\tmaxsize       : %5u\n", ((EGhash_t *) hash)->maxsize);
	fprintf (stderr, "\tsize          : %5u\n", ((EGhash_t *) hash)->size);
	fprintf (stderr, "\taverage ratio : %5.3lf\n",
					 ((double) ((EGhash_t *) (hash))->size) /
					 ((EGhash_t *) (hash))->maxsize);
	fprintf (stderr, "\treal average  : %5.3lf\n",
					 ((double) ((EGhash_t *) (hash))->size) / (rusage + 1));
#endif

	/* and now we store the hash in the hash memory */
	EGmemPoolFree (hash, sizeof (EGhash_t), ((EGhash_t *) hash)->mempool);

	/* ending */
	return;
}

/* ========================================================================= */
/* this is to insert a new element to the hash */
int EGhashInsert (EGhash_t * hash,
									const EGkey_t key,
									void *data)
{

	/* local variables */
	unsigned int pos;

	/* now we compute the hash function of the key */
	EGhashFunction (key, hash->maxsize, &pos);

	/* now we insert at the head of the corresponding list */
	if (!(hash->entry[pos]))
		hash->entry[pos] = EGnewHashList (hash);
#if ((__EG_HASH_CHECK__)&&(__EG_HASH_ALLOW_REPETITION__==0))
	TEST (EGhashFind (hash, key), "Adding an already added entry");
#endif
	__EGlistIntPushBack (hash->entry[pos], data,;, hash->mempool,
											 EGhashListNode_t);
	(hash->entry[pos])->end->key = key;
	hash->size++;

	/* ending */
	return 0;
}

/* ========================================================================= */
/* this is to find an element in the hash table, if the key is not found, it return 0 */
EGhashData_t *EGhashFind (const EGhash_t * hash,
													const EGkey_t key)
{

	/* local variables */
	unsigned int pos;
	EGhashListNode_t *It;

	/* we compute the hash function */
	EGhashFunction (key, hash->maxsize, &pos);

	/* now we lookup for the key in the corresponding list */
	if (!hash->entry[pos])
		return 0;
	for (It = hash->entry[pos]->begin; It; It = It->next)
	{
		if (It->key == key)
		{
			return It;
		}
	}

	/* ending */
	return 0;
}

/* ========================================================================= */
/* this is to eliminate an element from the hash table, if the key weren't there it will return */
/* a nonzero value, if it success, it will return zero */
int EGhashErase (EGhash_t * hash,
								 EGkey_t key)
{

	/* local variables */
	int pos;
	EGhashListNode_t *It;

	/* we compute the hash function */
	EGhashFunction (key, hash->maxsize, &pos);

	/* now we lookup for the key in the corresponding list and erase it if we found it */
	if (!hash->entry[pos])
		return 1;
	for (It = hash->entry[pos]->begin; It; It = It->next)
	{
		if (It->key == key)
		{
			__EGlistIntErase (hash->entry[pos], It, nullFree,;, hash->mempool,
												EGhashListNode_t);
			hash->size--;
			return 0;
		}
	}

	/* ending */
	return 1;
}

/* ========================================================================= */
int EGhashDisplay (const EGhash_t * hash,
									 FILE * display)
{

	/* local variables */
	register unsigned int t;
	EGhashListNode_t *n;

	/* loop */
	fprintf (display, "Hash %p Size: %zu MaxSize: %zu\n", (const void *) hash,
					 hash->size, hash->maxsize);
	for (t = hash->maxsize; t--;)
	{
		if (hash->entry[t])
		{
			n = hash->entry[t]->begin;
			fprintf (display, "entry %u (%p): ", t, (void *) (hash->entry[t]));
			while (n)
			{
				fprintf (display, "(%p){key=%ju this=%p prev=%p next=%p} ", (void *) n,
								 n->key, n->this, (void *) n->prev, (void *) n->next);
				n = n->next;
			}													/* end while n */
			fprintf (display, "\n");
		}														/* end if hash->entry[t] */
	}															/* end for t */

	/* ending */
	return 0;
}

/* ========================================================================= */
/* just to make some testing */
void EGhashTest (void)
{

	/* local data */
	EGhash_t *myhash;
	EGhashData_t *mydata;
	unsigned int maxsize = 50000;
	int key[50000],
	  i;
	EGmemPool_t *mempool;

	/* now we initialize the keys */
	mempool = EGnewMemPool (100, EGmemPoolNewSize, EGmemPoolNewSize (1));
	for (i = maxsize; i;)
	{
		i--;
		key[i] = rand ();
	}

	/* now we create the hash table */
	myhash = EGnewHash (2 * maxsize + 1, mempool);

	/* and add all entries to it */
	for (i = maxsize; i;)
	{
		i--;
		EGhashInsert (myhash, (EGkey_t) i, (void *) (key + i));
	}

	/* and find them */
	for (i = maxsize; i;)
	{
		i--;
		mydata = EGhashFind (myhash, (EGkey_t) i);
		if ((mydata == 0) || (mydata->this != key + i))
			EXIT (1, "internal data is mess-up");
	}

	/* and finally we erase the hash */
	EGfreeHash (myhash);

	/* now we create the hash table */
	myhash = EGnewHash (maxsize / 2 + 1, mempool);

	/* and add all entries to it */
	for (i = maxsize; i;)
	{
		i--;
		EGhashInsert (myhash, (EGkey_t) i, (void *) (key + i));
	}

	/* and erase half of them */
	for (i = maxsize; i;)
	{
		i--;
		if (EGhashErase (myhash, (EGkey_t) i))
			EXIT (1, "internal data is mess-up");
	}

	/* and reinsert them */
	for (i = maxsize; i;)
	{
		i--;
		EGhashInsert (myhash, (EGkey_t) i, key + i);
	}

	/* and finally we erase the hash */
	EGfreeHash (myhash);

	/* ending */
	EGfreeMemPool (mempool);
	return;
}

/* ========================================================================= */
/* this function leaves the hash table 'as new', it recives a free function for
 * the internal data */
void EGhashClear (EGhash_t * hash,
									EGfree_f dataFree)
{
	/* local variables */
	register int i;

	/* we loop through the entries of the hash */
	for (i = hash->maxsize; i;)
	{
		i--;
		if (((EGhash_t *) hash)->entry[i])
		{
			__EGlistIntClear (hash->entry[i], dataFree,;, hash->mempool,
												EGhashListNode_t);
			__EGlistIntFreeList (hash->entry[i], hash->mempool, EGhashList_t,
													 EGhashListNode_t);
			hash->entry[i] = 0;
		}														/* end if */
	}															/* end for i */

	/* ending */
	return;
}
