/* EGlib "Efficient General Library" provides some basic structures and
 * algorithms commons in many optimization algorithms.
 *
 * Copyright (C) 2005 Daniel Espinoza and Marcos Goycoolea.
 * 
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but 
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY 
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public 
 * License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA 
 * */
/* ========================================================================= */
/* This header include definitions of macros to work with fixed point
 * representation of decimal numbers.
 *
 * - 2003-06-26
 * 					- First Implementation
 * */
/* ========================================================================= */
#ifndef __EG_FP_H__
#define __EG_FP_H__
#include <limits.h>
#include <stdio.h>
#include <stdlib.h>
//#include <stdint.h>
#include "eg_config.h"
#include "eg_macros.h"
/* ========================================================================= */
/* local definitions, this implies acuracy */
/* ========================================================================= */

/* we choose the size_t integers because this is the usual natural length for
 * the machine, thus should be fastest choice. */
typedef long EGfp10_t;
typedef long EGfp20_t;
typedef long EGfp25_t;
typedef long EGfp28_t;

/* define how many bits there are in our structure */
#define EGFP_BIT CHAR_BIT*sizeof(long)

/* we reserve the second to last bit as an overflow mark, we are assuming that
 * the sign bit is the last one... this may be ot true in some machines, but
 * should be according to the STDC99 */
#define EGFP_OFBIT (((long)1)<<(EGFP_BIT-2))

/* this is the sign bit of the representation */
#define EGFP_SGBIT (1<<(EGFP_BIT-1))

/* this is how many bits we use for fractional representation */
#define EGFP_FRBIT10 10
#define EGFP_FRBIT20 20
#define EGFP_FRBIT25 25
#define EGFP_FRBIT28 28

/* this is how many bits we use for the integer parrt, we compute it as the
 * resoult of substract the other's fields to the total space */
#define EGFP_INBIT10 (EGFP_BIT-2-EGFP_FRBIT10)
#define EGFP_INBIT20 (EGFP_BIT-2-EGFP_FRBIT20)
#define EGFP_INBIT25 (EGFP_BIT-2-EGFP_FRBIT25)
#define EGFP_INBIT28 (EGFP_BIT-2-EGFP_FRBIT28)

/* now we define the maximum representable number */
#define EGFP_MAX10 ((double)((((long)1)<<30)-1))/((long)1<<EGFP_FRBIT10)
#define EGFP_MAX20 ((double)((((long)1)<<30)-1))/((long)1<<EGFP_FRBIT20)
#define EGFP_MAX25 ((double)((((long)1)<<30)-1))/((long)1<<EGFP_FRBIT25)
#define EGFP_MAX28 ((double)((((long)1)<<30)-1))/((long)1<<EGFP_FRBIT28)

/* now the minimum representable value */
#define EGFP_MIN10 (-EGFP_MAX10)
#define EGFP_MIN20 (-EGFP_MAX20)
#define EGFP_MIN25 (-EGFP_MAX25)
#define EGFP_MIN28 (-EGFP_MAX28)

/* this is the conversion factor used to transform a real type to our fixed
 * point representation. */
#define EGFP_FACTOR10 (1LL<<EGFP_FRBIT10)
#define EGFP_FACTOR20 (1LL<<EGFP_FRBIT20)
#define EGFP_FACTOR25 (1LL<<EGFP_FRBIT25)
#define EGFP_FACTOR28 (1LL<<EGFP_FRBIT28)

/* the difference between one and the nearest representable number. Note that in
 * our case this number may be seen as the discretization for the numbers that
 * can be represented */
#define EGFP10_EPSILON (((double)1)/(EGFP_FACTOR10))
#define EGFP20_EPSILON (((double)1)/(EGFP_FACTOR20))
#define EGFP25_EPSILON (((double)1)/(EGFP_FACTOR25))
#define EGFP28_EPSILON (((double)1)/(EGFP_FACTOR28))

/* ========================================================================= */
/* now we define some function to manipulate this numbers */
/* ========================================================================= */

/* this check and maintain the overflow bit if the checking was enable, see the
 * eg_config.h to see the default value */
#if __EGFP_CHECK_OVERFLOW__
/* chek if a FP_T number is overflow */
#define __EGFP_ABS__(x) (x>0?x:-x)
#define __EGFP_SGN__(x) (x<0?-1:1)
#if __EGFP_CHECK_VERBOSE__
#define EGFP_OCHK(x) (((__EGFP_ABS__(x))&EGFP_OFBIT)?fprintf(stderr,"Overflow in %s:%d\n",__FILE__,__LINE__),EGFP_OFBIT&(__EGFP_ABS__(x)):0)
#else
#define EGFP_OCHK(x) ((__EGFP_ABS__(x))&EGFP_OFBIT)
#endif
/* define range check */
#define EGFP_RCHK10(x) (((EGfp10_t)(((x)>EGFP_MAX10)||((x)<EGFP_MIN10)))<<(EGFP_BIT-2))
#define EGFP_RCHK20(x) (((EGfp20_t)(((x)>EGFP_MAX20)||((x)<EGFP_MIN20)))<<(EGFP_BIT-2))
#define EGFP_RCHK25(x) (((EGfp25_t)(((x)>EGFP_MAX25)||((x)<EGFP_MIN25)))<<(EGFP_BIT-2))
#define EGFP_RCHK28(x) (((EGfp28_t)(((x)>EGFP_MAX28)||((x)<EGFP_MIN28)))<<(EGFP_BIT-2))
#else
#define EGFP_RCHK10(x) 0
#define EGFP_RCHK20(x) 0
#define EGFP_RCHK25(x) 0
#define EGFP_RCHK28(x) 0
#define EGFP_OCHK(x) 0
#define __EGFP_SGN__(x) 1
#endif

/* this function check if a EG_fp_t is overflown */
#define EGfpCheckOverflow10(x) EGFP_OCHK(x)
#define EGfpCheckOverflow20(x) EGFP_OCHK(x)
#define EGfpCheckOverflow25(x) EGFP_OCHK(x)
#define EGfpCheckOverflow28(x) EGFP_OCHK(x)

/* convert a float to our fixed point representation */
#define ftofp10(f) (__EGFP_SGN__(f)*(((EGfp10_t)((f*__EGFP_SGN__(f))*EGFP_FACTOR10))|EGFP_RCHK10(f)))
#define ftofp20(f) (__EGFP_SGN__(f)*(((EGfp20_t)((f*__EGFP_SGN__(f))*EGFP_FACTOR20))|EGFP_RCHK20(f)))
#define ftofp25(f) (__EGFP_SGN__(f)*(((EGfp25_t)((f*__EGFP_SGN__(f))*EGFP_FACTOR25))|EGFP_RCHK25(f)))
#define ftofp28(f) (__EGFP_SGN__(f)*(((EGfp28_t)((f*__EGFP_SGN__(f))*EGFP_FACTOR28))|EGFP_RCHK28(f)))

/* convert a double to our fixed point representation */
#define lftofp10(f) (__EGFP_SGN__(f)*(((EGfp10_t)((f*__EGFP_SGN__(f))*EGFP_FACTOR10))|EGFP_RCHK10(f)))
#define lftofp20(f) (__EGFP_SGN__(f)*(((EGfp20_t)((f*__EGFP_SGN__(f))*EGFP_FACTOR20))|EGFP_RCHK20(f)))
#define lftofp25(f) (__EGFP_SGN__(f)*(((EGfp25_t)((f*__EGFP_SGN__(f))*EGFP_FACTOR25))|EGFP_RCHK25(f)))
#define lftofp28(f) (__EGFP_SGN__(f)*(((EGfp28_t)((f*__EGFP_SGN__(f))*EGFP_FACTOR28))|EGFP_RCHK28(f)))

/* convert a long double to our fixed point representation */
#define llftofp10(f) (__EGFP_SGN__(f)*(((EGfp10_t)((f*__EGFP_SGN__(f))*EGFP_FACTOR10))|EGFP_RCHK10(f)))
#define llftofp20(f) (__EGFP_SGN__(f)*(((EGfp20_t)((f*__EGFP_SGN__(f))*EGFP_FACTOR20))|EGFP_RCHK20(f)))
#define llftofp25(f) (__EGFP_SGN__(f)*(((EGfp25_t)((f*__EGFP_SGN__(f))*EGFP_FACTOR25))|EGFP_RCHK25(f)))
#define llftofp28(f) (__EGFP_SGN__(f)*(((EGfp28_t)((f*__EGFP_SGN__(f))*EGFP_FACTOR28))|EGFP_RCHK28(f)))

/* convert an integer to our fixed point representation */
#define itofp10(f) (__EGFP_SGN__(f)*(((EGfp10_t)((f*__EGFP_SGN__(f))*EGFP_FACTOR10))|EGFP_RCHK10(f)))
#define itofp20(f) (__EGFP_SGN__(f)*(((EGfp20_t)((f*__EGFP_SGN__(f))*EGFP_FACTOR20))|EGFP_RCHK20(f)))
#define itofp25(f) (__EGFP_SGN__(f)*(((EGfp25_t)((f*__EGFP_SGN__(f))*EGFP_FACTOR25))|EGFP_RCHK25(f)))
#define itofp28(f) (__EGFP_SGN__(f)*(((EGfp28_t)((f*__EGFP_SGN__(f))*EGFP_FACTOR28))|EGFP_RCHK28(f)))

/* convert an integer to our fixed point representation */
#define ltofp10(f) (__EGFP_SGN__(f)*(((EGfp10_t)((f*__EGFP_SGN__(f))*EGFP_FACTOR10))|EGFP_RCHK10(f)))
#define ltofp20(f) (__EGFP_SGN__(f)*(((EGfp20_t)((f*__EGFP_SGN__(f))*EGFP_FACTOR20))|EGFP_RCHK20(f)))
#define ltofp25(f) (__EGFP_SGN__(f)*(((EGfp25_t)((f*__EGFP_SGN__(f))*EGFP_FACTOR25))|EGFP_RCHK25(f)))
#define ltofp28(f) (__EGFP_SGN__(f)*(((EGfp28_t)((f*__EGFP_SGN__(f))*EGFP_FACTOR28))|EGFP_RCHK28(f)))

/* convert an integer to our fixed point representation */
#define lltofp10(f) (__EGFP_SGN__(f)*(((EGfp10_t)((f*__EGFP_SGN__(f))*EGFP_FACTOR10))|EGFP_RCHK10(f)))
#define lltofp20(f) (__EGFP_SGN__(f)*(((EGfp20_t)((f*__EGFP_SGN__(f))*EGFP_FACTOR20))|EGFP_RCHK20(f)))
#define lltofp25(f) (__EGFP_SGN__(f)*(((EGfp25_t)((f*__EGFP_SGN__(f))*EGFP_FACTOR25))|EGFP_RCHK25(f)))
#define lltofp28(f) (__EGFP_SGN__(f)*(((EGfp28_t)((f*__EGFP_SGN__(f))*EGFP_FACTOR28))|EGFP_RCHK28(f)))

/* convert anEGfp_t to double */
#define fptolf10(f) (((double)f)/EGFP_FACTOR10)
#define fptolf20(f) (((double)f)/EGFP_FACTOR20)
#define fptolf25(f) (((double)f)/EGFP_FACTOR25)
#define fptolf28(f) (((double)f)/EGFP_FACTOR28)

/* convert anEGfp_t to float */
#define fptof10(f) (((float)f)/EGFP_FACTOR10)
#define fptof20(f) (((float)f)/EGFP_FACTOR20)
#define fptof25(f) (((float)f)/EGFP_FACTOR25)
#define fptof28(f) (((float)f)/EGFP_FACTOR28)

/* convert anEGfp_t to int */
#define fptoi10(f) (((int)f)/EGFP_FACTOR10)
#define fptoi20(f) (((int)f)/EGFP_FACTOR20)
#define fptoi25(f) (((int)f)/EGFP_FACTOR25)
#define fptoi28(f) (((int)f)/EGFP_FACTOR28)

/* convert anEGfp_t to long */
#define fptol10(f) (((long)f)/EGFP_FACTOR10)
#define fptol20(f) (((long)f)/EGFP_FACTOR20)
#define fptol25(f) (((long)f)/EGFP_FACTOR25)
#define fptol28(f) (((long)f)/EGFP_FACTOR28)

/* convert anEGfp_t to long long */
#define fptoll10(f) (((long long)f)/EGFP_FACTOR10)
#define fptoll20(f) (((long long)f)/EGFP_FACTOR20)
#define fptoll25(f) (((long long)f)/EGFP_FACTOR25)
#define fptoll28(f) (((long long)f)/EGFP_FACTOR28)

/* this function add to FP numbers */
#if __EGFP_CHECK_OVERFLOW__
#define EGfpAdd(a,b) ((EGFP_OCHK(a)|EGFP_OCHK(b)|EGFP_OCHK((a+b))|((a+b)*__EGFP_SGN__(a+b)))*__EGFP_SGN__(a+b))
#else
#define EGfpAdd(a,b) (a+b)
#endif
#define EGfpAdd10(a,b) EGfpAdd(a,b)
#define EGfpAdd20(a,b) EGfpAdd(a,b)
#define EGfpAdd25(a,b) EGfpAdd(a,b)
#define EGfpAdd28(a,b) EGfpAdd(a,b)

/* this function substract to FP numbers */
#if __EGFP_CHECK_OVERFLOW__
#define EGfpSub(a,b) ((EGFP_OCHK(a)|EGFP_OCHK(b)|EGFP_OCHK((a-b))|((a-b)*__EGFP_SGN__(a-b)))*__EGFP_SGN__(a-b))
#else
#define EGfpSub(a,b) (a-b)
#endif
#define EGfpSub10(a,b) EGfpSub(a,b)
#define EGfpSub20(a,b) EGfpSub(a,b)
#define EGfpSub25(a,b) EGfpSub(a,b)
#define EGfpSub28(a,b) EGfpSub(a,b)

/* this define multiplication of FP numbers */
#define __EGfpIntMul10__(a,b) ((long)(((long long)(a))*((long long)(b))>>EGFP_FRBIT10))
#define __EGfpIntMul20__(a,b) ((long)(((long long)(a))*((long long)(b))>>EGFP_FRBIT20))
#define __EGfpIntMul25__(a,b) ((long)(((long long)(a))*((long long)(b))>>EGFP_FRBIT25))
#define __EGfpIntMul28__(a,b) ((long)(((long long)(a))*((long long)(b))>>EGFP_FRBIT28))
#if __EGFP_CHECK_OVERFLOW__
#define EGfpMul10(a,b) ((\
		(EGFP_OCHK(a))|\
		(EGFP_OCHK(b))|\
		(EGFP_RCHK10((((double)(a))*((double)(b)))/(EGFP_FACTOR10*EGFP_FACTOR10)))|\
		(__EGfpIntMul10__(a,b)*__EGFP_SGN__(__EGfpIntMul10__(a,b))))*__EGFP_SGN__(__EGfpIntMul10__(a,b)))
#define EGfpMul20(a,b) ((\
		(EGFP_OCHK(a))|\
		(EGFP_OCHK(b))|\
		(EGFP_RCHK20((((double)(a))*((double)(b)))/(EGFP_FACTOR20*EGFP_FACTOR20)))|\
		(__EGfpIntMul20__(a,b)*__EGFP_SGN__(__EGfpIntMul20__(a,b))))*__EGFP_SGN__(__EGfpIntMul20__(a,b)))
#define EGfpMul25(a,b) ((\
		(EGFP_OCHK(a))|\
		(EGFP_OCHK(b))|\
		(EGFP_RCHK25((((double)(a))*((double)(b)))/(EGFP_FACTOR25*EGFP_FACTOR25)))|\
		(__EGfpIntMul25__(a,b)*__EGFP_SGN__(__EGfpIntMul25__(a,b))))*__EGFP_SGN__(__EGfpIntMul25__(a,b)))
#define EGfpMul28(a,b) ((\
		(EGFP_OCHK(a))|\
		(EGFP_OCHK(b))|\
		(EGFP_RCHK28((((double)(a))*((double)(b)))/(EGFP_FACTOR28*EGFP_FACTOR28)))|\
		(__EGfpIntMul28__(a,b)*__EGFP_SGN__(__EGfpIntMul28__(a,b))))*__EGFP_SGN__(__EGfpIntMul28__(a,b)))
#else
#define EGfpMul10(a,b) __EGfpIntMul10__(a,b)
#define EGfpMul20(a,b) __EGfpIntMul20__(a,b)
#define EGfpMul25(a,b) __EGfpIntMul25__(a,b)
#define EGfpMul28(a,b) __EGfpIntMul28__(a,b)
#endif

/* this define divition of FP numbers */
#define __EGfpIntDiv10__(a,b) ((EGfp10_t)((((long long)(a))<<EGFP_FRBIT10)/(b)))
#define __EGfpIntDiv20__(a,b) ((EGfp20_t)((((long long)(a))<<EGFP_FRBIT20)/(b)))
#define __EGfpIntDiv25__(a,b) ((EGfp25_t)((((long long)(a))<<EGFP_FRBIT25)/(b)))
#define __EGfpIntDiv28__(a,b) ((EGfp28_t)((((long long)(a))<<EGFP_FRBIT28)/(b)))
#if __EGFP_CHECK_OVERFLOW__
#define EGfpDiv10(a,b) ((\
		(EGFP_OCHK(a))|\
		(EGFP_OCHK(b))|\
		(EGFP_RCHK10(((((((double)(a))*EGFP_FACTOR10)/(b)))/EGFP_FACTOR10)))|\
		(__EGfpIntDiv10__(a,b)*__EGFP_SGN__(__EGfpIntDiv10__(a,b))))*__EGFP_SGN__(__EGfpIntDiv10__(a,b)))
#define EGfpDiv20(a,b) ((\
		(EGFP_OCHK(a))|\
		(EGFP_OCHK(b))|\
		(EGFP_RCHK20(((((((double)(a))*EGFP_FACTOR20)/(b)))/EGFP_FACTOR20)))|\
		(__EGfpIntDiv20__(a,b)*__EGFP_SGN__(__EGfpIntDiv20__(a,b))))*__EGFP_SGN__(__EGfpIntDiv20__(a,b)))
#define EGfpDiv25(a,b) ((\
		(EGFP_OCHK(a))|\
		(EGFP_OCHK(b))|\
		(EGFP_RCHK25(((((((double)(a))*EGFP_FACTOR25)/(b)))/EGFP_FACTOR25)))|\
		(__EGfpIntDiv25__(a,b)*__EGFP_SGN__(__EGfpIntDiv25__(a,b))))*__EGFP_SGN__(__EGfpIntDiv25__(a,b)))
#define EGfpDiv28(a,b) ((\
		(EGFP_OCHK(a))|\
		(EGFP_OCHK(b))|\
		(EGFP_RCHK28(((((((double)(a))*EGFP_FACTOR28)/(b)))/EGFP_FACTOR28)))|\
		(__EGfpIntDiv28__(a,b)*__EGFP_SGN__(__EGfpIntDiv28__(a,b))))*__EGFP_SGN__(__EGfpIntDiv28__(a,b)))
#else
#define EGfpDiv10(a,b) __EGfpIntDiv10__(a,b)
#define EGfpDiv20(a,b) __EGfpIntDiv20__(a,b)
#define EGfpDiv25(a,b) __EGfpIntDiv25__(a,b)
#define EGfpDiv28(a,b) __EGfpIntDiv28__(a,b)
#endif

/* this defines the change of sign of a FP number */
#define EGfpMinus(a) (a*-1)
#define EGfpMinus10(a) EGfpMinus(a)
#define EGfpMinus20(a) EGfpMinus(a)
#define EGfpMinus25(a) EGfpMinus(a)
#define EGfpMinus28(a) EGfpMinus(a)

/* end of eg_fp.h */
#endif
