/* EGlib "Efficient General Library" provides some basic structures and
 * algorithms commons in many optimization algorithms.
 *
 * Copyright (C) 2005 Daniel Espinoza and Marcos Goycoolea.
 * 
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but 
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY 
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public 
 * License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA 
 * */
#include "eg_aequiset.h"
/* ========================================================================= */
/*  this function copy one aequiset to another new set */
EGaequiSet_t *EGaequiSetCopyMP (EGaequiSet_t * U,
																EGcopyMP_f data_copy,
																EGmemPool_t * mem)
{
	/* local variables */
	EGaequiSet_t *orig = (EGaequiSet_t *) U,
	 *copy;
	register unsigned i;

	/* create basic copy of the set */
	copy = EGmemPoolSMalloc (orig->mem, EGaequiSet_t, 1);
	copy->mem = orig->mem;
	copy->size = orig->size;
	copy->set = EGmemPoolSMalloc (orig->mem, EGaequiSetElem_t, orig->size);
	memcpy (copy->set, orig->set, sizeof (EGaequiSetElem_t) * orig->size);

	/* if we have a copy function we use it */
	if (data_copy)
		for (i = copy->size; i--;)
			copy->set[i].this = data_copy (orig->set[i].this, mem);

	/* ending */
	return copy;
}

/* ========================================================================= */
/*  this function copy one aequiset to another new set */
EGaequiSet_t *EGaequiSetCopy (EGaequiSet_t * U,
															EGcopy_f data_copy)
{
	/* local variables */
	EGaequiSet_t *orig = (EGaequiSet_t *) U,
	 *copy;
	register unsigned i;

	/* create basic copy of the set */
	copy = EGmemPoolSMalloc (orig->mem, EGaequiSet_t, 1);
	copy->mem = orig->mem;
	copy->size = orig->size;
	copy->set = EGmemPoolSMalloc (orig->mem, EGaequiSetElem_t, orig->size);
	memcpy (copy->set, orig->set, sizeof (EGaequiSetElem_t) * orig->size);

	/* if we have a copy function we use it */
	if (data_copy)
		for (i = copy->size; i--;)
			copy->set[i].this = data_copy (orig->set[i].this);

	/* ending */
	return copy;
}

/* ========================================================================= */
/* this procedure make a new set of size 'size' and with all elements as
 * singletons and without internal data. */
EGaequiSet_t *EGnewAEquiSet (EGmemPool_t * mem,
														 const unsigned size)
{
	/* local variables */
	EGaequiSet_t *v;
	register unsigned i;
	EGaequiSetElem_t *elem;

	/* basic set-up */
	v = EGmemPoolSMalloc (mem, EGaequiSet_t, 1);
	v->mem = mem;
	v->size = size;
	v->set = EGmemPoolSMalloc (mem, EGaequiSetElem_t, size);
	for (elem = v->set + size, i = size; i--;)
	{
		elem--;
		elem->father = i;
		elem->this = 0;
		elem->rank = 0;
	}

	/* ending */
	return v;
}

/* ========================================================================= */
/* this function find the representant of this element in his equivalence set */
unsigned EGaequiSetFind (EGaequiSet_t * const s,
												 const unsigned elem)
{
	/* local variables */
	unsigned father;

	/* some tests */
	EXITL (EG_AEQUISET_DLEVEL, elem >= s->size,
				 "elem (%u) is out of range (size = %u)", elem, s->size);

	/* computing fahter */
	father = s->set[elem].father;
	return (father == elem) ? elem : (s->set[elem].father =
																		EGaequiSetFind (s, father));
}

/* ========================================================================= */
/* this functionlink two sets or equivalence clases */
unsigned EGaequiSetLink (EGaequiSet_t * const U,
												 const unsigned u,
												 const unsigned v)
{
	/* local variables */
	EGaequiSetElem_t *uptr,
	 *vptr,
	 *w;

	/* some tests */
	EXITL (EG_AEQUISET_DLEVEL, u >= U->size,
				 "elem (%u) is out of range (size = %u)", u, U->size);
	EXITL (EG_AEQUISET_DLEVEL, v >= U->size,
				 "elem (%u) is out of range (size = %u)", v, U->size);
	EXITL (EG_AEQUISET_DLEVEL, v != U->set[v].father,
				 "v is not a representant for its class");
	EXITL (EG_AEQUISET_DLEVEL, u != U->set[u].father,
				 "u is not a representant for its class");

	/* pick the biggest one */
	uptr = U->set + u;
	vptr = U->set + v;
	if (uptr->rank < vptr->rank)
	{
		w = uptr;
		uptr = vptr;
		vptr = w;
	}

	if (uptr->rank == vptr->rank)
	{
		uptr->rank++;
	}

	/* now we know that rank->u > rank->v */
	vptr->father = uptr - U->set;

	/* ending */
	return vptr->father;
}

/* ========================================================================= */
/* this function reset a Universe set to all singleton-state */
void EGaequiSetReset (EGaequiSet_t * const U)
{
	unsigned register i;
	EGaequiSetElem_t *elem;
	for (i = U->size, elem = U->set + U->size; i--;)
	{
		elem--;
		elem->father = i;
		elem->this = 0;
		elem->rank = 0;
	}
	return;
}

/* ========================================================================= */
/* destructor for the structure, note that his only eliminate the given element
 * in the set, but no the whole structure */
void EGfreeAEquiSet (void *u)
{
	EGmemPoolSFree (((EGaequiSet_t *) u)->set,
									EGaequiSetElem_t,
									((EGaequiSet_t *) u)->size, ((EGaequiSet_t *) u)->mem);
	EGmemPoolSFree (((EGaequiSet_t *) u),
									EGaequiSet_t, 1, ((EGaequiSet_t *) u)->mem);
	return;
}

/* ========================================================================= */

/* end eg_equiset.c */
