program nbody2
use def
use ic
use energy
use output
implicit none
integer :: j
type(node), pointer :: t 
double precision :: Fx,Fy,tstart,tend

allocate(pdata(npts))
allocate(KE(nsteps/out_frq))
allocate(PE(nsteps/out_frq))

call cpu_time(tstart)

call initialize_random_seed(rseed)
call initialize_particles_colliding(pdata,npts)

do time=1,nsteps
   nullify(t)
   call compute_bounding_box(pdata,left,right,bottom,top)
   call create_tree(t,pdata)
   call compute_mass_distribution(t)
   do j=1,npts
      Fx = 0. ; Fy = 0.
      call compute_forces(t,pdata(j),Fx,Fy)
      call time_step(pdata(j),Fx,Fy)
   enddo
   
   if (mod(time,out_frq)==0) then
      call create_output_file(time/out_frq)
      !call total_kinetic_energy(KE(time/out_frq),pdata)
      !call total_potential_energy(PE(time/out_frq),pdata)
   endif
enddo
call cpu_time(tend)
print *,tend-tstart




contains
   subroutine compute_bounding_box(pdata,left,right,bottom,top)
   type(particle), dimension(npts), intent(in) :: pdata
   double precision, intent(inout) :: left,right,bottom,top

   integer :: i
  
   left = pdata(1)%x(1)
   right = pdata(1)%x(1)
   bottom = pdata(1)%x(2)
   top = pdata(1)%x(2)
   do i=1,npts
      left = min(left,pdata(i)%x(1))
      right = max(right,pdata(i)%x(1))
      bottom = min(bottom,pdata(i)%x(2))
      top = max(top,pdata(i)%x(2))
   enddo
   end subroutine compute_bounding_box


   subroutine create_tree(tree,pdata)
   type(node), pointer :: tree
   type(particle), dimension(npts) :: pdata   
   integer :: i
   double precision :: l,r,b,t
 
   do i=1,npts
      l=left ; r=right ; b=bottom ; t=top
      call insert(tree,pdata(i),l,r,b,t)
   enddo
   end subroutine create_tree  
 

   recursive subroutine insert(tree,new_particle,l,r,b,t)
   type(node), pointer, intent(inout) :: tree
   type(particle), intent(in) :: new_particle
   double precision, intent(inout) :: l,r,b,t

   integer :: noc,quad
   type(particle) :: existing_particle

   noc=num_of_part_in_quad(tree)
   if (noc>=1) then
      if (noc==1) then
         existing_particle = tree % value
         quad = get_quad(existing_particle,l,r,b,t)
         if (quad==1) then
            allocate(tree%NW)
            tree%NW%value=existing_particle ; tree%NW%node_diameter=0.5*(r-l)
            nullify(tree%NW%NW) ; nullify(tree%NW%NE) ; nullify(tree%NW%SW) ; nullify(tree%NW%SE)
         elseif (quad==2) then
            allocate(tree%NE)
            tree%NE%value=existing_particle ; tree%NE%node_diameter=0.5*(r-l)
            nullify(tree%NE%NW) ; nullify(tree%NE%NE) ; nullify(tree%NE%SW) ; nullify(tree%NE%SE)
         elseif (quad==3) then
            allocate(tree%SW)
            tree%SW%value=existing_particle ; tree%SW%node_diameter=0.5*(r-l)
            nullify(tree%SW%NW) ; nullify(tree%SW%NE) ; nullify(tree%SW%SW) ; nullify(tree%SW%SE)
         elseif (quad==4) then
            allocate(tree%SE)
            tree%SE%value=existing_particle ; tree%SE%node_diameter=0.5*(r-l)
            nullify(tree%SE%NW) ; nullify(tree%SE%NE) ; nullify(tree%SE%SW) ; nullify(tree%SE%SE)
         endif
      endif
      
      quad = get_quad(new_particle,l,r,b,t)
      if (quad==1.and.(.not.associated(tree%NW))) then
         allocate(tree%NW)
         tree%NW%value=new_particle ; tree%NW%node_diameter=0.5*(r-l)
         nullify(tree%NW%NW) ; nullify(tree%NW%NE) ; nullify(tree%NW%SW) ; nullify(tree%NW%SE)
      elseif (quad==2.and.(.not.associated(tree%NE))) then
         allocate(tree%NE)
         tree%NE%value=new_particle ; tree%NE%node_diameter=0.5*(r-l)
         nullify(tree%NE%NW) ; nullify(tree%NE%NE) ; nullify(tree%NE%SW) ; nullify(tree%NE%SE)
      elseif (quad==3.and.(.not.associated(tree%SW))) then
         allocate(tree%SW)
         tree%SW%value=new_particle ; tree%SW%node_diameter=0.5*(r-l)
         nullify(tree%SW%NW) ; nullify(tree%SW%NE) ; nullify(tree%SW%SW) ; nullify(tree%SW%SE)
      elseif (quad==4.and.(.not.associated(tree%SE))) then
         allocate(tree%SE)
         tree%SE%value=new_particle ; tree%SE%node_diameter=0.5*(r-l)
         nullify(tree%SE%NW) ; nullify(tree%SE%NE) ; nullify(tree%SE%SW) ; nullify(tree%SE%SE)
      else
         if (quad==1.and.associated(tree%NW)) then
            r = (l+r)/2. ; b = (t+b)/2.
            call insert(tree%NW,new_particle,l,r,b,t)
         elseif (quad==2.and.associated(tree%NE)) then
            l = (l+r)/2. ; b = (t+b)/2.
            call insert(tree%NE,new_particle,l,r,b,t)
         elseif (quad==3.and.associated(tree%SW)) then
            r = (l+r)/2. ; t = (t+b)/2.
            call insert(tree%SW,new_particle,l,r,b,t)
         elseif (quad==4.and.associated(tree%SE)) then
            l = (l+r)/2. ; t = (t+b)/2.
            call insert(tree%SE,new_particle,l,r,b,t)
         endif
      endif
   elseif (noc==0) then
      allocate(tree)
      tree % value=new_particle ; tree % node_diameter=(r-l)
      nullify(tree%NW) ; nullify(tree%NE) ; nullify(tree%SW) ; nullify(tree%SE)
   endif
   end subroutine insert


   recursive subroutine compute_mass_distribution(tree)
   type(node), pointer, intent(inout) :: tree
   double precision :: node_mass
   double precision, dimension(2) :: com
   integer :: noc
   
   noc = num_of_part_in_quad(tree)
   if (noc==1) then
      tree%node_mass = tree%value%mass
      tree%centre_of_mass(1) = tree%value%x(1)
      tree%centre_of_mass(2) = tree%value%x(2)
   else
      node_mass = 0. ; com(1) = 0. ; com(2) = 0.
      call compute_node_mass(tree,node_mass,com)
      tree%node_mass = node_mass 
      tree%centre_of_mass(1) = com(1)/node_mass
      tree%centre_of_mass(2) = com(2)/node_mass
   
      if (associated(tree%NW)) then
         call compute_mass_distribution(tree%NW)
      endif
      if (associated(tree%NE)) then 
         call compute_mass_distribution(tree%NE)
      endif
      if (associated(tree%SW)) then
         call compute_mass_distribution(tree%SW)
      endif
      if (associated(tree%SE)) then
         call compute_mass_distribution(tree%SE) 
      endif  
   endif
   end subroutine compute_mass_distribution


   recursive subroutine compute_node_mass(tree,node_mass,com)
   type(node), pointer, intent(in) :: tree
   double precision, intent(inout) :: node_mass 
   double precision, dimension(2), intent(inout) :: com
   integer :: noc

   
   if (associated(tree%NW)) then
      call compute_node_mass(tree%NW,node_mass,com)
   endif
   if (associated(tree%NE)) then
      call compute_node_mass(tree%NE,node_mass,com)
   endif
   if (associated(tree%SW)) then
      call compute_node_mass(tree%SW,node_mass,com)
   endif
   if (associated(tree%SE)) then
      call compute_node_mass(tree%SE,node_mass,com)
   endif
   if ((.not.associated(tree%NW)).and.(.not.associated(tree%NE)).and.(.not.associated(tree%SW)).and.(.not.associated(tree%SE))) then
      node_mass=node_mass+tree%value%mass
      com(1)=com(1)+(tree%value%mass)*(tree%value%x(1))
      com(2)=com(2)+(tree%value%mass)*(tree%value%x(2))
   endif
   end subroutine compute_node_mass 


   recursive subroutine compute_forces(tree,current_particle,fx,fy)
   type(node), pointer, intent(in) :: tree
   type(particle), intent(in) :: current_particle
   double precision, intent(inout) :: fx,fy

   double precision :: current_theta,px,py,qx,qy,r
 
   current_theta = get_theta(current_particle%x,tree%centre_of_mass,tree%node_diameter)

   if (current_theta<=theta) then
      px = current_particle%x(1) ; py = current_particle%x(2)
      qx = tree%centre_of_mass(1) ; qy = tree%centre_of_mass(2)
      r = sqrt((qx-px)**2 + (qy-py)**2 + eps**2)
      fx = fx + G*(tree%node_mass)*(current_particle%mass)*(qx-px)/r**3
      fy = fy + G*(tree%node_mass)*(current_particle%mass)*(qy-py)/r**3
   else
      if (associated(tree%NW)) then
         call compute_forces(tree%NW,current_particle,fx,fy)
      endif
      if (associated(tree%NE)) then
         call compute_forces(tree%NE,current_particle,fx,fy)
      endif
      if (associated(tree%SW)) then
         call compute_forces(tree%SW,current_particle,fx,fy)
      endif
      if (associated(tree%SE)) then
         call compute_forces(tree%SE,current_particle,fx,fy)   
      endif
      if ((.not.associated(tree%NW)).and.(.not.associated(tree%NE))) then
         if ((.not.associated(tree%SW)).and.(.not.associated(tree%SE))) then 
            px = current_particle%x(1) ; py = current_particle%x(2)
            qx = tree%centre_of_mass(1) ; qy = tree%centre_of_mass(2)
            r = sqrt((qx-px)**2 + (qy-py)**2 + eps**2)
            fx = fx + G*(tree%node_mass)*(current_particle%mass)*(qx-px)/r**3
            fy = fy + G*(tree%node_mass)*(current_particle%mass)*(qy-py)/r**3
         endif
      endif
   endif
   end subroutine compute_forces

   
   subroutine time_step(current_particle,fx,fy)
   type(particle), intent(inout) :: current_particle
   double precision, intent(in) :: fx,fy
 
   double precision :: x0,y0,x1,y1,m
   double precision :: vx0,vy0,vx1,vy1

   x0 = current_particle%x(1) ; y0 = current_particle%x(2) ; m = current_particle%mass
   vx0 = current_particle%vel(1) ; vy0 = current_particle%vel(2)
   if (time==1) then
      vx1 = vx0 + 2.*dt*(fx/m)      !
      vy1 = vy0 + 2.*dt*(fy/m)      !
                                    ! Euler integration
      x1 = x0 + vx1*dt              !
      y1 = y0 + vy1*dt              !
   else
      vx1 = vx0 + dt*(fx/m)         !
      vy1 = vy0 + dt*(fy/m)         !
                                    ! Verlet Leapfrog integration
      x1 = x0 + vx1*dt              !
      y1 = y0 + vy1*dt              !
   endif
   current_particle%x(1) = x1 ; current_particle%x(2) = y1
   current_particle%vel(1) = vx1 ; current_particle%vel(2) = vy1
   end subroutine time_step


   function num_of_part_in_quad(t)
   type(node), pointer :: t
   integer :: num_of_part_in_quad 

   num_of_part_in_quad=0
   if (.not.associated(t)) then
      return
   elseif (associated(t%NW).or.associated(t%NE).or.associated(t%SW).or.associated(t%SE)) then
      num_of_part_in_quad=2
   else
      num_of_part_in_quad=1
   endif
   end function

   function get_quad(p,l,r,b,t)
   type(particle) :: p
   integer :: get_quad
   double precision :: l,r,b,t
   double precision :: x1,x2

   x1 = p % x(1) ; x2 = p % x(2)
   if (x1>=l.and.x1<=0.5*(r+l)) then
      if (x2>=b.and.x2<=0.5*(t+b)) then
         get_quad=3
      elseif (x2>0.5*(t+b).and.x2<=t) then
         get_quad=1
      endif
   elseif (x1>0.5*(r+l).and.x1<=r) then
      if (x2>=b.and.x2<=0.5*(t+b)) then
         get_quad=4
      elseif (x2>0.5*(t+b).and.x2<=t) then
         get_quad=2
      endif
   endif
   end function get_quad

   function get_theta(a,b,d)
   double precision, dimension(2) :: a,b
   double precision :: get_theta,d,r
   r = sqrt((b(1)-a(1))**2+(b(2)-a(2))**2)
   if (r<0.000000001) then
      get_theta=0.
   else
      get_theta = d/r
   endif
   end function get_theta

   subroutine display_particles(pdata,n)
   integer :: i,n
   type(particle), intent(inout), dimension(n) :: pdata
   print *, 'displaying particles: '
   print *,'x','                   ','y','                       ','vx','                         ','vy','   ','mass'
   do i=1,n
      print *, pdata(i)%x(1), '   ', pdata(i)%x(2),'   ',pdata(i)%vel(1),'   ',pdata(i)%vel(2),'   ',pdata(i)%mass
   enddo
   print *,''
   end subroutine display_particles
 
end program nbody2
